% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hp_close.R
\name{hp_close}
\alias{hp_close}
\title{HP Model: Fitting the advanced ages of the life tables.}
\usage{
hp_close (fit, method = c("hp", "plateau", "linear", "gompertz"),
 x0 = max(fit$data$x), max_age = 120, k = 7,
 weights = seq(from = 0, to = 1, length.out = 2*k+1),
 new_Ex = NULL, new_Dx = NULL)
}
\arguments{
\item{fit}{Object of the class \code{HP} fitted by the hp() function}

\item{method}{Character string specifying the closing method to be fitted, with them being: 'hp', 'plateau', 'linear' or 'gompertz'.}

\item{x0}{Integer with the starting age the closing method will be fitted from. Default is the last age fitted by the 'HP' object.}

\item{max_age}{Integer with the maximum age the closing method will be fitted. Default age is '120'.}

\item{k}{Integer representing the size of the age interval to be mixed with the 'linear' or 'gompertz' closing methods for smooth graduation. If k = 0, no mixing will be applied.}

\item{weights}{Vector of weights to be applied in the mixing of the life tables. Vector's size should be equal to 2*k+1.}

\item{new_Ex}{Exposure in the period after the x0 input. This argument is necessary for HP objects following the binomial and poisson distributions, as well as the 'linear' and 'gompertz' closing methods (optional).}

\item{new_Dx}{Vector containing the death counts in the period after the x0 input. This argument is necessary in the 'linear' and 'gompertz' closing methods, being equal in length with the new_Ex argument.}
}
\value{
Returns a \code{ClosedHP} class object with the predictive chains of the death probability
(qx) from first fitted age to max_age argument, the data utilized by the function and the
closing method chosen.
}
\description{
This function receives an object of the class \code{HP} fitted by the hp() function
and fits a closing method to expand the life tables dataset to a maximum age argument inputted
by the user.
There are four closing methods available: 'hp', 'plateau', 'linear', and 'gompertz'.
The 'linear' method can only be applied with HP objects following the lognormal variant of
the HP mortality law.
}
\details{
The four closing methods for life tables are:
1.'hp' method: Expands the previously adjusted HP model until the max_age argument.
2.'plateau' method: Keeps the death probability (qx) constant after the x0 argument.
3.'linear' method: Fits a linear regression starting at age x0 - k until the last age with data available (lognormal only).
4.'gompertz' method: Adopted as the closing method of the 2010-2012 English Life Table No. 17, fits the Gompertz mortality law via SIR using the same available data as the 'linear' method.
}
\examples{
## Importing mortality data from the USA available on the Human Mortality Database (HMD):
data(USA)

## Selecting the exposure and the death count of the year 2010, ranging from 0 to 90 years old:
USA2010 = USA[USA$Year == 2010,]
x = 0:90
Ex = USA2010$Ex.Male[x+1]
Dx = USA2010$Dx.Male[x+1]

## Fitting a lognormal HP model:
fit = hp(x = x, Ex = Ex, Dx = Dx, model = "lognormal",
         M = 1000, bn = 0, thin = 10)

## Applying the closing function with different methods:
close1 = hp_close(fit, method = "hp", x0 = 90)
\donttest{close2 = hp_close(fit, method = "plateau", x0 = 90)
close3 = hp_close(fit, method = "linear", x0 = 80,
                  new_Ex = USA2010$Ex.Male[82:101],
                  new_Dx = USA2010$Dx.Male[82:101])
close4 = hp_close(fit, method = "gompertz", x0 = 70,
                  new_Ex = USA2010$Ex.Male[72:101],
                  new_Dx = USA2010$Dx.Male[72:101],
                  k = 5, max_age = 120)

#### Using the other functions available in the package with the 'ClosedHP' object:

## credible intervals (See "?qx_ci" for more options):
qx_ci(close1)

## qx estimation (See "?fitted.HP" in the BayesMortalityPlus package for more options):
fitted(close2)

## life expectancy (See "?expectancy.HP" for more options)
expectancy(close3, age = 0:110)

## plotting (See "?plot.HP" in the BayesMortalityPlus package for more options):
plot(close4)
g <- plot(list(close4, fit),
          colors = c("seagreen", "blue"),
          labels = c("Closed", "Model"))
# plotly::ggplotly(g)
}

}
\references{
Dodd, Erengul, Forster, Jonathan, Bijak, Jakub, & Smith, Peter 2018. “Smoothing mortality data: the English life table, 2010-12.” \emph{Journal of the Royal Statistical Society: Series A (Statistics in Society)}, 181(3), 717-735.
}
\seealso{
\code{\link[=fitted.HP]{fitted.HP()}}, \code{\link[=plot.HP]{plot.HP()}}, \code{\link[=print.HP]{print.HP()}} and \code{\link[=summary.HP]{summary.HP()}} for \code{ClosedHP} methods to native R functions \code{\link[=fitted]{fitted()}},
\code{\link[=plot]{plot()}}, \code{\link[=print]{print()}} and \code{\link[=summary]{summary()}}.

\code{\link[=expectancy.HP]{expectancy.HP()}} and \code{\link[=Heatmap.HP]{Heatmap.HP()}} for \code{ClosedHP} methods to compute and visualise the truncated life expectancy
via \code{\link[=expectancy]{expectancy()}} and \code{\link[=Heatmap]{Heatmap()}} functions.

\code{\link[=qx_ci]{qx_ci()}} to compute credible intervals.
}
