\name{impdiagnosticconversion}
\alias{impdiagnosticconversion}

\title{Conversion from BBPMM output to mice's mids object or prepares imputed data for coda's mcmc or mcmc.list objects}

\description{\sQuote{impdiagnosticconversion} prepares the output of \code{\link{BBPMM}} for \pkg{coda}'s (Plummer et al. 2006) \code{\link[coda]{mcmc}} or \code{\link[coda]{mcmc.list}} object. It is also possible to convert the output into an object of the type \code{\link[mice]{mids}} provided by the package \pkg{mice} (van Buuren, Groothuis-Oudshoorn 2011).}

\usage{impdiagnosticconversion(imputed.data, type=c("mcmc.list","mcmc","mids"))}

\arguments{
  \item{imputed.data}{An object of the type \sQuote{imp} returned by \pkg{BaBooN}'s \code{BBPMM} imputation method.}
  \item{type}{A string choosing the type of output: Either \kbd{"mcmc.list"}, \kbd{"mcmc"} or \kbd{"mids"}. \var{Default}=\kbd{"mcmc.list"}.}
}

\details{To provide a wider variety of analysis and diagnostic tools and facilitate the collaboration between different imputation packages, we started to create a conversion tool.  
At the moment it is possible to transform an output created by \pkg{BaBooN}'s \code{BBPMM} function into a mids-like object returned from \code{\link[mice]{mice}} (van Buuren, Groothuis-Oudshoorn 2011).  
For diagnostics with the help of the functions provided by the package \pkg{coda} (Plummer et al. 2006), this function prepares the output. It returns a list, that isn't directly of the desired class, but results in a list with four elements, containing the chains for the means, variances, medians and standard deviations of the imputed values transformed to the specified type respectively. Hence, accessing the first element of the list returns actual the converted means, the second element contains the converted variances and so on.  

We hope it is somehow an impulse for future standardisation or interchangeability of outputs generated by different multiple imputation routines and analysing the imputed values.}

\value{
  See for details \code{\link[coda]{mcmc}}, \code{\link[coda]{mcmc.list}} or \code{\link[mice]{mids}}.
  Remark: Converting to \code{\link[coda]{mcmc}} or \code{\link[coda]{mcmc.list}} results in a list with four elements (medians, vars, medians, sds), containing the chains for the means, variances, medians and standard deviations of the imputed values transformed to the specified type. See the examples.
}

\author{Thorsten Schnapp, Florian Meinfelder [ctb]}

\seealso{\code{\link[coda]{mcmc}}, \code{\link[coda]{mcmc.list}}, \code{\link[mice]{mids}}, \code{\link[mice]{mids-class}}, \code{\link[mice]{mice}}, \code{\link[mice]{plot.mids}}, \code{\link{BBPMM}}}

\references{

  Cowles, M.K. and Carlin, B.P. (1996) Markov Chain Monte Carlo Convergence Diagnostics: A Comparative Review.
  \emph{Journal of the American Statistical Association}, Vol. \bold{91}, pp. 883--904.
  
  Koller-Meinfelder, F. (2009) \emph{Analysis of Incomplete Survey Data
    -- Multiple Imputation Via Bayesian Bootstrap Predictive Mean
    Matching}, doctoral thesis. 
    
  Plummer, M. and Best, N. and Cowles, K. and Vines, K. (2006) CODA: Convergence 
    Diagnosis and Output Analysis for MCMC, \emph{R News}, Vol. \bold{6}, pp. 7--11
    
  Van Buuren, S. and Groothuis-Oudshoorn, K. (2011) mice: Multivariate Imputation by Chained Equations in R. \emph{Journal of
    Statistical Software}, Vol. \bold{45}, No. 3, pp. 1--67. URL http://www.jstatsoft.org/v45/i03/.
  
  Van Buuren, S. (2012) \emph{Flexible imputation of missing data}. Boca Raton: CRC Press.

  Sarkar, Deepayan (2008) \emph{Lattice: Multivariate Data Visualization with R}. New York: Springer.

}

\keyword{conversion}

\examples{
\dontrun{
### sample data set with non-normal variables
set.seed(1000)
n <- 50
x1 <- round(runif(n,0.5,3.5))
x2 <- as.factor(c(rep(1,10),rep(2,25),rep(3,15)))
x3 <- round(rnorm(n,0,3))
y1 <- round(x1-0.25*(x2==2)+0.5*x3+rnorm(n,0,1))
y1 <- ifelse(y1<1,1,y1)
y1 <- as.factor(ifelse(y1>4,5,y1))
y2 <- x1+rnorm(n,0,0.5)
y3 <- round(x3+rnorm(n,0,2))
data1 <- as.data.frame(cbind(x1,x2,x3,y1,y2,y3))
misrow1 <- sample(n,20)
misrow2 <- sample(n,15)
misrow3 <- sample(n,10)
is.na(data1[misrow1, 4]) <- TRUE
is.na(data1[misrow2, 5]) <- TRUE
is.na(data1[misrow2, 6]) <- TRUE

### imputation
imputed.data <- BBPMM(data1, nIter=3, M=3)

### Test Conversion
if(!require(coda)) install.packages("coda")
if(!require(mice)) install.packages("mice")

require(coda) ## see references
require(mice) ## see references
require(lattice) ## see references

## conversion to mcmc
imp.to.mcmc <- impdiagnosticconversion(imputed.data,
                                       type="mcmc")

## conversion to mcmc.list
imp.to.mcmc.list <- impdiagnosticconversion(imputed.data,
                                            type="mcmc.list")

## conversion to mids
imp.to.mids <- impdiagnosticconversion(imputed.data,
                                       type="mids")

### Test

## mcmc:
plot(imp.to.mcmc$means[[1]])
acfplot(imp.to.mcmc$vars[[1]])
plot(imp.to.mcmc$medians[[1]])
acfplot(imp.to.mcmc$sds[[1]])

## mcmc.list:
xyplot(imp.to.mcmc.list[[1]]) ## Mean
qqmath(imp.to.mcmc.list[[2]]) ## Variance
xyplot(imp.to.mcmc.list[[3]]) ## Median
qqmath(imp.to.mcmc.list[[4]]) ## Std.dev.

## mids:
# Chain-plot from mice
mice:::plot.mids(imp.to.mids)

}
}
