# ----------------------------------------------------------------------------
#
#  Copyright (C) 2010-2022 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


from zita_jacktools.jackclient import JackClient
from zita_jacktools import jackpeaklim_ext


class JackPeaklim(JackClient):

    """
    Multichannel look-ahead peak limiter, equivalent to zita-dpl1. 

    Channels are coupled and will have the same gain at all times.
    This is the type of limiter used to avoid downstream clipping.
    It can be safely used with a short release time. If the limiting
    action is due to low frequency content, a longer release is used
    automatically to avoid following the waveform rather than its
    envelope.
    """

    def __init__(self, nchan, client_name, server_name = None):
        """
        Create a new JackPeaklim instance with 'nchan' channels.

        The result should be checked using get_state().
        The optional 'server_name' allows to select between running
        Jack servers.
        """
        assert (nchan > 0)
        self._jpeaklim, base = jackpeaklim_ext.makecaps (self, client_name, server_name, nchan)
        super().__init__(base)


    def set_inpgain(self, inpgain) :
        """
        Set input gain in dB.

        The range is -30 to +30 dB, the intial value is 0 dB
        """
        return jackpeaklim_ext.set_inpgain (self._jpeaklim, inpgain)


    def set_threshold(self, threshold) :
        """
        Set threshold in dB.

        To accomodate inter-sample peaks on typical program
        material, set this 0.3 dB below the required maximum
        output sample level. The initial value is 0 dB.
        """
        return jackpeaklim_ext.set_threshold (self._jpeaklim, threshold)


    def set_release(self, release) :
        """
        Set release time in seconds.

        The range is 1 millisecond to 1 second. The initial
        release time is 10 ms.
        """
        return jackpeaklim_ext.set_release (self._jpeaklim, release)


