CREATE SCHEMA gitlab_partitions_dynamic;

COMMENT ON SCHEMA gitlab_partitions_dynamic IS 'Schema to hold partitions managed dynamically from the application, e.g. for time space partitioning.';

CREATE SCHEMA gitlab_partitions_static;

COMMENT ON SCHEMA gitlab_partitions_static IS 'Schema to hold static partitions, e.g. for hash partitioning';

CREATE EXTENSION IF NOT EXISTS btree_gist;

CREATE EXTENSION IF NOT EXISTS pg_trgm;

CREATE FUNCTION assign_ci_runner_machines_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_runner_machines_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_ci_runner_taggings_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_runner_taggings_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_ci_runners_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_runners_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_build_tags_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('p_ci_build_tags_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_builds_execution_configs_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('p_ci_builds_execution_configs_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_builds_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_builds_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_job_annotations_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('p_ci_job_annotations_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_job_artifacts_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_job_artifacts_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_pipeline_variables_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_pipeline_variables_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_pipelines_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_pipelines_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_ci_stages_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('ci_stages_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_p_duo_workflows_checkpoints_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('p_duo_workflows_checkpoints_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION assign_zoekt_tasks_id_value() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."id" IS NOT NULL THEN
  RAISE WARNING 'Manually assigning ids is not allowed, the value will be ignored';
END IF;
NEW."id" := nextval('zoekt_tasks_id_seq'::regclass);
RETURN NEW;

END
$$;

CREATE FUNCTION bulk_import_batch_trackers_sharding_key() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF num_nonnulls(NEW.organization_id, NEW.namespace_id, NEW.project_id) != 1 THEN
    SELECT "organization_id", "namespace_id", "project_id"
    INTO NEW."organization_id", NEW."namespace_id", NEW."project_id"
    FROM "bulk_import_trackers"
    WHERE "bulk_import_trackers"."id" = NEW."tracker_id";
  END IF;

  RETURN NEW;
END
$$;

CREATE FUNCTION bulk_import_trackers_sharding_key() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF num_nonnulls(NEW.namespace_id, NEW.organization_id, NEW.project_id) != 1 THEN
    SELECT "organization_id", "namespace_id", "project_id"
    INTO NEW."organization_id", NEW."namespace_id", NEW."project_id"
    FROM "bulk_import_entities"
    WHERE "bulk_import_entities"."id" = NEW."bulk_import_entity_id";
  END IF;

  RETURN NEW;
END
$$;

CREATE FUNCTION cleanup_pipeline_iid_after_delete() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF OLD.iid IS NOT NULL THEN
    DELETE FROM p_ci_pipeline_iids
    WHERE project_id = OLD.project_id AND iid = OLD.iid;
  END IF;
  RETURN OLD;
END;
$$;

CREATE FUNCTION custom_dashboard_search_vector_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  INSERT INTO custom_dashboard_search_data (
    custom_dashboard_id,
    organization_id,
    search_vector,
    created_at,
    updated_at
  )
  VALUES (
    NEW.id,
    NEW.organization_id,
    to_tsvector('english', NEW.name || ' ' || NEW.description),
    CURRENT_TIMESTAMP,
    CURRENT_TIMESTAMP
  )
  ON CONFLICT (custom_dashboard_id) DO UPDATE
  SET search_vector = EXCLUDED.search_vector,
      updated_at = CURRENT_TIMESTAMP;

  RETURN NEW;
END
$$;

CREATE FUNCTION delete_associated_project_namespace() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
DELETE FROM namespaces
WHERE namespaces.id = OLD.project_namespace_id AND
namespaces.type = 'Project';
RETURN NULL;

END
$$;

CREATE FUNCTION delete_orphaned_granular_scopes() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
DELETE FROM granular_scopes
WHERE id = OLD.granular_scope_id
AND NOT EXISTS (
  SELECT 1
  FROM personal_access_token_granular_scopes
  WHERE granular_scope_id = OLD.granular_scope_id
);
RETURN OLD;

END
$$;

CREATE FUNCTION ensure_note_diff_files_sharding_key() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  note_project_id BIGINT;
  note_namespace_id BIGINT;
BEGIN
  SELECT "project_id", "namespace_id"
  INTO note_project_id, note_namespace_id
  FROM "notes"
  WHERE "id" = NEW."diff_note_id";

  IF note_project_id IS NOT NULL THEN
    SELECT "project_namespace_id" FROM "projects"
    INTO NEW."namespace_id" WHERE "projects"."id" = note_project_id;
  ELSE
    NEW."namespace_id" := note_namespace_id;
  END IF;

  RETURN NEW;
END
$$;

CREATE FUNCTION ensure_pipeline_iid_uniqueness_before_insert() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF NEW.iid IS NOT NULL THEN
    BEGIN
      INSERT INTO p_ci_pipeline_iids (project_id, iid)
      VALUES (NEW.project_id, NEW.iid);
    EXCEPTION WHEN unique_violation THEN
      RAISE EXCEPTION 'Pipeline with iid % already exists for project %',
        NEW.iid, NEW.project_id
        USING ERRCODE = 'unique_violation',
              DETAIL = 'The iid must be unique within a project',
              HINT = 'Use a different iid or let the system generate one';
    END;
  END IF;

  RETURN NEW;
END;
$$;

CREATE FUNCTION ensure_pipeline_iid_uniqueness_before_update_iid() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF NEW.iid IS DISTINCT FROM OLD.iid THEN
    IF NEW.iid IS NOT NULL THEN
      BEGIN
        INSERT INTO p_ci_pipeline_iids (project_id, iid)
        VALUES (NEW.project_id, NEW.iid);
      EXCEPTION WHEN unique_violation THEN
        RAISE EXCEPTION 'Pipeline with iid % already exists for project %',
          NEW.iid, NEW.project_id
          USING ERRCODE = 'unique_violation',
                DETAIL = 'The iid must be unique within a project',
                HINT = 'Use a different iid or let the system generate one';
      END;
    END IF;

    IF OLD.iid IS NOT NULL THEN
      DELETE FROM p_ci_pipeline_iids
      WHERE project_id = OLD.project_id AND iid = OLD.iid;
    END IF;
  END IF;
  RETURN NEW;
END;
$$;

CREATE TABLE namespaces (
    id bigint NOT NULL,
    name character varying NOT NULL,
    path character varying NOT NULL,
    owner_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    type character varying DEFAULT 'User'::character varying NOT NULL,
    description character varying DEFAULT ''::character varying NOT NULL,
    avatar character varying,
    membership_lock boolean DEFAULT false,
    share_with_group_lock boolean DEFAULT false,
    visibility_level integer DEFAULT 20 NOT NULL,
    request_access_enabled boolean DEFAULT true NOT NULL,
    ldap_sync_status character varying DEFAULT 'ready'::character varying NOT NULL,
    ldap_sync_error character varying,
    ldap_sync_last_update_at timestamp without time zone,
    ldap_sync_last_successful_update_at timestamp without time zone,
    ldap_sync_last_sync_at timestamp without time zone,
    description_html text,
    lfs_enabled boolean,
    parent_id bigint,
    shared_runners_minutes_limit integer,
    repository_size_limit bigint,
    require_two_factor_authentication boolean DEFAULT false NOT NULL,
    two_factor_grace_period integer DEFAULT 48 NOT NULL,
    cached_markdown_version integer,
    project_creation_level integer,
    runners_token character varying,
    file_template_project_id bigint,
    saml_discovery_token character varying,
    runners_token_encrypted character varying,
    custom_project_templates_group_id bigint,
    auto_devops_enabled boolean,
    extra_shared_runners_minutes_limit integer,
    last_ci_minutes_notification_at timestamp with time zone,
    last_ci_minutes_usage_notification_level integer,
    subgroup_creation_level integer DEFAULT 1,
    max_pages_size integer,
    max_artifacts_size integer,
    mentions_disabled boolean,
    default_branch_protection smallint,
    max_personal_access_token_lifetime integer,
    push_rule_id bigint,
    shared_runners_enabled boolean DEFAULT true NOT NULL,
    allow_descendants_override_disabled_shared_runners boolean DEFAULT false NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    organization_id bigint,
    state smallint,
    CONSTRAINT check_2eae3bdf93 CHECK ((organization_id IS NOT NULL))
);

CREATE FUNCTION find_namespaces_by_id(namespaces_id bigint) RETURNS namespaces
    LANGUAGE plpgsql STABLE COST 1 PARALLEL SAFE
    AS $$
BEGIN
  return (SELECT namespaces FROM namespaces WHERE id = namespaces_id LIMIT 1);
END;
$$;

CREATE TABLE projects (
    id bigint NOT NULL,
    name character varying,
    path character varying,
    description text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    creator_id bigint,
    namespace_id bigint NOT NULL,
    last_activity_at timestamp without time zone,
    import_url character varying,
    visibility_level integer DEFAULT 0 NOT NULL,
    archived boolean DEFAULT false NOT NULL,
    avatar character varying,
    merge_requests_template text,
    star_count integer DEFAULT 0 NOT NULL,
    merge_requests_rebase_enabled boolean DEFAULT false,
    import_type character varying,
    import_source character varying,
    approvals_before_merge integer DEFAULT 0 NOT NULL,
    reset_approvals_on_push boolean DEFAULT true,
    merge_requests_ff_only_enabled boolean DEFAULT false,
    issues_template text,
    mirror boolean DEFAULT false NOT NULL,
    mirror_last_update_at timestamp without time zone,
    mirror_last_successful_update_at timestamp without time zone,
    mirror_user_id bigint,
    shared_runners_enabled boolean DEFAULT true NOT NULL,
    runners_token character varying,
    build_allow_git_fetch boolean DEFAULT true NOT NULL,
    build_timeout integer DEFAULT 3600 NOT NULL,
    mirror_trigger_builds boolean DEFAULT false NOT NULL,
    pending_delete boolean DEFAULT false,
    public_builds boolean DEFAULT true NOT NULL,
    last_repository_check_failed boolean,
    last_repository_check_at timestamp without time zone,
    only_allow_merge_if_pipeline_succeeds boolean DEFAULT false NOT NULL,
    has_external_issue_tracker boolean,
    repository_storage character varying DEFAULT 'default'::character varying NOT NULL,
    repository_read_only boolean,
    request_access_enabled boolean DEFAULT true NOT NULL,
    has_external_wiki boolean,
    ci_config_path character varying,
    lfs_enabled boolean,
    description_html text,
    only_allow_merge_if_all_discussions_are_resolved boolean,
    repository_size_limit bigint,
    printing_merge_request_link_enabled boolean DEFAULT true NOT NULL,
    auto_cancel_pending_pipelines integer DEFAULT 1 NOT NULL,
    service_desk_enabled boolean DEFAULT true,
    cached_markdown_version integer,
    delete_error text,
    last_repository_updated_at timestamp without time zone,
    disable_overriding_approvers_per_merge_request boolean,
    storage_version smallint,
    resolve_outdated_diff_discussions boolean,
    remote_mirror_available_overridden boolean,
    only_mirror_protected_branches boolean,
    pull_mirror_available_overridden boolean,
    jobs_cache_index integer,
    external_authorization_classification_label character varying,
    mirror_overwrites_diverged_branches boolean,
    pages_https_only boolean DEFAULT true,
    external_webhook_token character varying,
    packages_enabled boolean,
    merge_requests_author_approval boolean DEFAULT false,
    pool_repository_id bigint,
    runners_token_encrypted character varying,
    bfg_object_map character varying,
    detected_repository_languages boolean,
    merge_requests_disable_committers_approval boolean,
    require_password_to_approve boolean,
    emails_disabled boolean,
    max_pages_size integer,
    max_artifacts_size integer,
    pull_mirror_branch_prefix character varying(50),
    remove_source_branch_after_merge boolean,
    marked_for_deletion_at date,
    marked_for_deletion_by_user_id bigint,
    autoclose_referenced_issues boolean,
    suggestion_commit_message character varying(255),
    project_namespace_id bigint,
    hidden boolean DEFAULT false NOT NULL,
    organization_id bigint,
    CONSTRAINT check_1a6f946a8a CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_fa75869cb1 CHECK ((project_namespace_id IS NOT NULL))
);

CREATE FUNCTION find_projects_by_id(projects_id bigint) RETURNS projects
    LANGUAGE plpgsql STABLE COST 1 PARALLEL SAFE
    AS $$
BEGIN
  return (SELECT projects FROM projects WHERE id = projects_id LIMIT 1);
END;
$$;

CREATE TABLE users (
    id bigint NOT NULL,
    email character varying DEFAULT ''::character varying NOT NULL,
    encrypted_password character varying DEFAULT ''::character varying NOT NULL,
    reset_password_token character varying,
    reset_password_sent_at timestamp without time zone,
    remember_created_at timestamp without time zone,
    sign_in_count integer DEFAULT 0,
    current_sign_in_at timestamp without time zone,
    last_sign_in_at timestamp without time zone,
    current_sign_in_ip character varying,
    last_sign_in_ip character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    name character varying,
    admin boolean DEFAULT false NOT NULL,
    projects_limit integer NOT NULL,
    failed_attempts integer DEFAULT 0,
    locked_at timestamp without time zone,
    username character varying,
    can_create_group boolean DEFAULT true NOT NULL,
    can_create_team boolean DEFAULT true NOT NULL,
    state character varying,
    color_scheme_id bigint DEFAULT 1 NOT NULL,
    password_expires_at timestamp without time zone,
    created_by_id bigint,
    last_credential_check_at timestamp without time zone,
    avatar character varying,
    confirmation_token character varying,
    confirmed_at timestamp without time zone,
    confirmation_sent_at timestamp without time zone,
    unconfirmed_email character varying,
    hide_no_ssh_key boolean DEFAULT false,
    admin_email_unsubscribed_at timestamp without time zone,
    notification_email character varying,
    hide_no_password boolean DEFAULT false,
    password_automatically_set boolean DEFAULT false,
    encrypted_otp_secret character varying,
    encrypted_otp_secret_iv character varying,
    encrypted_otp_secret_salt character varying,
    otp_required_for_login boolean DEFAULT false NOT NULL,
    otp_backup_codes text,
    public_email character varying,
    dashboard integer DEFAULT 0,
    project_view integer DEFAULT 2,
    consumed_timestep integer,
    layout integer DEFAULT 0,
    hide_project_limit boolean DEFAULT false,
    note text,
    unlock_token character varying,
    otp_grace_period_started_at timestamp without time zone,
    external boolean DEFAULT false,
    incoming_email_token character varying,
    auditor boolean DEFAULT false NOT NULL,
    require_two_factor_authentication_from_group boolean DEFAULT false NOT NULL,
    two_factor_grace_period integer DEFAULT 48 NOT NULL,
    last_activity_on date,
    notified_of_own_activity boolean DEFAULT false,
    preferred_language character varying,
    theme_id smallint,
    accepted_term_id bigint,
    feed_token character varying,
    private_profile boolean DEFAULT false NOT NULL,
    roadmap_layout smallint,
    include_private_contributions boolean,
    commit_email character varying,
    group_view integer,
    managing_group_id bigint,
    first_name character varying(255),
    last_name character varying(255),
    static_object_token character varying(255),
    user_type smallint DEFAULT 0,
    static_object_token_encrypted text,
    otp_secret_expires_at timestamp with time zone,
    onboarding_in_progress boolean DEFAULT false NOT NULL,
    color_mode_id smallint DEFAULT 1 NOT NULL,
    composite_identity_enforced boolean DEFAULT false NOT NULL,
    organization_id bigint NOT NULL,
    CONSTRAINT check_061f6f1c91 CHECK ((project_view IS NOT NULL)),
    CONSTRAINT check_0dd5948e38 CHECK ((user_type IS NOT NULL)),
    CONSTRAINT check_3a60c18afc CHECK ((hide_no_password IS NOT NULL)),
    CONSTRAINT check_693c6f3aab CHECK ((hide_no_ssh_key IS NOT NULL)),
    CONSTRAINT check_7bde697e8e CHECK ((char_length(static_object_token_encrypted) <= 255)),
    CONSTRAINT check_c737c04b87 CHECK ((notified_of_own_activity IS NOT NULL))
);

CREATE FUNCTION find_users_by_id(users_id bigint) RETURNS users
    LANGUAGE plpgsql STABLE COST 1 PARALLEL SAFE
    AS $$
BEGIN
  return (SELECT users FROM users WHERE id = users_id LIMIT 1);
END;
$$;

CREATE FUNCTION function_for_trigger_03be0f8add7e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  NEW."all_unarchived_project_ids" := NEW."all_active_project_ids";
  RETURN NEW;
END
$$;

CREATE FUNCTION function_for_trigger_7d6a4f5b82c2() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  NEW."all_active_project_ids" := NEW."all_unarchived_project_ids";
  RETURN NEW;
END
$$;

CREATE FUNCTION function_for_trigger_de99bb993511() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF NEW."all_active_project_ids" IS NOT DISTINCT FROM '{}' AND NEW."all_unarchived_project_ids" IS DISTINCT FROM '{}' THEN
    NEW."all_active_project_ids" = NEW."all_unarchived_project_ids";
  END IF;

  IF NEW."all_unarchived_project_ids" IS NOT DISTINCT FROM '{}' AND NEW."all_active_project_ids" IS DISTINCT FROM '{}' THEN
    NEW."all_unarchived_project_ids" = NEW."all_active_project_ids";
  END IF;

  RETURN NEW;
END
$$;

CREATE FUNCTION gitlab_schema_prevent_write() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
    IF COALESCE(NULLIF(current_setting(CONCAT('lock_writes.', TG_TABLE_NAME), true), ''), 'true') THEN
      RAISE EXCEPTION 'Table: "%" is write protected within this Gitlab database.', TG_TABLE_NAME
        USING ERRCODE = 'modifying_sql_data_not_permitted',
        HINT = 'Make sure you are using the right database connection';
    END IF;
    RETURN NEW;
END
$$;

CREATE FUNCTION insert_catalog_resource_sync_event() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO p_catalog_resource_sync_events (catalog_resource_id, project_id)
SELECT id, OLD.id FROM catalog_resources
WHERE project_id = OLD.id;
RETURN NULL;

END
$$;

CREATE FUNCTION insert_into_loose_foreign_keys_deleted_records() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  INSERT INTO loose_foreign_keys_deleted_records
  (fully_qualified_table_name, primary_key_value)
  SELECT TG_TABLE_SCHEMA || '.' || TG_TABLE_NAME, old_table.id FROM old_table;

  RETURN NULL;
END
$$;

CREATE FUNCTION insert_into_loose_foreign_keys_deleted_records_override_table() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  INSERT INTO loose_foreign_keys_deleted_records
  (fully_qualified_table_name, primary_key_value)
  SELECT current_schema() || '.' || TG_ARGV[0], old_table.id FROM old_table;

  RETURN NULL;
END
$$;

CREATE FUNCTION insert_namespaces_sync_event() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO namespaces_sync_events (namespace_id)
VALUES(COALESCE(NEW.id, OLD.id));
RETURN NULL;

END
$$;

CREATE FUNCTION insert_or_update_vulnerability_reads() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  severity smallint;
  state smallint;
  report_type smallint;
  resolved_on_default_branch boolean;
  present_on_default_branch boolean;
  has_issues boolean;
  has_merge_request boolean;
BEGIN
  IF (SELECT current_setting('vulnerability_management.dont_execute_db_trigger', true) = 'true') THEN
    RETURN NULL;
  END IF;

  IF (NEW.vulnerability_id IS NULL AND (TG_OP = 'INSERT' OR TG_OP = 'UPDATE')) THEN
    RETURN NULL;
  END IF;

  IF (TG_OP = 'UPDATE' AND OLD.vulnerability_id IS NOT NULL AND NEW.vulnerability_id IS NOT NULL) THEN
    RETURN NULL;
  END IF;

  SELECT
    vulnerabilities.severity, vulnerabilities.state, vulnerabilities.report_type, vulnerabilities.resolved_on_default_branch, vulnerabilities.present_on_default_branch
  INTO
    severity, state, report_type, resolved_on_default_branch, present_on_default_branch
  FROM
    vulnerabilities
  WHERE
    vulnerabilities.id = NEW.vulnerability_id;

  IF present_on_default_branch IS NOT true THEN
    RETURN NULL;
  END IF;

  SELECT
    EXISTS (SELECT 1 FROM vulnerability_issue_links WHERE vulnerability_issue_links.vulnerability_id = NEW.vulnerability_id)
  INTO
    has_issues;

  SELECT
    EXISTS (SELECT 1 FROM vulnerability_merge_request_links WHERE vulnerability_merge_request_links.vulnerability_id = NEW.vulnerability_id)
  INTO
    has_merge_request;

  INSERT INTO vulnerability_reads (vulnerability_id, project_id, scanner_id, report_type, severity, state, resolved_on_default_branch, uuid, location_image, cluster_agent_id, casted_cluster_agent_id, has_issues, has_merge_request)
    VALUES (NEW.vulnerability_id, NEW.project_id, NEW.scanner_id, report_type, severity, state, resolved_on_default_branch, NEW.uuid::uuid, NEW.location->>'image', NEW.location->'kubernetes_resource'->>'agent_id', CAST(NEW.location->'kubernetes_resource'->>'agent_id' AS bigint), has_issues, has_merge_request)
    ON CONFLICT(vulnerability_id) DO NOTHING;
  RETURN NULL;
END
$$;

CREATE FUNCTION insert_projects_sync_event() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO projects_sync_events (project_id)
VALUES(COALESCE(NEW.id, OLD.id));
RETURN NULL;

END
$$;

CREATE FUNCTION insert_vulnerability_reads_from_vulnerability() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  scanner_id bigint;
  uuid uuid;
  location_image text;
  cluster_agent_id text;
  casted_cluster_agent_id bigint;
  has_issues boolean;
  has_merge_request boolean;
BEGIN
  IF (SELECT current_setting('vulnerability_management.dont_execute_db_trigger', true) = 'true') THEN
    RETURN NULL;
  END IF;

  SELECT
    v_o.scanner_id, v_o.uuid, v_o.location->>'image', v_o.location->'kubernetes_resource'->>'agent_id', CAST(v_o.location->'kubernetes_resource'->>'agent_id' AS bigint)
  INTO
    scanner_id, uuid, location_image, cluster_agent_id, casted_cluster_agent_id
  FROM
    vulnerability_occurrences v_o
  WHERE
    v_o.vulnerability_id = NEW.id
  LIMIT 1;

  SELECT
    EXISTS (SELECT 1 FROM vulnerability_issue_links WHERE vulnerability_issue_links.vulnerability_id = NEW.id)
  INTO
    has_issues;

  SELECT
    EXISTS (SELECT 1 FROM vulnerability_merge_request_links WHERE vulnerability_merge_request_links.vulnerability_id = NEW.id)
  INTO
    has_merge_request;

  INSERT INTO vulnerability_reads (vulnerability_id, project_id, scanner_id, report_type, severity, state, resolved_on_default_branch, uuid, location_image, cluster_agent_id, casted_cluster_agent_id, has_issues, has_merge_request)
    VALUES (NEW.id, NEW.project_id, scanner_id, NEW.report_type, NEW.severity, NEW.state, NEW.resolved_on_default_branch, uuid::uuid, location_image, cluster_agent_id, casted_cluster_agent_id, has_issues, has_merge_request)
    ON CONFLICT(vulnerability_id) DO NOTHING;
  RETURN NULL;
END
$$;

CREATE FUNCTION next_traversal_ids_sibling(traversal_ids bigint[]) RETURNS bigint[]
    LANGUAGE plpgsql IMMUTABLE STRICT
    AS $$
BEGIN
  return traversal_ids[1:array_length(traversal_ids, 1)-1] ||
  ARRAY[traversal_ids[array_length(traversal_ids, 1)]+1];
END;
$$;

CREATE FUNCTION nullify_merge_request_metrics_build_data() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF (OLD.pipeline_id IS NOT NULL) AND (NEW.pipeline_id IS NULL) THEN
  NEW.latest_build_started_at = NULL;
  NEW.latest_build_finished_at = NULL;
END IF;
RETURN NEW;

END
$$;

CREATE FUNCTION pool_repositories_sharding_key() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF NEW.organization_id IS NOT NULL THEN
    RETURN NEW;
  END IF;

  IF NEW.source_project_id IS NOT NULL THEN
    SELECT p.organization_id
    INTO NEW.organization_id
    FROM projects p
    WHERE p.id = NEW.source_project_id;
  END IF;

  RETURN NEW;
END;
$$;

CREATE FUNCTION postgres_pg_stat_activity_autovacuum() RETURNS TABLE(query text, query_start timestamp with time zone)
    LANGUAGE sql SECURITY DEFINER
    SET search_path TO 'pg_catalog', 'pg_temp'
    AS $$
  SELECT query, query_start
  FROM pg_stat_activity
  WHERE datname = current_database()
    AND state = 'active'
    AND backend_type = 'autovacuum worker'
$$;

CREATE FUNCTION prevent_delete_of_default_organization() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF OLD.id = 1 THEN
  RAISE EXCEPTION 'Deletion of the default Organization is not allowed.';
END IF;
RETURN OLD;

END
$$;

CREATE FUNCTION set_has_external_issue_tracker() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE projects SET has_external_issue_tracker = (
  EXISTS
  (
    SELECT 1
    FROM integrations
    WHERE project_id = COALESCE(NEW.project_id, OLD.project_id)
      AND active = TRUE
      AND category = 'issue_tracker'
  )
)
WHERE projects.id = COALESCE(NEW.project_id, OLD.project_id);
RETURN NULL;

END
$$;

CREATE FUNCTION set_has_external_wiki() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE projects SET has_external_wiki = COALESCE(NEW.active, FALSE)
WHERE projects.id = COALESCE(NEW.project_id, OLD.project_id);
RETURN NULL;

END
$$;

CREATE FUNCTION set_has_issues_on_vulnerability_reads() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF (SELECT current_setting('vulnerability_management.dont_execute_db_trigger', true) = 'true') THEN
    RETURN NULL;
  END IF;

  UPDATE
    vulnerability_reads
  SET
    has_issues = true
  WHERE
    vulnerability_id = NEW.vulnerability_id AND has_issues IS FALSE;
  RETURN NULL;
END
$$;

CREATE FUNCTION set_has_merge_request_on_vulnerability_reads() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF (SELECT current_setting('vulnerability_management.dont_execute_db_trigger', true) = 'true') THEN
    RETURN NULL;
  END IF;

  UPDATE
    vulnerability_reads
  SET
    has_merge_request = true
  WHERE
    vulnerability_id = NEW.vulnerability_id AND has_merge_request IS FALSE;
  RETURN NULL;
END
$$;

CREATE FUNCTION sync_issues_dates_with_work_item_dates_sources() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE
  issues
SET
  start_date = NEW.start_date,
  due_date = NEW.due_date
WHERE
  issues.id = NEW.issue_id;

RETURN NULL;

END
$$;

CREATE FUNCTION sync_namespace_to_group_push_rules() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  push_rule RECORD;
BEGIN
  IF NEW.type != 'Group' THEN
    RETURN NEW;
  END IF;

  IF OLD.push_rule_id IS NOT NULL AND NEW.push_rule_id IS NULL THEN
    DELETE FROM group_push_rules WHERE group_id = NEW.id;
    RETURN NEW;
  END IF;

  IF NEW.push_rule_id IS NOT NULL AND (OLD.push_rule_id IS NULL OR OLD.push_rule_id != NEW.push_rule_id) THEN
    SELECT * INTO push_rule FROM push_rules WHERE id = NEW.push_rule_id;

    IF FOUND THEN
      INSERT INTO group_push_rules (
        id,
        group_id,
        max_file_size,
        member_check,
        prevent_secrets,
        commit_committer_name_check,
        deny_delete_tag,
        reject_unsigned_commits,
        commit_committer_check,
        reject_non_dco_commits,
        commit_message_regex,
        branch_name_regex,
        commit_message_negative_regex,
        author_email_regex,
        file_name_regex,
        created_at,
        updated_at
      ) VALUES (
        push_rule.id,
        NEW.id,
        push_rule.max_file_size,
        push_rule.member_check,
        push_rule.prevent_secrets,
        push_rule.commit_committer_name_check,
        push_rule.deny_delete_tag,
        push_rule.reject_unsigned_commits,
        push_rule.commit_committer_check,
        push_rule.reject_non_dco_commits,
        push_rule.commit_message_regex,
        push_rule.branch_name_regex,
        push_rule.commit_message_negative_regex,
        push_rule.author_email_regex,
        push_rule.file_name_regex,
        push_rule.created_at,
        push_rule.updated_at
      )
      ON CONFLICT (group_id) DO UPDATE SET
        id = push_rule.id,
        max_file_size = push_rule.max_file_size,
        member_check = push_rule.member_check,
        prevent_secrets = push_rule.prevent_secrets,
        reject_unsigned_commits = push_rule.reject_unsigned_commits,
        commit_committer_check = push_rule.commit_committer_check,
        deny_delete_tag = push_rule.deny_delete_tag,
        reject_non_dco_commits = push_rule.reject_non_dco_commits,
        commit_committer_name_check = push_rule.commit_committer_name_check,
        commit_message_regex = push_rule.commit_message_regex,
        branch_name_regex = push_rule.branch_name_regex,
        commit_message_negative_regex = push_rule.commit_message_negative_regex,
        author_email_regex = push_rule.author_email_regex,
        file_name_regex = push_rule.file_name_regex,
        updated_at = push_rule.updated_at;
    END IF;
  END IF;

  RETURN NEW;
END;
$$;

CREATE FUNCTION sync_organization_push_rules_on_delete() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
  BEGIN
    IF (OLD.organization_id IS NOT NULL AND OLD.is_sample = true) THEN
      DELETE FROM organization_push_rules WHERE organization_id = OLD.organization_id;
    END IF;
    RETURN OLD;
  END;
$$;

CREATE FUNCTION sync_organization_push_rules_on_insert_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
 BEGIN
    IF (NEW.organization_id IS NOT NULL AND NEW.is_sample = TRUE) THEN
      INSERT INTO organization_push_rules (
        id,
        organization_id,
        max_file_size,
        member_check,
        prevent_secrets,
        reject_unsigned_commits,
        commit_committer_check,
        deny_delete_tag,
        reject_non_dco_commits,
        commit_committer_name_check,
        commit_message_regex,
        branch_name_regex,
        commit_message_negative_regex,
        author_email_regex,
        file_name_regex,
        created_at,
        updated_at
      ) VALUES (
        NEW.id,
        NEW.organization_id,
        NEW.max_file_size,
        NEW.member_check,
        NEW.prevent_secrets,
        NEW.reject_unsigned_commits,
        NEW.commit_committer_check,
        NEW.deny_delete_tag,
        NEW.reject_non_dco_commits,
        NEW.commit_committer_name_check,
        NEW.commit_message_regex,
        NEW.branch_name_regex,
        NEW.commit_message_negative_regex,
        NEW.author_email_regex,
        NEW.file_name_regex,
        NEW.created_at,
        NEW.updated_at
      )
      ON CONFLICT (organization_id) DO UPDATE SET
        id = NEW.id,
        max_file_size = NEW.max_file_size,
        member_check = NEW.member_check,
        prevent_secrets = NEW.prevent_secrets,
        reject_unsigned_commits = NEW.reject_unsigned_commits,
        commit_committer_check = NEW.commit_committer_check,
        deny_delete_tag = NEW.deny_delete_tag,
        reject_non_dco_commits = NEW.reject_non_dco_commits,
        commit_committer_name_check = NEW.commit_committer_name_check,
        commit_message_regex = NEW.commit_message_regex,
        branch_name_regex = NEW.branch_name_regex,
        commit_message_negative_regex = NEW.commit_message_negative_regex,
        author_email_regex = NEW.author_email_regex,
        file_name_regex = NEW.file_name_regex,
        updated_at = NEW.updated_at;
    END IF;
   RETURN NEW;
  END;
 $$;

CREATE FUNCTION sync_packages_composer_with_composer_metadata() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE "packages_composer_packages"
    SET target_sha = NEW.target_sha,
        composer_json = NEW.composer_json,
        version_cache_sha = NEW.version_cache_sha
    WHERE id = NEW.package_id;
RETURN NULL;

END
$$;

CREATE FUNCTION sync_packages_composer_with_packages() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF (COALESCE(NEW.package_type, OLD.package_type) = 6) THEN
  IF (TG_OP = 'INSERT') THEN
    INSERT INTO "packages_composer_packages" (id, project_id, created_at, updated_at, name, version, creator_id, status, last_downloaded_at, status_message)
      VALUES (NEW.id, NEW.project_id, NEW.created_at, NEW.updated_at, NEW.name, NEW.version, NEW.creator_id, NEW.status, NEW.last_downloaded_at, NEW.status_message);
  ELSIF (TG_OP = 'UPDATE') THEN
    UPDATE "packages_composer_packages"
        SET project_id = NEW.project_id,
            updated_at = NEW.updated_at,
            name = NEW.name,
            version = NEW.version,
            creator_id = NEW.creator_id,
            status = NEW.status,
            last_downloaded_at = NEW.last_downloaded_at,
            status_message = NEW.status_message
        WHERE id = OLD.id;
  ELSIF (TG_OP = 'DELETE') THEN
    DELETE FROM "packages_composer_packages" WHERE id = OLD.id;
  END IF;
END IF;
RETURN NULL;

END
$$;

CREATE FUNCTION sync_project_authorizations_to_migration_table() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF (TG_OP = 'INSERT' OR TG_OP = 'UPDATE') THEN
  INSERT INTO project_authorizations_for_migration (project_id, user_id, access_level)
  VALUES (NEW.project_id, NEW.user_id, NEW.access_level::smallint)
  ON CONFLICT (project_id, user_id) DO UPDATE
    SET access_level = NEW.access_level::smallint;
  RETURN NEW;

ELSIF (TG_OP = 'DELETE') THEN
  DELETE FROM project_authorizations_for_migration
  WHERE project_id = OLD.project_id AND user_id = OLD.user_id;
  RETURN OLD;
END IF;

RETURN NULL;

END
$$;

CREATE FUNCTION sync_project_push_rules_on_delete() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
 BEGIN
    IF (OLD.project_id IS NOT NULL) THEN
      DELETE FROM project_push_rules WHERE project_id = OLD.project_id;
    END IF;
   RETURN OLD;
  END;
 $$;

CREATE FUNCTION sync_project_push_rules_on_insert_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
 BEGIN
    IF (NEW.project_id IS NOT NULL) THEN
      IF EXISTS (SELECT 1 FROM project_push_rules WHERE id = NEW.id) THEN
        UPDATE project_push_rules SET
          max_file_size = NEW.max_file_size,
          member_check = NEW.member_check,
          prevent_secrets = NEW.prevent_secrets,
          commit_committer_name_check = NEW.commit_committer_name_check,
          deny_delete_tag = NEW.deny_delete_tag,
          reject_unsigned_commits = NEW.reject_unsigned_commits,
          commit_committer_check = NEW.commit_committer_check,
          reject_non_dco_commits = NEW.reject_non_dco_commits,
          commit_message_regex = NEW.commit_message_regex,
          branch_name_regex = NEW.branch_name_regex,
          commit_message_negative_regex = NEW.commit_message_negative_regex,
          author_email_regex = NEW.author_email_regex,
          file_name_regex = NEW.file_name_regex,
          updated_at = NEW.updated_at
        WHERE id = NEW.id;
      ELSE
        INSERT INTO project_push_rules (
          id,
          project_id,
          max_file_size,
          member_check,
          prevent_secrets,
          commit_committer_name_check,
          deny_delete_tag,
          reject_unsigned_commits,
          commit_committer_check,
          reject_non_dco_commits,
          commit_message_regex,
          branch_name_regex,
          commit_message_negative_regex,
          author_email_regex,
          file_name_regex,
          created_at,
          updated_at
        ) VALUES (
          NEW.id,
          NEW.project_id,
          NEW.max_file_size,
          NEW.member_check,
          NEW.prevent_secrets,
          NEW.commit_committer_name_check,
          NEW.deny_delete_tag,
          NEW.reject_unsigned_commits,
          NEW.commit_committer_check,
          NEW.reject_non_dco_commits,
          NEW.commit_message_regex,
          NEW.branch_name_regex,
          NEW.commit_message_negative_regex,
          NEW.author_email_regex,
          NEW.file_name_regex,
          NEW.created_at,
          NEW.updated_at
        )
        ON CONFLICT (project_id) DO UPDATE SET
          id = EXCLUDED.id,
          max_file_size = EXCLUDED.max_file_size,
          member_check = EXCLUDED.member_check,
          prevent_secrets = EXCLUDED.prevent_secrets,
          commit_committer_name_check = EXCLUDED.commit_committer_name_check,
          deny_delete_tag = EXCLUDED.deny_delete_tag,
          reject_unsigned_commits = EXCLUDED.reject_unsigned_commits,
          commit_committer_check = EXCLUDED.commit_committer_check,
          reject_non_dco_commits = EXCLUDED.reject_non_dco_commits,
          commit_message_regex = EXCLUDED.commit_message_regex,
          branch_name_regex = EXCLUDED.branch_name_regex,
          commit_message_negative_regex = EXCLUDED.commit_message_negative_regex,
          author_email_regex = EXCLUDED.author_email_regex,
          file_name_regex = EXCLUDED.file_name_regex,
          updated_at = EXCLUDED.updated_at
        WHERE NOT EXISTS (SELECT 1 FROM project_push_rules WHERE id = EXCLUDED.id AND project_id != EXCLUDED.project_id);
      END IF;
    END IF;
   RETURN NEW;
  END;
 $$;

CREATE FUNCTION sync_push_rules_to_group_push_rules() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  UPDATE group_push_rules
  SET
    max_file_size = NEW.max_file_size,
    member_check = NEW.member_check,
    prevent_secrets = NEW.prevent_secrets,
    commit_committer_name_check = NEW.commit_committer_name_check,
    deny_delete_tag = NEW.deny_delete_tag,
    reject_unsigned_commits = NEW.reject_unsigned_commits,
    commit_committer_check = NEW.commit_committer_check,
    reject_non_dco_commits = NEW.reject_non_dco_commits,
    commit_message_regex = NEW.commit_message_regex,
    branch_name_regex = NEW.branch_name_regex,
    commit_message_negative_regex = NEW.commit_message_negative_regex,
    author_email_regex = NEW.author_email_regex,
    file_name_regex = NEW.file_name_regex,
    updated_at = NEW.updated_at
  FROM namespaces
  WHERE
    namespaces.push_rule_id = NEW.id
    AND namespaces.type = 'Group'
    AND group_push_rules.group_id = namespaces.id;

  RETURN NEW;
END;
$$;

CREATE FUNCTION sync_redirect_routes_namespace_id() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."source_type" = 'Namespace' THEN
  NEW."namespace_id" = NEW."source_id";
ELSIF NEW."source_type" = 'Project' THEN
  NEW."namespace_id" = (SELECT project_namespace_id FROM projects WHERE id = NEW.source_id);
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION sync_sharding_key_with_notes_table() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  note_project_id BIGINT;
  note_namespace_id BIGINT;
BEGIN
  IF NEW."note_id" IS NULL OR NEW."namespace_id" IS NOT NULL THEN
    RETURN NEW;
  END IF;

  SELECT "project_id", "namespace_id"
  INTO note_project_id, note_namespace_id
  FROM "notes"
  WHERE "id" = NEW."note_id";

  IF note_project_id IS NOT NULL THEN
    SELECT "project_namespace_id" FROM "projects"
    INTO NEW."namespace_id" WHERE "projects"."id" = note_project_id;
  ELSE
    NEW."namespace_id" := note_namespace_id;
  END IF;

  RETURN NEW;
END
$$;

CREATE FUNCTION sync_user_id_from_gpg_keys_table() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF NEW."gpg_key_id" IS NULL OR NEW."user_id" IS NOT NULL THEN
    RETURN NEW;
  END IF;

  SELECT "user_id"
  INTO NEW."user_id"
  FROM "gpg_keys"
  WHERE "id" = NEW."gpg_key_id";

  RETURN NEW;
END
$$;

CREATE FUNCTION sync_work_item_transitions_from_issues() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO work_item_transitions (
  work_item_id,
  namespace_id,
  moved_to_id,
  duplicated_to_id,
  promoted_to_epic_id
)
VALUES (
  NEW.id,
  NEW.namespace_id,
  NEW.moved_to_id,
  NEW.duplicated_to_id,
  NEW.promoted_to_epic_id
)
ON CONFLICT (work_item_id)
DO UPDATE SET
  moved_to_id = EXCLUDED.moved_to_id,
  duplicated_to_id = EXCLUDED.duplicated_to_id,
  promoted_to_epic_id = EXCLUDED.promoted_to_epic_id,
  namespace_id = EXCLUDED.namespace_id;
RETURN NULL;

END
$$;

CREATE FUNCTION table_sync_function_3f39f64fc3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF (TG_OP = 'DELETE') THEN
  DELETE FROM merge_request_diff_files_99208b8fac where "merge_request_diff_id" = OLD."merge_request_diff_id" AND "relative_order" = OLD."relative_order";
ELSIF (TG_OP = 'UPDATE') THEN
  UPDATE merge_request_diff_files_99208b8fac
  SET "new_file" = NEW."new_file",
    "renamed_file" = NEW."renamed_file",
    "deleted_file" = NEW."deleted_file",
    "too_large" = NEW."too_large",
    "a_mode" = NEW."a_mode",
    "b_mode" = NEW."b_mode",
    "new_path" = NULLIF(NEW."new_path", NEW."old_path"),
    "old_path" = NEW."old_path",
    "diff" = NEW."diff",
    "binary" = NEW."binary",
    "external_diff_offset" = NEW."external_diff_offset",
    "external_diff_size" = NEW."external_diff_size",
    "generated" = NEW."generated",
    "encoded_file_path" = NEW."encoded_file_path",
    "project_id" = COALESCE(NEW."project_id", (SELECT mrd.project_id FROM merge_request_diffs mrd WHERE mrd.id = NEW."merge_request_diff_id"))
  WHERE merge_request_diff_files_99208b8fac."merge_request_diff_id" = NEW."merge_request_diff_id" AND merge_request_diff_files_99208b8fac."relative_order" = NEW."relative_order";
ELSIF (TG_OP = 'INSERT') THEN
  INSERT INTO merge_request_diff_files_99208b8fac ("new_file",
    "renamed_file",
    "deleted_file",
    "too_large",
    "a_mode",
    "b_mode",
    "new_path",
    "old_path",
    "diff",
    "binary",
    "external_diff_offset",
    "external_diff_size",
    "generated",
    "encoded_file_path",
    "project_id",
    "merge_request_diff_id",
    "relative_order")
  VALUES (NEW."new_file",
    NEW."renamed_file",
    NEW."deleted_file",
    NEW."too_large",
    NEW."a_mode",
    NEW."b_mode",
    NULLIF(NEW."new_path", NEW."old_path"),
    NEW."old_path",
    NEW."diff",
    NEW."binary",
    NEW."external_diff_offset",
    NEW."external_diff_size",
    NEW."generated",
    NEW."encoded_file_path",
    COALESCE(NEW."project_id", (SELECT mrd.project_id FROM merge_request_diffs mrd WHERE mrd.id = NEW."merge_request_diff_id")),
    NEW."merge_request_diff_id",
    NEW."relative_order");
END IF;
RETURN NULL;

END
$$;

COMMENT ON FUNCTION table_sync_function_3f39f64fc3() IS 'Partitioning migration: table sync for merge_request_diff_files table';

CREATE FUNCTION table_sync_function_40ecbfb353() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF (TG_OP = 'DELETE') THEN
  DELETE FROM uploads_9ba88c4165 where "id" = OLD."id";
ELSIF (TG_OP = 'UPDATE') THEN
  UPDATE uploads_9ba88c4165
  SET "size" = NEW."size",
    "model_id" = NEW."model_id",
    "uploaded_by_user_id" = NEW."uploaded_by_user_id",
    "organization_id" = NEW."organization_id",
    "namespace_id" = NEW."namespace_id",
    "project_id" = NEW."project_id",
    "created_at" = NEW."created_at",
    "store" = NEW."store",
    "version" = NEW."version",
    "path" = NEW."path",
    "checksum" = NEW."checksum",
    "model_type" = NEW."model_type",
    "uploader" = NEW."uploader",
    "mount_point" = NEW."mount_point",
    "secret" = NEW."secret"
  WHERE uploads_9ba88c4165."id" = NEW."id";
ELSIF (TG_OP = 'INSERT') THEN
  INSERT INTO uploads_9ba88c4165 ("id",
    "size",
    "model_id",
    "uploaded_by_user_id",
    "organization_id",
    "namespace_id",
    "project_id",
    "created_at",
    "store",
    "version",
    "path",
    "checksum",
    "model_type",
    "uploader",
    "mount_point",
    "secret")
  VALUES (NEW."id",
    NEW."size",
    NEW."model_id",
    NEW."uploaded_by_user_id",
    NEW."organization_id",
    NEW."namespace_id",
    NEW."project_id",
    NEW."created_at",
    NEW."store",
    NEW."version",
    NEW."path",
    NEW."checksum",
    NEW."model_type",
    NEW."uploader",
    NEW."mount_point",
    NEW."secret");
END IF;
RETURN NULL;

END
$$;

COMMENT ON FUNCTION table_sync_function_40ecbfb353() IS 'Partitioning migration: table sync for uploads table';

CREATE FUNCTION table_sync_function_c237afdf68() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF (TG_OP = 'DELETE') THEN
  DELETE FROM project_daily_statistics_archived where "id" = OLD."id";
ELSIF (TG_OP = 'UPDATE') THEN
  UPDATE project_daily_statistics_archived
  SET "project_id" = NEW."project_id",
    "fetch_count" = NEW."fetch_count",
    "date" = NEW."date"
  WHERE project_daily_statistics_archived."id" = NEW."id";
ELSIF (TG_OP = 'INSERT') THEN
  INSERT INTO project_daily_statistics_archived ("id",
    "project_id",
    "fetch_count",
    "date")
  VALUES (NEW."id",
    NEW."project_id",
    NEW."fetch_count",
    NEW."date");
END IF;
RETURN NULL;

END
$$;

COMMENT ON FUNCTION table_sync_function_c237afdf68() IS 'Partitioning migration: table sync for project_daily_statistics table';

CREATE FUNCTION timestamp_coalesce(t1 timestamp with time zone, t2 anyelement) RETURNS timestamp without time zone
    LANGUAGE plpgsql IMMUTABLE
    AS $$
BEGIN
  RETURN COALESCE(t1::TIMESTAMP, t2);
END;
$$;

CREATE FUNCTION todos_sharding_key() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  IF num_nonnulls(NEW.organization_id, NEW.group_id, NEW.project_id) != 1 THEN
    IF NEW.project_id IS NOT NULL THEN
      NEW.organization_id := NULL;
      NEW.group_id := NULL;
    ELSIF NEW.group_id IS NOT NULL THEN
      NEW.organization_id := NULL;
      NEW.project_id := NULL;
    ELSE
      SELECT "organization_id", NULL, NULL
      INTO NEW."organization_id", NEW."group_id", NEW."project_id"
      FROM "users"
      WHERE "users"."id" = NEW."user_id";
    END IF;
  END IF;

  RETURN NEW;
END
$$;

CREATE FUNCTION trigger_009314eae986() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."protected_branch_project_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_01b3fc052119() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_02450faab875() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_occurrences"
  WHERE "vulnerability_occurrences"."id" = NEW."occurrence_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_038fe84feff7() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_05cc4448a8aa() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."protected_branch_namespace_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_05ce163deddf() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0a1b0adcf686() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_debian_project_distributions"
  WHERE "packages_debian_project_distributions"."id" = NEW."distribution_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0a29d4d42b62() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "approval_project_rules"
  WHERE "approval_project_rules"."id" = NEW."approval_project_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0aea02e5a699() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."protected_branch_project_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0af180e1ec89() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_debian_project_components"
  WHERE "packages_debian_project_components"."id" = NEW."component_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0c326daf67cf() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."namespace_id"
  FROM "analytics_cycle_analytics_group_value_streams"
  WHERE "analytics_cycle_analytics_group_value_streams"."id" = NEW."value_stream_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0d96daa4d734() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "bulk_import_exports"
  WHERE "bulk_import_exports"."id" = NEW."export_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0da002390fdc() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "operations_feature_flags"
  WHERE "operations_feature_flags"."id" = NEW."feature_flag_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0ddb594934c9() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "incident_management_oncall_rotations"
  WHERE "incident_management_oncall_rotations"."id" = NEW."oncall_rotation_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0e13f214e504() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_0f38e5af9adf() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ml_candidates"
  WHERE "ml_candidates"."id" = NEW."candidate_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_13d4aa8fe3dd() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_14a39509be0a() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1513378d715d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_158ac875f254() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "approval_group_rules"
  WHERE "approval_group_rules"."id" = NEW."approval_group_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_174b23fa3dfb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "approval_project_rules"
  WHERE "approval_project_rules"."id" = NEW."approval_project_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1825cdc71779() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "organization_details"
  WHERE "organization_details"."organization_id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_18bc439a6741() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1996c9e5bea0() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "abuse_reports"
  WHERE "abuse_reports"."id" = NEW."abuse_report_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1a052e65e9d9() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."namespace_id"
  FROM "import_export_uploads"
  WHERE "import_export_uploads"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1a41d368edd5() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "import_export_uploads"
  WHERE "import_export_uploads"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1c0f1ca199a3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ci_resource_groups"
  WHERE "ci_resource_groups"."id" = NEW."resource_group_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1ed40f4d5f4e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1eda1bc6ef53() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "merge_request_diffs"
  WHERE "merge_request_diffs"."id" = NEW."merge_request_diff_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_1f57c71a69fb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."snippet_organization_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_206cbe2dc1a2() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_207005e8e995() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "operations_feature_flags"
  WHERE "operations_feature_flags"."id" = NEW."feature_flag_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_218433b4faa5() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_package_files"
  WHERE "packages_package_files"."id" = NEW."package_file_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_219952df8fc4() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."blocking_merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_22262f5f16d8() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  NEW."author_id_convert_to_bigint" := NEW."author_id";
  NEW."closed_by_id_convert_to_bigint" := NEW."closed_by_id";
  NEW."duplicated_to_id_convert_to_bigint" := NEW."duplicated_to_id";
  NEW."id_convert_to_bigint" := NEW."id";
  NEW."last_edited_by_id_convert_to_bigint" := NEW."last_edited_by_id";
  NEW."milestone_id_convert_to_bigint" := NEW."milestone_id";
  NEW."moved_to_id_convert_to_bigint" := NEW."moved_to_id";
  NEW."project_id_convert_to_bigint" := NEW."project_id";
  NEW."promoted_to_epic_id_convert_to_bigint" := NEW."promoted_to_epic_id";
  NEW."updated_by_id_convert_to_bigint" := NEW."updated_by_id";
  RETURN NEW;
END;
$$;

CREATE FUNCTION trigger_238f37f25bb2() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "boards_epic_lists"
  WHERE "boards_epic_lists"."id" = NEW."epic_list_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_243aecba8654() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_site_profiles"
  WHERE "dast_site_profiles"."id" = NEW."dast_site_profile_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_248cafd363ff() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_2514245c7fc5() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_site_profiles"
  WHERE "dast_site_profiles"."id" = NEW."dast_site_profile_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_25ba78722e56() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "users"
  WHERE "users"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_25c44c30884f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."work_item_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_25d35f02ab55() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ml_candidates"
  WHERE "ml_candidates"."id" = NEW."candidate_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_25fe4f7da510() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerabilities"
  WHERE "vulnerabilities"."id" = NEW."vulnerability_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_29128c51c7c6() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_pre_scan_verifications"
  WHERE "dast_pre_scan_verifications"."id" = NEW."dast_pre_scan_verification_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_292097dea85c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "terraform_state_versions"
  WHERE "terraform_state_versions"."id" = NEW."terraform_state_version_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_2a994bb5629f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "alert_management_alerts"
  WHERE "alert_management_alerts"."id" = NEW."alert_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_2b8fdc9b4a4e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ml_experiments"
  WHERE "ml_experiments"."id" = NEW."experiment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_2cb7e7147818() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "notes"
  WHERE "notes"."id" = NEW."note_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_2dafd0d13605() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "pages_domains"
  WHERE "pages_domains"."id" = NEW."pages_domain_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_2e4861e8640c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_package_files"
  WHERE "packages_package_files"."id" = NEW."package_file_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_30209d0fba3e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "alert_management_alerts"
  WHERE "alert_management_alerts"."id" = NEW."alert_management_alert_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_309294c3b889() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."snippet_project_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3434b82e5e12() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "abuse_reports"
  WHERE "abuse_reports"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_363d0fd35f2c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_dependency_links"
  WHERE "packages_dependency_links"."id" = NEW."dependency_link_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3691f9f6a69f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "cluster_agents"
  WHERE "cluster_agents"."id" = NEW."cluster_agent_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_36cb404f9a02() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "bulk_import_entities"
  WHERE "bulk_import_entities"."id" = NEW."bulk_import_entity_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_388de55cd36c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "p_ci_builds"
  WHERE "p_ci_builds"."id" = NEW."build_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_38b6d9d97935() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "id"
  INTO NEW."project_id"
  FROM "projects"
  WHERE "projects"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_38bfee591e40() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "dependency_proxy_blobs"
  WHERE "dependency_proxy_blobs"."id" = NEW."dependency_proxy_blob_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_397d1b13068e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3be1956babdb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."snippet_organization_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3c1a5f58a668() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "incident_management_oncall_rotations"
  WHERE "incident_management_oncall_rotations"."id" = NEW."rotation_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3d1a58344b29() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "alert_management_alerts"
  WHERE "alert_management_alerts"."id" = NEW."alert_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3e067fa9bfe3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "incident_management_timeline_event_tags"
  WHERE "incident_management_timeline_event_tags"."id" = NEW."timeline_event_tag_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3f28a0bfdb16() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_3fe922f4db67() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerabilities"
  WHERE "vulnerabilities"."id" = NEW."vulnerability_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_41eaf23bf547() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "releases"
  WHERE "releases"."id" = NEW."release_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_43484cb41aca() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "project_wiki_repositories"
  WHERE "project_wiki_repositories"."id" = NEW."project_wiki_repository_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_442d030cfdfe() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "id"
  INTO NEW."namespace_id"
  FROM "namespaces"
  WHERE "namespaces"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_44558add1625() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_44ff19ad0ab2() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_468b8554e533() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_scanners"
  WHERE "vulnerability_scanners"."id" = NEW."scanner_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_46ebe375f632() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_47b402bdab5f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "bulk_import_exports"
  WHERE "bulk_import_exports"."id" = NEW."export_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_47c43d40f0d2() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "alert_management_alert_metric_images"
  WHERE "alert_management_alert_metric_images"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_49862b4b3035() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "approval_group_rules"
  WHERE "approval_group_rules"."id" = NEW."approval_group_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_49b563d0130b() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_scanner_profiles"
  WHERE "dast_scanner_profiles"."id" = NEW."dast_scanner_profile_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_49e070da6320() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4ad9a52a6614() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "sbom_occurrences"
  WHERE "sbom_occurrences"."id" = NEW."sbom_occurrence_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4b43790d717f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_environment_group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."protected_environment_group_id"
  FROM "protected_environments"
  WHERE "protected_environments"."id" = NEW."protected_environment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4c320a13bc8d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "security_orchestration_policy_configurations"
  WHERE "security_orchestration_policy_configurations"."id" = NEW."security_orchestration_policy_configuration_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4cc5c3ac4d7f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "bulk_import_exports"
  WHERE "bulk_import_exports"."id" = NEW."export_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4dc8ec48e038() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4f1b6c76fdfc() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "topics"
  WHERE "topics"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_4fc14aa830b1() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."work_item_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_54707c384ad7() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "security_orchestration_policy_configurations"
  WHERE "security_orchestration_policy_configurations"."id" = NEW."security_orchestration_policy_configuration_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_56d49f4ed623() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "workspaces"
  WHERE "workspaces"."id" = NEW."workspace_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_57ad2742ac16() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "achievements"
  WHERE "achievements"."id" = NEW."achievement_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_57d53b2ab135() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_589db52d2d69() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_5ca97b87ee30() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_5cf44cd40f22() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "operations_strategies"
  WHERE "operations_strategies"."id" = NEW."strategy_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_5ed68c226e97() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "approval_merge_request_rules"
  WHERE "approval_merge_request_rules"."id" = NEW."approval_merge_request_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_5f6432d2dccc() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "operations_user_lists"
  WHERE "operations_user_lists"."id" = NEW."user_list_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_627949f72f05() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_664594a3d0a7() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_67d0d39e2f41() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."uploaded_by_user_id" IS NULL THEN
  SELECT "user_id"
  INTO NEW."uploaded_by_user_id"
  FROM "user_permission_export_uploads"
  WHERE "user_permission_export_uploads"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_68435a54ee2b() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_debian_project_distributions"
  WHERE "packages_debian_project_distributions"."id" = NEW."distribution_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_6bf50b363152() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "security_orchestration_policy_configurations"
  WHERE "security_orchestration_policy_configurations"."id" = NEW."policy_configuration_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_6c38ba395cc1() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "error_tracking_errors"
  WHERE "error_tracking_errors"."id" = NEW."error_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_6cdea9559242() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."source_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_6d6c79ce74e1() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_environment_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."protected_environment_project_id"
  FROM "protected_environments"
  WHERE "protected_environments"."id" = NEW."protected_environment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_6fc75a2395f3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_package_files"
  WHERE "packages_package_files"."id" = NEW."package_file_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_700f29b1312e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_70d3f0bba1de() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "security_orchestration_policy_configurations"
  WHERE "security_orchestration_policy_configurations"."id" = NEW."policy_configuration_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_738125833856() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "bulk_imports"
  WHERE "bulk_imports"."id" = NEW."bulk_import_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_740afa9807b8() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "subscription_add_on_purchases"
  WHERE "subscription_add_on_purchases"."id" = NEW."add_on_purchase_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_744ab45ee5ac() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."protected_branch_namespace_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7495f5e0efcb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."snippet_project_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_77d9fbad5b12() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_debian_project_distributions"
  WHERE "packages_debian_project_distributions"."id" = NEW."distribution_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_78c85ddc4031() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7943cb549289() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7a6d75e9eecd() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "project_relation_exports"
  WHERE "project_relation_exports"."id" = NEW."project_relation_export_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7a8b08eed782() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "boards_epic_boards"
  WHERE "boards_epic_boards"."id" = NEW."epic_board_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7b21c87a1f91() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "bulk_import_entities"
  WHERE "bulk_import_entities"."id" = NEW."bulk_import_entity_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7b378a0c402b() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7de792ddbc05() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_site_tokens"
  WHERE "dast_site_tokens"."id" = NEW."dast_site_token_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_7e2eed79e46e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  NEW."assignee_id_convert_to_bigint" := NEW."assignee_id";
  NEW."id_convert_to_bigint" := NEW."id";
  NEW."reporter_id_convert_to_bigint" := NEW."reporter_id";
  NEW."resolved_by_id_convert_to_bigint" := NEW."resolved_by_id";
  NEW."user_id_convert_to_bigint" := NEW."user_id";
  RETURN NEW;
END;
$$;

CREATE FUNCTION trigger_80578cfbdaf9() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "events"
  WHERE "events"."id" = NEW."event_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_81b4c93e7133() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "pages_deployments"
  WHERE "pages_deployments"."id" = NEW."pages_deployment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_81b53b626109() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_package_files"
  WHERE "packages_package_files"."id" = NEW."package_file_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8204480b3a2e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "incident_management_escalation_policies"
  WHERE "incident_management_escalation_policies"."id" = NEW."policy_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_84d67ad63e93() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "wiki_page_meta"
  WHERE "wiki_page_meta"."id" = NEW."wiki_page_meta_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_85d89f0f11db() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8a11b103857c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "packages_debian_group_components"
  WHERE "packages_debian_group_components"."id" = NEW."component_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8a38ce2327de() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "epics"
  WHERE "epics"."id" = NEW."epic_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8ac78f164b2d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "project_namespace_id"
  INTO NEW."namespace_id"
  FROM "projects"
  WHERE "projects"."id" = NEW."project_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8b39d532224c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ci_secure_files"
  WHERE "ci_secure_files"."id" = NEW."ci_secure_file_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8ba074736a77() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."snippet_project_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8cb8ad095bf6() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "bulk_import_entities"
  WHERE "bulk_import_entities"."id" = NEW."bulk_import_entity_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8cf1745cf163() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "design_management_repositories"
  WHERE "design_management_repositories"."id" = NEW."design_management_repository_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8d002f38bdef() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "packages_debian_group_distributions"
  WHERE "packages_debian_group_distributions"."id" = NEW."distribution_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8d17725116fe() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8d661362aa1a() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8e66b994e8f0() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."group_id"
  FROM "audit_events_external_audit_event_destinations"
  WHERE "audit_events_external_audit_event_destinations"."id" = NEW."external_audit_event_destination_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_8fbb044c64ad() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "project_namespace_id"
  INTO NEW."namespace_id"
  FROM "projects"
  WHERE "projects"."id" = NEW."project_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_90fa5c6951f1() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_profiles"
  WHERE "dast_profiles"."id" = NEW."dast_profile_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_91e1012b9851() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "merge_request_context_commits"
  WHERE "merge_request_context_commits"."id" = NEW."merge_request_context_commit_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9259aae92378() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_93a5b044f4e8() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."snippet_organization_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_94514aeadc50() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "deployments"
  WHERE "deployments"."id" = NEW."deployment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_951ac22c24d7() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."protected_branch_namespace_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_96298f7da5d3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."protected_branch_project_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9699ea03bb37() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "epics"
  WHERE "epics"."id" = NEW."source_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_96a76ee9f147() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_979e7f45114f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ml_candidates"
  WHERE "ml_candidates"."id" = NEW."candidate_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_97e9245e767d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_98ad3a4c1d35() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "reviews"
  WHERE "reviews"."id" = NEW."review_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9b944f36fdac() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "approval_merge_request_rules"
  WHERE "approval_merge_request_rules"."id" = NEW."approval_merge_request_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9e137c16de79() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_occurrences"
  WHERE "vulnerability_occurrences"."id" = NEW."vulnerability_occurrence_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9e875cabe9c9() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "wiki_page_meta"
  WHERE "wiki_page_meta"."id" = NEW."wiki_page_meta_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9f3745f8fe32() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_9f3de326ea61() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ci_pipeline_schedules"
  WHERE "ci_pipeline_schedules"."id" = NEW."pipeline_schedule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a1bc7c70cbdf() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerabilities"
  WHERE "vulnerabilities"."id" = NEW."vulnerability_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a22be47501db() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "group_wiki_repositories"
  WHERE "group_wiki_repositories"."group_id" = NEW."group_wiki_repository_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a253cb3cacdf() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "environments"
  WHERE "environments"."id" = NEW."environment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a465de38164e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "p_ci_job_artifacts"
  WHERE "p_ci_job_artifacts"."id" = NEW."job_artifact_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a4e4fb2451d9() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "epics"
  WHERE "epics"."id" = NEW."epic_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a68471fea292() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_a7e0fb195210() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_occurrences"
  WHERE "vulnerability_occurrences"."id" = NEW."vulnerability_occurrence_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_ad05b7ebe49b() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "deployments"
  WHERE "deployments"."id" = NEW."deployment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_af3f17817e4d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "protected_tags"
  WHERE "protected_tags"."id" = NEW."protected_tag_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b046dd50c711() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "incident_management_oncall_schedules"
  WHERE "incident_management_oncall_schedules"."id" = NEW."oncall_schedule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b0f4298cadff() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."protected_branch_project_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b2612138515d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "project_export_jobs"
  WHERE "project_export_jobs"."id" = NEW."project_export_job_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b4520c29ea74() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "approval_project_rules"
  WHERE "approval_project_rules"."id" = NEW."approval_project_rule_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b75e5731e305() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_profiles"
  WHERE "dast_profiles"."id" = NEW."dast_profile_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b7abb8fc4cf0() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b83b7e51e2f5() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "security_orchestration_policy_configurations"
  WHERE "security_orchestration_policy_configurations"."id" = NEW."security_orchestration_policy_configuration_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_b8eecea7f351() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "dependency_proxy_manifests"
  WHERE "dependency_proxy_manifests"."id" = NEW."dependency_proxy_manifest_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c17a166692a2() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."group_id"
  FROM "audit_events_external_audit_event_destinations"
  WHERE "audit_events_external_audit_event_destinations"."id" = NEW."external_audit_event_destination_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c24a252f7b04() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "design_management_designs_versions"
  WHERE "design_management_designs_versions"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c40a5bb7c1c3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "bulk_import_export_uploads"
  WHERE "bulk_import_export_uploads"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c48e4298f362() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  row_data JSONB;
BEGIN
  row_data := to_jsonb(NEW);
  IF row_data ? 'company' THEN
    NEW."company" := NEW."organization";
  END IF;
  RETURN NEW;
END;
$$;

CREATE FUNCTION trigger_c52d215d50a1() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c59fe6f31e71() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "security_orchestration_policy_configurations"
  WHERE "security_orchestration_policy_configurations"."id" = NEW."security_orchestration_policy_configuration_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c5eec113ea76() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "dast_profiles"
  WHERE "dast_profiles"."id" = NEW."dast_profile_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c6728503decb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "design_management_designs"
  WHERE "design_management_designs"."id" = NEW."design_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c8bc8646bce9() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerabilities"
  WHERE "vulnerabilities"."id" = NEW."vulnerability_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_c9090feed334() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "boards_epic_boards"
  WHERE "boards_epic_boards"."id" = NEW."epic_board_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_ca93521f3a6d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "abuse_reports"
  WHERE "abuse_reports"."id" = NEW."abuse_report_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_cac7c0698291() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "releases"
  WHERE "releases"."id" = NEW."release_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_cca6a43d90dd() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "achievements"
  WHERE "achievements"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_cd50823537a3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_cdfa6500a121() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."snippet_organization_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_cf646a118cbb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "releases"
  WHERE "releases"."id" = NEW."release_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_cfbec3f07e2b() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "deployments"
  WHERE "deployments"."id" = NEW."deployment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_d32ff9d5c63d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."namespace_id"
  FROM "bulk_import_export_uploads"
  WHERE "bulk_import_export_uploads"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_d4487a75bd44() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "terraform_states"
  WHERE "terraform_states"."id" = NEW."terraform_state_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_d5c895007948() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_environment_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."protected_environment_project_id"
  FROM "protected_environments"
  WHERE "protected_environments"."id" = NEW."protected_environment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_d8c2de748d8c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "target_project_id"
  INTO NEW."project_id"
  FROM "merge_requests"
  WHERE "merge_requests"."id" = NEW."merge_request_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_d9468bfbb0b4() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."snippet_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."snippet_project_id"
  FROM "snippets"
  WHERE "snippets"."id" = NEW."snippet_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_da5fd3d6d75c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_dadd660afe2c() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "packages_debian_group_distributions"
  WHERE "packages_debian_group_distributions"."id" = NEW."distribution_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_dbdd61a66a91() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."agent_project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."agent_project_id"
  FROM "cluster_agents"
  WHERE "cluster_agents"."id" = NEW."agent_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_dbe374a57cbb() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_dc13168b8025() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_occurrences"
  WHERE "vulnerability_occurrences"."id" = NEW."vulnerability_occurrence_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_de59b81d3044() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "bulk_import_exports"
  WHERE "bulk_import_exports"."id" = NEW."export_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_dfad97659d5f() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_e0864d1cff37() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "packages_debian_group_distributions"
  WHERE "packages_debian_group_distributions"."id" = NEW."distribution_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_e1da4a738230() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerabilities"
  WHERE "vulnerabilities"."id" = NEW."vulnerability_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_e49ab4d904a0() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_occurrences"
  WHERE "vulnerability_occurrences"."id" = NEW."vulnerability_occurrence_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_e4a6cde57b42() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_e740510cfd33() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issuable_metric_images"
  WHERE "issuable_metric_images"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_e815625b59fa() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_ebab34f83f1d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "packages_packages"
  WHERE "packages_packages"."id" = NEW."package_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_ec1934755627() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "alert_management_alerts"
  WHERE "alert_management_alerts"."id" = NEW."alert_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_ed554313ca66() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_branch_namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."protected_branch_namespace_id"
  FROM "protected_branches"
  WHERE "protected_branches"."id" = NEW."protected_branch_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_efb9d354f05a() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "issues"
  WHERE "issues"."id" = NEW."issue_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_eff80ead42ac() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ci_unit_tests"
  WHERE "ci_unit_tests"."id" = NEW."unit_test_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_f468204dcd5d() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "project_relation_export_uploads"
  WHERE "project_relation_export_uploads"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_f6c61cdddf31() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ml_models"
  WHERE "ml_models"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_f6f59d8216b3() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."protected_environment_group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."protected_environment_group_id"
  FROM "protected_environments"
  WHERE "protected_environments"."id" = NEW."protected_environment_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_f7464057d53e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."organization_id" IS NULL THEN
  SELECT "organization_id"
  INTO NEW."organization_id"
  FROM "users"
  WHERE "users"."id" = NEW."reporter_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_fac444e0cae6() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."namespace_id" IS NULL THEN
  SELECT "namespace_id"
  INTO NEW."namespace_id"
  FROM "design_management_designs"
  WHERE "design_management_designs"."id" = NEW."design_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_fbd42ed69453() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "external_status_checks"
  WHERE "external_status_checks"."id" = NEW."external_status_check_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_fbd8825b3057() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."group_id" IS NULL THEN
  SELECT "group_id"
  INTO NEW."group_id"
  FROM "boards_epic_boards"
  WHERE "boards_epic_boards"."id" = NEW."epic_board_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_fcc3ea1f9d4e() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "ai_vectorizable_files"
  WHERE "ai_vectorizable_files"."id" = NEW."model_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_fd4a1be98713() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "container_repositories"
  WHERE "container_repositories"."id" = NEW."container_repository_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION trigger_fff8735b6b9a() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
IF NEW."project_id" IS NULL THEN
  SELECT "project_id"
  INTO NEW."project_id"
  FROM "vulnerability_occurrences"
  WHERE "vulnerability_occurrences"."id" = NEW."finding_id";
END IF;

RETURN NEW;

END
$$;

CREATE FUNCTION unset_has_issues_on_vulnerability_reads() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  has_issue_links integer;
BEGIN
  IF (SELECT current_setting('vulnerability_management.dont_execute_db_trigger', true) = 'true') THEN
    RETURN NULL;
  END IF;

  PERFORM 1
  FROM
    vulnerability_reads
  WHERE
    vulnerability_id = OLD.vulnerability_id
  FOR UPDATE;

  SELECT 1 INTO has_issue_links FROM vulnerability_issue_links WHERE vulnerability_id = OLD.vulnerability_id LIMIT 1;

  IF (has_issue_links = 1) THEN
    RETURN NULL;
  END IF;

  UPDATE
    vulnerability_reads
  SET
    has_issues = false
  WHERE
    vulnerability_id = OLD.vulnerability_id;

  RETURN NULL;
END
$$;

CREATE FUNCTION unset_has_merge_request_on_vulnerability_reads() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  has_merge_request_links integer;
BEGIN
  IF (SELECT current_setting('vulnerability_management.dont_execute_db_trigger', true) = 'true') THEN
    RETURN NULL;
  END IF;

  PERFORM 1
  FROM
    vulnerability_reads
  WHERE
    vulnerability_id = OLD.vulnerability_id
  FOR UPDATE;

  SELECT 1 INTO has_merge_request_links FROM vulnerability_merge_request_links WHERE vulnerability_id = OLD.vulnerability_id LIMIT 1;

  IF (has_merge_request_links = 1) THEN
    RETURN NULL;
  END IF;

  UPDATE
    vulnerability_reads
  SET
    has_merge_request = false
  WHERE
    vulnerability_id = OLD.vulnerability_id;

  RETURN NULL;
END
$$;

CREATE FUNCTION update_jira_tracker_data_sharding_key() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
SELECT
  "integrations"."project_id",
  "integrations"."group_id",
  "integrations"."organization_id"
INTO
  NEW."project_id",
  NEW."group_id",
  NEW."organization_id"
FROM "integrations"
WHERE "integrations"."id" = NEW."integration_id";
RETURN NEW;

END
$$;

CREATE FUNCTION update_location_from_vulnerability_occurrences() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE
  vulnerability_reads
SET
  location_image = NEW.location->>'image',
  casted_cluster_agent_id = CAST(NEW.location->'kubernetes_resource'->>'agent_id' AS bigint),
  cluster_agent_id = NEW.location->'kubernetes_resource'->>'agent_id'
WHERE
  vulnerability_id = NEW.vulnerability_id;
RETURN NULL;

END
$$;

CREATE FUNCTION update_namespace_details_from_namespaces() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO
  namespace_details (
    description,
    description_html,
    cached_markdown_version,
    updated_at,
    created_at,
    namespace_id
  )
VALUES
  (
    NEW.description,
    NEW.description_html,
    NEW.cached_markdown_version,
    NEW.updated_at,
    NEW.updated_at,
    NEW.id
  ) ON CONFLICT (namespace_id) DO
UPDATE
SET
  description = NEW.description,
  description_html = NEW.description_html,
  cached_markdown_version = NEW.cached_markdown_version,
  updated_at = NEW.updated_at
WHERE
  namespace_details.namespace_id = NEW.id;RETURN NULL;

END
$$;

CREATE FUNCTION update_namespace_details_from_projects() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO
  namespace_details (
    description,
    description_html,
    cached_markdown_version,
    updated_at,
    created_at,
    namespace_id
  )
VALUES
  (
    NEW.description,
    NEW.description_html,
    NEW.cached_markdown_version,
    NEW.updated_at,
    NEW.updated_at,
    NEW.project_namespace_id
  ) ON CONFLICT (namespace_id) DO
UPDATE
SET
  description = NEW.description,
  description_html = NEW.description_html,
  cached_markdown_version = NEW.cached_markdown_version,
  updated_at = NEW.updated_at
WHERE
  namespace_details.namespace_id = NEW.project_namespace_id;RETURN NULL;

END
$$;

CREATE FUNCTION update_vulnerability_reads_from_vulnerability() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE
  vulnerability_reads
SET
  severity = NEW.severity,
  state = NEW.state,
  resolved_on_default_branch = NEW.resolved_on_default_branch,
  auto_resolved = NEW.auto_resolved
WHERE vulnerability_id = NEW.id;
RETURN NULL;

END
$$;

CREATE TABLE ai_events_counts (
    id bigint NOT NULL,
    events_date date NOT NULL,
    namespace_id bigint,
    user_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    event smallint NOT NULL,
    total_occurrences integer DEFAULT 0 NOT NULL
)
PARTITION BY RANGE (events_date);

CREATE TABLE ai_usage_events (
    id bigint NOT NULL,
    "timestamp" timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    event smallint NOT NULL,
    extras jsonb DEFAULT '{}'::jsonb NOT NULL,
    namespace_id bigint
)
PARTITION BY RANGE ("timestamp");

CREATE TABLE audit_events (
    id bigint NOT NULL,
    author_id bigint NOT NULL,
    entity_id bigint NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    entity_path text,
    target_details text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
)
PARTITION BY RANGE (created_at);

CREATE TABLE background_operation_jobs (
    id uuid DEFAULT gen_random_uuid() NOT NULL,
    worker_id uuid NOT NULL,
    organization_id bigint NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    worker_partition bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone,
    finished_at timestamp with time zone,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    pause_ms integer DEFAULT 100 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    attempts smallint DEFAULT 0 NOT NULL,
    metrics jsonb DEFAULT '{}'::jsonb NOT NULL,
    min_cursor jsonb,
    max_cursor jsonb,
    CONSTRAINT check_22e75767e4 CHECK (((jsonb_typeof(min_cursor) = 'array'::text) AND (jsonb_typeof(max_cursor) = 'array'::text))),
    CONSTRAINT check_b922a72749 CHECK ((pause_ms >= 100)),
    CONSTRAINT check_fc1d4517f5 CHECK ((num_nonnulls(min_cursor, max_cursor) = 2))
)
PARTITION BY LIST (partition);

CREATE TABLE background_operation_jobs_cell_local (
    id bigint NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    worker_id bigint NOT NULL,
    worker_partition bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone,
    finished_at timestamp with time zone,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    pause_ms integer DEFAULT 100 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    attempts smallint DEFAULT 0 NOT NULL,
    metrics jsonb DEFAULT '{}'::jsonb NOT NULL,
    min_cursor jsonb,
    max_cursor jsonb,
    CONSTRAINT check_00bb39bb33 CHECK ((pause_ms >= 100)),
    CONSTRAINT check_5b84acc749 CHECK ((num_nonnulls(min_cursor, max_cursor) = 2)),
    CONSTRAINT check_ebc3302442 CHECK (((jsonb_typeof(min_cursor) = 'array'::text) AND (jsonb_typeof(max_cursor) = 'array'::text)))
)
PARTITION BY LIST (partition);

CREATE TABLE background_operation_workers (
    id uuid DEFAULT gen_random_uuid() NOT NULL,
    organization_id bigint NOT NULL,
    user_id bigint NOT NULL,
    total_tuple_count bigint,
    partition bigint DEFAULT 1 NOT NULL,
    started_at timestamp with time zone,
    on_hold_until timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    finished_at timestamp with time zone,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    pause_ms integer DEFAULT 100 NOT NULL,
    max_batch_size integer,
    priority smallint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    "interval" smallint NOT NULL,
    job_class_name text NOT NULL,
    batch_class_name text NOT NULL,
    table_name text NOT NULL,
    column_name text NOT NULL,
    gitlab_schema text NOT NULL,
    job_arguments jsonb DEFAULT '"[]"'::jsonb,
    min_cursor jsonb,
    max_cursor jsonb,
    CONSTRAINT check_10f672741a CHECK ((char_length(column_name) <= 63)),
    CONSTRAINT check_510f6260d5 CHECK ((char_length(gitlab_schema) <= 255)),
    CONSTRAINT check_63fe8b8121 CHECK ((sub_batch_size > 0)),
    CONSTRAINT check_7f88b7751b CHECK ((char_length(job_class_name) <= 100)),
    CONSTRAINT check_91cc32fc67 CHECK ((char_length(batch_class_name) <= 100)),
    CONSTRAINT check_c316362d95 CHECK ((char_length(table_name) <= 63)),
    CONSTRAINT check_c74b62c410 CHECK ((batch_size >= sub_batch_size)),
    CONSTRAINT check_e91dfde154 CHECK ((num_nonnulls(min_cursor, max_cursor) = 2)),
    CONSTRAINT check_f1affe613c CHECK (((jsonb_typeof(min_cursor) = 'array'::text) AND (jsonb_typeof(max_cursor) = 'array'::text)))
)
PARTITION BY LIST (partition);

CREATE TABLE background_operation_workers_cell_local (
    id bigint NOT NULL,
    total_tuple_count bigint,
    partition bigint DEFAULT 1 NOT NULL,
    started_at timestamp with time zone,
    on_hold_until timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    finished_at timestamp with time zone,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    pause_ms integer DEFAULT 100 NOT NULL,
    max_batch_size integer,
    priority smallint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    "interval" smallint NOT NULL,
    job_class_name text NOT NULL,
    batch_class_name text NOT NULL,
    table_name text NOT NULL,
    column_name text NOT NULL,
    gitlab_schema text NOT NULL,
    job_arguments jsonb DEFAULT '"[]"'::jsonb,
    min_cursor jsonb,
    max_cursor jsonb,
    CONSTRAINT check_1da63db6a8 CHECK ((char_length(table_name) <= 63)),
    CONSTRAINT check_4cc5ecb4f2 CHECK ((char_length(column_name) <= 63)),
    CONSTRAINT check_5f184cd88f CHECK ((char_length(gitlab_schema) <= 255)),
    CONSTRAINT check_9d0c37a905 CHECK ((char_length(batch_class_name) <= 100)),
    CONSTRAINT check_be878382ae CHECK ((batch_size >= sub_batch_size)),
    CONSTRAINT check_d94474cbf2 CHECK ((char_length(job_class_name) <= 100)),
    CONSTRAINT check_e40b641a88 CHECK ((num_nonnulls(min_cursor, max_cursor) = 2)),
    CONSTRAINT check_f9383a3f2e CHECK ((sub_batch_size > 0)),
    CONSTRAINT check_f9caba0499 CHECK (((jsonb_typeof(min_cursor) = 'array'::text) AND (jsonb_typeof(max_cursor) = 'array'::text)))
)
PARTITION BY LIST (partition);

CREATE TABLE backup_finding_evidences (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_finding_flags (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_finding_identifiers (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_finding_links (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_finding_remediations (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_finding_signatures (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_findings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerabilities (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_external_issue_links (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_issue_links (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_merge_request_links (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_reads (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_severity_overrides (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_state_transitions (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE backup_vulnerability_user_mentions (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    original_record_identifier bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    data jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE batched_background_migration_job_transition_logs (
    id bigint NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
)
PARTITION BY RANGE (created_at);

CREATE TABLE p_ci_build_names (
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    search_vector tsvector GENERATED ALWAYS AS (to_tsvector('english'::regconfig, COALESCE(name, ''::text))) STORED,
    CONSTRAINT check_1722c96346 CHECK ((char_length(name) <= 255))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_build_sources (
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    source smallint,
    pipeline_source smallint
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_build_tags (
    id bigint NOT NULL,
    tag_id bigint NOT NULL,
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_build_trace_metadata (
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    trace_artifact_id bigint,
    last_archival_attempt_at timestamp with time zone,
    archived_at timestamp with time zone,
    archival_attempts smallint DEFAULT 0 NOT NULL,
    checksum bytea,
    remote_checksum bytea,
    project_id bigint
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_builds (
    status character varying,
    finished_at timestamp without time zone,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    started_at timestamp without time zone,
    coverage double precision,
    name character varying,
    options text,
    allow_failure boolean DEFAULT false NOT NULL,
    stage_idx integer,
    tag boolean,
    ref character varying,
    type character varying,
    target_url character varying,
    description character varying,
    erased_at timestamp without time zone,
    artifacts_expire_at timestamp without time zone,
    environment character varying,
    "when" character varying,
    yaml_variables text,
    queued_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    coverage_regex character varying,
    retried boolean,
    protected boolean,
    failure_reason integer,
    scheduled_at timestamp with time zone,
    token_encrypted character varying,
    resource_group_id bigint,
    waiting_for_resource_at timestamp with time zone,
    processed boolean,
    scheduling_type smallint,
    id bigint NOT NULL,
    stage_id bigint,
    partition_id bigint NOT NULL,
    auto_canceled_by_partition_id bigint,
    auto_canceled_by_id bigint,
    commit_id bigint,
    erased_by_id bigint,
    project_id bigint,
    runner_id bigint,
    upstream_pipeline_id bigint,
    user_id bigint,
    execution_config_id bigint,
    upstream_pipeline_partition_id bigint,
    scoped_user_id bigint,
    timeout integer,
    timeout_source smallint,
    exit_code smallint,
    debug_trace_enabled boolean,
    CONSTRAINT check_1e2fbd1b39 CHECK ((lock_version IS NOT NULL)),
    CONSTRAINT check_9aa9432137 CHECK ((project_id IS NOT NULL))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_builds_execution_configs (
    id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    pipeline_id bigint NOT NULL,
    run_steps jsonb DEFAULT '{}'::jsonb NOT NULL
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_annotations (
    id bigint NOT NULL,
    partition_id bigint NOT NULL,
    job_id bigint NOT NULL,
    name text NOT NULL,
    data jsonb DEFAULT '[]'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_375bb9900a CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_bac9224e45 CHECK ((char_length(name) <= 255)),
    CONSTRAINT data_is_array CHECK ((jsonb_typeof(data) = 'array'::text))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_artifact_reports (
    job_artifact_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    status smallint NOT NULL,
    validation_error text,
    CONSTRAINT check_1c55aee894 CHECK ((char_length(validation_error) <= 255))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_artifacts (
    project_id bigint NOT NULL,
    file_type integer NOT NULL,
    size bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    expire_at timestamp with time zone,
    file character varying,
    file_store integer DEFAULT 1,
    file_sha256 bytea,
    file_format smallint,
    file_location smallint,
    id bigint NOT NULL,
    job_id bigint NOT NULL,
    locked smallint DEFAULT 2,
    partition_id bigint NOT NULL,
    accessibility smallint DEFAULT 0 NOT NULL,
    file_final_path text,
    exposed_as text,
    exposed_paths text[],
    CONSTRAINT check_27f0f6dbab CHECK ((file_store IS NOT NULL)),
    CONSTRAINT check_9f04410cf4 CHECK ((char_length(file_final_path) <= 1024)),
    CONSTRAINT check_b8fac815e7 CHECK ((char_length(exposed_as) <= 100))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_definition_instances (
    job_id bigint NOT NULL,
    job_definition_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_definitions (
    id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone,
    interruptible boolean DEFAULT false NOT NULL,
    checksum bytea NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_inputs (
    id bigint NOT NULL,
    job_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    value jsonb,
    CONSTRAINT check_007134e1cd CHECK ((char_length(name) <= 255))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_job_messages (
    id bigint NOT NULL,
    job_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    severity smallint DEFAULT 0 NOT NULL,
    content text,
    CONSTRAINT check_6b838ff738 CHECK ((char_length(content) <= 10000))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_pipeline_artifact_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    pipeline_artifact_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_ee83b93f85 CHECK ((char_length(verification_failure) <= 255))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_pipeline_variables (
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    variable_type smallint DEFAULT 1 NOT NULL,
    partition_id bigint NOT NULL,
    raw boolean DEFAULT false NOT NULL,
    id bigint NOT NULL,
    pipeline_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_6e932dbabf CHECK ((project_id IS NOT NULL))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_pipelines (
    ref character varying,
    sha character varying,
    before_sha character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    tag boolean DEFAULT false,
    yaml_errors text,
    committed_at timestamp without time zone,
    project_id bigint,
    status character varying,
    started_at timestamp without time zone,
    finished_at timestamp without time zone,
    duration integer,
    user_id bigint,
    lock_version integer DEFAULT 0,
    pipeline_schedule_id bigint,
    source integer,
    config_source integer,
    protected boolean,
    failure_reason integer,
    iid integer,
    merge_request_id bigint,
    source_sha bytea,
    target_sha bytea,
    external_pull_request_id bigint,
    ci_ref_id bigint,
    locked smallint DEFAULT 1 NOT NULL,
    partition_id bigint NOT NULL,
    id bigint NOT NULL,
    auto_canceled_by_id bigint,
    auto_canceled_by_partition_id bigint,
    trigger_id bigint,
    CONSTRAINT check_2ba2a044b9 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_d7e99a025e CHECK ((lock_version IS NOT NULL))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_runner_machine_builds (
    partition_id bigint NOT NULL,
    build_id bigint NOT NULL,
    runner_machine_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_149ee35c38 CHECK ((project_id IS NOT NULL))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_stages (
    project_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    name character varying,
    status integer,
    lock_version integer DEFAULT 0,
    "position" integer,
    id bigint NOT NULL,
    partition_id bigint NOT NULL,
    pipeline_id bigint,
    CONSTRAINT check_74835fc631 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_81b431e49b CHECK ((lock_version IS NOT NULL))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_workload_variable_inclusions (
    id bigint NOT NULL,
    workload_id bigint,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    variable_name text NOT NULL,
    CONSTRAINT check_9c26407a8f CHECK ((char_length(variable_name) <= 255))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_workloads (
    id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint NOT NULL,
    pipeline_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    branch_name text,
    status smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_f2fe503728 CHECK ((char_length(branch_name) <= 255))
)
PARTITION BY LIST (partition_id);

CREATE SEQUENCE shared_audit_event_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE group_audit_events (
    id bigint DEFAULT nextval('shared_audit_event_id_seq'::regclass) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    author_id bigint NOT NULL,
    target_id bigint,
    event_name text,
    details text,
    ip_address inet,
    author_name text,
    entity_path text,
    target_details text,
    target_type text,
    CONSTRAINT group_audit_events_author_name_check CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT group_audit_events_entity_path_check CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT group_audit_events_event_name_check CHECK ((char_length(event_name) <= 255)),
    CONSTRAINT group_audit_events_target_details_check CHECK ((char_length(target_details) <= 5500)),
    CONSTRAINT group_audit_events_target_type_check CHECK ((char_length(target_type) <= 255))
)
PARTITION BY RANGE (created_at);

CREATE TABLE groups_visits (
    id bigint NOT NULL,
    entity_id bigint NOT NULL,
    user_id bigint NOT NULL,
    visited_at timestamp with time zone NOT NULL
)
PARTITION BY RANGE (visited_at);

CREATE TABLE incident_management_pending_alert_escalations (
    id bigint NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    CONSTRAINT check_0012942e48 CHECK ((project_id IS NOT NULL))
)
PARTITION BY RANGE (process_at);

CREATE TABLE incident_management_pending_issue_escalations (
    id bigint NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_dff22b638a CHECK ((namespace_id IS NOT NULL))
)
PARTITION BY RANGE (process_at);

CREATE TABLE instance_audit_events (
    id bigint DEFAULT nextval('shared_audit_event_id_seq'::regclass) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    author_id bigint NOT NULL,
    target_id bigint,
    event_name text,
    details text,
    ip_address inet,
    author_name text,
    entity_path text,
    target_details text,
    target_type text,
    CONSTRAINT instance_audit_events_author_name_check CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT instance_audit_events_entity_path_check CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT instance_audit_events_event_name_check CHECK ((char_length(event_name) <= 255)),
    CONSTRAINT instance_audit_events_target_details_check CHECK ((char_length(target_details) <= 5500)),
    CONSTRAINT instance_audit_events_target_type_check CHECK ((char_length(target_type) <= 255))
)
PARTITION BY RANGE (created_at);

CREATE TABLE loose_foreign_keys_deleted_records (
    id bigint NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    primary_key_value bigint NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    fully_qualified_table_name text NOT NULL,
    consume_after timestamp with time zone DEFAULT now(),
    cleanup_attempts smallint DEFAULT 0,
    CONSTRAINT check_1a541f3235 CHECK ((char_length(fully_qualified_table_name) <= 150))
)
PARTITION BY LIST (partition);

CREATE TABLE merge_request_commits_metadata (
    authored_date timestamp without time zone,
    committed_date timestamp without time zone,
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    commit_author_id bigint NOT NULL,
    committer_id bigint NOT NULL,
    sha bytea NOT NULL,
    message text
)
PARTITION BY RANGE (project_id);

CREATE TABLE merge_request_diff_files_99208b8fac (
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    too_large boolean NOT NULL,
    a_mode character varying NOT NULL,
    b_mode character varying NOT NULL,
    new_path text,
    old_path text NOT NULL,
    diff text,
    "binary" boolean,
    external_diff_offset integer,
    external_diff_size integer,
    generated boolean,
    encoded_file_path boolean DEFAULT false NOT NULL,
    project_id bigint,
    merge_request_diff_id bigint NOT NULL,
    relative_order integer NOT NULL
)
PARTITION BY RANGE (merge_request_diff_id);

CREATE TABLE merge_requests_merge_data (
    merge_request_id bigint NOT NULL,
    project_id bigint NOT NULL,
    merge_user_id bigint,
    merge_params text,
    merge_error text,
    merge_jid text,
    merge_commit_sha bytea,
    merged_commit_sha bytea,
    merge_ref_sha bytea,
    squash_commit_sha bytea,
    in_progress_merge_commit_sha bytea,
    merge_status smallint DEFAULT 0 NOT NULL,
    auto_merge_enabled boolean DEFAULT false NOT NULL,
    squash boolean DEFAULT false NOT NULL,
    CONSTRAINT check_d25e93fc19 CHECK ((char_length(merge_jid) <= 255))
)
PARTITION BY RANGE (merge_request_id);

CREATE TABLE p_ai_active_context_code_enabled_namespaces (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    connection_id bigint NOT NULL,
    metadata jsonb DEFAULT '{}'::jsonb NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
)
PARTITION BY RANGE (namespace_id);

CREATE TABLE p_ai_active_context_code_repositories (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    connection_id bigint,
    enabled_namespace_id bigint,
    metadata jsonb DEFAULT '{}'::jsonb NOT NULL,
    last_commit text,
    state smallint DEFAULT 0 NOT NULL,
    indexed_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    last_queried_at timestamp with time zone,
    CONSTRAINT check_b253d453c7 CHECK ((char_length(last_commit) <= 64))
)
PARTITION BY RANGE (project_id);

CREATE TABLE p_batched_git_ref_updates_deletions (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    partition_id bigint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    ref text NOT NULL,
    CONSTRAINT check_f322d53b92 CHECK ((char_length(ref) <= 1024))
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_catalog_resource_sync_events (
    id bigint NOT NULL,
    catalog_resource_id bigint NOT NULL,
    project_id bigint NOT NULL,
    partition_id bigint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL
)
PARTITION BY LIST (partition_id);

CREATE TABLE p_ci_finished_build_ch_sync_events (
    build_id bigint NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    build_finished_at timestamp without time zone NOT NULL,
    processed boolean DEFAULT false NOT NULL,
    project_id bigint NOT NULL
)
PARTITION BY LIST (partition);

CREATE TABLE p_ci_finished_pipeline_ch_sync_events (
    pipeline_id bigint NOT NULL,
    project_namespace_id bigint NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    pipeline_finished_at timestamp without time zone NOT NULL,
    processed boolean DEFAULT false NOT NULL
)
PARTITION BY LIST (partition);

CREATE TABLE p_duo_workflows_checkpoints (
    id bigint NOT NULL,
    workflow_id bigint NOT NULL,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint,
    thread_ts text NOT NULL,
    parent_ts text,
    checkpoint jsonb NOT NULL,
    metadata jsonb NOT NULL,
    ui_chat_log jsonb,
    CONSTRAINT check_70d1d05b50 CHECK ((num_nonnulls(namespace_id, project_id) = 1)),
    CONSTRAINT check_b55c120f3f CHECK ((char_length(thread_ts) <= 255)),
    CONSTRAINT check_e63817afa6 CHECK ((char_length(parent_ts) <= 255))
)
PARTITION BY RANGE (created_at);

CREATE TABLE p_generated_ref_commits (
    id bigint NOT NULL,
    merge_request_iid bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    commit_sha bytea NOT NULL
)
PARTITION BY RANGE (project_id);

CREATE SEQUENCE sent_notifications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE p_sent_notifications (
    id bigint DEFAULT nextval('sent_notifications_id_seq'::regclass) NOT NULL,
    project_id bigint,
    noteable_id bigint,
    recipient_id bigint,
    issue_email_participant_id bigint,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    partition integer DEFAULT 1 NOT NULL,
    noteable_type text,
    commit_id text,
    reply_key text NOT NULL,
    in_reply_to_discussion_id text,
    CONSTRAINT check_18a72130c5 CHECK ((char_length(commit_id) <= 255)),
    CONSTRAINT check_21ab645552 CHECK ((char_length(reply_key) <= 255)),
    CONSTRAINT check_4c25a085c6 CHECK ((char_length(noteable_type) <= 255)),
    CONSTRAINT check_7e7dc3845d CHECK ((char_length(in_reply_to_discussion_id) <= 255))
)
PARTITION BY LIST (partition);

CREATE TABLE project_audit_events (
    id bigint DEFAULT nextval('shared_audit_event_id_seq'::regclass) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    author_id bigint NOT NULL,
    target_id bigint,
    event_name text,
    details text,
    ip_address inet,
    author_name text,
    entity_path text,
    target_details text,
    target_type text,
    CONSTRAINT project_audit_events_author_name_check CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT project_audit_events_entity_path_check CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT project_audit_events_event_name_check CHECK ((char_length(event_name) <= 255)),
    CONSTRAINT project_audit_events_target_details_check CHECK ((char_length(target_details) <= 5500)),
    CONSTRAINT project_audit_events_target_type_check CHECK ((char_length(target_type) <= 255))
)
PARTITION BY RANGE (created_at);

CREATE TABLE project_daily_statistics (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    fetch_count integer NOT NULL,
    date date NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE projects_visits (
    id bigint NOT NULL,
    entity_id bigint NOT NULL,
    user_id bigint NOT NULL,
    visited_at timestamp with time zone NOT NULL
)
PARTITION BY RANGE (visited_at);

CREATE TABLE security_findings (
    id bigint NOT NULL,
    scan_id bigint NOT NULL,
    scanner_id bigint NOT NULL,
    severity smallint NOT NULL,
    deduplicated boolean DEFAULT false NOT NULL,
    uuid uuid,
    overridden_uuid uuid,
    partition_number integer DEFAULT 1 NOT NULL,
    finding_data jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_6c2851a8c9 CHECK ((uuid IS NOT NULL))
)
PARTITION BY LIST (partition_number);

CREATE TABLE user_audit_events (
    id bigint DEFAULT nextval('shared_audit_event_id_seq'::regclass) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    author_id bigint NOT NULL,
    target_id bigint,
    event_name text,
    details text,
    ip_address inet,
    author_name text,
    entity_path text,
    target_details text,
    target_type text,
    CONSTRAINT user_audit_events_author_name_check CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT user_audit_events_entity_path_check CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT user_audit_events_event_name_check CHECK ((char_length(event_name) <= 255)),
    CONSTRAINT user_audit_events_target_details_check CHECK ((char_length(target_details) <= 5500)),
    CONSTRAINT user_audit_events_target_type_check CHECK ((char_length(target_type) <= 255))
)
PARTITION BY RANGE (created_at);

CREATE TABLE value_stream_dashboard_counts (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    count bigint NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    metric smallint NOT NULL
)
PARTITION BY RANGE (recorded_at);

CREATE TABLE verification_codes (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 50))
)
PARTITION BY RANGE (created_at);

COMMENT ON TABLE verification_codes IS 'JiHu-specific table';

CREATE TABLE vulnerability_archive_exports (
    id bigint NOT NULL,
    partition_number bigint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone,
    finished_at timestamp with time zone,
    project_id bigint NOT NULL,
    author_id bigint NOT NULL,
    date_range daterange NOT NULL,
    file_store smallint,
    format smallint,
    file text,
    status text,
    CONSTRAINT check_3423276100 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_aada0b0f45 CHECK ((char_length(status) <= 8))
)
PARTITION BY LIST (partition_number);

CREATE TABLE vulnerability_archived_records (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    id bigint NOT NULL,
    date date NOT NULL,
    project_id bigint NOT NULL,
    archive_id bigint NOT NULL,
    vulnerability_identifier bigint NOT NULL,
    data jsonb DEFAULT '{}'::jsonb NOT NULL
)
PARTITION BY RANGE (date);

CREATE TABLE vulnerability_archives (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    archived_records_count integer DEFAULT 0 NOT NULL,
    date date NOT NULL,
    CONSTRAINT chk_rails_6b9e2d707f CHECK ((archived_records_count >= 0))
)
PARTITION BY RANGE (date);

CREATE TABLE web_hook_logs_daily (
    id bigint NOT NULL,
    web_hook_id bigint NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL,
    url_hash text,
    organization_id bigint,
    group_id bigint,
    project_id bigint,
    CONSTRAINT check_df72cb58f5 CHECK ((char_length(url_hash) <= 44))
)
PARTITION BY RANGE (created_at);

CREATE TABLE zoekt_tasks (
    id bigint NOT NULL,
    partition_id bigint DEFAULT 1 NOT NULL,
    zoekt_node_id bigint NOT NULL,
    zoekt_repository_id bigint NOT NULL,
    project_identifier bigint NOT NULL,
    perform_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    task_type smallint NOT NULL,
    retries_left smallint DEFAULT 5 NOT NULL,
    CONSTRAINT c_zoekt_tasks_on_retries_left CHECK (((retries_left > 0) OR ((retries_left = 0) AND (state = 255))))
)
PARTITION BY LIST (partition_id);

CREATE TABLE analytics_cycle_analytics_issue_stage_events (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
)
PARTITION BY HASH (stage_event_hash_id);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_00 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_01 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_02 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_03 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_04 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_05 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_06 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_07 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_08 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_09 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_10 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_11 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_12 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_13 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_14 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_15 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_16 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_17 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_18 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_19 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_20 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_21 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_22 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_23 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_24 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_25 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_26 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_27 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_28 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_29 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_30 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_31 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    weight integer,
    sprint_id bigint,
    duration_in_milliseconds bigint
);

CREATE TABLE analytics_cycle_analytics_merge_request_stage_events (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
)
PARTITION BY HASH (stage_event_hash_id);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_00 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_01 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_02 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_03 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_04 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_05 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_06 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_07 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_08 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_09 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_10 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_11 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_12 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_13 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_14 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_15 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_16 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_17 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_18 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_19 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_20 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_21 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_22 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_23 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_24 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_25 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_26 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_27 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_28 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_29 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_30 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_31 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL,
    duration_in_milliseconds bigint
);

CREATE TABLE issue_search_data (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
)
PARTITION BY HASH (project_id);

CREATE TABLE gitlab_partitions_static.issue_search_data_00 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_01 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_02 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_03 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_04 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_05 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_06 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_07 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_08 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_09 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_10 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_11 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_12 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_13 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_14 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_15 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_16 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_17 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_18 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_19 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_20 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_21 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_22 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_23 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_24 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_25 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_26 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_27 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_28 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_29 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_30 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_31 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_32 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_33 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_34 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_35 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_36 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_37 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_38 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_39 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_40 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_41 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_42 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_43 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_44 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_45 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_46 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_47 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_48 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_49 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_50 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_51 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_52 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_53 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_54 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_55 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_56 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_57 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_58 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_59 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_60 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_61 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_62 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE gitlab_partitions_static.issue_search_data_63 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector,
    namespace_id bigint
);

CREATE TABLE namespace_descendants (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
)
PARTITION BY HASH (namespace_id);

CREATE TABLE gitlab_partitions_static.namespace_descendants_00 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_01 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_02 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_03 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_04 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_05 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_06 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_07 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_08 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_09 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_10 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_11 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_12 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_13 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_14 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_15 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_16 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_17 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_18 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_19 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_20 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_21 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_22 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_23 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_24 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_25 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_26 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_27 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_28 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_29 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_30 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE gitlab_partitions_static.namespace_descendants_31 (
    namespace_id bigint NOT NULL,
    self_and_descendant_group_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    all_project_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    traversal_ids bigint[] DEFAULT ARRAY[]::bigint[] NOT NULL,
    outdated_at timestamp with time zone,
    calculated_at timestamp with time zone,
    all_active_project_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    all_unarchived_project_ids bigint[] DEFAULT '{}'::bigint[],
    self_and_descendant_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    CONSTRAINT check_60ae9ef706 CHECK ((all_unarchived_project_ids IS NOT NULL))
);

CREATE TABLE p_ci_pipeline_iids (
    project_id bigint NOT NULL,
    iid integer NOT NULL
)
PARTITION BY HASH (project_id);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_00 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_01 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_02 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_03 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_04 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_05 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_06 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_07 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_08 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_09 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_10 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_11 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_12 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_13 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_14 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_15 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_16 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_17 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_18 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_19 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_20 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_21 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_22 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_23 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_24 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_25 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_26 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_27 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_28 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_29 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_30 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_31 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_32 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_33 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_34 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_35 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_36 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_37 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_38 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_39 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_40 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_41 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_42 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_43 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_44 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_45 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_46 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_47 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_48 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_49 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_50 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_51 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_52 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_53 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_54 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_55 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_56 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_57 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_58 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_59 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_60 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_61 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_62 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE gitlab_partitions_static.p_ci_pipeline_iids_63 (
    project_id bigint NOT NULL,
    iid integer NOT NULL
);

CREATE TABLE virtual_registries_container_cache_entries (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
)
PARTITION BY HASH (relative_path);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_00 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_01 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_02 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_03 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_04 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_05 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_06 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_07 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_08 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_09 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_10 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_11 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_12 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_13 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_14 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_container_cache_entries_15 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_0bd0ebf413 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_0f76ce8cd8 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_844f00a59d CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_be077e7ebe CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_f776d05cc6 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_cf0e2a2780 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE virtual_registries_packages_maven_cache_entries (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
)
PARTITION BY HASH (relative_path);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_00 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_01 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_02 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_03 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_04 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_05 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_06 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_07 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_08 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_09 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_10 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_11 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_12 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_13 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_14 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_entries_15 (
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    downloaded_at timestamp with time zone,
    CONSTRAINT check_215f531366 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_2a52b4e0fc CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_36391449ea CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_45d3174f8a CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_cc222855d6 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT check_f2ea43b900 CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT check_fd9fc90696 CHECK ((char_length(upstream_etag) <= 255))
);

CREATE SEQUENCE virtual_registries_packages_maven_cache_remote_entries_iid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE virtual_registries_packages_maven_cache_remote_entries (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
)
PARTITION BY HASH (group_id);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_00 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_01 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_02 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_03 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_04 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_05 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_06 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_07 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_08 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_09 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_10 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_11 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_12 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_13 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_14 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_maven_cache_remote_entries_15 (
    iid bigint DEFAULT nextval('virtual_registries_packages_maven_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_sha1 bytea NOT NULL,
    file_md5 bytea,
    relative_path text NOT NULL,
    object_storage_key text NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_3b002d9c70 CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_3c247dc29d CHECK ((char_length(file) <= 1024)),
    CONSTRAINT check_8d84955388 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_f699e9ea8b CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_fa0d05ab4f CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT chk_rails_176c0ec5da CHECK ((octet_length(file_sha1) = 20)),
    CONSTRAINT chk_rails_f9a2aeef13 CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16)))
);

CREATE SEQUENCE virtual_registries_packages_npm_cache_local_entries_iid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE virtual_registries_packages_npm_cache_local_entries (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
)
PARTITION BY HASH (group_id);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_00 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_01 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_02 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_03 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_04 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_05 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_06 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_07 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_08 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_09 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_10 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_11 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_12 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_13 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_14 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_local_entries_15 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_local_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_path text NOT NULL,
    CONSTRAINT check_ea7974004d CHECK ((char_length(relative_path) <= 1024))
);

CREATE SEQUENCE virtual_registries_packages_npm_cache_remote_entries_iid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE virtual_registries_packages_npm_cache_remote_entries (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
)
PARTITION BY HASH (group_id);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_00 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_01 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_02 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_03 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_04 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_05 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_06 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_07 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_08 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_09 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_10 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_11 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_12 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_13 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_14 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE gitlab_partitions_static.virtual_registries_packages_npm_cache_remote_entries_15 (
    iid bigint DEFAULT nextval('virtual_registries_packages_npm_cache_remote_entries_iid_seq'::regclass) NOT NULL,
    group_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    downloads_count bigint DEFAULT 0 NOT NULL,
    upstream_checked_at timestamp with time zone DEFAULT now() NOT NULL,
    downloaded_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer DEFAULT 1 NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_md5 bytea,
    file_sha1 bytea NOT NULL,
    upstream_etag text,
    content_type text DEFAULT 'application/octet-stream'::text NOT NULL,
    relative_path text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_2b69b2b072 CHECK ((char_length(upstream_etag) <= 255)),
    CONSTRAINT check_3220f927bf CHECK ((char_length(relative_path) <= 1024)),
    CONSTRAINT check_7f8673b467 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_8e92860258 CHECK ((char_length(object_storage_key) <= 1024)),
    CONSTRAINT check_9cee66a1e9 CHECK ((char_length(file) <= 1024)),
    CONSTRAINT chk_rails_39aff89dfc CHECK (((file_md5 IS NULL) OR (octet_length(file_md5) = 16))),
    CONSTRAINT chk_rails_c878fc3684 CHECK ((octet_length(file_sha1) = 20))
);

CREATE TABLE work_item_descriptions (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
)
PARTITION BY HASH (namespace_id);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_00 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_01 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_02 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_03 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_04 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_05 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_06 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_07 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_08 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_09 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_10 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_11 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_12 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_13 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_14 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_15 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_16 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_17 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_18 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_19 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_20 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_21 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_22 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_23 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_24 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_25 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_26 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_27 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_28 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_29 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_30 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_31 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_32 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_33 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_34 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_35 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_36 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_37 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_38 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_39 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_40 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_41 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_42 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_43 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_44 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_45 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_46 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_47 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_48 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_49 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_50 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_51 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_52 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_53 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_54 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_55 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_56 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_57 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_58 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_59 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_60 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_61 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_62 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE gitlab_partitions_static.work_item_descriptions_63 (
    work_item_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    last_edited_by_id bigint,
    last_edited_at timestamp without time zone,
    lock_version integer DEFAULT 0,
    cached_markdown_version integer,
    search_vector tsvector,
    description text,
    description_html text
);

CREATE TABLE abuse_events (
    id bigint NOT NULL,
    user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    abuse_report_id bigint,
    source smallint NOT NULL,
    category smallint,
    metadata jsonb,
    organization_id bigint,
    CONSTRAINT check_9b41e64a86 CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE abuse_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE abuse_events_id_seq OWNED BY abuse_events.id;

CREATE TABLE abuse_report_events (
    id bigint NOT NULL,
    abuse_report_id bigint NOT NULL,
    user_id bigint,
    created_at timestamp with time zone NOT NULL,
    action smallint DEFAULT 1 NOT NULL,
    reason smallint,
    comment text,
    organization_id bigint,
    CONSTRAINT check_bb4cd85618 CHECK ((char_length(comment) <= 1024)),
    CONSTRAINT check_ed2aa0210e CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE abuse_report_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE abuse_report_events_id_seq OWNED BY abuse_report_events.id;

CREATE TABLE uploads_9ba88c4165 (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
)
PARTITION BY LIST (model_type);

CREATE TABLE abuse_report_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_2db09e0e37 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE abuse_report_user_mentions (
    id bigint NOT NULL,
    abuse_report_id bigint NOT NULL,
    note_id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    organization_id bigint,
    CONSTRAINT check_f0d6e86b14 CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE abuse_report_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE abuse_report_user_mentions_id_seq OWNED BY abuse_report_user_mentions.id;

CREATE TABLE abuse_reports (
    id bigint NOT NULL,
    reporter_id bigint,
    user_id bigint,
    message text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    message_html text,
    cached_markdown_version integer,
    category smallint DEFAULT 1 NOT NULL,
    reported_from_url text DEFAULT ''::text NOT NULL,
    links_to_spam text[] DEFAULT '{}'::text[] NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    resolved_at timestamp with time zone,
    screenshot text,
    resolved_by_id bigint,
    assignee_id bigint,
    mitigation_steps text,
    evidence jsonb,
    assignee_id_convert_to_bigint bigint,
    id_convert_to_bigint bigint DEFAULT 0 NOT NULL,
    reporter_id_convert_to_bigint bigint,
    resolved_by_id_convert_to_bigint bigint,
    user_id_convert_to_bigint bigint,
    organization_id bigint,
    CONSTRAINT abuse_reports_links_to_spam_length_check CHECK ((cardinality(links_to_spam) <= 20)),
    CONSTRAINT check_1e642c5f94 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_4b0a5120e0 CHECK ((char_length(screenshot) <= 255)),
    CONSTRAINT check_ab1260fa6c CHECK ((char_length(reported_from_url) <= 512)),
    CONSTRAINT check_f3c0947a2d CHECK ((char_length(mitigation_steps) <= 1000)),
    CONSTRAINT check_fc643d4880 CHECK ((reporter_id IS NOT NULL))
);

CREATE SEQUENCE abuse_reports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE abuse_reports_id_seq OWNED BY abuse_reports.id;

CREATE TABLE achievement_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_9dac0178ee CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE achievements (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    avatar text,
    description text,
    CONSTRAINT check_5171b03f22 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_a7a7b84a80 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_e174e93a9e CHECK ((char_length(avatar) <= 255))
);

CREATE SEQUENCE achievements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE achievements_id_seq OWNED BY achievements.id;

CREATE TABLE activation_metrics (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    namespace_id bigint,
    metric smallint NOT NULL
);

CREATE SEQUENCE activation_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE activation_metrics_id_seq OWNED BY activation_metrics.id;

CREATE TABLE activity_pub_releases_subscriptions (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    shared_inbox_url text,
    subscriber_inbox_url text,
    subscriber_url text NOT NULL,
    payload jsonb,
    CONSTRAINT check_0ebf38bcaa CHECK ((char_length(subscriber_inbox_url) <= 1024)),
    CONSTRAINT check_2afd35ba17 CHECK ((char_length(subscriber_url) <= 1024)),
    CONSTRAINT check_61b77ced49 CHECK ((char_length(shared_inbox_url) <= 1024))
);

CREATE SEQUENCE activity_pub_releases_subscriptions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE activity_pub_releases_subscriptions_id_seq OWNED BY activity_pub_releases_subscriptions.id;

CREATE TABLE admin_roles (
    id bigint NOT NULL,
    name text NOT NULL,
    description text,
    permissions jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint,
    CONSTRAINT check_1c6a3bcca1 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_89a2f4f799 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_a8c6d1de58 CHECK ((char_length(description) <= 255))
);

CREATE SEQUENCE admin_roles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE admin_roles_id_seq OWNED BY admin_roles.id;

CREATE TABLE agent_activity_events (
    id bigint NOT NULL,
    agent_id bigint NOT NULL,
    user_id bigint,
    project_id bigint,
    merge_request_id bigint,
    agent_token_id bigint,
    recorded_at timestamp with time zone NOT NULL,
    kind smallint NOT NULL,
    level smallint NOT NULL,
    sha bytea,
    detail text,
    agent_project_id bigint,
    CONSTRAINT check_068205e735 CHECK ((char_length(detail) <= 255)),
    CONSTRAINT check_9e09ffbd0f CHECK ((agent_project_id IS NOT NULL))
);

CREATE SEQUENCE agent_activity_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_activity_events_id_seq OWNED BY agent_activity_events.id;

CREATE TABLE agent_group_authorizations (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_group_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_group_authorizations_id_seq OWNED BY agent_group_authorizations.id;

CREATE TABLE agent_organization_authorizations (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_organization_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_organization_authorizations_id_seq OWNED BY agent_organization_authorizations.id;

CREATE TABLE agent_project_authorizations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_project_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_project_authorizations_id_seq OWNED BY agent_project_authorizations.id;

CREATE TABLE agent_user_access_group_authorizations (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_user_access_group_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_user_access_group_authorizations_id_seq OWNED BY agent_user_access_group_authorizations.id;

CREATE TABLE agent_user_access_project_authorizations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_user_access_project_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_user_access_project_authorizations_id_seq OWNED BY agent_user_access_project_authorizations.id;

CREATE TABLE ai_active_context_collections (
    id bigint NOT NULL,
    name text NOT NULL,
    metadata jsonb DEFAULT '{}'::jsonb NOT NULL,
    number_of_partitions integer DEFAULT 1 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    connection_id bigint NOT NULL,
    options jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_fe84a77f95 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE ai_active_context_collections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_active_context_collections_id_seq OWNED BY ai_active_context_collections.id;

CREATE TABLE ai_active_context_connections (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT false NOT NULL,
    name text NOT NULL,
    prefix text,
    adapter_class text NOT NULL,
    options jsonb NOT NULL,
    CONSTRAINT check_7bb86d51bc CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b1c9c4af95 CHECK ((char_length(adapter_class) <= 255)),
    CONSTRAINT check_ffbd5301a3 CHECK ((char_length(prefix) <= 255))
);

CREATE SEQUENCE ai_active_context_connections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_active_context_connections_id_seq OWNED BY ai_active_context_connections.id;

CREATE TABLE ai_active_context_migrations (
    id bigint NOT NULL,
    connection_id bigint NOT NULL,
    started_at timestamp with time zone,
    completed_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    retries_left smallint NOT NULL,
    version text NOT NULL,
    metadata jsonb DEFAULT '{}'::jsonb NOT NULL,
    error_message text,
    CONSTRAINT c_ai_active_context_migrations_on_retries_left CHECK (((retries_left > 0) OR ((retries_left = 0) AND (status = 255)))),
    CONSTRAINT c_ai_active_context_migrations_version_format CHECK ((version ~ '^[0-9]{14}$'::text)),
    CONSTRAINT check_184ab3430e CHECK ((char_length(error_message) <= 1024)),
    CONSTRAINT check_b2e8a34818 CHECK ((char_length(version) <= 255))
);

CREATE SEQUENCE ai_active_context_migrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_active_context_migrations_id_seq OWNED BY ai_active_context_migrations.id;

CREATE TABLE ai_agent_version_attachments (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    ai_agent_version_id bigint NOT NULL,
    ai_vectorizable_file_id bigint NOT NULL
);

CREATE SEQUENCE ai_agent_version_attachments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_agent_version_attachments_id_seq OWNED BY ai_agent_version_attachments.id;

CREATE TABLE ai_agent_versions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    prompt text NOT NULL,
    model text NOT NULL,
    CONSTRAINT check_8cda7448e9 CHECK ((char_length(model) <= 255)),
    CONSTRAINT check_d7a4fc9834 CHECK ((char_length(prompt) <= 5000))
);

CREATE SEQUENCE ai_agent_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_agent_versions_id_seq OWNED BY ai_agent_versions.id;

CREATE TABLE ai_agents (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_67934c8e85 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE ai_agents_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_agents_id_seq OWNED BY ai_agents.id;

CREATE TABLE ai_catalog_item_consumers (
    id bigint NOT NULL,
    ai_catalog_item_id bigint NOT NULL,
    organization_id bigint,
    group_id bigint,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    locked boolean DEFAULT true NOT NULL,
    pinned_version_prefix text,
    service_account_id bigint,
    parent_item_consumer_id bigint,
    CONSTRAINT check_55026cf703 CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1)),
    CONSTRAINT check_a788d1fdfa CHECK ((char_length(pinned_version_prefix) <= 50))
);

CREATE SEQUENCE ai_catalog_item_consumers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_catalog_item_consumers_id_seq OWNED BY ai_catalog_item_consumers.id;

CREATE TABLE ai_catalog_item_version_dependencies (
    id bigint NOT NULL,
    ai_catalog_item_version_id bigint NOT NULL,
    dependency_id bigint NOT NULL,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE ai_catalog_item_version_dependencies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_catalog_item_version_dependencies_id_seq OWNED BY ai_catalog_item_version_dependencies.id;

CREATE TABLE ai_catalog_item_versions (
    id bigint NOT NULL,
    release_date timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint NOT NULL,
    ai_catalog_item_id bigint NOT NULL,
    schema_version smallint NOT NULL,
    version text NOT NULL,
    definition jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_by_id bigint,
    CONSTRAINT check_8cabb46fa3 CHECK ((char_length(version) <= 50))
);

CREATE SEQUENCE ai_catalog_item_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_catalog_item_versions_id_seq OWNED BY ai_catalog_item_versions.id;

CREATE TABLE ai_catalog_items (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    item_type smallint NOT NULL,
    description text NOT NULL,
    name text NOT NULL,
    public boolean DEFAULT false NOT NULL,
    deleted_at timestamp with time zone,
    latest_version_id bigint,
    latest_released_version_id bigint,
    verification_level smallint DEFAULT 0 NOT NULL,
    identifier text,
    foundational_flow_reference text,
    CONSTRAINT check_5a87fd2753 CHECK ((char_length(identifier) <= 255)),
    CONSTRAINT check_7e02a4805b CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_804e59e032 CHECK ((char_length(foundational_flow_reference) <= 255)),
    CONSTRAINT check_edddd6e1fe CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE ai_catalog_items_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_catalog_items_id_seq OWNED BY ai_catalog_items.id;

CREATE TABLE ai_code_suggestion_events (
    id bigint NOT NULL,
    "timestamp" timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    event smallint NOT NULL,
    namespace_path text,
    payload jsonb,
    CONSTRAINT check_ba9ae3f258 CHECK ((char_length(namespace_path) <= 255))
)
PARTITION BY RANGE ("timestamp");

CREATE SEQUENCE ai_code_suggestion_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_code_suggestion_events_id_seq OWNED BY ai_code_suggestion_events.id;

CREATE TABLE ai_conversation_messages (
    id bigint NOT NULL,
    thread_id bigint NOT NULL,
    agent_version_id bigint,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    role smallint NOT NULL,
    has_feedback boolean DEFAULT false,
    extras jsonb DEFAULT '{}'::jsonb NOT NULL,
    error_details jsonb DEFAULT '{}'::jsonb NOT NULL,
    content text NOT NULL,
    request_xid text,
    message_xid text,
    referer_url text,
    CONSTRAINT check_0fe78937e4 CHECK ((char_length(content) <= 524288)),
    CONSTRAINT check_8daec62ec9 CHECK ((char_length(request_xid) <= 255)),
    CONSTRAINT check_b14b137e02 CHECK ((char_length(message_xid) <= 255)),
    CONSTRAINT check_f36c73d1d9 CHECK ((char_length(referer_url) <= 255))
);

CREATE SEQUENCE ai_conversation_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_conversation_messages_id_seq OWNED BY ai_conversation_messages.id;

CREATE TABLE ai_conversation_threads (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    last_updated_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    conversation_type smallint NOT NULL
);

CREATE SEQUENCE ai_conversation_threads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_conversation_threads_id_seq OWNED BY ai_conversation_threads.id;

CREATE TABLE ai_duo_chat_events (
    id bigint NOT NULL,
    "timestamp" timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    personal_namespace_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    event smallint NOT NULL,
    namespace_path text,
    payload jsonb,
    organization_id bigint,
    CONSTRAINT check_628cdfbf3f CHECK ((char_length(namespace_path) <= 255)),
    CONSTRAINT check_f759f45177 CHECK ((organization_id IS NOT NULL))
)
PARTITION BY RANGE ("timestamp");

CREATE SEQUENCE ai_duo_chat_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_duo_chat_events_id_seq OWNED BY ai_duo_chat_events.id;

CREATE SEQUENCE ai_events_counts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_events_counts_id_seq OWNED BY ai_events_counts.id;

CREATE TABLE ai_feature_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    ai_self_hosted_model_id bigint,
    feature smallint NOT NULL,
    provider smallint NOT NULL
);

CREATE SEQUENCE ai_feature_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_feature_settings_id_seq OWNED BY ai_feature_settings.id;

CREATE TABLE ai_flow_triggers (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint,
    config_path text,
    description text NOT NULL,
    event_types smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    ai_catalog_item_consumer_id bigint,
    CONSTRAINT check_87b77d9d54 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_f3a5b0bd6e CHECK ((char_length(config_path) <= 255))
);

CREATE SEQUENCE ai_flow_triggers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_flow_triggers_id_seq OWNED BY ai_flow_triggers.id;

CREATE TABLE ai_instance_accessible_entity_rules (
    id bigint NOT NULL,
    through_namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    accessible_entity text NOT NULL,
    CONSTRAINT check_e8f4eb707f CHECK ((char_length(accessible_entity) <= 255))
);

CREATE SEQUENCE ai_instance_accessible_entity_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_instance_accessible_entity_rules_id_seq OWNED BY ai_instance_accessible_entity_rules.id;

CREATE TABLE ai_namespace_feature_access_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    root_namespace_id bigint NOT NULL,
    through_namespace_id bigint NOT NULL,
    accessible_entity text NOT NULL,
    CONSTRAINT check_ca828b88ca CHECK ((char_length(accessible_entity) <= 255))
);

CREATE SEQUENCE ai_namespace_feature_access_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_namespace_feature_access_rules_id_seq OWNED BY ai_namespace_feature_access_rules.id;

CREATE TABLE ai_namespace_feature_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    feature smallint NOT NULL,
    offered_model_ref text,
    offered_model_name text,
    CONSTRAINT check_14e81e87bc CHECK ((char_length(offered_model_ref) <= 255)),
    CONSTRAINT check_c850e74656 CHECK ((char_length(offered_model_name) <= 255))
);

CREATE SEQUENCE ai_namespace_feature_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_namespace_feature_settings_id_seq OWNED BY ai_namespace_feature_settings.id;

CREATE TABLE ai_self_hosted_models (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    model smallint NOT NULL,
    endpoint text NOT NULL,
    name text NOT NULL,
    encrypted_api_token bytea,
    encrypted_api_token_iv bytea,
    identifier text,
    CONSTRAINT check_a28005edb2 CHECK ((char_length(endpoint) <= 2048)),
    CONSTRAINT check_cccb37e0de CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_d1e593d04d CHECK ((char_length(identifier) <= 255))
);

CREATE SEQUENCE ai_self_hosted_models_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_self_hosted_models_id_seq OWNED BY ai_self_hosted_models.id;

CREATE TABLE ai_settings (
    id bigint NOT NULL,
    ai_gateway_url text,
    singleton boolean DEFAULT true NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    amazon_q_oauth_application_id bigint,
    amazon_q_service_account_user_id bigint,
    amazon_q_ready boolean DEFAULT false NOT NULL,
    amazon_q_role_arn text,
    duo_workflow_service_account_user_id bigint,
    duo_workflow_oauth_application_id bigint,
    enabled_instance_verbose_ai_logs boolean,
    duo_core_features_enabled boolean,
    duo_agent_platform_service_url text,
    duo_agent_platform_request_count integer DEFAULT 0 NOT NULL,
    foundational_agents_default_enabled boolean DEFAULT true,
    ai_gateway_timeout_seconds integer DEFAULT 60,
    minimum_access_level_execute smallint,
    minimum_access_level_manage smallint,
    minimum_access_level_enable_on_projects smallint,
    minimum_access_level_execute_async smallint,
    feature_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_3cf9826589 CHECK ((char_length(ai_gateway_url) <= 2048)),
    CONSTRAINT check_900d7a89b3 CHECK ((char_length(duo_agent_platform_service_url) <= 2048)),
    CONSTRAINT check_a02bd8868c CHECK ((char_length(amazon_q_role_arn) <= 2048)),
    CONSTRAINT check_ai_settings_feature_settings_is_hash CHECK ((jsonb_typeof(feature_settings) = 'object'::text)),
    CONSTRAINT check_singleton CHECK ((singleton IS TRUE))
);

COMMENT ON COLUMN ai_settings.singleton IS 'Always true, used for singleton enforcement';

CREATE SEQUENCE ai_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_settings_id_seq OWNED BY ai_settings.id;

CREATE TABLE ai_testing_terms_acceptances (
    created_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    user_email text NOT NULL,
    CONSTRAINT check_5efe98894e CHECK ((char_length(user_email) <= 255))
);

CREATE TABLE ai_troubleshoot_job_events (
    id bigint NOT NULL,
    "timestamp" timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    job_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    event smallint NOT NULL,
    namespace_path text,
    payload jsonb,
    CONSTRAINT check_29d6dbc329 CHECK ((char_length(namespace_path) <= 255))
)
PARTITION BY RANGE ("timestamp");

CREATE SEQUENCE ai_troubleshoot_job_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_troubleshoot_job_events_id_seq OWNED BY ai_troubleshoot_job_events.id;

CREATE SEQUENCE ai_usage_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_usage_events_id_seq OWNED BY ai_usage_events.id;

CREATE TABLE ai_user_metrics (
    user_id bigint NOT NULL,
    last_duo_activity_on date NOT NULL
);

CREATE TABLE ai_vectorizable_file_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_73db12226b CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE ai_vectorizable_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_c2ad8df0bf CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_fc6abf5b01 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE ai_vectorizable_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ai_vectorizable_files_id_seq OWNED BY ai_vectorizable_files.id;

CREATE TABLE alert_management_alert_assignees (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_f3efe02c81 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE alert_management_alert_assignees_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alert_assignees_id_seq OWNED BY alert_management_alert_assignees.id;

CREATE TABLE alert_management_alert_metric_image_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_49e9003aa6 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE alert_management_alert_metric_images (
    id bigint NOT NULL,
    alert_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint,
    file text NOT NULL,
    url text,
    url_text text,
    project_id bigint,
    CONSTRAINT check_2587666252 CHECK ((char_length(url_text) <= 128)),
    CONSTRAINT check_4d811d9007 CHECK ((char_length(url) <= 255)),
    CONSTRAINT check_70fafae519 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_7e9ef22adc CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE alert_management_alert_metric_images_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alert_metric_images_id_seq OWNED BY alert_management_alert_metric_images.id;

CREATE TABLE alert_management_alert_user_mentions (
    id bigint NOT NULL,
    alert_management_alert_id bigint NOT NULL,
    note_id bigint,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    project_id bigint,
    CONSTRAINT check_5094a9a20a CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE alert_management_alert_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alert_user_mentions_id_seq OWNED BY alert_management_alert_user_mentions.id;

CREATE TABLE alert_management_alerts (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone NOT NULL,
    ended_at timestamp with time zone,
    events integer DEFAULT 1 NOT NULL,
    iid integer NOT NULL,
    severity smallint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    fingerprint bytea,
    issue_id bigint,
    project_id bigint NOT NULL,
    title text NOT NULL,
    description text,
    service text,
    monitoring_tool text,
    hosts text[] DEFAULT '{}'::text[] NOT NULL,
    payload jsonb DEFAULT '{}'::jsonb NOT NULL,
    environment_id bigint,
    domain smallint DEFAULT 0,
    CONSTRAINT check_2df3e2fdc1 CHECK ((char_length(monitoring_tool) <= 100)),
    CONSTRAINT check_5e9e57cadb CHECK ((char_length(description) <= 1000)),
    CONSTRAINT check_bac14dddde CHECK ((char_length(service) <= 100)),
    CONSTRAINT check_d1d1c2d14c CHECK ((char_length(title) <= 200))
);

CREATE SEQUENCE alert_management_alerts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alerts_id_seq OWNED BY alert_management_alerts.id;

CREATE TABLE alert_management_http_integrations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    active boolean DEFAULT false NOT NULL,
    encrypted_token text NOT NULL,
    encrypted_token_iv text NOT NULL,
    endpoint_identifier text NOT NULL,
    name text NOT NULL,
    payload_example jsonb DEFAULT '{}'::jsonb NOT NULL,
    payload_attribute_mapping jsonb DEFAULT '{}'::jsonb NOT NULL,
    type_identifier smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_286943b636 CHECK ((char_length(encrypted_token_iv) <= 255)),
    CONSTRAINT check_392143ccf4 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e270820180 CHECK ((char_length(endpoint_identifier) <= 255)),
    CONSTRAINT check_f68577c4af CHECK ((char_length(encrypted_token) <= 255))
);

CREATE SEQUENCE alert_management_http_integrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_http_integrations_id_seq OWNED BY alert_management_http_integrations.id;

CREATE TABLE allowed_email_domains (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    domain character varying(255) NOT NULL
);

CREATE SEQUENCE allowed_email_domains_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE allowed_email_domains_id_seq OWNED BY allowed_email_domains.id;

CREATE TABLE analytics_cycle_analytics_aggregations (
    group_id bigint NOT NULL,
    incremental_runtimes_in_seconds integer[] DEFAULT '{}'::integer[] NOT NULL,
    incremental_processed_records integer[] DEFAULT '{}'::integer[] NOT NULL,
    last_incremental_issues_id bigint,
    last_incremental_merge_requests_id bigint,
    last_incremental_run_at timestamp with time zone,
    last_incremental_issues_updated_at timestamp with time zone,
    last_incremental_merge_requests_updated_at timestamp with time zone,
    last_full_run_at timestamp with time zone,
    last_consistency_check_updated_at timestamp with time zone,
    enabled boolean DEFAULT true NOT NULL,
    full_runtimes_in_seconds integer[] DEFAULT '{}'::integer[] NOT NULL,
    full_processed_records integer[] DEFAULT '{}'::integer[] NOT NULL,
    last_full_merge_requests_updated_at timestamp with time zone,
    last_full_issues_updated_at timestamp with time zone,
    last_full_issues_id bigint,
    last_full_merge_requests_id bigint,
    last_consistency_check_issues_stage_event_hash_id bigint,
    last_consistency_check_issues_start_event_timestamp timestamp with time zone,
    last_consistency_check_issues_end_event_timestamp timestamp with time zone,
    last_consistency_check_issues_issuable_id bigint,
    last_consistency_check_merge_requests_stage_event_hash_id bigint,
    last_consistency_check_merge_requests_start_event_timestamp timestamp with time zone,
    last_consistency_check_merge_requests_end_event_timestamp timestamp with time zone,
    last_consistency_check_merge_requests_issuable_id bigint,
    CONSTRAINT chk_rails_1ef688e577 CHECK ((cardinality(incremental_runtimes_in_seconds) <= 10)),
    CONSTRAINT chk_rails_e16bf3913a CHECK ((cardinality(incremental_processed_records) <= 10)),
    CONSTRAINT full_processed_records_size CHECK ((cardinality(full_processed_records) <= 10)),
    CONSTRAINT full_runtimes_in_seconds_size CHECK ((cardinality(full_runtimes_in_seconds) <= 10))
);

CREATE TABLE analytics_cycle_analytics_group_stages (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_position integer,
    start_event_identifier integer NOT NULL,
    end_event_identifier integer NOT NULL,
    group_id bigint NOT NULL,
    start_event_label_id bigint,
    end_event_label_id bigint,
    hidden boolean DEFAULT false NOT NULL,
    custom boolean DEFAULT true NOT NULL,
    name character varying(255) NOT NULL,
    group_value_stream_id bigint NOT NULL,
    stage_event_hash_id bigint,
    CONSTRAINT check_e6bd4271b5 CHECK ((stage_event_hash_id IS NOT NULL))
);

CREATE SEQUENCE analytics_cycle_analytics_group_stages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_group_stages_id_seq OWNED BY analytics_cycle_analytics_group_stages.id;

CREATE TABLE analytics_cycle_analytics_group_value_streams (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_bc1ed5f1f7 CHECK ((char_length(name) <= 100))
);

CREATE SEQUENCE analytics_cycle_analytics_group_value_streams_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_group_value_streams_id_seq OWNED BY analytics_cycle_analytics_group_value_streams.id;

CREATE TABLE analytics_cycle_analytics_stage_aggregations (
    group_id bigint NOT NULL,
    stage_id bigint NOT NULL,
    runtimes_in_seconds integer[] DEFAULT '{}'::integer[] NOT NULL,
    processed_records integer[] DEFAULT '{}'::integer[] NOT NULL,
    last_issues_id bigint,
    last_merge_requests_id bigint,
    last_run_at timestamp with time zone,
    last_issues_updated_at timestamp with time zone,
    last_merge_requests_updated_at timestamp with time zone,
    last_completed_at timestamp with time zone,
    enabled boolean DEFAULT true NOT NULL,
    CONSTRAINT chk_rails_50ee1c451c CHECK ((cardinality(runtimes_in_seconds) <= 10)),
    CONSTRAINT chk_rails_7586ea045d CHECK ((cardinality(processed_records) <= 10))
);

CREATE TABLE analytics_cycle_analytics_stage_event_hashes (
    id bigint NOT NULL,
    hash_sha256 bytea,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE analytics_cycle_analytics_stage_event_hashes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_stage_event_hashes_id_seq OWNED BY analytics_cycle_analytics_stage_event_hashes.id;

CREATE TABLE analytics_cycle_analytics_value_stream_settings (
    value_stream_id bigint NOT NULL,
    project_ids_filter bigint[] DEFAULT '{}'::bigint[],
    namespace_id bigint NOT NULL,
    CONSTRAINT project_ids_filter_array_check CHECK (((cardinality(project_ids_filter) <= 100) AND (array_position(project_ids_filter, NULL::bigint) IS NULL)))
);

CREATE TABLE analytics_dashboards_pointers (
    id bigint NOT NULL,
    namespace_id bigint,
    project_id bigint,
    target_project_id bigint NOT NULL,
    CONSTRAINT chk_analytics_dashboards_pointers_project_or_namespace CHECK (((project_id IS NULL) <> (namespace_id IS NULL)))
);

CREATE SEQUENCE analytics_dashboards_pointers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_dashboards_pointers_id_seq OWNED BY analytics_dashboards_pointers.id;

CREATE TABLE analytics_devops_adoption_segments (
    id bigint NOT NULL,
    last_recorded_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint,
    display_namespace_id bigint,
    CONSTRAINT check_fc420e89ee CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE analytics_devops_adoption_segments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_devops_adoption_segments_id_seq OWNED BY analytics_devops_adoption_segments.id;

CREATE TABLE analytics_devops_adoption_snapshots (
    id bigint NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    issue_opened boolean NOT NULL,
    merge_request_opened boolean NOT NULL,
    merge_request_approved boolean NOT NULL,
    runner_configured boolean NOT NULL,
    pipeline_succeeded boolean NOT NULL,
    deploy_succeeded boolean NOT NULL,
    end_time timestamp with time zone NOT NULL,
    total_projects_count integer,
    code_owners_used_count integer,
    namespace_id bigint,
    sast_enabled_count integer,
    dast_enabled_count integer,
    dependency_scanning_enabled_count integer,
    coverage_fuzzing_enabled_count integer,
    vulnerability_management_used_count integer,
    CONSTRAINT check_3f472de131 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE analytics_devops_adoption_snapshots_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_devops_adoption_snapshots_id_seq OWNED BY analytics_devops_adoption_snapshots.id;

CREATE TABLE analytics_language_trend_repository_languages (
    file_count integer DEFAULT 0 NOT NULL,
    programming_language_id bigint NOT NULL,
    project_id bigint NOT NULL,
    loc integer DEFAULT 0 NOT NULL,
    bytes integer DEFAULT 0 NOT NULL,
    percentage smallint DEFAULT 0 NOT NULL,
    snapshot_date date NOT NULL
);

CREATE TABLE analytics_usage_trends_measurements (
    id bigint NOT NULL,
    count bigint NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    identifier smallint NOT NULL
);

CREATE SEQUENCE analytics_usage_trends_measurements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_usage_trends_measurements_id_seq OWNED BY analytics_usage_trends_measurements.id;

CREATE TABLE analyzer_namespace_statuses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    analyzer_type smallint NOT NULL,
    success bigint DEFAULT 0 NOT NULL,
    failure bigint DEFAULT 0 NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL
);

CREATE SEQUENCE analyzer_namespace_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analyzer_namespace_statuses_id_seq OWNED BY analyzer_namespace_statuses.id;

CREATE TABLE analyzer_project_statuses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    analyzer_type smallint NOT NULL,
    status smallint NOT NULL,
    last_call timestamp with time zone NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    build_id bigint,
    archived boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE analyzer_project_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analyzer_project_statuses_id_seq OWNED BY analyzer_project_statuses.id;

CREATE TABLE appearance_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_84e287699b CHECK ((num_nonnulls(namespace_id, organization_id, project_id) = 0)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE appearances (
    id bigint NOT NULL,
    title character varying NOT NULL,
    description text NOT NULL,
    logo character varying,
    updated_by integer,
    header_logo character varying,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    description_html text,
    cached_markdown_version integer,
    new_project_guidelines text,
    new_project_guidelines_html text,
    header_message text,
    header_message_html text,
    footer_message text,
    footer_message_html text,
    message_background_color text,
    message_font_color text,
    favicon character varying,
    email_header_and_footer_enabled boolean DEFAULT false NOT NULL,
    profile_image_guidelines text,
    profile_image_guidelines_html text,
    pwa_short_name text,
    pwa_icon text,
    pwa_name text,
    pwa_description text,
    member_guidelines text,
    member_guidelines_html text,
    CONSTRAINT appearances_profile_image_guidelines CHECK ((char_length(profile_image_guidelines) <= 4096)),
    CONSTRAINT check_13b2165eca CHECK ((char_length(pwa_name) <= 255)),
    CONSTRAINT check_50e9b69ab6 CHECK ((char_length(member_guidelines) <= 4096)),
    CONSTRAINT check_5c3fd63577 CHECK ((char_length(pwa_short_name) <= 255)),
    CONSTRAINT check_5e0e6f24ed CHECK ((char_length(pwa_description) <= 2048)),
    CONSTRAINT check_5e5b7ac344 CHECK ((char_length(pwa_icon) <= 1024))
);

CREATE SEQUENCE appearances_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE appearances_id_seq OWNED BY appearances.id;

CREATE TABLE application_setting_terms (
    id bigint NOT NULL,
    cached_markdown_version integer,
    terms text NOT NULL,
    terms_html text
);

CREATE SEQUENCE application_setting_terms_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE application_setting_terms_id_seq OWNED BY application_setting_terms.id;

CREATE TABLE application_settings (
    id bigint NOT NULL,
    default_projects_limit integer,
    signup_enabled boolean,
    gravatar_enabled boolean,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    home_page_url character varying,
    default_branch_protection integer DEFAULT 2,
    restricted_visibility_levels text,
    version_check_enabled boolean DEFAULT true,
    max_attachment_size integer DEFAULT 100 NOT NULL,
    default_project_visibility integer DEFAULT 0 NOT NULL,
    default_snippet_visibility integer DEFAULT 0 NOT NULL,
    user_oauth_applications boolean DEFAULT true,
    after_sign_out_path character varying,
    session_expire_delay integer DEFAULT 10080 NOT NULL,
    import_sources text,
    help_page_text text,
    require_admin_two_factor_authentication boolean DEFAULT false NOT NULL,
    shared_runners_enabled boolean DEFAULT true NOT NULL,
    max_artifacts_size integer DEFAULT 100 NOT NULL,
    max_pages_size integer DEFAULT 100 NOT NULL,
    require_two_factor_authentication boolean DEFAULT false,
    two_factor_grace_period integer DEFAULT 48,
    metrics_enabled boolean DEFAULT false,
    metrics_host character varying DEFAULT 'localhost'::character varying,
    metrics_pool_size integer DEFAULT 16,
    metrics_timeout integer DEFAULT 10,
    metrics_method_call_threshold integer DEFAULT 10,
    recaptcha_enabled boolean DEFAULT false,
    metrics_port integer DEFAULT 8089,
    akismet_enabled boolean DEFAULT false,
    metrics_sample_interval integer DEFAULT 15,
    email_author_in_body boolean DEFAULT false,
    default_group_visibility integer,
    repository_checks_enabled boolean DEFAULT false,
    shared_runners_text text,
    metrics_packet_size integer DEFAULT 1,
    disabled_oauth_sign_in_sources text,
    health_check_access_token character varying,
    container_registry_token_expire_delay integer DEFAULT 5,
    after_sign_up_text text,
    user_default_external boolean DEFAULT false NOT NULL,
    repository_storages character varying DEFAULT 'default'::character varying,
    enabled_git_access_protocol character varying,
    usage_ping_enabled boolean DEFAULT true NOT NULL,
    help_page_text_html text,
    shared_runners_text_html text,
    after_sign_up_text_html text,
    rsa_key_restriction integer DEFAULT 0 NOT NULL,
    dsa_key_restriction integer DEFAULT '-1'::integer NOT NULL,
    ecdsa_key_restriction integer DEFAULT 0 NOT NULL,
    ed25519_key_restriction integer DEFAULT 0 NOT NULL,
    housekeeping_enabled boolean DEFAULT true NOT NULL,
    housekeeping_bitmaps_enabled boolean DEFAULT true NOT NULL,
    housekeeping_incremental_repack_period integer DEFAULT 10 NOT NULL,
    housekeeping_full_repack_period integer DEFAULT 50 NOT NULL,
    housekeeping_gc_period integer DEFAULT 200 NOT NULL,
    html_emails_enabled boolean DEFAULT true,
    plantuml_url character varying,
    plantuml_enabled boolean,
    shared_runners_minutes integer DEFAULT 0 NOT NULL,
    repository_size_limit bigint DEFAULT 0,
    terminal_max_session_time integer DEFAULT 0 NOT NULL,
    unique_ips_limit_per_user integer,
    unique_ips_limit_time_window integer,
    unique_ips_limit_enabled boolean DEFAULT false NOT NULL,
    default_artifacts_expire_in character varying DEFAULT '0'::character varying NOT NULL,
    elasticsearch_url character varying DEFAULT 'http://localhost:9200'::character varying,
    geo_status_timeout integer DEFAULT 10,
    uuid character varying,
    polling_interval_multiplier numeric DEFAULT 1.0 NOT NULL,
    cached_markdown_version integer,
    check_namespace_plan boolean DEFAULT false NOT NULL,
    mirror_max_delay integer DEFAULT 300 NOT NULL,
    mirror_max_capacity integer DEFAULT 100 NOT NULL,
    mirror_capacity_threshold integer DEFAULT 50 NOT NULL,
    prometheus_metrics_enabled boolean DEFAULT true NOT NULL,
    authorized_keys_enabled boolean DEFAULT true NOT NULL,
    help_page_hide_commercial_content boolean DEFAULT false,
    help_page_support_url character varying,
    slack_app_enabled boolean DEFAULT false,
    slack_app_id character varying,
    performance_bar_allowed_group_id bigint,
    allow_group_owners_to_manage_ldap boolean DEFAULT true NOT NULL,
    hashed_storage_enabled boolean DEFAULT true NOT NULL,
    project_export_enabled boolean DEFAULT true NOT NULL,
    auto_devops_enabled boolean DEFAULT true NOT NULL,
    throttle_unauthenticated_enabled boolean DEFAULT false NOT NULL,
    throttle_unauthenticated_requests_per_period integer DEFAULT 3600 NOT NULL,
    throttle_unauthenticated_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_api_requests_per_period integer DEFAULT 7200 NOT NULL,
    throttle_authenticated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_web_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_web_requests_per_period integer DEFAULT 7200 NOT NULL,
    throttle_authenticated_web_period_in_seconds integer DEFAULT 3600 NOT NULL,
    gitaly_timeout_default integer DEFAULT 55 NOT NULL,
    gitaly_timeout_medium integer DEFAULT 30 NOT NULL,
    gitaly_timeout_fast integer DEFAULT 10 NOT NULL,
    mirror_available boolean DEFAULT true NOT NULL,
    password_authentication_enabled_for_web boolean,
    password_authentication_enabled_for_git boolean DEFAULT true NOT NULL,
    auto_devops_domain character varying,
    external_authorization_service_enabled boolean DEFAULT false NOT NULL,
    external_authorization_service_url character varying,
    external_authorization_service_default_label character varying,
    pages_domain_verification_enabled boolean DEFAULT true NOT NULL,
    user_default_internal_regex character varying,
    external_authorization_service_timeout double precision DEFAULT 0.5,
    external_auth_client_cert text,
    encrypted_external_auth_client_key text,
    encrypted_external_auth_client_key_iv character varying,
    encrypted_external_auth_client_key_pass character varying,
    encrypted_external_auth_client_key_pass_iv character varying,
    email_additional_text character varying,
    enforce_terms boolean DEFAULT false,
    file_template_project_id bigint,
    pseudonymizer_enabled boolean DEFAULT false NOT NULL,
    hide_third_party_offers boolean DEFAULT false NOT NULL,
    snowplow_enabled boolean DEFAULT false NOT NULL,
    snowplow_collector_hostname character varying,
    snowplow_cookie_domain character varying,
    user_show_add_ssh_key_message boolean DEFAULT true NOT NULL,
    custom_project_templates_group_id bigint,
    usage_stats_set_by_user_id bigint,
    receive_max_input_size integer,
    diff_max_patch_bytes integer DEFAULT 204800 NOT NULL,
    archive_builds_in_seconds integer,
    commit_email_hostname character varying,
    protected_ci_variables boolean DEFAULT true NOT NULL,
    runners_registration_token_encrypted character varying,
    local_markdown_version integer DEFAULT 0 NOT NULL,
    first_day_of_week integer DEFAULT 0 NOT NULL,
    default_project_creation integer DEFAULT 2 NOT NULL,
    lets_encrypt_notification_email character varying,
    lets_encrypt_terms_of_service_accepted boolean DEFAULT false NOT NULL,
    geo_node_allowed_ips character varying DEFAULT '0.0.0.0/0, ::/0'::character varying,
    encrypted_lets_encrypt_private_key text,
    encrypted_lets_encrypt_private_key_iv text,
    dns_rebinding_protection_enabled boolean DEFAULT true NOT NULL,
    default_project_deletion_protection boolean DEFAULT false NOT NULL,
    grafana_enabled boolean DEFAULT false NOT NULL,
    lock_memberships_to_ldap boolean DEFAULT false NOT NULL,
    time_tracking_limit_to_hours boolean DEFAULT false NOT NULL,
    grafana_url character varying DEFAULT '/-/grafana'::character varying NOT NULL,
    login_recaptcha_protection_enabled boolean DEFAULT false NOT NULL,
    outbound_local_requests_whitelist character varying(255)[] DEFAULT '{}'::character varying[] NOT NULL,
    raw_blob_request_limit integer DEFAULT 300 NOT NULL,
    allow_local_requests_from_web_hooks_and_services boolean DEFAULT false NOT NULL,
    allow_local_requests_from_system_hooks boolean DEFAULT true NOT NULL,
    asset_proxy_enabled boolean DEFAULT false NOT NULL,
    asset_proxy_url character varying,
    encrypted_asset_proxy_secret_key text,
    encrypted_asset_proxy_secret_key_iv character varying,
    static_objects_external_storage_url character varying(255),
    max_personal_access_token_lifetime integer,
    throttle_protected_paths_enabled boolean DEFAULT false NOT NULL,
    throttle_protected_paths_requests_per_period integer DEFAULT 10 NOT NULL,
    throttle_protected_paths_period_in_seconds integer DEFAULT 60 NOT NULL,
    protected_paths character varying(255)[] DEFAULT '{/users/password,/users/sign_in,/api/v3/session.json,/api/v3/session,/api/v4/session.json,/api/v4/session,/users,/users/confirmation,/unsubscribes/,/import/github/personal_access_token,/admin/session}'::character varying[],
    throttle_incident_management_notification_enabled boolean DEFAULT false NOT NULL,
    throttle_incident_management_notification_period_in_seconds integer DEFAULT 3600,
    throttle_incident_management_notification_per_period integer DEFAULT 3600,
    push_event_hooks_limit integer DEFAULT 3 NOT NULL,
    push_event_activities_limit integer DEFAULT 3 NOT NULL,
    custom_http_clone_url_root character varying(511),
    deletion_adjourned_period integer DEFAULT 30 NOT NULL,
    license_trial_ends_on date,
    eks_integration_enabled boolean DEFAULT false NOT NULL,
    eks_account_id character varying(128),
    eks_access_key_id character varying(128),
    encrypted_eks_secret_access_key_iv character varying(255),
    encrypted_eks_secret_access_key text,
    snowplow_app_id character varying,
    productivity_analytics_start_date timestamp with time zone,
    default_ci_config_path character varying(255),
    sourcegraph_enabled boolean DEFAULT false NOT NULL,
    sourcegraph_url character varying(255),
    sourcegraph_public_only boolean DEFAULT true NOT NULL,
    snippet_size_limit bigint DEFAULT 52428800 NOT NULL,
    minimum_password_length integer DEFAULT 8 NOT NULL,
    encrypted_akismet_api_key text,
    encrypted_akismet_api_key_iv character varying(255),
    encrypted_elasticsearch_aws_secret_access_key text,
    encrypted_elasticsearch_aws_secret_access_key_iv character varying(255),
    encrypted_recaptcha_private_key text,
    encrypted_recaptcha_private_key_iv character varying(255),
    encrypted_recaptcha_site_key text,
    encrypted_recaptcha_site_key_iv character varying(255),
    encrypted_slack_app_secret text,
    encrypted_slack_app_secret_iv character varying(255),
    encrypted_slack_app_verification_token text,
    encrypted_slack_app_verification_token_iv character varying(255),
    force_pages_access_control boolean DEFAULT false NOT NULL,
    updating_name_disabled_for_users boolean DEFAULT false NOT NULL,
    disable_overriding_approvers_per_merge_request boolean DEFAULT false NOT NULL,
    prevent_merge_requests_author_approval boolean DEFAULT false NOT NULL,
    prevent_merge_requests_committers_approval boolean DEFAULT false NOT NULL,
    email_restrictions_enabled boolean DEFAULT false NOT NULL,
    email_restrictions text,
    container_expiration_policies_enable_historic_entries boolean DEFAULT false NOT NULL,
    issues_create_limit integer DEFAULT 0 NOT NULL,
    push_rule_id bigint,
    group_owners_can_manage_default_branch_protection boolean DEFAULT true NOT NULL,
    container_registry_vendor text DEFAULT ''::text NOT NULL,
    container_registry_version text DEFAULT ''::text NOT NULL,
    container_registry_features text[] DEFAULT '{}'::text[] NOT NULL,
    spam_check_endpoint_url text,
    spam_check_endpoint_enabled boolean DEFAULT false NOT NULL,
    repository_storages_weighted jsonb DEFAULT '{}'::jsonb NOT NULL,
    max_import_size integer DEFAULT 0 NOT NULL,
    compliance_frameworks smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    notify_on_unknown_sign_in boolean DEFAULT true NOT NULL,
    default_branch_name text,
    project_import_limit integer DEFAULT 6 NOT NULL,
    project_export_limit integer DEFAULT 6 NOT NULL,
    project_download_export_limit integer DEFAULT 1 NOT NULL,
    group_import_limit integer DEFAULT 6 NOT NULL,
    group_export_limit integer DEFAULT 6 NOT NULL,
    group_download_export_limit integer DEFAULT 1 NOT NULL,
    maintenance_mode boolean DEFAULT false NOT NULL,
    maintenance_mode_message text,
    wiki_page_max_content_bytes bigint DEFAULT 5242880 NOT NULL,
    enforce_namespace_storage_limit boolean DEFAULT false NOT NULL,
    container_registry_delete_tags_service_timeout integer DEFAULT 250 NOT NULL,
    kroki_url character varying,
    kroki_enabled boolean,
    gitpod_enabled boolean DEFAULT false NOT NULL,
    gitpod_url text DEFAULT 'https://gitpod.io/'::text,
    abuse_notification_email character varying,
    require_admin_approval_after_user_signup boolean DEFAULT true NOT NULL,
    help_page_documentation_base_url text,
    automatic_purchased_storage_allocation boolean DEFAULT false NOT NULL,
    encrypted_ci_jwt_signing_key text,
    encrypted_ci_jwt_signing_key_iv text,
    container_registry_expiration_policies_worker_capacity integer DEFAULT 4 NOT NULL,
    secret_detection_token_revocation_enabled boolean DEFAULT false NOT NULL,
    secret_detection_token_revocation_url text,
    encrypted_secret_detection_token_revocation_token text,
    encrypted_secret_detection_token_revocation_token_iv text,
    domain_denylist_enabled boolean DEFAULT false,
    domain_denylist text,
    domain_allowlist text,
    new_user_signups_cap integer,
    encrypted_cloud_license_auth_token text,
    encrypted_cloud_license_auth_token_iv text,
    secret_detection_revocation_token_types_url text,
    disable_feed_token boolean DEFAULT false NOT NULL,
    personal_access_token_prefix text DEFAULT 'glpat-'::text,
    rate_limiting_response_text text,
    invisible_captcha_enabled boolean DEFAULT false NOT NULL,
    container_registry_cleanup_tags_service_max_list_size integer DEFAULT 200 NOT NULL,
    git_two_factor_session_expiry integer DEFAULT 15 NOT NULL,
    keep_latest_artifact boolean DEFAULT true NOT NULL,
    notes_create_limit integer DEFAULT 300 NOT NULL,
    notes_create_limit_allowlist text[] DEFAULT '{}'::text[] NOT NULL,
    kroki_formats jsonb DEFAULT '{}'::jsonb NOT NULL,
    asset_proxy_whitelist text,
    admin_mode boolean DEFAULT false NOT NULL,
    external_pipeline_validation_service_timeout integer,
    encrypted_external_pipeline_validation_service_token text,
    encrypted_external_pipeline_validation_service_token_iv text,
    external_pipeline_validation_service_url text,
    throttle_unauthenticated_packages_api_requests_per_period integer DEFAULT 800 NOT NULL,
    throttle_unauthenticated_packages_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_authenticated_packages_api_requests_per_period integer DEFAULT 1000 NOT NULL,
    throttle_authenticated_packages_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_unauthenticated_packages_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_packages_api_enabled boolean DEFAULT false NOT NULL,
    deactivate_dormant_users boolean DEFAULT false NOT NULL,
    whats_new_variant smallint DEFAULT 0,
    encrypted_spam_check_api_key bytea,
    encrypted_spam_check_api_key_iv bytea,
    floc_enabled boolean DEFAULT false NOT NULL,
    encrypted_elasticsearch_password bytea,
    encrypted_elasticsearch_password_iv bytea,
    diff_max_lines integer DEFAULT 50000 NOT NULL,
    diff_max_files integer DEFAULT 1000 NOT NULL,
    valid_runner_registrars character varying[] DEFAULT '{project,group}'::character varying[],
    encrypted_mailgun_signing_key bytea,
    encrypted_mailgun_signing_key_iv bytea,
    mailgun_events_enabled boolean DEFAULT false NOT NULL,
    usage_ping_features_enabled boolean DEFAULT false NOT NULL,
    encrypted_customers_dot_jwt_signing_key bytea,
    encrypted_customers_dot_jwt_signing_key_iv bytea,
    throttle_unauthenticated_files_api_requests_per_period integer DEFAULT 125 NOT NULL,
    throttle_unauthenticated_files_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_authenticated_files_api_requests_per_period integer DEFAULT 500 NOT NULL,
    throttle_authenticated_files_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_unauthenticated_files_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_files_api_enabled boolean DEFAULT false NOT NULL,
    max_yaml_size_bytes bigint DEFAULT 2097152 NOT NULL,
    max_yaml_depth integer DEFAULT 100 NOT NULL,
    throttle_authenticated_git_lfs_requests_per_period integer DEFAULT 1000 NOT NULL,
    throttle_authenticated_git_lfs_period_in_seconds integer DEFAULT 60 NOT NULL,
    throttle_authenticated_git_lfs_enabled boolean DEFAULT false NOT NULL,
    user_deactivation_emails_enabled boolean DEFAULT true NOT NULL,
    throttle_unauthenticated_api_enabled boolean DEFAULT false NOT NULL,
    throttle_unauthenticated_api_requests_per_period integer DEFAULT 3600 NOT NULL,
    throttle_unauthenticated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    jobs_per_stage_page_size integer DEFAULT 200 NOT NULL,
    sidekiq_job_limiter_mode smallint DEFAULT 1 NOT NULL,
    sidekiq_job_limiter_compression_threshold_bytes integer DEFAULT 100000 NOT NULL,
    sidekiq_job_limiter_limit_bytes integer DEFAULT 0 NOT NULL,
    suggest_pipeline_enabled boolean DEFAULT true NOT NULL,
    throttle_unauthenticated_deprecated_api_requests_per_period integer DEFAULT 1800 NOT NULL,
    throttle_unauthenticated_deprecated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_unauthenticated_deprecated_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_deprecated_api_requests_per_period integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_deprecated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_deprecated_api_enabled boolean DEFAULT false NOT NULL,
    dependency_proxy_ttl_group_policy_worker_capacity smallint DEFAULT 2 NOT NULL,
    content_validation_endpoint_url text,
    encrypted_content_validation_api_key bytea,
    encrypted_content_validation_api_key_iv bytea,
    content_validation_endpoint_enabled boolean DEFAULT false NOT NULL,
    sentry_enabled boolean DEFAULT false NOT NULL,
    sentry_dsn text,
    sentry_clientside_dsn text,
    sentry_environment text,
    max_ssh_key_lifetime integer,
    static_objects_external_storage_auth_token_encrypted text,
    future_subscriptions jsonb DEFAULT '[]'::jsonb NOT NULL,
    runner_token_expiration_interval integer,
    group_runner_token_expiration_interval integer,
    project_runner_token_expiration_interval integer,
    ecdsa_sk_key_restriction integer DEFAULT 0 NOT NULL,
    ed25519_sk_key_restriction integer DEFAULT 0 NOT NULL,
    users_get_by_id_limit integer DEFAULT 300 NOT NULL,
    users_get_by_id_limit_allowlist text[] DEFAULT '{}'::text[] NOT NULL,
    container_registry_expiration_policies_caching boolean DEFAULT true NOT NULL,
    search_rate_limit integer DEFAULT 300 NOT NULL,
    search_rate_limit_unauthenticated integer DEFAULT 100 NOT NULL,
    encrypted_database_grafana_api_key bytea,
    encrypted_database_grafana_api_key_iv bytea,
    database_grafana_api_url text,
    database_grafana_tag text,
    public_runner_releases_url text DEFAULT 'https://gitlab.com/api/v4/projects/gitlab-org%2Fgitlab-runner/releases'::text NOT NULL,
    password_uppercase_required boolean DEFAULT false NOT NULL,
    password_lowercase_required boolean DEFAULT false NOT NULL,
    password_number_required boolean DEFAULT false NOT NULL,
    password_symbol_required boolean DEFAULT false NOT NULL,
    encrypted_arkose_labs_public_api_key bytea,
    encrypted_arkose_labs_public_api_key_iv bytea,
    encrypted_arkose_labs_private_api_key bytea,
    encrypted_arkose_labs_private_api_key_iv bytea,
    delete_inactive_projects boolean DEFAULT false NOT NULL,
    inactive_projects_delete_after_months integer DEFAULT 2 NOT NULL,
    inactive_projects_min_size_mb integer DEFAULT 0 NOT NULL,
    inactive_projects_send_warning_email_after_months integer DEFAULT 1 NOT NULL,
    arkose_labs_namespace text DEFAULT 'client'::text NOT NULL,
    max_export_size integer DEFAULT 0,
    encrypted_slack_app_signing_secret bytea,
    encrypted_slack_app_signing_secret_iv bytea,
    pipeline_limit_per_project_user_sha integer DEFAULT 0 NOT NULL,
    dingtalk_integration_enabled boolean DEFAULT false NOT NULL,
    encrypted_dingtalk_corpid bytea,
    encrypted_dingtalk_corpid_iv bytea,
    encrypted_dingtalk_app_key bytea,
    encrypted_dingtalk_app_key_iv bytea,
    encrypted_dingtalk_app_secret bytea,
    encrypted_dingtalk_app_secret_iv bytea,
    jira_connect_application_key text,
    globally_allowed_ips text DEFAULT ''::text NOT NULL,
    license_usage_data_exported boolean DEFAULT false NOT NULL,
    phone_verification_code_enabled boolean DEFAULT false NOT NULL,
    max_number_of_repository_downloads smallint DEFAULT 0 NOT NULL,
    max_number_of_repository_downloads_within_time_period integer DEFAULT 0 NOT NULL,
    feishu_integration_enabled boolean DEFAULT false NOT NULL,
    encrypted_feishu_app_key bytea,
    encrypted_feishu_app_key_iv bytea,
    encrypted_feishu_app_secret bytea,
    encrypted_feishu_app_secret_iv bytea,
    error_tracking_enabled boolean DEFAULT false NOT NULL,
    error_tracking_api_url text,
    git_rate_limit_users_allowlist text[] DEFAULT '{}'::text[] NOT NULL,
    error_tracking_access_token_encrypted text,
    invitation_flow_enforcement boolean DEFAULT false NOT NULL,
    deactivate_dormant_users_period integer DEFAULT 90 NOT NULL,
    auto_ban_user_on_excessive_projects_download boolean DEFAULT false NOT NULL,
    max_pages_custom_domains_per_project integer DEFAULT 0 NOT NULL,
    cube_api_base_url text,
    encrypted_cube_api_key bytea,
    encrypted_cube_api_key_iv bytea,
    dashboard_limit_enabled boolean DEFAULT false NOT NULL,
    dashboard_limit integer DEFAULT 0 NOT NULL,
    can_create_group boolean DEFAULT true NOT NULL,
    jira_connect_proxy_url text,
    password_expiration_enabled boolean DEFAULT false NOT NULL,
    password_expires_in_days integer DEFAULT 90 NOT NULL,
    password_expires_notice_before_days integer DEFAULT 7 NOT NULL,
    product_analytics_enabled boolean DEFAULT false NOT NULL,
    email_confirmation_setting smallint DEFAULT 0,
    disable_admin_oauth_scopes boolean DEFAULT false NOT NULL,
    default_preferred_language text DEFAULT 'en'::text NOT NULL,
    disable_download_button boolean DEFAULT false NOT NULL,
    encrypted_telesign_customer_xid bytea,
    encrypted_telesign_customer_xid_iv bytea,
    encrypted_telesign_api_key bytea,
    encrypted_telesign_api_key_iv bytea,
    disable_personal_access_tokens boolean DEFAULT false NOT NULL,
    max_terraform_state_size_bytes integer DEFAULT 0 NOT NULL,
    bulk_import_enabled boolean DEFAULT false NOT NULL,
    allow_runner_registration_token boolean DEFAULT true NOT NULL,
    user_defaults_to_private_profile boolean DEFAULT false NOT NULL,
    allow_possible_spam boolean DEFAULT false NOT NULL,
    default_syntax_highlighting_theme integer DEFAULT 1 NOT NULL,
    search_max_shard_size_gb integer DEFAULT 50 NOT NULL,
    search_max_docs_denominator integer DEFAULT 5000000 NOT NULL,
    search_min_docs_before_rollover integer DEFAULT 100000 NOT NULL,
    deactivation_email_additional_text text,
    jira_connect_public_key_storage_enabled boolean DEFAULT false NOT NULL,
    git_rate_limit_users_alertlist integer[] DEFAULT '{}'::integer[] NOT NULL,
    allow_deploy_tokens_and_keys_with_external_authn boolean DEFAULT false NOT NULL,
    security_policy_global_group_approvers_enabled boolean DEFAULT true NOT NULL,
    projects_api_rate_limit_unauthenticated integer DEFAULT 400 NOT NULL,
    deny_all_requests_except_allowed boolean DEFAULT false NOT NULL,
    product_analytics_data_collector_host text,
    lock_memberships_to_saml boolean DEFAULT false NOT NULL,
    gitlab_dedicated_instance boolean DEFAULT false NOT NULL,
    update_runner_versions_enabled boolean DEFAULT true NOT NULL,
    database_max_running_batched_background_migrations integer DEFAULT 2 NOT NULL,
    encrypted_product_analytics_configurator_connection_string bytea,
    encrypted_product_analytics_configurator_connection_string_iv bytea,
    silent_mode_enabled boolean DEFAULT false NOT NULL,
    package_metadata_purl_types smallint[] DEFAULT '{1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,17}'::smallint[],
    ci_max_includes integer DEFAULT 150 NOT NULL,
    remember_me_enabled boolean DEFAULT true NOT NULL,
    diagramsnet_enabled boolean DEFAULT true NOT NULL,
    diagramsnet_url text DEFAULT 'https://embed.diagrams.net'::text,
    allow_account_deletion boolean DEFAULT true NOT NULL,
    wiki_asciidoc_allow_uri_includes boolean DEFAULT false NOT NULL,
    namespace_aggregation_schedule_lease_duration_in_seconds integer DEFAULT 300 NOT NULL,
    container_registry_data_repair_detail_worker_max_concurrency integer DEFAULT 2 NOT NULL,
    vertex_ai_host text,
    vertex_ai_project text,
    delete_unconfirmed_users boolean DEFAULT false NOT NULL,
    unconfirmed_users_delete_after_days integer DEFAULT 7 NOT NULL,
    default_branch_protection_defaults jsonb DEFAULT '{}'::jsonb NOT NULL,
    gitlab_shell_operation_limit integer DEFAULT 600,
    protected_paths_for_get_request text[] DEFAULT '{}'::text[] NOT NULL,
    namespace_storage_forks_cost_factor double precision DEFAULT 1.0 NOT NULL,
    bulk_import_max_download_file_size bigint DEFAULT 5120 NOT NULL,
    max_import_remote_file_size bigint DEFAULT 10240 NOT NULL,
    max_decompressed_archive_size integer DEFAULT 25600 NOT NULL,
    sentry_clientside_traces_sample_rate double precision DEFAULT 0.0 NOT NULL,
    prometheus_alert_db_indicators_settings jsonb,
    ci_max_total_yaml_size_bytes integer DEFAULT 314572800 NOT NULL,
    decompress_archive_file_timeout integer DEFAULT 210 NOT NULL,
    search_rate_limit_allowlist text[] DEFAULT '{}'::text[] NOT NULL,
    snowplow_database_collector_hostname text,
    container_registry_db_enabled boolean DEFAULT false NOT NULL,
    failed_login_attempts_unlock_period_in_minutes integer,
    max_login_attempts integer,
    project_jobs_api_rate_limit integer DEFAULT 600 NOT NULL,
    math_rendering_limits_enabled boolean DEFAULT true NOT NULL,
    service_access_tokens_expiration_enforced boolean DEFAULT true NOT NULL,
    make_profile_private boolean DEFAULT true NOT NULL,
    enable_artifact_external_redirect_warning_page boolean DEFAULT true NOT NULL,
    allow_project_creation_for_guest_and_below boolean DEFAULT true NOT NULL,
    update_namespace_name_rate_limit smallint DEFAULT 120 NOT NULL,
    can_create_organization boolean DEFAULT true NOT NULL,
    bulk_import_concurrent_pipeline_batch_limit smallint DEFAULT 25 NOT NULL,
    web_ide_oauth_application_id bigint,
    instance_level_ai_beta_features_enabled boolean DEFAULT false NOT NULL,
    security_txt_content text,
    encrypted_arkose_labs_data_exchange_key bytea,
    encrypted_arkose_labs_data_exchange_key_iv bytea,
    rate_limits jsonb DEFAULT '{}'::jsonb NOT NULL,
    enable_member_promotion_management boolean DEFAULT false NOT NULL,
    lock_math_rendering_limits_enabled boolean DEFAULT false NOT NULL,
    security_approval_policies_limit integer DEFAULT 5 NOT NULL,
    encrypted_arkose_labs_client_xid bytea,
    encrypted_arkose_labs_client_xid_iv bytea,
    encrypted_arkose_labs_client_secret bytea,
    encrypted_arkose_labs_client_secret_iv bytea,
    duo_features_enabled boolean DEFAULT true NOT NULL,
    lock_duo_features_enabled boolean DEFAULT false NOT NULL,
    asciidoc_max_includes smallint DEFAULT 32 NOT NULL,
    clickhouse jsonb DEFAULT '{}'::jsonb NOT NULL,
    include_optional_metrics_in_service_ping boolean DEFAULT true NOT NULL,
    zoekt_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    service_ping_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    package_registry jsonb DEFAULT '{}'::jsonb NOT NULL,
    rate_limits_unauthenticated_git_http jsonb DEFAULT '{}'::jsonb NOT NULL,
    importers jsonb DEFAULT '{}'::jsonb NOT NULL,
    code_creation jsonb DEFAULT '{}'::jsonb NOT NULL,
    code_suggestions_api_rate_limit integer DEFAULT 60 NOT NULL,
    ai_action_api_rate_limit integer DEFAULT 160 NOT NULL,
    require_personal_access_token_expiry boolean DEFAULT true NOT NULL,
    duo_workflow jsonb DEFAULT '{}'::jsonb,
    max_artifacts_content_include_size integer DEFAULT 5242880 NOT NULL,
    max_number_of_vulnerabilities_per_project integer,
    cluster_agents jsonb DEFAULT '{}'::jsonb NOT NULL,
    observability_backend_ssl_verification_enabled boolean DEFAULT true NOT NULL,
    pages jsonb DEFAULT '{}'::jsonb NOT NULL,
    security_policies jsonb DEFAULT '{}'::jsonb NOT NULL,
    spp_repository_pipeline_access boolean DEFAULT false NOT NULL,
    lock_spp_repository_pipeline_access boolean DEFAULT false NOT NULL,
    required_instance_ci_template text,
    enforce_ci_inbound_job_token_scope_enabled boolean DEFAULT true NOT NULL,
    allow_top_level_group_owners_to_create_service_accounts boolean DEFAULT false NOT NULL,
    sign_in_restrictions jsonb DEFAULT '{}'::jsonb NOT NULL,
    transactional_emails jsonb DEFAULT '{}'::jsonb NOT NULL,
    identity_verification_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    integrations jsonb DEFAULT '{}'::jsonb NOT NULL,
    user_seat_management jsonb DEFAULT '{}'::jsonb NOT NULL,
    secret_detection_service_url text DEFAULT ''::text NOT NULL,
    encrypted_secret_detection_service_auth_token bytea,
    encrypted_secret_detection_service_auth_token_iv bytea,
    resource_usage_limits jsonb DEFAULT '{}'::jsonb NOT NULL,
    show_migrate_from_jenkins_banner boolean DEFAULT true NOT NULL,
    encrypted_ci_job_token_signing_key bytea,
    encrypted_ci_job_token_signing_key_iv bytea,
    elasticsearch jsonb DEFAULT '{}'::jsonb NOT NULL,
    oauth_provider jsonb DEFAULT '{}'::jsonb NOT NULL,
    observability_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    search jsonb DEFAULT '{}'::jsonb NOT NULL,
    anti_abuse_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    secret_push_protection_available boolean DEFAULT false,
    vscode_extension_marketplace jsonb DEFAULT '{}'::jsonb NOT NULL,
    token_prefixes jsonb DEFAULT '{}'::jsonb NOT NULL,
    ci_cd_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    database_reindexing jsonb DEFAULT '{}'::jsonb NOT NULL,
    duo_chat jsonb DEFAULT '{}'::jsonb NOT NULL,
    group_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    model_prompt_cache_enabled boolean DEFAULT true NOT NULL,
    lock_model_prompt_cache_enabled boolean DEFAULT false NOT NULL,
    response_limits jsonb DEFAULT '{}'::jsonb NOT NULL,
    web_based_commit_signing_enabled boolean DEFAULT false NOT NULL,
    lock_web_based_commit_signing_enabled boolean DEFAULT false NOT NULL,
    tmp_asset_proxy_secret_key jsonb,
    editor_extensions jsonb DEFAULT '{}'::jsonb NOT NULL,
    security_and_compliance_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    sdrs_url text,
    default_profile_preferences jsonb DEFAULT '{}'::jsonb NOT NULL,
    sdrs_enabled boolean DEFAULT false NOT NULL,
    sdrs_jwt_signing_key jsonb,
    resource_access_tokens_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    auto_duo_code_review_enabled boolean DEFAULT false NOT NULL,
    lock_auto_duo_code_review_enabled boolean DEFAULT false NOT NULL,
    workspaces_oauth_application_id bigint,
    usage_ping_generation_enabled boolean DEFAULT true NOT NULL,
    duo_remote_flows_enabled boolean DEFAULT true NOT NULL,
    lock_duo_remote_flows_enabled boolean DEFAULT false NOT NULL,
    terraform_state_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    namespace_deletion_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    duo_foundational_flows_enabled boolean DEFAULT true NOT NULL,
    lock_duo_foundational_flows_enabled boolean DEFAULT false NOT NULL,
    duo_sast_fp_detection_enabled boolean DEFAULT true NOT NULL,
    lock_duo_sast_fp_detection_enabled boolean DEFAULT false NOT NULL,
    iframe_rendering_enabled boolean DEFAULT false NOT NULL,
    iframe_rendering_allowlist text,
    database_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    usage_billing jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT app_settings_container_reg_cleanup_tags_max_list_size_positive CHECK ((container_registry_cleanup_tags_service_max_list_size >= 0)),
    CONSTRAINT app_settings_dep_proxy_ttl_policies_worker_capacity_positive CHECK ((dependency_proxy_ttl_group_policy_worker_capacity >= 0)),
    CONSTRAINT app_settings_ext_pipeline_validation_service_url_text_limit CHECK ((char_length(external_pipeline_validation_service_url) <= 255)),
    CONSTRAINT app_settings_failed_login_attempts_unlock_period_positive CHECK ((failed_login_attempts_unlock_period_in_minutes > 0)),
    CONSTRAINT app_settings_git_rate_limit_users_alertlist_max_usernames CHECK ((cardinality(git_rate_limit_users_alertlist) <= 100)),
    CONSTRAINT app_settings_git_rate_limit_users_allowlist_max_usernames CHECK ((cardinality(git_rate_limit_users_allowlist) <= 100)),
    CONSTRAINT app_settings_max_decompressed_archive_size_positive CHECK ((max_decompressed_archive_size >= 0)),
    CONSTRAINT app_settings_max_login_attempts_positive CHECK ((max_login_attempts > 0)),
    CONSTRAINT app_settings_max_pages_custom_domains_per_project_check CHECK ((max_pages_custom_domains_per_project >= 0)),
    CONSTRAINT app_settings_max_terraform_state_size_bytes_check CHECK ((max_terraform_state_size_bytes >= 0)),
    CONSTRAINT app_settings_protected_paths_max_length CHECK ((cardinality(protected_paths_for_get_request) <= 100)),
    CONSTRAINT app_settings_registry_exp_policies_worker_capacity_positive CHECK ((container_registry_expiration_policies_worker_capacity >= 0)),
    CONSTRAINT app_settings_registry_repair_worker_max_concurrency_positive CHECK ((container_registry_data_repair_detail_worker_max_concurrency >= 0)),
    CONSTRAINT app_settings_yaml_max_depth_positive CHECK ((max_yaml_depth > 0)),
    CONSTRAINT app_settings_yaml_max_size_positive CHECK ((max_yaml_size_bytes > 0)),
    CONSTRAINT application_settings_oauth_provider_settings_hash CHECK ((jsonb_typeof(oauth_provider) = 'object'::text)),
    CONSTRAINT check_0542340619 CHECK ((char_length(diagramsnet_url) <= 2048)),
    CONSTRAINT check_12f01f1dcd CHECK ((char_length(vertex_ai_project) <= 255)),
    CONSTRAINT check_17d9558205 CHECK ((char_length((kroki_url)::text) <= 1024)),
    CONSTRAINT check_2b820eaac3 CHECK ((char_length(database_grafana_tag) <= 255)),
    CONSTRAINT check_2dba05b802 CHECK ((char_length(gitpod_url) <= 255)),
    CONSTRAINT check_32710817e9 CHECK ((char_length(static_objects_external_storage_auth_token_encrypted) <= 255)),
    CONSTRAINT check_3455368420 CHECK ((char_length(database_grafana_api_url) <= 255)),
    CONSTRAINT check_3b22213b72 CHECK ((char_length(snowplow_database_collector_hostname) <= 255)),
    CONSTRAINT check_3def0f1829 CHECK ((char_length(sentry_clientside_dsn) <= 255)),
    CONSTRAINT check_4847426287 CHECK ((char_length(jira_connect_proxy_url) <= 255)),
    CONSTRAINT check_492cc1354d CHECK ((char_length(error_tracking_api_url) <= 255)),
    CONSTRAINT check_4f8b811780 CHECK ((char_length(sentry_dsn) <= 255)),
    CONSTRAINT check_51700b31b5 CHECK ((char_length(default_branch_name) <= 255)),
    CONSTRAINT check_5688c70478 CHECK ((char_length(error_tracking_access_token_encrypted) <= 255)),
    CONSTRAINT check_56fb3d74a1 CHECK ((char_length(sdrs_url) <= 255)),
    CONSTRAINT check_57123c9593 CHECK ((char_length(help_page_documentation_base_url) <= 255)),
    CONSTRAINT check_5a84c3ffdc CHECK ((char_length(content_validation_endpoint_url) <= 255)),
    CONSTRAINT check_5bcba483c4 CHECK ((char_length(sentry_environment) <= 255)),
    CONSTRAINT check_718b4458ae CHECK ((char_length(personal_access_token_prefix) <= 20)),
    CONSTRAINT check_7227fad848 CHECK ((char_length(rate_limiting_response_text) <= 255)),
    CONSTRAINT check_72c984b2a5 CHECK ((char_length(product_analytics_data_collector_host) <= 255)),
    CONSTRAINT check_734cc9407a CHECK ((char_length(globally_allowed_ips) <= 255)),
    CONSTRAINT check_7ccfe2764a CHECK ((char_length(arkose_labs_namespace) <= 255)),
    CONSTRAINT check_85a39b68ff CHECK ((char_length(encrypted_ci_jwt_signing_key_iv) <= 255)),
    CONSTRAINT check_8dca35398a CHECK ((char_length(public_runner_releases_url) <= 255)),
    CONSTRAINT check_8e7df605a1 CHECK ((char_length(cube_api_base_url) <= 512)),
    CONSTRAINT check_987903e806 CHECK ((char_length(iframe_rendering_allowlist) <= 5000)),
    CONSTRAINT check_9a719834eb CHECK ((char_length(secret_detection_token_revocation_url) <= 255)),
    CONSTRAINT check_9c6c447a13 CHECK ((char_length(maintenance_mode_message) <= 255)),
    CONSTRAINT check_a5704163cc CHECK ((char_length(secret_detection_revocation_token_types_url) <= 255)),
    CONSTRAINT check_ae53cf7f82 CHECK ((char_length(vertex_ai_host) <= 255)),
    CONSTRAINT check_anti_abuse_settings_is_hash CHECK ((jsonb_typeof(anti_abuse_settings) = 'object'::text)),
    CONSTRAINT check_app_settings_namespace_storage_forks_cost_factor_range CHECK (((namespace_storage_forks_cost_factor >= (0)::double precision) AND (namespace_storage_forks_cost_factor <= (1)::double precision))),
    CONSTRAINT check_app_settings_sentry_clientside_traces_sample_rate_range CHECK (((sentry_clientside_traces_sample_rate >= (0)::double precision) AND (sentry_clientside_traces_sample_rate <= (1)::double precision))),
    CONSTRAINT check_application_settings_ci_cd_settings_is_hash CHECK ((jsonb_typeof(ci_cd_settings) = 'object'::text)),
    CONSTRAINT check_application_settings_clickhouse_is_hash CHECK ((jsonb_typeof(clickhouse) = 'object'::text)),
    CONSTRAINT check_application_settings_cluster_agents_is_hash CHECK ((jsonb_typeof(cluster_agents) = 'object'::text)),
    CONSTRAINT check_application_settings_code_creation_is_hash CHECK ((jsonb_typeof(code_creation) = 'object'::text)),
    CONSTRAINT check_application_settings_database_reindexing_is_hash CHECK ((jsonb_typeof(database_reindexing) = 'object'::text)),
    CONSTRAINT check_application_settings_database_settings_is_hash CHECK ((jsonb_typeof(database_settings) = 'object'::text)),
    CONSTRAINT check_application_settings_duo_chat_is_hash CHECK ((jsonb_typeof(duo_chat) = 'object'::text)),
    CONSTRAINT check_application_settings_duo_workflow_is_hash CHECK ((jsonb_typeof(duo_workflow) = 'object'::text)),
    CONSTRAINT check_application_settings_editor_extensions_is_hash CHECK ((jsonb_typeof(editor_extensions) = 'object'::text)),
    CONSTRAINT check_application_settings_elasticsearch_is_hash CHECK ((jsonb_typeof(elasticsearch) = 'object'::text)),
    CONSTRAINT check_application_settings_group_settings_is_hash CHECK ((jsonb_typeof(group_settings) = 'object'::text)),
    CONSTRAINT check_application_settings_importers_is_hash CHECK ((jsonb_typeof(importers) = 'object'::text)),
    CONSTRAINT check_application_settings_integrations_is_hash CHECK ((jsonb_typeof(integrations) = 'object'::text)),
    CONSTRAINT check_application_settings_namespace_deletion_settings_is_hash CHECK ((jsonb_typeof(namespace_deletion_settings) = 'object'::text)),
    CONSTRAINT check_application_settings_o11y_settings_is_hash CHECK ((jsonb_typeof(observability_settings) = 'object'::text)),
    CONSTRAINT check_application_settings_package_registry_is_hash CHECK ((jsonb_typeof(package_registry) = 'object'::text)),
    CONSTRAINT check_application_settings_rate_limits_is_hash CHECK ((jsonb_typeof(rate_limits) = 'object'::text)),
    CONSTRAINT check_application_settings_rate_limits_unauth_git_http_is_hash CHECK ((jsonb_typeof(rate_limits_unauthenticated_git_http) = 'object'::text)),
    CONSTRAINT check_application_settings_security_policies_is_hash CHECK ((jsonb_typeof(security_policies) = 'object'::text)),
    CONSTRAINT check_application_settings_service_ping_settings_is_hash CHECK ((jsonb_typeof(service_ping_settings) = 'object'::text)),
    CONSTRAINT check_application_settings_sign_in_restrictions_is_hash CHECK ((jsonb_typeof(sign_in_restrictions) = 'object'::text)),
    CONSTRAINT check_application_settings_token_prefixes_is_hash CHECK ((jsonb_typeof(token_prefixes) = 'object'::text)),
    CONSTRAINT check_application_settings_transactional_emails_is_hash CHECK ((jsonb_typeof(transactional_emails) = 'object'::text)),
    CONSTRAINT check_application_settings_usage_billing_is_hash CHECK ((jsonb_typeof(usage_billing) = 'object'::text)),
    CONSTRAINT check_application_settings_vscode_extension_marketplace_is_hash CHECK ((jsonb_typeof(vscode_extension_marketplace) = 'object'::text)),
    CONSTRAINT check_b8c74ea5b3 CHECK ((char_length(deactivation_email_additional_text) <= 1000)),
    CONSTRAINT check_babd774f3c CHECK ((char_length(secret_detection_service_url) <= 255)),
    CONSTRAINT check_be6ab41dcc CHECK ((secret_push_protection_available IS NOT NULL)),
    CONSTRAINT check_bf5157a366 CHECK ((char_length(required_instance_ci_template) <= 1024)),
    CONSTRAINT check_cdfbd99405 CHECK ((char_length(security_txt_content) <= 2048)),
    CONSTRAINT check_d03919528d CHECK ((char_length(container_registry_vendor) <= 255)),
    CONSTRAINT check_d820146492 CHECK ((char_length(spam_check_endpoint_url) <= 255)),
    CONSTRAINT check_e2692d7523 CHECK ((char_length(default_preferred_language) <= 32)),
    CONSTRAINT check_e2dd6e290a CHECK ((char_length(jira_connect_application_key) <= 255)),
    CONSTRAINT check_e5aba18f02 CHECK ((char_length(container_registry_version) <= 255)),
    CONSTRAINT check_ef6176834f CHECK ((char_length(encrypted_cloud_license_auth_token_iv) <= 255)),
    CONSTRAINT check_identity_verification_settings_is_hash CHECK ((jsonb_typeof(identity_verification_settings) = 'object'::text)),
    CONSTRAINT check_security_and_compliance_settings_is_hash CHECK ((jsonb_typeof(security_and_compliance_settings) = 'object'::text)),
    CONSTRAINT check_terraform_state_settings_is_hash CHECK ((jsonb_typeof(terraform_state_settings) = 'object'::text))
);

COMMENT ON COLUMN application_settings.content_validation_endpoint_url IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_content_validation_api_key IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_content_validation_api_key_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.content_validation_endpoint_enabled IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.dingtalk_integration_enabled IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_dingtalk_corpid IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_dingtalk_corpid_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_dingtalk_app_key IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_dingtalk_app_key_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_dingtalk_app_secret IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_dingtalk_app_secret_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.phone_verification_code_enabled IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.feishu_integration_enabled IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_feishu_app_key IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_feishu_app_key_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_feishu_app_secret IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_feishu_app_secret_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.password_expiration_enabled IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.password_expires_in_days IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.password_expires_notice_before_days IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.disable_download_button IS 'JiHu-specific column';

CREATE SEQUENCE application_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE application_settings_id_seq OWNED BY application_settings.id;

CREATE TABLE approval_group_rules (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    approvals_required smallint DEFAULT 0 NOT NULL,
    report_type smallint,
    rule_type smallint DEFAULT 1 NOT NULL,
    security_orchestration_policy_configuration_id bigint,
    scan_result_policy_id bigint,
    name text NOT NULL,
    applies_to_all_protected_branches boolean DEFAULT false NOT NULL,
    approval_policy_rule_id bigint,
    CONSTRAINT check_25d42add43 CHECK ((char_length(name) <= 255))
);

CREATE TABLE approval_group_rules_groups (
    id bigint NOT NULL,
    approval_group_rule_id bigint NOT NULL,
    group_id bigint NOT NULL
);

CREATE SEQUENCE approval_group_rules_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_group_rules_groups_id_seq OWNED BY approval_group_rules_groups.id;

CREATE SEQUENCE approval_group_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_group_rules_id_seq OWNED BY approval_group_rules.id;

CREATE TABLE approval_group_rules_protected_branches (
    id bigint NOT NULL,
    approval_group_rule_id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    group_id bigint,
    CONSTRAINT check_b14ec67f68 CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE approval_group_rules_protected_branches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_group_rules_protected_branches_id_seq OWNED BY approval_group_rules_protected_branches.id;

CREATE TABLE approval_group_rules_users (
    id bigint NOT NULL,
    approval_group_rule_id bigint NOT NULL,
    user_id bigint NOT NULL,
    group_id bigint,
    CONSTRAINT check_6db3034f1c CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE approval_group_rules_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_group_rules_users_id_seq OWNED BY approval_group_rules_users.id;

CREATE TABLE approval_merge_request_rule_sources (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    approval_project_rule_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_f82666a937 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE approval_merge_request_rule_sources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rule_sources_id_seq OWNED BY approval_merge_request_rule_sources.id;

CREATE TABLE approval_merge_request_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    merge_request_id bigint NOT NULL,
    approvals_required smallint DEFAULT 0 NOT NULL,
    name character varying NOT NULL,
    rule_type smallint DEFAULT 1 NOT NULL,
    report_type smallint,
    section text,
    modified_from_project_rule boolean DEFAULT false NOT NULL,
    orchestration_policy_idx smallint,
    vulnerabilities_allowed smallint DEFAULT 0 NOT NULL,
    scanners text[] DEFAULT '{}'::text[] NOT NULL,
    severity_levels text[] DEFAULT '{}'::text[] NOT NULL,
    vulnerability_states text[] DEFAULT '{new_needs_triage,new_dismissed}'::text[] NOT NULL,
    security_orchestration_policy_configuration_id bigint,
    scan_result_policy_id bigint,
    applicable_post_merge boolean,
    project_id bigint,
    approval_policy_rule_id bigint,
    role_approvers integer[] DEFAULT '{}'::integer[] NOT NULL,
    approval_policy_action_idx smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_6fca5928b2 CHECK ((char_length(section) <= 255)),
    CONSTRAINT check_90caab37e0 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_approval_m_r_rules_allowed_role_approvers_valid_entries CHECK (((role_approvers = '{}'::integer[]) OR (role_approvers <@ ARRAY[20, 30, 40, 50, 60])))
);

CREATE TABLE approval_merge_request_rules_approved_approvers (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_4e73655ce3 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE approval_merge_request_rules_approved_approvers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_approved_approvers_id_seq OWNED BY approval_merge_request_rules_approved_approvers.id;

CREATE TABLE approval_merge_request_rules_groups (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    group_id bigint NOT NULL
);

CREATE SEQUENCE approval_merge_request_rules_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_groups_id_seq OWNED BY approval_merge_request_rules_groups.id;

CREATE SEQUENCE approval_merge_request_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_id_seq OWNED BY approval_merge_request_rules.id;

CREATE TABLE approval_merge_request_rules_users (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_eca70345f1 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE approval_merge_request_rules_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_users_id_seq OWNED BY approval_merge_request_rules_users.id;

CREATE TABLE approval_policy_merge_request_bypass_events (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    security_policy_id bigint NOT NULL,
    user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    reason text NOT NULL,
    CONSTRAINT check_3169f0d109 CHECK (((length(TRIM(BOTH FROM reason)) >= 1) AND (length(TRIM(BOTH FROM reason)) <= 1024)))
);

CREATE SEQUENCE approval_policy_merge_request_bypass_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_policy_merge_request_bypass_events_id_seq OWNED BY approval_policy_merge_request_bypass_events.id;

CREATE TABLE approval_policy_rule_project_links (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    approval_policy_rule_id bigint NOT NULL
);

CREATE SEQUENCE approval_policy_rule_project_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_policy_rule_project_links_id_seq OWNED BY approval_policy_rule_project_links.id;

CREATE TABLE approval_policy_rules (
    id bigint NOT NULL,
    security_policy_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    rule_index smallint NOT NULL,
    type smallint NOT NULL,
    content jsonb DEFAULT '{}'::jsonb NOT NULL,
    security_policy_management_project_id bigint NOT NULL
);

CREATE SEQUENCE approval_policy_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_policy_rules_id_seq OWNED BY approval_policy_rules.id;

CREATE TABLE approval_project_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    approvals_required smallint DEFAULT 0 NOT NULL,
    name character varying NOT NULL,
    rule_type smallint DEFAULT 0 NOT NULL,
    scanners text[] DEFAULT '{}'::text[],
    vulnerabilities_allowed smallint DEFAULT 0 NOT NULL,
    severity_levels text[] DEFAULT '{}'::text[] NOT NULL,
    report_type smallint,
    vulnerability_states text[] DEFAULT '{new_needs_triage,new_dismissed}'::text[] NOT NULL,
    orchestration_policy_idx smallint,
    applies_to_all_protected_branches boolean DEFAULT false NOT NULL,
    security_orchestration_policy_configuration_id bigint,
    scan_result_policy_id bigint,
    approval_policy_rule_id bigint,
    approval_policy_action_idx smallint DEFAULT 0 NOT NULL
);

CREATE TABLE approval_project_rules_groups (
    id bigint NOT NULL,
    approval_project_rule_id bigint NOT NULL,
    group_id bigint NOT NULL
);

CREATE SEQUENCE approval_project_rules_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_project_rules_groups_id_seq OWNED BY approval_project_rules_groups.id;

CREATE SEQUENCE approval_project_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_project_rules_id_seq OWNED BY approval_project_rules.id;

CREATE TABLE approval_project_rules_protected_branches (
    approval_project_rule_id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_3cae655261 CHECK ((project_id IS NOT NULL))
);

CREATE TABLE approval_project_rules_users (
    id bigint NOT NULL,
    approval_project_rule_id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_26058e3982 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE approval_project_rules_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_project_rules_users_id_seq OWNED BY approval_project_rules_users.id;

CREATE TABLE approvals (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    patch_id_sha bytea,
    project_id bigint,
    CONSTRAINT check_9da7c942dc CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE approvals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approvals_id_seq OWNED BY approvals.id;

CREATE TABLE ar_internal_metadata (
    key character varying NOT NULL,
    value character varying,
    created_at timestamp(6) without time zone NOT NULL,
    updated_at timestamp(6) without time zone NOT NULL
);

CREATE TABLE arkose_sessions (
    id bigint NOT NULL,
    session_created_at timestamp with time zone,
    checked_answer_at timestamp with time zone,
    verified_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    global_score integer,
    custom_score integer,
    challenge_shown boolean DEFAULT false NOT NULL,
    challenge_solved boolean DEFAULT false NOT NULL,
    session_is_legit boolean DEFAULT true NOT NULL,
    is_tor boolean DEFAULT false NOT NULL,
    is_vpn boolean DEFAULT false NOT NULL,
    is_proxy boolean DEFAULT false NOT NULL,
    is_bot boolean DEFAULT false NOT NULL,
    session_xid text NOT NULL,
    telltale_user text,
    user_agent text,
    user_language_shown text,
    device_xid text,
    telltale_list text[] DEFAULT '{}'::text[] NOT NULL,
    user_ip text,
    country text,
    region text,
    city text,
    isp text,
    connection_type text,
    risk_band text,
    risk_category text,
    CONSTRAINT check_1a6f4682be CHECK ((char_length(user_agent) <= 255)),
    CONSTRAINT check_1ccf4778d0 CHECK ((char_length(telltale_user) <= 128)),
    CONSTRAINT check_20eae4e360 CHECK ((char_length(risk_band) <= 64)),
    CONSTRAINT check_394c3c0153 CHECK ((char_length(session_xid) <= 64)),
    CONSTRAINT check_5a92894aa9 CHECK ((char_length(device_xid) <= 64)),
    CONSTRAINT check_8d83d12f95 CHECK ((char_length(user_ip) <= 64)),
    CONSTRAINT check_940ffc498d CHECK ((char_length(risk_category) <= 64)),
    CONSTRAINT check_9b4c7551e7 CHECK ((char_length(city) <= 64)),
    CONSTRAINT check_b81756eb85 CHECK ((char_length(isp) <= 128)),
    CONSTRAINT check_ba409cc401 CHECK ((char_length(region) <= 64)),
    CONSTRAINT check_c745f5db92 CHECK ((char_length(country) <= 64)),
    CONSTRAINT check_cd4cc1f7dc CHECK ((char_length(user_language_shown) <= 64)),
    CONSTRAINT check_d4fd1df18c CHECK ((char_length(connection_type) <= 64))
);

CREATE SEQUENCE arkose_sessions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE arkose_sessions_id_seq OWNED BY arkose_sessions.id;

CREATE TABLE atlassian_identities (
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    expires_at timestamp with time zone,
    extern_uid text NOT NULL,
    encrypted_token bytea,
    encrypted_token_iv bytea,
    encrypted_refresh_token bytea,
    encrypted_refresh_token_iv bytea,
    CONSTRAINT atlassian_identities_refresh_token_iv_length_constraint CHECK ((octet_length(encrypted_refresh_token_iv) <= 12)),
    CONSTRAINT atlassian_identities_refresh_token_length_constraint CHECK ((octet_length(encrypted_refresh_token) <= 5000)),
    CONSTRAINT atlassian_identities_token_iv_length_constraint CHECK ((octet_length(encrypted_token_iv) <= 12)),
    CONSTRAINT atlassian_identities_token_length_constraint CHECK ((octet_length(encrypted_token) <= 5120)),
    CONSTRAINT check_32f5779763 CHECK ((char_length(extern_uid) <= 255))
);

CREATE SEQUENCE atlassian_identities_user_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE atlassian_identities_user_id_seq OWNED BY atlassian_identities.user_id;

CREATE TABLE audit_events_amazon_s3_configurations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    access_key_xid text NOT NULL,
    name text NOT NULL,
    bucket_name text NOT NULL,
    aws_region text NOT NULL,
    encrypted_secret_access_key bytea NOT NULL,
    encrypted_secret_access_key_iv bytea NOT NULL,
    stream_destination_id bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_3a41f4ea06 CHECK ((char_length(bucket_name) <= 63)),
    CONSTRAINT check_72b5aaa71b CHECK ((char_length(aws_region) <= 50)),
    CONSTRAINT check_90505816db CHECK ((char_length(name) <= 72)),
    CONSTRAINT check_ec46f06e01 CHECK ((char_length(access_key_xid) <= 128))
);

CREATE SEQUENCE audit_events_amazon_s3_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_amazon_s3_configurations_id_seq OWNED BY audit_events_amazon_s3_configurations.id;

CREATE TABLE audit_events_external_audit_event_destinations (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    destination_url text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    verification_token text,
    name text NOT NULL,
    stream_destination_id bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_2feafb9daf CHECK ((char_length(destination_url) <= 255)),
    CONSTRAINT check_8ec80a7d06 CHECK ((char_length(verification_token) <= 24)),
    CONSTRAINT check_c52ff8e90e CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE audit_events_external_audit_event_destinations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_external_audit_event_destinations_id_seq OWNED BY audit_events_external_audit_event_destinations.id;

CREATE TABLE audit_events_google_cloud_logging_configurations (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    google_project_id_name text NOT NULL,
    client_email text NOT NULL,
    log_id_name text DEFAULT 'audit_events'::text,
    encrypted_private_key bytea NOT NULL,
    encrypted_private_key_iv bytea NOT NULL,
    name text NOT NULL,
    stream_destination_id bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_0ef835c61e CHECK ((char_length(client_email) <= 254)),
    CONSTRAINT check_55783c7c19 CHECK ((char_length(google_project_id_name) <= 30)),
    CONSTRAINT check_898a76b005 CHECK ((char_length(log_id_name) <= 511)),
    CONSTRAINT check_cdf6883cd6 CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE audit_events_google_cloud_logging_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_google_cloud_logging_configurations_id_seq OWNED BY audit_events_google_cloud_logging_configurations.id;

CREATE TABLE audit_events_group_external_streaming_destinations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    category smallint NOT NULL,
    name text NOT NULL,
    config jsonb NOT NULL,
    encrypted_secret_token bytea NOT NULL,
    encrypted_secret_token_iv bytea NOT NULL,
    legacy_destination_ref bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_97d157fbd0 CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE audit_events_group_external_streaming_destinations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_group_external_streaming_destinations_id_seq OWNED BY audit_events_group_external_streaming_destinations.id;

CREATE TABLE audit_events_group_streaming_event_type_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_streaming_destination_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    audit_event_type text NOT NULL,
    CONSTRAINT check_389708af23 CHECK ((char_length(audit_event_type) <= 255))
);

CREATE SEQUENCE audit_events_group_streaming_event_type_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_group_streaming_event_type_filters_id_seq OWNED BY audit_events_group_streaming_event_type_filters.id;

CREATE SEQUENCE audit_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_id_seq OWNED BY audit_events.id;

CREATE TABLE audit_events_instance_amazon_s3_configurations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    access_key_xid text NOT NULL,
    name text NOT NULL,
    bucket_name text NOT NULL,
    aws_region text NOT NULL,
    encrypted_secret_access_key bytea NOT NULL,
    encrypted_secret_access_key_iv bytea NOT NULL,
    stream_destination_id bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_1a908bd36f CHECK ((char_length(name) <= 72)),
    CONSTRAINT check_8083750c42 CHECK ((char_length(bucket_name) <= 63)),
    CONSTRAINT check_d2ca3eb90e CHECK ((char_length(aws_region) <= 50)),
    CONSTRAINT check_d6d6bd8e8b CHECK ((char_length(access_key_xid) <= 128))
);

CREATE SEQUENCE audit_events_instance_amazon_s3_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_instance_amazon_s3_configurations_id_seq OWNED BY audit_events_instance_amazon_s3_configurations.id;

CREATE TABLE audit_events_instance_external_audit_event_destinations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    destination_url text NOT NULL,
    encrypted_verification_token bytea NOT NULL,
    encrypted_verification_token_iv bytea NOT NULL,
    name text NOT NULL,
    stream_destination_id bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_433fbb3305 CHECK ((char_length(name) <= 72)),
    CONSTRAINT check_4dc67167ce CHECK ((char_length(destination_url) <= 255))
);

CREATE SEQUENCE audit_events_instance_external_audit_event_destinations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_instance_external_audit_event_destinations_id_seq OWNED BY audit_events_instance_external_audit_event_destinations.id;

CREATE TABLE audit_events_instance_external_streaming_destinations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    category smallint NOT NULL,
    name text NOT NULL,
    config jsonb NOT NULL,
    encrypted_secret_token bytea NOT NULL,
    encrypted_secret_token_iv bytea NOT NULL,
    legacy_destination_ref bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_219decfb51 CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE audit_events_instance_external_streaming_destinations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_instance_external_streaming_destinations_id_seq OWNED BY audit_events_instance_external_streaming_destinations.id;

CREATE TABLE audit_events_instance_google_cloud_logging_configurations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    google_project_id_name text NOT NULL,
    client_email text NOT NULL,
    log_id_name text DEFAULT 'audit_events'::text,
    name text NOT NULL,
    encrypted_private_key bytea NOT NULL,
    encrypted_private_key_iv bytea NOT NULL,
    stream_destination_id bigint,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_0da5c76c49 CHECK ((char_length(client_email) <= 254)),
    CONSTRAINT check_74fd943192 CHECK ((char_length(log_id_name) <= 511)),
    CONSTRAINT check_ab65f57721 CHECK ((char_length(google_project_id_name) <= 30)),
    CONSTRAINT check_ac42ad3ca2 CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE audit_events_instance_google_cloud_logging_configuration_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_instance_google_cloud_logging_configuration_id_seq OWNED BY audit_events_instance_google_cloud_logging_configurations.id;

CREATE TABLE audit_events_instance_streaming_event_type_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_streaming_destination_id bigint NOT NULL,
    audit_event_type text NOT NULL,
    CONSTRAINT check_4a5e8e01b5 CHECK ((char_length(audit_event_type) <= 255))
);

CREATE SEQUENCE audit_events_instance_streaming_event_type_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_instance_streaming_event_type_filters_id_seq OWNED BY audit_events_instance_streaming_event_type_filters.id;

CREATE TABLE audit_events_streaming_event_type_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_audit_event_destination_id bigint NOT NULL,
    audit_event_type text NOT NULL,
    group_id bigint,
    CONSTRAINT check_9eb6a21b47 CHECK ((group_id IS NOT NULL)),
    CONSTRAINT check_d20c8e5a51 CHECK ((char_length(audit_event_type) <= 255))
);

CREATE SEQUENCE audit_events_streaming_event_type_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_event_type_filters_id_seq OWNED BY audit_events_streaming_event_type_filters.id;

CREATE TABLE audit_events_streaming_group_namespace_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_streaming_destination_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE audit_events_streaming_group_namespace_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_group_namespace_filters_id_seq OWNED BY audit_events_streaming_group_namespace_filters.id;

CREATE TABLE audit_events_streaming_headers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_audit_event_destination_id bigint NOT NULL,
    key text NOT NULL,
    value text NOT NULL,
    active boolean DEFAULT true NOT NULL,
    group_id bigint,
    CONSTRAINT check_3feba4e364 CHECK ((group_id IS NOT NULL)),
    CONSTRAINT check_53c3152034 CHECK ((char_length(key) <= 255)),
    CONSTRAINT check_ac213cca22 CHECK ((char_length(value) <= 2000))
);

CREATE SEQUENCE audit_events_streaming_headers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_headers_id_seq OWNED BY audit_events_streaming_headers.id;

CREATE TABLE audit_events_streaming_http_group_namespace_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_audit_event_destination_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE audit_events_streaming_http_group_namespace_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_http_group_namespace_filters_id_seq OWNED BY audit_events_streaming_http_group_namespace_filters.id;

CREATE TABLE audit_events_streaming_http_instance_namespace_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    audit_events_instance_external_audit_event_destination_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE audit_events_streaming_http_instance_namespace_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_http_instance_namespace_filters_id_seq OWNED BY audit_events_streaming_http_instance_namespace_filters.id;

CREATE TABLE audit_events_streaming_instance_event_type_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    instance_external_audit_event_destination_id bigint NOT NULL,
    audit_event_type text NOT NULL,
    CONSTRAINT check_249c9370cc CHECK ((char_length(audit_event_type) <= 255))
);

CREATE SEQUENCE audit_events_streaming_instance_event_type_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_instance_event_type_filters_id_seq OWNED BY audit_events_streaming_instance_event_type_filters.id;

CREATE TABLE audit_events_streaming_instance_namespace_filters (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_streaming_destination_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE audit_events_streaming_instance_namespace_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_streaming_instance_namespace_filters_id_seq OWNED BY audit_events_streaming_instance_namespace_filters.id;

CREATE TABLE authentication_events (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    user_id bigint,
    result smallint NOT NULL,
    ip_address inet,
    provider text NOT NULL,
    user_name text NOT NULL,
    organization_id bigint NOT NULL,
    CONSTRAINT check_45a6cc4e80 CHECK ((char_length(user_name) <= 255)),
    CONSTRAINT check_c64f424630 CHECK ((char_length(provider) <= 64))
);

CREATE SEQUENCE authentication_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE authentication_events_id_seq OWNED BY authentication_events.id;

CREATE TABLE automation_rules (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    issues_events boolean DEFAULT false NOT NULL,
    merge_requests_events boolean DEFAULT false NOT NULL,
    permanently_disabled boolean DEFAULT false NOT NULL,
    name text NOT NULL,
    rule text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_0be3e2c953 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_ed5a4fcbd5 CHECK ((char_length(rule) <= 2048))
);

CREATE SEQUENCE automation_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE automation_rules_id_seq OWNED BY automation_rules.id;

CREATE TABLE award_emoji (
    id bigint NOT NULL,
    name character varying,
    user_id bigint,
    awardable_type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    awardable_id bigint,
    namespace_id bigint,
    organization_id bigint
);

CREATE TABLE award_emoji_archived (
    id bigint NOT NULL,
    name character varying,
    user_id bigint,
    awardable_type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    awardable_id bigint,
    namespace_id bigint,
    organization_id bigint,
    archived_at timestamp with time zone DEFAULT CURRENT_TIMESTAMP
);

COMMENT ON TABLE award_emoji_archived IS 'Temporary table for storing orphaned award_emoji during sharding key backfill. To be dropped after migration completion.';

CREATE SEQUENCE award_emoji_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE award_emoji_id_seq OWNED BY award_emoji.id;

CREATE TABLE aws_roles (
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    role_arn character varying(2048),
    role_external_id character varying(64) NOT NULL,
    region text,
    CONSTRAINT check_57adedab55 CHECK ((char_length(region) <= 255))
);

CREATE SEQUENCE background_operation_jobs_cell_local_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE background_operation_jobs_cell_local_id_seq OWNED BY background_operation_jobs_cell_local.id;

CREATE SEQUENCE background_operation_workers_cell_local_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE background_operation_workers_cell_local_id_seq OWNED BY background_operation_workers_cell_local.id;

CREATE TABLE badges (
    id bigint NOT NULL,
    link_url character varying NOT NULL,
    image_url character varying NOT NULL,
    project_id bigint,
    group_id bigint,
    type character varying NOT NULL,
    name character varying(255),
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_22ac1b6d3a CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE badges_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE badges_id_seq OWNED BY badges.id;

CREATE TABLE banned_users (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    projects_deleted boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE batched_background_migration_job_transition_logs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE batched_background_migration_job_transition_logs_id_seq OWNED BY batched_background_migration_job_transition_logs.id;

CREATE TABLE batched_background_migration_jobs (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone,
    finished_at timestamp with time zone,
    batched_background_migration_id bigint NOT NULL,
    min_value bigint,
    max_value bigint,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    attempts smallint DEFAULT 0 NOT NULL,
    metrics jsonb DEFAULT '{}'::jsonb NOT NULL,
    pause_ms integer DEFAULT 100 NOT NULL,
    min_cursor jsonb,
    max_cursor jsonb,
    CONSTRAINT check_18d498ea58 CHECK (((jsonb_typeof(min_cursor) = 'array'::text) AND (jsonb_typeof(max_cursor) = 'array'::text))),
    CONSTRAINT check_ba39c36ddb CHECK ((pause_ms >= 100)),
    CONSTRAINT check_c1ce96fe3b CHECK (((num_nonnulls(min_value, max_value) = 2) OR (num_nonnulls(min_cursor, max_cursor) = 2)))
);

CREATE SEQUENCE batched_background_migration_jobs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE batched_background_migration_jobs_id_seq OWNED BY batched_background_migration_jobs.id;

CREATE TABLE batched_background_migrations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    min_value bigint DEFAULT 1,
    max_value bigint,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    "interval" smallint NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    job_class_name text NOT NULL,
    batch_class_name text DEFAULT 'PrimaryKeyBatchingStrategy'::text NOT NULL,
    table_name text NOT NULL,
    column_name text NOT NULL,
    job_arguments jsonb DEFAULT '[]'::jsonb NOT NULL,
    total_tuple_count bigint,
    pause_ms integer DEFAULT 100 NOT NULL,
    max_batch_size integer,
    started_at timestamp with time zone,
    on_hold_until timestamp with time zone,
    gitlab_schema text NOT NULL,
    finished_at timestamp with time zone,
    queued_migration_version text,
    min_cursor jsonb,
    max_cursor jsonb,
    CONSTRAINT check_0406d9776f CHECK ((char_length(gitlab_schema) <= 255)),
    CONSTRAINT check_122750e705 CHECK (((jsonb_typeof(min_cursor) = 'array'::text) AND (jsonb_typeof(max_cursor) = 'array'::text))),
    CONSTRAINT check_5bb0382d6f CHECK ((char_length(column_name) <= 63)),
    CONSTRAINT check_69e4fcc53b CHECK ((pause_ms >= 100)),
    CONSTRAINT check_6b6a06254a CHECK ((char_length(table_name) <= 63)),
    CONSTRAINT check_713f147aea CHECK ((char_length(queued_migration_version) <= 14)),
    CONSTRAINT check_batch_size_in_range CHECK ((batch_size >= sub_batch_size)),
    CONSTRAINT check_e6c75b1e29 CHECK ((char_length(job_class_name) <= 100)),
    CONSTRAINT check_f5158baa12 CHECK (((num_nonnulls(min_value, max_value) = 2) OR (num_nonnulls(min_cursor, max_cursor) = 2))),
    CONSTRAINT check_fe10674721 CHECK ((char_length(batch_class_name) <= 100)),
    CONSTRAINT check_max_value_in_range CHECK ((max_value >= min_value)),
    CONSTRAINT check_positive_min_value CHECK ((min_value > 0)),
    CONSTRAINT check_positive_sub_batch_size CHECK ((sub_batch_size > 0))
);

COMMENT ON COLUMN batched_background_migrations.on_hold_until IS 'execution of this migration is on hold until this time';

CREATE SEQUENCE batched_background_migrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE batched_background_migrations_id_seq OWNED BY batched_background_migrations.id;

CREATE TABLE board_assignees (
    id bigint NOT NULL,
    board_id bigint NOT NULL,
    assignee_id bigint NOT NULL,
    group_id bigint,
    project_id bigint,
    CONSTRAINT check_b56ef26337 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE board_assignees_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_assignees_id_seq OWNED BY board_assignees.id;

CREATE TABLE board_group_recent_visits (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint,
    board_id bigint,
    group_id bigint,
    CONSTRAINT check_409f6caea4 CHECK ((user_id IS NOT NULL)),
    CONSTRAINT check_ddc74243ef CHECK ((group_id IS NOT NULL)),
    CONSTRAINT check_fa7711a898 CHECK ((board_id IS NOT NULL))
);

CREATE SEQUENCE board_group_recent_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_group_recent_visits_id_seq OWNED BY board_group_recent_visits.id;

CREATE TABLE board_labels (
    id bigint NOT NULL,
    board_id bigint NOT NULL,
    label_id bigint NOT NULL,
    group_id bigint,
    project_id bigint,
    CONSTRAINT check_b8d32bd9fe CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE board_labels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_labels_id_seq OWNED BY board_labels.id;

CREATE TABLE board_project_recent_visits (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint,
    project_id bigint,
    board_id bigint,
    CONSTRAINT check_0386e26981 CHECK ((board_id IS NOT NULL)),
    CONSTRAINT check_d9cc9b79da CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_df7762a99a CHECK ((user_id IS NOT NULL))
);

CREATE SEQUENCE board_project_recent_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_project_recent_visits_id_seq OWNED BY board_project_recent_visits.id;

CREATE TABLE board_user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    board_id bigint NOT NULL,
    hide_labels boolean,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint,
    project_id bigint,
    CONSTRAINT check_86d6706b52 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE board_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_user_preferences_id_seq OWNED BY board_user_preferences.id;

CREATE TABLE boards (
    id bigint NOT NULL,
    project_id bigint,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    name character varying DEFAULT 'Development'::character varying NOT NULL,
    milestone_id bigint,
    group_id bigint,
    weight integer,
    hide_backlog_list boolean DEFAULT false NOT NULL,
    hide_closed_list boolean DEFAULT false NOT NULL,
    iteration_id bigint,
    iteration_cadence_id bigint,
    CONSTRAINT check_a60857cc50 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE TABLE boards_epic_board_labels (
    id bigint NOT NULL,
    epic_board_id bigint NOT NULL,
    label_id bigint NOT NULL,
    group_id bigint,
    CONSTRAINT check_c71449be47 CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE boards_epic_board_labels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_board_labels_id_seq OWNED BY boards_epic_board_labels.id;

CREATE TABLE boards_epic_board_positions (
    id bigint NOT NULL,
    epic_board_id bigint NOT NULL,
    epic_id bigint NOT NULL,
    relative_position integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint,
    CONSTRAINT check_9d94ce874e CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE boards_epic_board_positions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_board_positions_id_seq OWNED BY boards_epic_board_positions.id;

CREATE TABLE boards_epic_board_recent_visits (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    epic_board_id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE boards_epic_board_recent_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_board_recent_visits_id_seq OWNED BY boards_epic_board_recent_visits.id;

CREATE TABLE boards_epic_boards (
    id bigint NOT NULL,
    hide_backlog_list boolean DEFAULT false NOT NULL,
    hide_closed_list boolean DEFAULT false NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text DEFAULT 'Development'::text NOT NULL,
    display_colors boolean DEFAULT true NOT NULL,
    CONSTRAINT check_bcbbffe601 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE boards_epic_boards_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_boards_id_seq OWNED BY boards_epic_boards.id;

CREATE TABLE boards_epic_list_user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    epic_list_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    collapsed boolean DEFAULT false NOT NULL,
    group_id bigint,
    CONSTRAINT check_c18068ad9c CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE boards_epic_list_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_list_user_preferences_id_seq OWNED BY boards_epic_list_user_preferences.id;

CREATE TABLE boards_epic_lists (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    epic_board_id bigint NOT NULL,
    label_id bigint,
    "position" integer,
    list_type smallint DEFAULT 1 NOT NULL,
    group_id bigint,
    CONSTRAINT boards_epic_lists_position_constraint CHECK (((list_type <> 1) OR (("position" IS NOT NULL) AND ("position" >= 0)))),
    CONSTRAINT check_86641111a7 CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE boards_epic_lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_lists_id_seq OWNED BY boards_epic_lists.id;

CREATE TABLE boards_epic_user_preferences (
    id bigint NOT NULL,
    board_id bigint NOT NULL,
    user_id bigint NOT NULL,
    epic_id bigint NOT NULL,
    collapsed boolean DEFAULT false NOT NULL,
    group_id bigint,
    CONSTRAINT check_e23d7636e9 CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE boards_epic_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_user_preferences_id_seq OWNED BY boards_epic_user_preferences.id;

CREATE SEQUENCE boards_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_id_seq OWNED BY boards.id;

CREATE TABLE broadcast_messages (
    id bigint NOT NULL,
    message text NOT NULL,
    starts_at timestamp without time zone NOT NULL,
    ends_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    color character varying,
    font character varying,
    message_html text NOT NULL,
    cached_markdown_version integer,
    target_path character varying(255),
    broadcast_type smallint DEFAULT 1 NOT NULL,
    dismissable boolean,
    target_access_levels integer[] DEFAULT '{}'::integer[] NOT NULL,
    theme smallint DEFAULT 0 NOT NULL,
    show_in_cli boolean DEFAULT true NOT NULL
);

CREATE SEQUENCE broadcast_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE broadcast_messages_id_seq OWNED BY broadcast_messages.id;

CREATE TABLE bulk_import_batch_trackers (
    id bigint NOT NULL,
    tracker_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    batch_number integer DEFAULT 0 NOT NULL,
    fetched_objects_count integer DEFAULT 0 NOT NULL,
    imported_objects_count integer DEFAULT 0 NOT NULL,
    error text,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    CONSTRAINT check_3d6963a51f CHECK ((char_length(error) <= 255))
);

CREATE SEQUENCE bulk_import_batch_trackers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_batch_trackers_id_seq OWNED BY bulk_import_batch_trackers.id;

CREATE TABLE bulk_import_configurations (
    id bigint NOT NULL,
    bulk_import_id bigint NOT NULL,
    encrypted_url text,
    encrypted_url_iv text,
    encrypted_access_token text,
    encrypted_access_token_iv text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint,
    CONSTRAINT check_cd24605431 CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE bulk_import_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_configurations_id_seq OWNED BY bulk_import_configurations.id;

CREATE TABLE bulk_import_entities (
    id bigint NOT NULL,
    bulk_import_id bigint NOT NULL,
    parent_id bigint,
    namespace_id bigint,
    project_id bigint,
    source_type smallint NOT NULL,
    source_full_path text NOT NULL,
    destination_name text NOT NULL,
    destination_namespace text NOT NULL,
    status smallint NOT NULL,
    jid text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    source_xid integer,
    migrate_projects boolean DEFAULT true NOT NULL,
    has_failures boolean DEFAULT false,
    migrate_memberships boolean DEFAULT true NOT NULL,
    organization_id bigint,
    CONSTRAINT check_13f279f7da CHECK ((char_length(source_full_path) <= 255)),
    CONSTRAINT check_469f9235c5 CHECK ((num_nonnulls(namespace_id, organization_id, project_id) = 1)),
    CONSTRAINT check_715d725ea2 CHECK ((char_length(destination_name) <= 255)),
    CONSTRAINT check_796a4d9cc6 CHECK ((char_length(jid) <= 255)),
    CONSTRAINT check_b834fff4d9 CHECK ((char_length(destination_namespace) <= 255))
);

CREATE SEQUENCE bulk_import_entities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_entities_id_seq OWNED BY bulk_import_entities.id;

CREATE TABLE bulk_import_export_batches (
    id bigint NOT NULL,
    export_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    batch_number integer DEFAULT 0 NOT NULL,
    objects_count integer DEFAULT 0 NOT NULL,
    error text,
    project_id bigint,
    group_id bigint,
    CONSTRAINT check_046dc60dfe CHECK ((char_length(error) <= 255)),
    CONSTRAINT check_31f6b54459 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE bulk_import_export_batches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_export_batches_id_seq OWNED BY bulk_import_export_batches.id;

CREATE TABLE bulk_import_export_upload_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64)),
    CONSTRAINT check_fd3aa31edf CHECK ((num_nonnulls(namespace_id, project_id) = 1))
);

CREATE TABLE bulk_import_export_uploads (
    id bigint NOT NULL,
    export_id bigint NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    export_file text,
    batch_id bigint,
    project_id bigint,
    group_id bigint,
    CONSTRAINT check_5add76239d CHECK ((char_length(export_file) <= 255)),
    CONSTRAINT check_e1d215df28 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE bulk_import_export_uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_export_uploads_id_seq OWNED BY bulk_import_export_uploads.id;

CREATE TABLE bulk_import_exports (
    id bigint NOT NULL,
    group_id bigint,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relation text NOT NULL,
    jid text,
    error text,
    batched boolean DEFAULT false NOT NULL,
    batches_count integer DEFAULT 0 NOT NULL,
    total_objects_count integer DEFAULT 0 NOT NULL,
    user_id bigint,
    offline_export_id bigint,
    CONSTRAINT check_24cb010672 CHECK ((char_length(relation) <= 255)),
    CONSTRAINT check_8f0f357334 CHECK ((char_length(error) <= 255)),
    CONSTRAINT check_9ee6d14d33 CHECK ((char_length(jid) <= 255)),
    CONSTRAINT check_e84b7c0730 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE bulk_import_exports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_exports_id_seq OWNED BY bulk_import_exports.id;

CREATE TABLE bulk_import_failures (
    id bigint NOT NULL,
    bulk_import_entity_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    pipeline_class text NOT NULL,
    exception_class text NOT NULL,
    exception_message text NOT NULL,
    correlation_id_value text,
    pipeline_step text,
    source_url text,
    source_title text,
    subrelation text,
    project_id bigint,
    namespace_id bigint,
    organization_id bigint,
    CONSTRAINT check_053d65c7a4 CHECK ((char_length(pipeline_class) <= 255)),
    CONSTRAINT check_6eca8f972e CHECK ((char_length(exception_message) <= 255)),
    CONSTRAINT check_721a422375 CHECK ((char_length(pipeline_step) <= 255)),
    CONSTRAINT check_74414228d4 CHECK ((char_length(source_title) <= 255)),
    CONSTRAINT check_c7dba8398e CHECK ((char_length(exception_class) <= 255)),
    CONSTRAINT check_e035a720ad CHECK ((char_length(source_url) <= 255)),
    CONSTRAINT check_e787285882 CHECK ((char_length(correlation_id_value) <= 255)),
    CONSTRAINT check_ebd770e350 CHECK ((num_nonnulls(namespace_id, organization_id, project_id) = 1)),
    CONSTRAINT check_f99665a440 CHECK ((char_length(subrelation) <= 255))
);

CREATE SEQUENCE bulk_import_failures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_failures_id_seq OWNED BY bulk_import_failures.id;

CREATE TABLE bulk_import_trackers (
    id bigint NOT NULL,
    bulk_import_entity_id bigint NOT NULL,
    relation text NOT NULL,
    next_page text,
    has_next_page boolean DEFAULT false NOT NULL,
    jid text,
    stage smallint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    batched boolean DEFAULT false,
    source_objects_count bigint DEFAULT 0 NOT NULL,
    fetched_objects_count bigint DEFAULT 0 NOT NULL,
    imported_objects_count bigint DEFAULT 0 NOT NULL,
    project_id bigint,
    namespace_id bigint,
    organization_id bigint,
    CONSTRAINT check_2d45cae629 CHECK ((char_length(relation) <= 255)),
    CONSTRAINT check_40aeaa600b CHECK ((char_length(next_page) <= 255)),
    CONSTRAINT check_5f034e7cad CHECK ((num_nonnulls(namespace_id, organization_id, project_id) = 1)),
    CONSTRAINT check_603f91cb06 CHECK ((char_length(jid) <= 255)),
    CONSTRAINT check_next_page_requirement CHECK (((has_next_page IS FALSE) OR (next_page IS NOT NULL)))
);

CREATE SEQUENCE bulk_import_trackers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_trackers_id_seq OWNED BY bulk_import_trackers.id;

CREATE TABLE bulk_imports (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    source_type smallint NOT NULL,
    status smallint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    source_version text,
    source_enterprise boolean DEFAULT true NOT NULL,
    has_failures boolean DEFAULT false,
    organization_id bigint NOT NULL,
    CONSTRAINT check_ea4e58775a CHECK ((char_length(source_version) <= 63))
);

CREATE SEQUENCE bulk_imports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_imports_id_seq OWNED BY bulk_imports.id;

CREATE TABLE catalog_resource_component_last_usages (
    id bigint NOT NULL,
    component_id bigint NOT NULL,
    catalog_resource_id bigint NOT NULL,
    component_project_id bigint NOT NULL,
    used_by_project_id bigint NOT NULL,
    last_used_date date NOT NULL
);

CREATE SEQUENCE catalog_resource_component_last_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE catalog_resource_component_last_usages_id_seq OWNED BY catalog_resource_component_last_usages.id;

CREATE TABLE catalog_resource_components (
    id bigint NOT NULL,
    catalog_resource_id bigint NOT NULL,
    version_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    spec jsonb DEFAULT '{}'::jsonb NOT NULL,
    last_30_day_usage_count integer DEFAULT 0 NOT NULL,
    last_30_day_usage_count_updated_at timestamp with time zone DEFAULT '1970-01-01 00:00:00+00'::timestamp with time zone NOT NULL,
    component_type smallint DEFAULT 1,
    CONSTRAINT check_47c5537132 CHECK ((component_type IS NOT NULL)),
    CONSTRAINT check_ddca729980 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE catalog_resource_components_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE catalog_resource_components_id_seq OWNED BY catalog_resource_components.id;

CREATE TABLE catalog_resource_versions (
    id bigint NOT NULL,
    release_id bigint NOT NULL,
    catalog_resource_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    released_at timestamp with time zone DEFAULT '1970-01-01 00:00:00+00'::timestamp with time zone NOT NULL,
    semver_major integer,
    semver_minor integer,
    semver_patch integer,
    semver_prerelease text,
    semver_prefixed boolean DEFAULT false,
    cached_markdown_version integer,
    readme text,
    readme_html text,
    published_by_id bigint,
    CONSTRAINT check_701bdce47b CHECK ((char_length(semver_prerelease) <= 255))
);

CREATE SEQUENCE catalog_resource_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE catalog_resource_versions_id_seq OWNED BY catalog_resource_versions.id;

CREATE TABLE catalog_resources (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    latest_released_at timestamp with time zone,
    name character varying,
    description text,
    visibility_level integer DEFAULT 0 NOT NULL,
    search_vector tsvector GENERATED ALWAYS AS ((setweight(to_tsvector('english'::regconfig, (COALESCE(name, ''::character varying))::text), 'A'::"char") || setweight(to_tsvector('english'::regconfig, COALESCE(description, ''::text)), 'B'::"char"))) STORED,
    verification_level smallint DEFAULT 0,
    last_30_day_usage_count integer DEFAULT 0 NOT NULL,
    last_30_day_usage_count_updated_at timestamp with time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE catalog_resources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE catalog_resources_id_seq OWNED BY catalog_resources.id;

CREATE TABLE catalog_verified_namespaces (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    verification_level smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE catalog_verified_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE catalog_verified_namespaces_id_seq OWNED BY catalog_verified_namespaces.id;

CREATE TABLE cells_outstanding_leases (
    uuid uuid NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE chat_names (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    team_id character varying NOT NULL,
    team_domain character varying,
    chat_id character varying NOT NULL,
    chat_name character varying,
    last_used_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    encrypted_token bytea,
    encrypted_token_iv bytea
);

CREATE SEQUENCE chat_names_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE chat_names_id_seq OWNED BY chat_names.id;

CREATE TABLE chat_teams (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    team_id character varying,
    name character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE chat_teams_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE chat_teams_id_seq OWNED BY chat_teams.id;

CREATE TABLE ci_build_needs (
    name text NOT NULL,
    artifacts boolean DEFAULT true NOT NULL,
    optional boolean DEFAULT false NOT NULL,
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_4fab85ecdc CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_build_needs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_needs_id_seq OWNED BY ci_build_needs.id;

CREATE TABLE ci_build_pending_states (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    build_id bigint NOT NULL,
    state smallint,
    failure_reason smallint,
    trace_checksum bytea,
    trace_bytesize bigint,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_20b28e5e16 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_build_pending_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_pending_states_id_seq OWNED BY ci_build_pending_states.id;

CREATE TABLE ci_build_report_results (
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    data jsonb DEFAULT '{}'::jsonb NOT NULL,
    partition_id bigint NOT NULL
);

CREATE TABLE ci_build_trace_chunks (
    id bigint NOT NULL,
    chunk_index integer NOT NULL,
    data_store integer NOT NULL,
    raw_data bytea,
    checksum bytea,
    lock_version integer DEFAULT 0 NOT NULL,
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_b374316678 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_build_trace_chunks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_trace_chunks_id_seq OWNED BY ci_build_trace_chunks.id;

CREATE SEQUENCE ci_builds_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_builds_id_seq OWNED BY p_ci_builds.id;

CREATE TABLE p_ci_builds_metadata (
    project_id bigint NOT NULL,
    timeout integer,
    timeout_source integer DEFAULT 1 NOT NULL,
    interruptible boolean,
    config_options jsonb,
    config_variables jsonb,
    has_exposed_artifacts boolean,
    environment_auto_stop_in character varying(255),
    expanded_environment_name character varying(255),
    secrets jsonb DEFAULT '{}'::jsonb NOT NULL,
    build_id bigint NOT NULL,
    id bigint NOT NULL,
    id_tokens jsonb DEFAULT '{}'::jsonb NOT NULL,
    partition_id bigint NOT NULL,
    debug_trace_enabled boolean DEFAULT false NOT NULL,
    exit_code smallint
)
PARTITION BY LIST (partition_id);

CREATE SEQUENCE ci_builds_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_builds_metadata_id_seq OWNED BY p_ci_builds_metadata.id;

CREATE TABLE ci_builds_runner_session (
    id bigint NOT NULL,
    url character varying NOT NULL,
    certificate character varying,
    "authorization" character varying,
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_2eb15fa9f3 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_builds_runner_session_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_builds_runner_session_id_seq OWNED BY ci_builds_runner_session.id;

CREATE TABLE ci_cost_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    runner_id bigint NOT NULL,
    standard_factor double precision DEFAULT 1.0 NOT NULL,
    os_contribution_factor double precision DEFAULT 0.008 NOT NULL,
    os_plan_factor double precision DEFAULT 0.5 NOT NULL
);

CREATE TABLE ci_daily_build_group_report_results (
    id bigint NOT NULL,
    date date NOT NULL,
    project_id bigint NOT NULL,
    last_pipeline_id bigint NOT NULL,
    ref_path text NOT NULL,
    group_name text NOT NULL,
    data jsonb NOT NULL,
    default_branch boolean DEFAULT false NOT NULL,
    group_id bigint,
    partition_id bigint NOT NULL
);

CREATE SEQUENCE ci_daily_build_group_report_results_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_daily_build_group_report_results_id_seq OWNED BY ci_daily_build_group_report_results.id;

CREATE TABLE ci_deleted_objects (
    id bigint NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    pick_up_at timestamp with time zone DEFAULT now() NOT NULL,
    store_dir text NOT NULL,
    file text NOT NULL,
    project_id bigint,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_5e151d6912 CHECK ((char_length(store_dir) <= 1024)),
    CONSTRAINT check_98f90d6c53 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_deleted_objects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_deleted_objects_id_seq OWNED BY ci_deleted_objects.id;

CREATE TABLE ci_freeze_periods (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    freeze_start character varying(998) NOT NULL,
    freeze_end character varying(998) NOT NULL,
    cron_timezone character varying(255) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE ci_freeze_periods_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_freeze_periods_id_seq OWNED BY ci_freeze_periods.id;

CREATE TABLE ci_gitlab_hosted_runner_monthly_usages (
    id bigint NOT NULL,
    runner_id bigint NOT NULL,
    runner_duration_seconds bigint DEFAULT 0 NOT NULL,
    project_id bigint NOT NULL,
    root_namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    billing_month date NOT NULL,
    notification_level integer DEFAULT 100 NOT NULL,
    compute_minutes_used numeric(18,4) DEFAULT 0.0 NOT NULL,
    CONSTRAINT ci_hosted_runner_monthly_usages_month_constraint CHECK ((billing_month = date_trunc('month'::text, (billing_month)::timestamp with time zone)))
);

CREATE SEQUENCE ci_gitlab_hosted_runner_monthly_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_gitlab_hosted_runner_monthly_usages_id_seq OWNED BY ci_gitlab_hosted_runner_monthly_usages.id;

CREATE TABLE ci_group_variables (
    id bigint NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    group_id bigint NOT NULL,
    protected boolean DEFAULT false NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    masked boolean DEFAULT false NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    environment_scope text DEFAULT '*'::text NOT NULL,
    raw boolean DEFAULT false NOT NULL,
    description text,
    hidden boolean DEFAULT false NOT NULL,
    CONSTRAINT check_dfe009485a CHECK ((char_length(environment_scope) <= 255)),
    CONSTRAINT check_e2e50ff879 CHECK ((char_length(description) <= 255))
);

CREATE SEQUENCE ci_group_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_group_variables_id_seq OWNED BY ci_group_variables.id;

CREATE TABLE ci_hosted_runners (
    runner_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE ci_instance_runner_monthly_usages (
    id bigint NOT NULL,
    runner_id bigint,
    runner_duration_seconds bigint DEFAULT 0 NOT NULL,
    project_id bigint,
    root_namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    billing_month date NOT NULL,
    notification_level integer DEFAULT 100 NOT NULL,
    compute_minutes_used numeric(18,4) DEFAULT 0.0 NOT NULL,
    CONSTRAINT ci_instance_runner_monthly_usages_year_month_constraint CHECK ((billing_month = date_trunc('month'::text, (billing_month)::timestamp with time zone)))
);

CREATE SEQUENCE ci_instance_runner_monthly_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_instance_runner_monthly_usages_id_seq OWNED BY ci_instance_runner_monthly_usages.id;

CREATE TABLE ci_instance_variables (
    id bigint NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    masked boolean DEFAULT false,
    protected boolean DEFAULT false,
    key text NOT NULL,
    encrypted_value text,
    encrypted_value_iv text,
    raw boolean DEFAULT false NOT NULL,
    description text,
    CONSTRAINT check_07a45a5bcb CHECK ((char_length(encrypted_value_iv) <= 255)),
    CONSTRAINT check_5aede12208 CHECK ((char_length(key) <= 255)),
    CONSTRAINT check_956afd70f2 CHECK ((((variable_type = 2) AND (char_length(encrypted_value) <= 67800)) OR (char_length(encrypted_value) <= 13579))),
    CONSTRAINT check_a0a9762afa CHECK ((char_length(description) <= 255))
);

CREATE SEQUENCE ci_instance_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_instance_variables_id_seq OWNED BY ci_instance_variables.id;

CREATE TABLE ci_job_artifact_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    job_artifact_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_df832b66ea CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE ci_job_artifacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_artifacts_id_seq OWNED BY p_ci_job_artifacts.id;

CREATE TABLE ci_job_token_authorizations (
    id bigint NOT NULL,
    accessed_project_id bigint NOT NULL,
    origin_project_id bigint NOT NULL,
    last_authorized_at timestamp with time zone NOT NULL,
    job_token_policies jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE ci_job_token_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_token_authorizations_id_seq OWNED BY ci_job_token_authorizations.id;

CREATE TABLE ci_job_token_group_scope_links (
    id bigint NOT NULL,
    source_project_id bigint NOT NULL,
    target_group_id bigint NOT NULL,
    added_by_id bigint,
    created_at timestamp with time zone NOT NULL,
    job_token_policies jsonb DEFAULT '[]'::jsonb,
    default_permissions boolean DEFAULT true NOT NULL,
    autopopulated boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE ci_job_token_group_scope_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_token_group_scope_links_id_seq OWNED BY ci_job_token_group_scope_links.id;

CREATE TABLE ci_job_token_project_scope_links (
    id bigint NOT NULL,
    source_project_id bigint NOT NULL,
    target_project_id bigint NOT NULL,
    added_by_id bigint,
    created_at timestamp with time zone NOT NULL,
    direction smallint DEFAULT 0 NOT NULL,
    job_token_policies jsonb DEFAULT '[]'::jsonb,
    default_permissions boolean DEFAULT true NOT NULL,
    autopopulated boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE ci_job_token_project_scope_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_token_project_scope_links_id_seq OWNED BY ci_job_token_project_scope_links.id;

CREATE TABLE ci_job_variables (
    id bigint NOT NULL,
    key character varying NOT NULL,
    encrypted_value text,
    encrypted_value_iv character varying,
    job_id bigint NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    source smallint DEFAULT 0 NOT NULL,
    raw boolean DEFAULT false NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_567d1ccb72 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_job_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_variables_id_seq OWNED BY ci_job_variables.id;

CREATE TABLE ci_minutes_additional_packs (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    expires_at date,
    number_of_minutes integer NOT NULL,
    purchase_xid text,
    CONSTRAINT check_d7ef254af0 CHECK ((char_length(purchase_xid) <= 50))
);

CREATE SEQUENCE ci_minutes_additional_packs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_minutes_additional_packs_id_seq OWNED BY ci_minutes_additional_packs.id;

CREATE TABLE ci_namespace_mirrors (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL
);

CREATE SEQUENCE ci_namespace_mirrors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_namespace_mirrors_id_seq OWNED BY ci_namespace_mirrors.id;

CREATE TABLE ci_namespace_monthly_usages (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    date date NOT NULL,
    notification_level smallint DEFAULT 100 NOT NULL,
    created_at timestamp with time zone,
    amount_used numeric(18,4) DEFAULT 0.0 NOT NULL,
    shared_runners_duration bigint DEFAULT 0 NOT NULL,
    CONSTRAINT ci_namespace_monthly_usages_year_month_constraint CHECK ((date = date_trunc('month'::text, (date)::timestamp with time zone)))
);

CREATE SEQUENCE ci_namespace_monthly_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_namespace_monthly_usages_id_seq OWNED BY ci_namespace_monthly_usages.id;

CREATE TABLE ci_partitions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL
);

CREATE TABLE ci_pending_builds (
    id bigint NOT NULL,
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    protected boolean DEFAULT false NOT NULL,
    instance_runners_enabled boolean DEFAULT false NOT NULL,
    namespace_id bigint,
    minutes_exceeded boolean DEFAULT false NOT NULL,
    tag_ids bigint[] DEFAULT '{}'::bigint[],
    namespace_traversal_ids bigint[] DEFAULT '{}'::bigint[],
    partition_id bigint NOT NULL,
    plan_id bigint,
    plan_name_uid smallint
);

CREATE SEQUENCE ci_pending_builds_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pending_builds_id_seq OWNED BY ci_pending_builds.id;

CREATE TABLE ci_pipeline_artifacts (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    pipeline_id bigint NOT NULL,
    project_id bigint NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    file_type smallint NOT NULL,
    file_format smallint NOT NULL,
    file text,
    expire_at timestamp with time zone,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    locked smallint DEFAULT 2,
    partition_id bigint NOT NULL,
    CONSTRAINT check_191b5850ec CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_abeeb71caf CHECK ((file IS NOT NULL)),
    CONSTRAINT ci_pipeline_artifacts_verification_failure_text_limit CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE ci_pipeline_artifacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_artifacts_id_seq OWNED BY ci_pipeline_artifacts.id;

CREATE TABLE ci_pipeline_chat_data (
    id bigint NOT NULL,
    chat_name_id bigint NOT NULL,
    response_url text NOT NULL,
    pipeline_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_f6412eda6f CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_pipeline_chat_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_chat_data_id_seq OWNED BY ci_pipeline_chat_data.id;

CREATE TABLE ci_pipeline_messages (
    id bigint NOT NULL,
    severity smallint DEFAULT 0 NOT NULL,
    content text NOT NULL,
    pipeline_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_58ca2981b2 CHECK ((char_length(content) <= 10000)),
    CONSTRAINT check_fe8ee122a2 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_pipeline_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_messages_id_seq OWNED BY ci_pipeline_messages.id;

CREATE TABLE ci_pipeline_metadata (
    project_id bigint NOT NULL,
    pipeline_id bigint NOT NULL,
    name text,
    auto_cancel_on_new_commit smallint DEFAULT 0 NOT NULL,
    auto_cancel_on_job_failure smallint DEFAULT 0 NOT NULL,
    partition_id bigint NOT NULL,
    CONSTRAINT check_9d3665463c CHECK ((char_length(name) <= 255))
);

CREATE TABLE ci_pipeline_schedule_inputs (
    id bigint NOT NULL,
    pipeline_schedule_id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    value jsonb,
    CONSTRAINT check_a340b48bb4 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE ci_pipeline_schedule_inputs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_schedule_inputs_id_seq OWNED BY ci_pipeline_schedule_inputs.id;

CREATE TABLE ci_pipeline_schedule_variables (
    id bigint NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    pipeline_schedule_id bigint NOT NULL,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    variable_type smallint DEFAULT 1 NOT NULL,
    raw boolean DEFAULT false NOT NULL,
    project_id bigint,
    CONSTRAINT check_17806054a8 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_pipeline_schedule_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_schedule_variables_id_seq OWNED BY ci_pipeline_schedule_variables.id;

CREATE TABLE ci_pipeline_schedules (
    id bigint NOT NULL,
    description character varying,
    ref character varying,
    cron character varying,
    cron_timezone character varying,
    next_run_at timestamp without time zone,
    project_id bigint,
    owner_id bigint,
    active boolean DEFAULT true,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    CONSTRAINT check_4a0f7b994d CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_pipeline_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_schedules_id_seq OWNED BY ci_pipeline_schedules.id;

CREATE SEQUENCE ci_pipeline_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_variables_id_seq OWNED BY p_ci_pipeline_variables.id;

CREATE SEQUENCE ci_pipelines_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipelines_id_seq OWNED BY p_ci_pipelines.id;

CREATE TABLE ci_project_mirrors (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE ci_project_mirrors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_project_mirrors_id_seq OWNED BY ci_project_mirrors.id;

CREATE TABLE ci_project_monthly_usages (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    created_at timestamp with time zone,
    amount_used numeric(18,4) DEFAULT 0.0 NOT NULL,
    shared_runners_duration bigint DEFAULT 0 NOT NULL,
    CONSTRAINT ci_project_monthly_usages_year_month_constraint CHECK ((date = date_trunc('month'::text, (date)::timestamp with time zone)))
);

CREATE SEQUENCE ci_project_monthly_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_project_monthly_usages_id_seq OWNED BY ci_project_monthly_usages.id;

CREATE TABLE ci_refs (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    lock_version integer DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    ref_path text NOT NULL
);

CREATE SEQUENCE ci_refs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_refs_id_seq OWNED BY ci_refs.id;

CREATE TABLE ci_resource_groups (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    key character varying(255) NOT NULL,
    process_mode smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE ci_resource_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_resource_groups_id_seq OWNED BY ci_resource_groups.id;

CREATE TABLE ci_resources (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    resource_group_id bigint NOT NULL,
    build_id bigint,
    partition_id bigint,
    project_id bigint,
    CONSTRAINT check_f3eccb9d35 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_resources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_resources_id_seq OWNED BY ci_resources.id;

CREATE TABLE ci_runner_controller_tokens (
    id bigint NOT NULL,
    description text,
    token_digest text NOT NULL,
    runner_controller_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_84d7d76c86 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_ec7c3fc764 CHECK ((char_length(token_digest) <= 255))
);

CREATE SEQUENCE ci_runner_controller_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_controller_tokens_id_seq OWNED BY ci_runner_controller_tokens.id;

CREATE TABLE ci_runner_controllers (
    id bigint NOT NULL,
    description text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    CONSTRAINT check_0d6af097e9 CHECK ((char_length(description) <= 1024))
);

CREATE SEQUENCE ci_runner_controllers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_controllers_id_seq OWNED BY ci_runner_controllers.id;

CREATE TABLE ci_runner_machines (
    id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    contacted_at timestamp with time zone,
    creation_state smallint DEFAULT 0 NOT NULL,
    executor_type smallint,
    runner_type smallint NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL,
    system_xid text NOT NULL,
    platform text,
    architecture text,
    revision text,
    ip_address text,
    version text,
    runtime_features jsonb DEFAULT '{}'::jsonb NOT NULL,
    organization_id bigint,
    labels jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_3d8736b3af CHECK ((char_length(system_xid) <= 64)),
    CONSTRAINT check_5bad2a6944 CHECK ((char_length(revision) <= 255)),
    CONSTRAINT check_7dc4eee8a5 CHECK ((char_length(version) <= 2048)),
    CONSTRAINT check_b1e456641b CHECK ((char_length(ip_address) <= 1024)),
    CONSTRAINT check_c788f4b18a CHECK ((char_length(platform) <= 255)),
    CONSTRAINT check_f3d25ab844 CHECK ((char_length(architecture) <= 255))
)
PARTITION BY LIST (runner_type);

CREATE SEQUENCE ci_runner_machines_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_machines_id_seq OWNED BY ci_runner_machines.id;

CREATE TABLE ci_runner_namespaces (
    id bigint NOT NULL,
    runner_id bigint,
    namespace_id bigint,
    CONSTRAINT check_5f3dce48df CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE ci_runner_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_namespaces_id_seq OWNED BY ci_runner_namespaces.id;

CREATE TABLE ci_runner_projects (
    id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id bigint,
    CONSTRAINT check_db297016c6 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_runner_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_projects_id_seq OWNED BY ci_runner_projects.id;

CREATE TABLE ci_runner_taggings (
    id bigint NOT NULL,
    tag_id bigint NOT NULL,
    runner_id bigint NOT NULL,
    runner_type smallint NOT NULL,
    organization_id bigint,
    tag_name text,
    CONSTRAINT check_tag_name_not_null CHECK ((tag_name IS NOT NULL)),
    CONSTRAINT ci_runner_taggings_tag_name_length CHECK ((char_length(tag_name) <= 1024))
)
PARTITION BY LIST (runner_type);

CREATE TABLE ci_runner_taggings_group_type (
    id bigint NOT NULL,
    tag_id bigint NOT NULL,
    runner_id bigint NOT NULL,
    runner_type smallint NOT NULL,
    organization_id bigint,
    tag_name text,
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_tag_name_not_null CHECK ((tag_name IS NOT NULL)),
    CONSTRAINT ci_runner_taggings_tag_name_length CHECK ((char_length(tag_name) <= 1024))
);

CREATE SEQUENCE ci_runner_taggings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_taggings_id_seq OWNED BY ci_runner_taggings.id;

CREATE TABLE ci_runner_taggings_instance_type (
    id bigint NOT NULL,
    tag_id bigint NOT NULL,
    runner_id bigint NOT NULL,
    runner_type smallint NOT NULL,
    organization_id bigint,
    tag_name text,
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NULL)),
    CONSTRAINT check_tag_name_not_null CHECK ((tag_name IS NOT NULL)),
    CONSTRAINT ci_runner_taggings_tag_name_length CHECK ((char_length(tag_name) <= 1024))
);

CREATE TABLE ci_runner_taggings_project_type (
    id bigint NOT NULL,
    tag_id bigint NOT NULL,
    runner_id bigint NOT NULL,
    runner_type smallint NOT NULL,
    organization_id bigint,
    tag_name text,
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_tag_name_not_null CHECK ((tag_name IS NOT NULL)),
    CONSTRAINT ci_runner_taggings_tag_name_length CHECK ((char_length(tag_name) <= 1024))
);

CREATE TABLE ci_runner_versions (
    version text NOT NULL,
    status smallint,
    CONSTRAINT check_b5a3714594 CHECK ((char_length(version) <= 2048))
);

CREATE TABLE ci_runners (
    id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    contacted_at timestamp with time zone,
    token_expires_at timestamp with time zone,
    public_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    private_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    access_level integer DEFAULT 0 NOT NULL,
    maximum_timeout integer,
    runner_type smallint NOT NULL,
    registration_type smallint DEFAULT 0 NOT NULL,
    creation_state smallint DEFAULT 0 NOT NULL,
    active boolean DEFAULT true NOT NULL,
    run_untagged boolean DEFAULT true NOT NULL,
    locked boolean DEFAULT false NOT NULL,
    name text,
    token_encrypted text,
    description text,
    maintainer_note text,
    allowed_plans text[] DEFAULT '{}'::text[] NOT NULL,
    allowed_plan_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    organization_id bigint,
    allowed_plan_name_uids smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    CONSTRAINT check_030ad0773d CHECK ((char_length(token_encrypted) <= 512)),
    CONSTRAINT check_1f8618ab23 CHECK ((char_length(name) <= 256)),
    CONSTRAINT check_24b281f5bf CHECK ((char_length(maintainer_note) <= 1024)),
    CONSTRAINT check_5db8ae9d30 CHECK ((char_length(description) <= 1024))
)
PARTITION BY LIST (runner_type);

CREATE SEQUENCE ci_runners_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runners_id_seq OWNED BY ci_runners.id;

CREATE TABLE ci_running_builds (
    id bigint NOT NULL,
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    runner_type smallint NOT NULL,
    partition_id bigint NOT NULL,
    runner_owner_namespace_xid bigint
);

CREATE SEQUENCE ci_running_builds_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_running_builds_id_seq OWNED BY ci_running_builds.id;

CREATE TABLE ci_secure_file_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    ci_secure_file_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    project_id bigint,
    CONSTRAINT check_a79e5a9261 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE ci_secure_file_states_ci_secure_file_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_secure_file_states_ci_secure_file_id_seq OWNED BY ci_secure_file_states.ci_secure_file_id;

CREATE TABLE ci_secure_files (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    name text NOT NULL,
    file text NOT NULL,
    checksum bytea NOT NULL,
    key_data text,
    metadata jsonb,
    expires_at timestamp with time zone,
    CONSTRAINT check_320790634d CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_402c7b4a56 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_7279b4e293 CHECK ((char_length(key_data) <= 128))
);

CREATE SEQUENCE ci_secure_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_secure_files_id_seq OWNED BY ci_secure_files.id;

CREATE TABLE ci_sources_pipelines (
    id bigint NOT NULL,
    project_id bigint,
    source_project_id bigint,
    source_job_id bigint,
    partition_id bigint NOT NULL,
    source_partition_id bigint NOT NULL,
    pipeline_id bigint,
    source_pipeline_id bigint,
    CONSTRAINT check_5a76e457e6 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_sources_pipelines_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_sources_pipelines_id_seq OWNED BY ci_sources_pipelines.id;

CREATE TABLE ci_sources_projects (
    id bigint NOT NULL,
    pipeline_id bigint NOT NULL,
    source_project_id bigint NOT NULL,
    partition_id bigint NOT NULL
);

CREATE SEQUENCE ci_sources_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_sources_projects_id_seq OWNED BY ci_sources_projects.id;

CREATE SEQUENCE ci_stages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_stages_id_seq OWNED BY p_ci_stages.id;

CREATE TABLE ci_subscriptions_projects (
    id bigint NOT NULL,
    downstream_project_id bigint NOT NULL,
    upstream_project_id bigint NOT NULL,
    author_id bigint
);

CREATE SEQUENCE ci_subscriptions_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_subscriptions_projects_id_seq OWNED BY ci_subscriptions_projects.id;

CREATE TABLE ci_triggers (
    id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id bigint,
    owner_id bigint NOT NULL,
    description character varying,
    expires_at timestamp with time zone,
    token_encrypted text,
    CONSTRAINT check_4905e4c2cb CHECK ((char_length(token_encrypted) <= 255)),
    CONSTRAINT check_8120a40ce8 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_triggers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_triggers_id_seq OWNED BY ci_triggers.id;

CREATE TABLE ci_unit_test_failures (
    id bigint NOT NULL,
    failed_at timestamp with time zone NOT NULL,
    unit_test_id bigint NOT NULL,
    build_id bigint NOT NULL,
    partition_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_5e4c2d7261 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ci_unit_test_failures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_unit_test_failures_id_seq OWNED BY ci_unit_test_failures.id;

CREATE TABLE ci_unit_tests (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    key_hash text NOT NULL,
    name text NOT NULL,
    suite_name text NOT NULL,
    CONSTRAINT check_248fae1a3b CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b288215ffe CHECK ((char_length(key_hash) <= 64)),
    CONSTRAINT check_c2d57b3c49 CHECK ((char_length(suite_name) <= 255))
);

CREATE SEQUENCE ci_unit_tests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_unit_tests_id_seq OWNED BY ci_unit_tests.id;

CREATE TABLE ci_variables (
    id bigint NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    project_id bigint NOT NULL,
    protected boolean DEFAULT false NOT NULL,
    environment_scope character varying DEFAULT '*'::character varying NOT NULL,
    masked boolean DEFAULT false NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    raw boolean DEFAULT false NOT NULL,
    description text,
    hidden boolean DEFAULT false NOT NULL,
    CONSTRAINT check_7e46c006aa CHECK ((char_length(description) <= 255))
);

CREATE SEQUENCE ci_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_variables_id_seq OWNED BY ci_variables.id;

CREATE TABLE cloud_connector_access (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    data jsonb,
    catalog jsonb
);

CREATE SEQUENCE cloud_connector_access_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cloud_connector_access_id_seq OWNED BY cloud_connector_access.id;

CREATE TABLE cloud_connector_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    secret_key jsonb
);

CREATE SEQUENCE cloud_connector_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cloud_connector_keys_id_seq OWNED BY cloud_connector_keys.id;

CREATE TABLE cluster_agent_migrations (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    project_id bigint NOT NULL,
    issue_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    agent_install_status smallint NOT NULL,
    agent_install_message text,
    CONSTRAINT check_110bed11f8 CHECK ((char_length(agent_install_message) <= 255))
);

CREATE SEQUENCE cluster_agent_migrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_agent_migrations_id_seq OWNED BY cluster_agent_migrations.id;

CREATE TABLE cluster_agent_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    agent_id bigint NOT NULL,
    token_encrypted text NOT NULL,
    created_by_user_id bigint,
    description text,
    name text,
    last_used_at timestamp with time zone,
    status smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_0fb634d04d CHECK ((name IS NOT NULL)),
    CONSTRAINT check_2b79dbb315 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_4e4ec5070a CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_5aff240050 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_c60daed227 CHECK ((char_length(token_encrypted) <= 255))
);

CREATE SEQUENCE cluster_agent_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_agent_tokens_id_seq OWNED BY cluster_agent_tokens.id;

CREATE TABLE cluster_agent_url_configurations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    agent_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_by_user_id bigint,
    status smallint DEFAULT 0 NOT NULL,
    url text NOT NULL,
    ca_cert text,
    client_key text,
    client_cert text,
    tls_host text,
    public_key bytea,
    encrypted_private_key bytea,
    encrypted_private_key_iv bytea,
    CONSTRAINT check_1ffcfef6d6 CHECK ((char_length(tls_host) <= 2048)),
    CONSTRAINT check_25ef8c679c CHECK ((char_length(ca_cert) <= 16384)),
    CONSTRAINT check_93a57284e5 CHECK ((char_length(client_cert) <= 16384)),
    CONSTRAINT check_e3736d97df CHECK ((char_length(client_key) <= 16384)),
    CONSTRAINT check_ed21ced327 CHECK ((char_length(url) <= 2048))
);

CREATE SEQUENCE cluster_agent_url_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_agent_url_configurations_id_seq OWNED BY cluster_agent_url_configurations.id;

CREATE TABLE cluster_agents (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    created_by_user_id bigint,
    has_vulnerabilities boolean DEFAULT false NOT NULL,
    connection_mode smallint DEFAULT 0 NOT NULL,
    is_receptive boolean DEFAULT false NOT NULL,
    CONSTRAINT check_3498369510 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE cluster_agents_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_agents_id_seq OWNED BY cluster_agents.id;

CREATE TABLE cluster_enabled_grants (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE cluster_enabled_grants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_enabled_grants_id_seq OWNED BY cluster_enabled_grants.id;

CREATE TABLE cluster_groups (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    group_id bigint NOT NULL
);

CREATE SEQUENCE cluster_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_groups_id_seq OWNED BY cluster_groups.id;

CREATE TABLE cluster_platforms_kubernetes (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    api_url text,
    ca_cert text,
    namespace character varying,
    username character varying,
    encrypted_password text,
    encrypted_password_iv character varying,
    encrypted_token text,
    encrypted_token_iv character varying,
    authorization_type smallint
);

CREATE SEQUENCE cluster_platforms_kubernetes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_platforms_kubernetes_id_seq OWNED BY cluster_platforms_kubernetes.id;

CREATE TABLE cluster_projects (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE cluster_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_projects_id_seq OWNED BY cluster_projects.id;

CREATE TABLE cluster_providers_aws (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    num_nodes integer NOT NULL,
    status integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    key_name character varying(255) NOT NULL,
    role_arn character varying(2048) NOT NULL,
    region character varying(255) NOT NULL,
    vpc_id character varying(255) NOT NULL,
    subnet_ids character varying(255)[] DEFAULT '{}'::character varying[] NOT NULL,
    security_group_id character varying(255) NOT NULL,
    instance_type character varying(255) NOT NULL,
    access_key_id character varying(255),
    encrypted_secret_access_key_iv character varying(255),
    encrypted_secret_access_key text,
    session_token text,
    status_reason text,
    kubernetes_version text DEFAULT '1.14'::text NOT NULL,
    CONSTRAINT check_f1f42cd85e CHECK ((char_length(kubernetes_version) <= 30))
);

CREATE SEQUENCE cluster_providers_aws_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_providers_aws_id_seq OWNED BY cluster_providers_aws.id;

CREATE TABLE cluster_providers_gcp (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    status integer,
    num_nodes integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    status_reason text,
    gcp_project_id character varying NOT NULL,
    zone character varying NOT NULL,
    machine_type character varying,
    operation_id character varying,
    endpoint character varying,
    encrypted_access_token text,
    encrypted_access_token_iv character varying,
    legacy_abac boolean DEFAULT false NOT NULL,
    cloud_run boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE cluster_providers_gcp_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_providers_gcp_id_seq OWNED BY cluster_providers_gcp.id;

CREATE TABLE clusters (
    id bigint NOT NULL,
    user_id bigint,
    provider_type integer,
    platform_type integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    enabled boolean DEFAULT true,
    name character varying NOT NULL,
    environment_scope character varying DEFAULT '*'::character varying NOT NULL,
    cluster_type smallint DEFAULT 3 NOT NULL,
    domain character varying,
    managed boolean DEFAULT true NOT NULL,
    namespace_per_environment boolean DEFAULT true NOT NULL,
    management_project_id bigint,
    cleanup_status smallint DEFAULT 1 NOT NULL,
    cleanup_status_reason text,
    helm_major_version integer DEFAULT 3 NOT NULL,
    project_id bigint,
    group_id bigint,
    organization_id bigint,
    CONSTRAINT check_1a12345d3a CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1))
);

CREATE SEQUENCE clusters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_id_seq OWNED BY clusters.id;

CREATE TABLE clusters_kubernetes_namespaces (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    project_id bigint,
    cluster_project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_service_account_token text,
    encrypted_service_account_token_iv character varying,
    namespace character varying NOT NULL,
    service_account_name character varying,
    environment_id bigint
);

CREATE SEQUENCE clusters_kubernetes_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_kubernetes_namespaces_id_seq OWNED BY clusters_kubernetes_namespaces.id;

CREATE TABLE clusters_managed_resources (
    id bigint NOT NULL,
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    environment_id bigint NOT NULL,
    cluster_agent_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    template_name text,
    tracked_objects jsonb DEFAULT '[]'::jsonb NOT NULL,
    deletion_strategy smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_4f81a98847 CHECK ((char_length(template_name) <= 1024))
);

CREATE SEQUENCE clusters_managed_resources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_managed_resources_id_seq OWNED BY clusters_managed_resources.id;

CREATE TABLE commit_user_mentions (
    id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    commit_id character varying NOT NULL,
    note_id bigint NOT NULL,
    namespace_id bigint
);

CREATE SEQUENCE commit_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE commit_user_mentions_id_seq OWNED BY commit_user_mentions.id;

CREATE TABLE compliance_framework_security_policies (
    id bigint NOT NULL,
    framework_id bigint NOT NULL,
    policy_configuration_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    policy_index smallint NOT NULL,
    project_id bigint,
    namespace_id bigint,
    security_policy_id bigint,
    CONSTRAINT check_473e5b2da9 CHECK ((num_nonnulls(namespace_id, project_id) = 1))
);

CREATE SEQUENCE compliance_framework_security_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE compliance_framework_security_policies_id_seq OWNED BY compliance_framework_security_policies.id;

CREATE TABLE compliance_management_frameworks (
    id bigint NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    color text NOT NULL,
    namespace_id bigint NOT NULL,
    pipeline_configuration_full_path text,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    source_id bigint,
    CONSTRAINT check_08cd34b2c2 CHECK ((char_length(color) <= 10)),
    CONSTRAINT check_1617e0b87e CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_ab00bc2193 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e7a9972435 CHECK ((char_length(pipeline_configuration_full_path) <= 255))
);

CREATE SEQUENCE compliance_management_frameworks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE compliance_management_frameworks_id_seq OWNED BY compliance_management_frameworks.id;

CREATE TABLE compliance_requirements (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    framework_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    CONSTRAINT check_71d7c59197 CHECK ((char_length(description) <= 500)),
    CONSTRAINT check_f1fb6fdd81 CHECK ((char_length(name) <= 255))
);

CREATE TABLE compliance_requirements_controls (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    compliance_requirement_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    name smallint NOT NULL,
    control_type smallint NOT NULL,
    expression text,
    encrypted_secret_token bytea,
    encrypted_secret_token_iv bytea,
    external_url text,
    external_control_name text,
    ping_enabled boolean DEFAULT true NOT NULL,
    CONSTRAINT check_110c87ed8d CHECK ((char_length(expression) <= 255)),
    CONSTRAINT check_5020dd6745 CHECK ((char_length(external_url) <= 1024)),
    CONSTRAINT check_e3c26a3c02 CHECK ((char_length(external_control_name) <= 255))
);

CREATE SEQUENCE compliance_requirements_controls_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE compliance_requirements_controls_id_seq OWNED BY compliance_requirements_controls.id;

CREATE SEQUENCE compliance_requirements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE compliance_requirements_id_seq OWNED BY compliance_requirements.id;

CREATE TABLE compromised_password_detections (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    resolved_at timestamp with time zone,
    user_id bigint NOT NULL
);

CREATE SEQUENCE compromised_password_detections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE compromised_password_detections_id_seq OWNED BY compromised_password_detections.id;

CREATE TABLE container_expiration_policies (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    next_run_at timestamp with time zone,
    name_regex character varying(255) DEFAULT '.*'::character varying,
    cadence character varying(12) DEFAULT '1d'::character varying NOT NULL,
    older_than character varying(12) DEFAULT '90d'::character varying,
    keep_n integer DEFAULT 10,
    enabled boolean DEFAULT false NOT NULL,
    name_regex_keep text,
    CONSTRAINT container_expiration_policies_name_regex_keep CHECK ((char_length(name_regex_keep) <= 255))
);

CREATE TABLE container_registry_data_repair_details (
    missing_count integer DEFAULT 0,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL
);

CREATE TABLE container_registry_protection_rules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    repository_path_pattern text,
    minimum_access_level_for_push smallint,
    minimum_access_level_for_delete smallint,
    CONSTRAINT check_0dc4ab5f43 CHECK ((num_nonnulls(minimum_access_level_for_delete, minimum_access_level_for_push) > 0)),
    CONSTRAINT check_3658b31291 CHECK ((repository_path_pattern IS NOT NULL)),
    CONSTRAINT check_d53a270af5 CHECK ((char_length(repository_path_pattern) <= 255))
);

CREATE SEQUENCE container_registry_protection_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE container_registry_protection_rules_id_seq OWNED BY container_registry_protection_rules.id;

CREATE TABLE container_registry_protection_tag_rules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    minimum_access_level_for_push smallint,
    minimum_access_level_for_delete smallint,
    tag_name_pattern text NOT NULL,
    CONSTRAINT check_50aba315a8 CHECK ((char_length(tag_name_pattern) <= 255)),
    CONSTRAINT check_ae29637175 CHECK ((num_nonnulls(minimum_access_level_for_delete, minimum_access_level_for_push) <> 1))
);

CREATE SEQUENCE container_registry_protection_tag_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE container_registry_protection_tag_rules_id_seq OWNED BY container_registry_protection_tag_rules.id;

CREATE TABLE container_repositories (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    status smallint,
    expiration_policy_started_at timestamp with time zone,
    expiration_policy_cleanup_status smallint DEFAULT 0 NOT NULL,
    expiration_policy_completed_at timestamp with time zone,
    last_cleanup_deleted_tags_count integer,
    delete_started_at timestamp with time zone,
    status_updated_at timestamp with time zone,
    failed_deletion_count integer DEFAULT 0 NOT NULL,
    next_delete_attempt_at timestamp with time zone,
    CONSTRAINT check_container_repositories_non_negative_failed_deletion_count CHECK ((failed_deletion_count >= 0))
);

CREATE SEQUENCE container_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE container_repositories_id_seq OWNED BY container_repositories.id;

CREATE TABLE container_repository_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    container_repository_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    project_id bigint,
    CONSTRAINT check_c96417dbc5 CHECK ((char_length(verification_failure) <= 255)),
    CONSTRAINT check_d65b1f0839 CHECK ((project_id IS NOT NULL))
);

CREATE TABLE content_blocked_states (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    commit_sha bytea NOT NULL,
    blob_sha bytea NOT NULL,
    path text NOT NULL,
    container_identifier text NOT NULL,
    CONSTRAINT check_023093d38f CHECK ((char_length(container_identifier) <= 255)),
    CONSTRAINT check_1870100678 CHECK ((char_length(path) <= 2048))
);

COMMENT ON TABLE content_blocked_states IS 'JiHu-specific table';

CREATE SEQUENCE content_blocked_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE content_blocked_states_id_seq OWNED BY content_blocked_states.id;

CREATE TABLE conversational_development_index_metrics (
    id bigint NOT NULL,
    leader_issues double precision NOT NULL,
    instance_issues double precision NOT NULL,
    leader_notes double precision NOT NULL,
    instance_notes double precision NOT NULL,
    leader_milestones double precision NOT NULL,
    instance_milestones double precision NOT NULL,
    leader_boards double precision NOT NULL,
    instance_boards double precision NOT NULL,
    leader_merge_requests double precision NOT NULL,
    instance_merge_requests double precision NOT NULL,
    leader_ci_pipelines double precision NOT NULL,
    instance_ci_pipelines double precision NOT NULL,
    leader_environments double precision NOT NULL,
    instance_environments double precision NOT NULL,
    leader_deployments double precision NOT NULL,
    instance_deployments double precision NOT NULL,
    leader_projects_prometheus_active double precision,
    instance_projects_prometheus_active double precision,
    leader_service_desk_issues double precision NOT NULL,
    instance_service_desk_issues double precision NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    percentage_boards double precision DEFAULT 0.0 NOT NULL,
    percentage_ci_pipelines double precision DEFAULT 0.0 NOT NULL,
    percentage_deployments double precision DEFAULT 0.0 NOT NULL,
    percentage_environments double precision DEFAULT 0.0 NOT NULL,
    percentage_issues double precision DEFAULT 0.0 NOT NULL,
    percentage_merge_requests double precision DEFAULT 0.0 NOT NULL,
    percentage_milestones double precision DEFAULT 0.0 NOT NULL,
    percentage_notes double precision DEFAULT 0.0 NOT NULL,
    percentage_projects_prometheus_active double precision DEFAULT 0.0 NOT NULL,
    percentage_service_desk_issues double precision DEFAULT 0.0 NOT NULL
);

CREATE SEQUENCE conversational_development_index_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE conversational_development_index_metrics_id_seq OWNED BY conversational_development_index_metrics.id;

CREATE TABLE country_access_logs (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    access_count_reset_at timestamp with time zone,
    first_access_at timestamp with time zone,
    last_access_at timestamp with time zone,
    user_id bigint NOT NULL,
    access_count integer DEFAULT 0 NOT NULL,
    country_code smallint NOT NULL
);

CREATE SEQUENCE country_access_logs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE country_access_logs_id_seq OWNED BY country_access_logs.id;

CREATE TABLE coverage_fuzzing_corpuses (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint,
    package_id bigint NOT NULL,
    file_updated_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE coverage_fuzzing_corpuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE coverage_fuzzing_corpuses_id_seq OWNED BY coverage_fuzzing_corpuses.id;

CREATE TABLE csv_issue_imports (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE csv_issue_imports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE csv_issue_imports_id_seq OWNED BY csv_issue_imports.id;

CREATE TABLE custom_dashboard_search_data (
    id bigint NOT NULL,
    custom_dashboard_id bigint NOT NULL,
    name text DEFAULT ''::text NOT NULL,
    description text DEFAULT ''::text NOT NULL,
    search_vector tsvector,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint NOT NULL,
    CONSTRAINT check_8a2d9ffee0 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_8b9b792eb5 CHECK ((char_length(description) <= 2048))
);

CREATE SEQUENCE custom_dashboard_search_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_dashboard_search_data_id_seq OWNED BY custom_dashboard_search_data.id;

CREATE TABLE custom_dashboard_versions (
    id bigint NOT NULL,
    custom_dashboard_id bigint NOT NULL,
    version_number integer NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL,
    updated_by_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint NOT NULL,
    CONSTRAINT chk_dashboard_versions_config_object CHECK ((jsonb_typeof(config) = 'object'::text))
);

CREATE SEQUENCE custom_dashboard_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_dashboard_versions_id_seq OWNED BY custom_dashboard_versions.id;

CREATE TABLE custom_dashboards (
    id bigint NOT NULL,
    namespace_id bigint,
    organization_id bigint NOT NULL,
    created_by_id bigint,
    updated_by_id bigint,
    lock_version integer DEFAULT 0 NOT NULL,
    name text NOT NULL,
    description text DEFAULT ''::text NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_4e13e6f9f9 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_686309f24f CHECK ((char_length(description) <= 2048)),
    CONSTRAINT chk_custom_dashboards_config_object CHECK ((jsonb_typeof(config) = 'object'::text))
);

CREATE SEQUENCE custom_dashboards_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_dashboards_id_seq OWNED BY custom_dashboards.id;

CREATE TABLE custom_emoji (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    file text NOT NULL,
    external boolean DEFAULT true NOT NULL,
    creator_id bigint NOT NULL,
    CONSTRAINT check_8c586dd507 CHECK ((char_length(name) <= 36)),
    CONSTRAINT check_dd5d60f1fb CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE custom_emoji_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_emoji_id_seq OWNED BY custom_emoji.id;

CREATE TABLE custom_field_select_options (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    custom_field_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    "position" smallint NOT NULL,
    value text NOT NULL,
    CONSTRAINT check_fbed57c2bf CHECK ((char_length(value) <= 255))
);

CREATE SEQUENCE custom_field_select_options_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_field_select_options_id_seq OWNED BY custom_field_select_options.id;

CREATE TABLE custom_fields (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    archived_at timestamp with time zone,
    field_type smallint NOT NULL,
    name text NOT NULL,
    created_by_id bigint,
    updated_by_id bigint,
    CONSTRAINT check_b047b04af9 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE custom_fields_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_fields_id_seq OWNED BY custom_fields.id;

CREATE TABLE custom_software_licenses (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_4e365784ce CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE custom_software_licenses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_software_licenses_id_seq OWNED BY custom_software_licenses.id;

CREATE TABLE customer_relations_contacts (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    organization_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    phone text,
    first_name text NOT NULL,
    last_name text NOT NULL,
    email text,
    description text,
    CONSTRAINT check_1195f4c929 CHECK ((char_length(first_name) <= 255)),
    CONSTRAINT check_40c70da037 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_cd2d67c484 CHECK ((char_length(last_name) <= 255)),
    CONSTRAINT check_f4b7f78c89 CHECK ((char_length(phone) <= 32)),
    CONSTRAINT check_fc0adabf60 CHECK ((char_length(email) <= 255))
);

CREATE SEQUENCE customer_relations_contacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE customer_relations_contacts_id_seq OWNED BY customer_relations_contacts.id;

CREATE TABLE customer_relations_organizations (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    default_rate numeric(18,2),
    name text NOT NULL,
    description text,
    CONSTRAINT check_2ba9ef1c4c CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e476b6058e CHECK ((char_length(description) <= 1024))
);

CREATE SEQUENCE customer_relations_organizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE customer_relations_organizations_id_seq OWNED BY customer_relations_organizations.id;

CREATE TABLE dast_pre_scan_verification_steps (
    id bigint NOT NULL,
    dast_pre_scan_verification_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text,
    verification_errors text[] DEFAULT '{}'::text[] NOT NULL,
    check_type smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_2cf67eeb54 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_cd216b95e4 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE dast_pre_scan_verification_steps_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_pre_scan_verification_steps_id_seq OWNED BY dast_pre_scan_verification_steps.id;

CREATE TABLE dast_pre_scan_verifications (
    id bigint NOT NULL,
    dast_profile_id bigint NOT NULL,
    ci_pipeline_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_a764bbc378 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE dast_pre_scan_verifications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_pre_scan_verifications_id_seq OWNED BY dast_pre_scan_verifications.id;

CREATE TABLE dast_profile_schedules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    dast_profile_id bigint NOT NULL,
    user_id bigint,
    next_run_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT true NOT NULL,
    cron text NOT NULL,
    cadence jsonb DEFAULT '{}'::jsonb NOT NULL,
    timezone text NOT NULL,
    starts_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_86531ea73f CHECK ((char_length(cron) <= 255)),
    CONSTRAINT check_be4d1c3af1 CHECK ((char_length(timezone) <= 255))
);

COMMENT ON TABLE dast_profile_schedules IS '{"owner":"group::dynamic analysis","description":"Scheduling for scans using DAST Profiles"}';

CREATE SEQUENCE dast_profile_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_profile_schedules_id_seq OWNED BY dast_profile_schedules.id;

CREATE TABLE dast_profiles (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    dast_site_profile_id bigint NOT NULL,
    dast_scanner_profile_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    branch_name text,
    CONSTRAINT check_5fcf73bf61 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_6c9d775949 CHECK ((char_length(branch_name) <= 255)),
    CONSTRAINT check_c34e505c24 CHECK ((char_length(description) <= 255))
);

COMMENT ON TABLE dast_profiles IS '{"owner":"group::dynamic analysis","description":"Profile used to run a DAST on-demand scan"}';

CREATE SEQUENCE dast_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_profiles_id_seq OWNED BY dast_profiles.id;

CREATE TABLE dast_profiles_pipelines (
    dast_profile_id bigint NOT NULL,
    ci_pipeline_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_39563bc8de CHECK ((project_id IS NOT NULL))
);

COMMENT ON TABLE dast_profiles_pipelines IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Profiles and CI Pipelines"}';

CREATE TABLE dast_profiles_tags (
    id bigint NOT NULL,
    dast_profile_id bigint NOT NULL,
    tag_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_b1aa92f799 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE dast_profiles_tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_profiles_tags_id_seq OWNED BY dast_profiles_tags.id;

CREATE TABLE dast_scanner_profiles (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    spider_timeout smallint,
    target_timeout smallint,
    name text NOT NULL,
    scan_type smallint DEFAULT 1 NOT NULL,
    use_ajax_spider boolean DEFAULT false NOT NULL,
    show_debug_messages boolean DEFAULT false NOT NULL,
    CONSTRAINT check_568568fabf CHECK ((char_length(name) <= 255))
);

CREATE TABLE dast_scanner_profiles_builds (
    dast_scanner_profile_id bigint NOT NULL,
    ci_build_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_2f3b8ab3e6 CHECK ((project_id IS NOT NULL))
);

COMMENT ON TABLE dast_scanner_profiles_builds IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Scanner Profiles and CI Builds"}';

CREATE SEQUENCE dast_scanner_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_scanner_profiles_id_seq OWNED BY dast_scanner_profiles.id;

CREATE TABLE dast_site_profile_secret_variables (
    id bigint NOT NULL,
    dast_site_profile_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    key text NOT NULL,
    encrypted_value bytea NOT NULL,
    encrypted_value_iv bytea NOT NULL,
    project_id bigint,
    CONSTRAINT check_236213f179 CHECK ((length(encrypted_value) <= 13352)),
    CONSTRAINT check_8cbef204b2 CHECK ((char_length(key) <= 255)),
    CONSTRAINT check_b49080abbf CHECK ((length(encrypted_value_iv) <= 17)),
    CONSTRAINT check_d972e5f59d CHECK ((project_id IS NOT NULL))
);

COMMENT ON TABLE dast_site_profile_secret_variables IS '{"owner":"group::dynamic analysis","description":"Secret variables used in DAST on-demand scans"}';

CREATE SEQUENCE dast_site_profile_secret_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_profile_secret_variables_id_seq OWNED BY dast_site_profile_secret_variables.id;

CREATE TABLE dast_site_profiles (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    dast_site_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    excluded_urls text[] DEFAULT '{}'::text[] NOT NULL,
    auth_enabled boolean DEFAULT false NOT NULL,
    auth_url text,
    auth_username_field text,
    auth_password_field text,
    auth_username text,
    target_type smallint DEFAULT 0 NOT NULL,
    scan_method smallint DEFAULT 0 NOT NULL,
    auth_submit_field text,
    scan_file_path text,
    optional_variables jsonb DEFAULT '[]'::jsonb NOT NULL,
    CONSTRAINT check_5203110fee CHECK ((char_length(auth_username_field) <= 255)),
    CONSTRAINT check_6cfab17b48 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_8d2aa0f66d CHECK ((char_length(scan_file_path) <= 1024)),
    CONSTRAINT check_af44f54c96 CHECK ((char_length(auth_submit_field) <= 255)),
    CONSTRAINT check_c329dffdba CHECK ((char_length(auth_password_field) <= 255)),
    CONSTRAINT check_d446f7047b CHECK ((char_length(auth_url) <= 1024)),
    CONSTRAINT check_f22f18002a CHECK ((char_length(auth_username) <= 255))
);

CREATE TABLE dast_site_profiles_builds (
    dast_site_profile_id bigint NOT NULL,
    ci_build_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_581c9bb699 CHECK ((project_id IS NOT NULL))
);

COMMENT ON TABLE dast_site_profiles_builds IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Site Profiles and CI Builds"}';

CREATE SEQUENCE dast_site_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_profiles_id_seq OWNED BY dast_site_profiles.id;

CREATE TABLE dast_site_tokens (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    expired_at timestamp with time zone,
    token text NOT NULL,
    url text NOT NULL,
    CONSTRAINT check_02a6bf20a7 CHECK ((char_length(token) <= 255)),
    CONSTRAINT check_69ab8622a6 CHECK ((char_length(url) <= 255))
);

CREATE SEQUENCE dast_site_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_tokens_id_seq OWNED BY dast_site_tokens.id;

CREATE TABLE dast_site_validations (
    id bigint NOT NULL,
    dast_site_token_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    validation_started_at timestamp with time zone,
    validation_passed_at timestamp with time zone,
    validation_failed_at timestamp with time zone,
    validation_last_retried_at timestamp with time zone,
    validation_strategy smallint NOT NULL,
    url_base text NOT NULL,
    url_path text NOT NULL,
    state text DEFAULT 'pending'::text NOT NULL,
    project_id bigint,
    CONSTRAINT check_13b34efe4b CHECK ((char_length(url_path) <= 255)),
    CONSTRAINT check_283be72e9b CHECK ((char_length(state) <= 255)),
    CONSTRAINT check_cb7514c7d3 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_cd3b538210 CHECK ((char_length(url_base) <= 255))
);

CREATE SEQUENCE dast_site_validations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_validations_id_seq OWNED BY dast_site_validations.id;

CREATE TABLE dast_sites (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    url text NOT NULL,
    dast_site_validation_id bigint,
    CONSTRAINT check_46df8b449c CHECK ((char_length(url) <= 255))
);

CREATE SEQUENCE dast_sites_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_sites_id_seq OWNED BY dast_sites.id;

CREATE TABLE dependency_list_export_part_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_86a96d8348 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE dependency_list_export_parts (
    id bigint NOT NULL,
    dependency_list_export_id bigint NOT NULL,
    start_id bigint NOT NULL,
    end_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store integer,
    file text,
    CONSTRAINT check_f799431fc1 CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE dependency_list_export_parts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_list_export_parts_id_seq OWNED BY dependency_list_export_parts.id;

CREATE TABLE dependency_list_export_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_889220aa2d CHECK ((num_nonnulls(namespace_id, organization_id, project_id) > 0)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE dependency_list_exports (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    user_id bigint,
    file_store integer,
    status smallint DEFAULT 0 NOT NULL,
    file text,
    group_id bigint,
    pipeline_id bigint,
    export_type smallint DEFAULT 0 NOT NULL,
    organization_id bigint,
    expires_at timestamp with time zone,
    send_email boolean DEFAULT false NOT NULL,
    CONSTRAINT check_67a9c23e79 CHECK ((num_nonnulls(group_id, organization_id, project_id) > 0)),
    CONSTRAINT check_fff6fc9b2f CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE dependency_list_exports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_list_exports_id_seq OWNED BY dependency_list_exports.id;

CREATE TABLE dependency_proxy_blob_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    dependency_proxy_blob_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    group_id bigint,
    CONSTRAINT check_8e4f76fffe CHECK ((char_length(verification_failure) <= 255))
);

COMMENT ON TABLE dependency_proxy_blob_states IS '{"owner":"group::geo","description":"Geo-specific table to store the verification state of DependencyProxy::Blob objects"}';

CREATE TABLE dependency_proxy_blobs (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    size bigint,
    file_store integer,
    file_name character varying NOT NULL,
    file text NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    read_at timestamp with time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE dependency_proxy_blobs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_proxy_blobs_id_seq OWNED BY dependency_proxy_blobs.id;

CREATE TABLE dependency_proxy_group_settings (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT true NOT NULL,
    identity jsonb,
    secret jsonb,
    CONSTRAINT check_7ed6c2f608 CHECK (((num_nonnulls(identity, secret) = 2) OR (num_nulls(identity, secret) = 2)))
);

CREATE SEQUENCE dependency_proxy_group_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_proxy_group_settings_id_seq OWNED BY dependency_proxy_group_settings.id;

CREATE TABLE dependency_proxy_image_ttl_group_policies (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    ttl integer DEFAULT 90,
    enabled boolean DEFAULT false NOT NULL
);

CREATE TABLE dependency_proxy_manifest_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    dependency_proxy_manifest_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    group_id bigint,
    CONSTRAINT check_fdd5d9791b CHECK ((char_length(verification_failure) <= 255))
);

CREATE TABLE dependency_proxy_manifests (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    size bigint,
    file_store smallint,
    file_name text NOT NULL,
    file text NOT NULL,
    digest text NOT NULL,
    content_type text,
    status smallint DEFAULT 0 NOT NULL,
    read_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_079b293a7b CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_167a9a8a91 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_c579e3f586 CHECK ((char_length(file_name) <= 255)),
    CONSTRAINT check_f5d9996bf1 CHECK ((char_length(digest) <= 255))
);

CREATE SEQUENCE dependency_proxy_manifests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_proxy_manifests_id_seq OWNED BY dependency_proxy_manifests.id;

CREATE TABLE dependency_proxy_packages_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    enabled boolean DEFAULT false,
    maven_external_registry_url text,
    encrypted_maven_external_registry_username bytea,
    encrypted_maven_external_registry_username_iv bytea,
    encrypted_maven_external_registry_password bytea,
    encrypted_maven_external_registry_password_iv bytea,
    npm_external_registry_url text,
    encrypted_npm_external_registry_basic_auth bytea,
    encrypted_npm_external_registry_basic_auth_iv bytea,
    encrypted_npm_external_registry_auth_token bytea,
    encrypted_npm_external_registry_auth_token_iv bytea,
    CONSTRAINT check_12c046b67f CHECK ((char_length(npm_external_registry_url) <= 255)),
    CONSTRAINT check_14a2818907 CHECK (((num_nulls(encrypted_maven_external_registry_username, encrypted_maven_external_registry_password) = 0) OR (num_nulls(encrypted_maven_external_registry_username, encrypted_maven_external_registry_password) = 2))),
    CONSTRAINT check_353c7ecafd CHECK ((octet_length(encrypted_maven_external_registry_username) <= 1020)),
    CONSTRAINT check_48643112c8 CHECK ((octet_length(encrypted_npm_external_registry_auth_token) <= 1020)),
    CONSTRAINT check_54126e21c1 CHECK ((octet_length(encrypted_npm_external_registry_basic_auth) <= 1020)),
    CONSTRAINT check_7fafb5606e CHECK ((octet_length(encrypted_npm_external_registry_basic_auth_iv) <= 1020)),
    CONSTRAINT check_93afb1690f CHECK ((num_nulls(encrypted_npm_external_registry_basic_auth, encrypted_npm_external_registry_auth_token) > 0)),
    CONSTRAINT check_ac55c514a5 CHECK ((char_length(maven_external_registry_url) <= 255)),
    CONSTRAINT check_c6f700648d CHECK ((octet_length(encrypted_maven_external_registry_password) <= 1020)),
    CONSTRAINT check_c8613a3d35 CHECK ((octet_length(encrypted_npm_external_registry_auth_token_iv) <= 1020)),
    CONSTRAINT check_cdf5f9a434 CHECK ((octet_length(encrypted_maven_external_registry_password_iv) <= 1020)),
    CONSTRAINT check_fd5def68ba CHECK ((octet_length(encrypted_maven_external_registry_username_iv) <= 1020))
);

CREATE TABLE deploy_keys_projects (
    id bigint NOT NULL,
    deploy_key_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    can_push boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE deploy_keys_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deploy_keys_projects_id_seq OWNED BY deploy_keys_projects.id;

CREATE TABLE deploy_tokens (
    id bigint NOT NULL,
    revoked boolean DEFAULT false,
    read_repository boolean DEFAULT false NOT NULL,
    read_registry boolean DEFAULT false NOT NULL,
    expires_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    username character varying,
    token_encrypted character varying(255),
    deploy_token_type smallint DEFAULT 2 NOT NULL,
    write_registry boolean DEFAULT false NOT NULL,
    read_package_registry boolean DEFAULT false NOT NULL,
    write_package_registry boolean DEFAULT false NOT NULL,
    creator_id bigint,
    read_virtual_registry boolean DEFAULT false NOT NULL,
    project_id bigint,
    group_id bigint,
    write_virtual_registry boolean DEFAULT false NOT NULL,
    seven_days_notification_sent_at timestamp with time zone,
    thirty_days_notification_sent_at timestamp with time zone,
    sixty_days_notification_sent_at timestamp with time zone,
    CONSTRAINT check_e2ab92a2f6 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE deploy_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deploy_tokens_id_seq OWNED BY deploy_tokens.id;

CREATE TABLE deployment_approvals (
    id bigint NOT NULL,
    deployment_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint NOT NULL,
    comment text,
    approval_rule_id bigint,
    project_id bigint,
    ci_build_id bigint,
    CONSTRAINT check_692c1d1b90 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_e2eb6a17d8 CHECK ((char_length(comment) <= 255))
);

CREATE SEQUENCE deployment_approvals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deployment_approvals_id_seq OWNED BY deployment_approvals.id;

CREATE TABLE deployment_clusters (
    deployment_id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    kubernetes_namespace character varying(255),
    project_id bigint
);

CREATE TABLE deployment_merge_requests (
    deployment_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    environment_id bigint,
    project_id bigint,
    CONSTRAINT check_5ae1fd5b0e CHECK ((project_id IS NOT NULL))
);

CREATE TABLE deployments (
    id bigint NOT NULL,
    iid integer NOT NULL,
    project_id bigint NOT NULL,
    environment_id bigint NOT NULL,
    ref character varying NOT NULL,
    tag boolean NOT NULL,
    sha character varying NOT NULL,
    user_id bigint,
    deployable_type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    on_stop character varying,
    status smallint NOT NULL,
    finished_at timestamp with time zone,
    deployable_id bigint,
    archived boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE deployments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deployments_id_seq OWNED BY deployments.id;

CREATE TABLE description_versions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_id bigint,
    merge_request_id bigint,
    epic_id bigint,
    description text,
    deleted_at timestamp with time zone,
    namespace_id bigint NOT NULL,
    CONSTRAINT check_76c1eb7122 CHECK ((num_nonnulls(epic_id, issue_id, merge_request_id) = 1))
);

CREATE SEQUENCE description_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE description_versions_id_seq OWNED BY description_versions.id;

CREATE TABLE design_management_action_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_63456653da CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE design_management_designs (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    issue_id bigint,
    filename character varying NOT NULL,
    relative_position integer,
    iid integer,
    cached_markdown_version integer,
    description text,
    description_html text,
    imported_from smallint DEFAULT 0 NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_07155e2715 CHECK ((char_length((filename)::text) <= 255)),
    CONSTRAINT check_aaf9fa6ae5 CHECK ((char_length(description) <= 1000000)),
    CONSTRAINT check_cfb92df01a CHECK ((iid IS NOT NULL)),
    CONSTRAINT check_ed4c70e3f1 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE design_management_designs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_designs_id_seq OWNED BY design_management_designs.id;

CREATE TABLE design_management_designs_versions (
    id bigint NOT NULL,
    design_id bigint NOT NULL,
    version_id bigint NOT NULL,
    event smallint DEFAULT 0 NOT NULL,
    image_v432x230 character varying(255),
    namespace_id bigint,
    CONSTRAINT check_ae7359f44b CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE design_management_designs_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_designs_versions_id_seq OWNED BY design_management_designs_versions.id;

CREATE TABLE design_management_repositories (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_4f95da7ef9 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE design_management_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_repositories_id_seq OWNED BY design_management_repositories.id;

CREATE TABLE design_management_repository_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    design_management_repository_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    namespace_id bigint,
    CONSTRAINT check_380bdde342 CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_bf1387c28b CHECK ((char_length(verification_failure) <= 255))
);

CREATE TABLE design_management_versions (
    id bigint NOT NULL,
    sha bytea NOT NULL,
    issue_id bigint,
    created_at timestamp with time zone NOT NULL,
    author_id bigint,
    namespace_id bigint,
    CONSTRAINT check_1d0291f47a CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE design_management_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_versions_id_seq OWNED BY design_management_versions.id;

CREATE TABLE design_user_mentions (
    id bigint NOT NULL,
    design_id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    note_id bigint NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_6e9fbd8673 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE design_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_user_mentions_id_seq OWNED BY design_user_mentions.id;

CREATE TABLE designated_beneficiaries (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT CURRENT_TIMESTAMP NOT NULL,
    updated_at timestamp with time zone DEFAULT CURRENT_TIMESTAMP NOT NULL,
    type smallint NOT NULL,
    name text NOT NULL,
    relationship text,
    email text,
    CONSTRAINT check_6f552fd9ec CHECK ((char_length(relationship) <= 255)),
    CONSTRAINT check_a9a32c82cb CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_db6bb32865 CHECK ((char_length(email) <= 255))
);

CREATE SEQUENCE designated_beneficiaries_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE designated_beneficiaries_id_seq OWNED BY designated_beneficiaries.id;

CREATE TABLE detached_partitions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    drop_after timestamp with time zone NOT NULL,
    table_name text NOT NULL,
    CONSTRAINT check_aecee24ba3 CHECK ((char_length(table_name) <= 63))
);

CREATE SEQUENCE detached_partitions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE detached_partitions_id_seq OWNED BY detached_partitions.id;

CREATE TABLE diff_note_positions (
    id bigint NOT NULL,
    note_id bigint NOT NULL,
    old_line integer,
    new_line integer,
    diff_content_type smallint NOT NULL,
    diff_type smallint NOT NULL,
    line_code character varying(255) NOT NULL,
    base_sha bytea NOT NULL,
    start_sha bytea NOT NULL,
    head_sha bytea NOT NULL,
    old_path text NOT NULL,
    new_path text NOT NULL,
    namespace_id bigint
);

CREATE SEQUENCE diff_note_positions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE diff_note_positions_id_seq OWNED BY diff_note_positions.id;

CREATE TABLE dingtalk_tracker_data (
    id bigint NOT NULL,
    integration_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    corpid text,
    CONSTRAINT check_d3fe332e6a CHECK ((char_length(corpid) <= 255))
);

COMMENT ON TABLE dingtalk_tracker_data IS 'JiHu-specific table';

COMMENT ON COLUMN dingtalk_tracker_data.integration_id IS 'JiHu-specific column';

COMMENT ON COLUMN dingtalk_tracker_data.corpid IS 'JiHu-specific column';

CREATE SEQUENCE dingtalk_tracker_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dingtalk_tracker_data_id_seq OWNED BY dingtalk_tracker_data.id;

CREATE TABLE dora_configurations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    branches_for_lead_time_for_changes text[] DEFAULT '{}'::text[] NOT NULL
);

CREATE SEQUENCE dora_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dora_configurations_id_seq OWNED BY dora_configurations.id;

CREATE TABLE dora_daily_metrics (
    id bigint NOT NULL,
    environment_id bigint NOT NULL,
    date date NOT NULL,
    deployment_frequency integer,
    lead_time_for_changes_in_seconds integer,
    time_to_restore_service_in_seconds integer,
    incidents_count integer,
    project_id bigint,
    CONSTRAINT check_25a1971e50 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT dora_daily_metrics_deployment_frequency_positive CHECK ((deployment_frequency >= 0)),
    CONSTRAINT dora_daily_metrics_lead_time_for_changes_in_seconds_positive CHECK ((lead_time_for_changes_in_seconds >= 0))
);

CREATE SEQUENCE dora_daily_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dora_daily_metrics_id_seq OWNED BY dora_daily_metrics.id;

CREATE TABLE dora_performance_scores (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    deployment_frequency smallint,
    lead_time_for_changes smallint,
    time_to_restore_service smallint,
    change_failure_rate smallint
);

CREATE SEQUENCE dora_performance_scores_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dora_performance_scores_id_seq OWNED BY dora_performance_scores.id;

CREATE TABLE draft_notes (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    author_id bigint NOT NULL,
    resolve_discussion boolean DEFAULT false NOT NULL,
    discussion_id character varying,
    note text NOT NULL,
    "position" text,
    original_position text,
    change_position text,
    commit_id bytea,
    line_code text,
    internal boolean DEFAULT false NOT NULL,
    note_type smallint,
    project_id bigint,
    CONSTRAINT check_2a752d05fe CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_c497a94a0e CHECK ((char_length(line_code) <= 255))
);

CREATE SEQUENCE draft_notes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE draft_notes_id_seq OWNED BY draft_notes.id;

CREATE TABLE duo_workflows_checkpoint_writes (
    id bigint NOT NULL,
    workflow_id bigint NOT NULL,
    project_id bigint,
    idx integer NOT NULL,
    thread_ts text NOT NULL,
    task text NOT NULL,
    channel text NOT NULL,
    write_type text NOT NULL,
    data text NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_38dc205bb2 CHECK ((char_length(data) <= 10000)),
    CONSTRAINT check_3d119c06ee CHECK ((num_nonnulls(namespace_id, project_id) = 1)),
    CONSTRAINT check_c64af76670 CHECK ((char_length(write_type) <= 255)),
    CONSTRAINT check_d66d09c813 CHECK ((char_length(task) <= 255)),
    CONSTRAINT check_ddb83bc2d5 CHECK ((char_length(channel) <= 255)),
    CONSTRAINT check_f12e0c8f88 CHECK ((char_length(thread_ts) <= 255))
);

CREATE SEQUENCE duo_workflows_checkpoint_writes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE duo_workflows_checkpoint_writes_id_seq OWNED BY duo_workflows_checkpoint_writes.id;

CREATE TABLE duo_workflows_events (
    id bigint NOT NULL,
    workflow_id bigint NOT NULL,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    event_type smallint NOT NULL,
    event_status smallint NOT NULL,
    message text,
    correlation_id_value text,
    namespace_id bigint,
    CONSTRAINT check_125840165c CHECK ((char_length(message) <= 16384)),
    CONSTRAINT check_5e35596b00 CHECK ((char_length(correlation_id_value) <= 128)),
    CONSTRAINT check_9014d33202 CHECK ((num_nonnulls(namespace_id, project_id) = 1))
);

CREATE SEQUENCE duo_workflows_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE duo_workflows_events_id_seq OWNED BY duo_workflows_events.id;

CREATE TABLE duo_workflows_workflows (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    goal text,
    agent_privileges smallint[] DEFAULT '{1,2}'::smallint[] NOT NULL,
    workflow_definition text DEFAULT 'software_development'::text NOT NULL,
    allow_agent_to_request_user boolean DEFAULT true NOT NULL,
    pre_approved_agent_privileges smallint[] DEFAULT '{1,2}'::smallint[] NOT NULL,
    image text,
    environment smallint,
    namespace_id bigint,
    ai_catalog_item_version_id bigint,
    issue_id bigint,
    merge_request_id bigint,
    CONSTRAINT check_30ca07a4ef CHECK ((char_length(goal) <= 16384)),
    CONSTRAINT check_3a9162f1ae CHECK ((char_length(image) <= 2048)),
    CONSTRAINT check_73884a5839 CHECK ((num_nonnulls(namespace_id, project_id) = 1)),
    CONSTRAINT check_ec723e2a1a CHECK ((char_length(workflow_definition) <= 255)),
    CONSTRAINT check_workflows_single_noteable CHECK ((num_nonnulls(issue_id, merge_request_id) <= 1))
);

CREATE SEQUENCE duo_workflows_workflows_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE duo_workflows_workflows_id_seq OWNED BY duo_workflows_workflows.id;

CREATE TABLE duo_workflows_workloads (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    workflow_id bigint NOT NULL,
    workload_id bigint NOT NULL
);

CREATE SEQUENCE duo_workflows_workloads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE duo_workflows_workloads_id_seq OWNED BY duo_workflows_workloads.id;

CREATE TABLE early_access_program_tracking_events (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    event_name text NOT NULL,
    event_label text,
    category text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_4faf711d09 CHECK ((char_length(event_label) <= 255)),
    CONSTRAINT check_573cf46c14 CHECK ((char_length(event_name) <= 255)),
    CONSTRAINT check_e631e07806 CHECK ((char_length(category) <= 255))
);

CREATE SEQUENCE early_access_program_tracking_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE early_access_program_tracking_events_id_seq OWNED BY early_access_program_tracking_events.id;

CREATE TABLE elastic_group_index_statuses (
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    wiki_indexed_at timestamp with time zone,
    last_wiki_commit bytea
);

CREATE TABLE elastic_index_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    number_of_replicas smallint DEFAULT 1 NOT NULL,
    number_of_shards smallint DEFAULT 5 NOT NULL,
    alias_name text NOT NULL,
    CONSTRAINT check_c30005c325 CHECK ((char_length(alias_name) <= 255))
);

CREATE SEQUENCE elastic_index_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_index_settings_id_seq OWNED BY elastic_index_settings.id;

CREATE TABLE elastic_reindexing_slices (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    elastic_reindexing_subtask_id bigint NOT NULL,
    elastic_slice smallint DEFAULT 0 NOT NULL,
    elastic_max_slice smallint DEFAULT 0 NOT NULL,
    retry_attempt smallint DEFAULT 0 NOT NULL,
    elastic_task text,
    CONSTRAINT check_ca30e1396e CHECK ((char_length(elastic_task) <= 255))
);

CREATE SEQUENCE elastic_reindexing_slices_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_reindexing_slices_id_seq OWNED BY elastic_reindexing_slices.id;

CREATE TABLE elastic_reindexing_subtasks (
    id bigint NOT NULL,
    elastic_reindexing_task_id bigint NOT NULL,
    alias_name text NOT NULL,
    index_name_from text NOT NULL,
    index_name_to text NOT NULL,
    elastic_task text,
    documents_count_target integer,
    documents_count integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_4910adc798 CHECK ((char_length(elastic_task) <= 255)),
    CONSTRAINT check_88f56216a4 CHECK ((char_length(alias_name) <= 255)),
    CONSTRAINT check_a1fbd9faa9 CHECK ((char_length(index_name_from) <= 255)),
    CONSTRAINT check_f456494bd8 CHECK ((char_length(index_name_to) <= 255))
);

CREATE SEQUENCE elastic_reindexing_subtasks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_reindexing_subtasks_id_seq OWNED BY elastic_reindexing_subtasks.id;

CREATE TABLE elastic_reindexing_tasks (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    in_progress boolean DEFAULT true NOT NULL,
    error_message text,
    delete_original_index_at timestamp with time zone,
    max_slices_running smallint DEFAULT 60 NOT NULL,
    slice_multiplier smallint DEFAULT 2 NOT NULL,
    targets text[],
    options jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_7f64acda8e CHECK ((char_length(error_message) <= 255))
);

CREATE SEQUENCE elastic_reindexing_tasks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_reindexing_tasks_id_seq OWNED BY elastic_reindexing_tasks.id;

CREATE TABLE elasticsearch_indexed_namespaces (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE TABLE elasticsearch_indexed_projects (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL
);

CREATE TABLE emails (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    email character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    confirmation_token character varying,
    confirmed_at timestamp without time zone,
    confirmation_sent_at timestamp without time zone,
    detumbled_email text,
    CONSTRAINT check_319f6999dc CHECK ((char_length(detumbled_email) <= 255))
);

CREATE SEQUENCE emails_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE emails_id_seq OWNED BY emails.id;

CREATE TABLE enabled_foundational_flows (
    id bigint NOT NULL,
    namespace_id bigint,
    project_id bigint,
    catalog_item_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_0a4362ce38 CHECK ((num_nonnulls(namespace_id, project_id) = 1))
);

CREATE SEQUENCE enabled_foundational_flows_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE enabled_foundational_flows_id_seq OWNED BY enabled_foundational_flows.id;

CREATE TABLE environments (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    external_url character varying,
    environment_type character varying,
    state character varying DEFAULT 'available'::character varying NOT NULL,
    slug character varying NOT NULL,
    auto_stop_at timestamp with time zone,
    auto_delete_at timestamp with time zone,
    tier smallint,
    merge_request_id bigint,
    cluster_agent_id bigint,
    kubernetes_namespace text,
    flux_resource_path text,
    description text,
    description_html text,
    cached_markdown_version integer,
    auto_stop_setting smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_23b1eb18a2 CHECK ((char_length(flux_resource_path) <= 255)),
    CONSTRAINT check_ad5e1ed5e1 CHECK ((char_length(description) <= 10000)),
    CONSTRAINT check_b5373a1804 CHECK ((char_length(kubernetes_namespace) <= 63)),
    CONSTRAINT check_d26221226f CHECK ((char_length(description_html) <= 50000))
);

CREATE SEQUENCE environments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE environments_id_seq OWNED BY environments.id;

CREATE TABLE epic_issues (
    id bigint NOT NULL,
    epic_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    relative_position integer,
    namespace_id bigint,
    work_item_parent_link_id bigint,
    CONSTRAINT check_885d672eec CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE epic_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epic_issues_id_seq OWNED BY epic_issues.id;

CREATE TABLE epic_user_mentions (
    id bigint NOT NULL,
    epic_id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    note_id bigint,
    group_id bigint,
    CONSTRAINT check_4865a37c73 CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE epic_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epic_user_mentions_id_seq OWNED BY epic_user_mentions.id;

CREATE TABLE epics (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    author_id bigint NOT NULL,
    assignee_id bigint,
    iid integer NOT NULL,
    cached_markdown_version integer,
    updated_by_id bigint,
    last_edited_by_id bigint,
    lock_version integer DEFAULT 0,
    start_date date,
    end_date date,
    last_edited_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    title character varying NOT NULL,
    title_html character varying NOT NULL,
    description text,
    description_html text,
    start_date_sourcing_milestone_id bigint,
    due_date_sourcing_milestone_id bigint,
    start_date_fixed date,
    due_date_fixed date,
    start_date_is_fixed boolean,
    due_date_is_fixed boolean,
    closed_by_id bigint,
    closed_at timestamp without time zone,
    parent_id bigint,
    relative_position integer,
    state_id smallint DEFAULT 1 NOT NULL,
    start_date_sourcing_epic_id bigint,
    due_date_sourcing_epic_id bigint,
    confidential boolean DEFAULT false NOT NULL,
    external_key character varying(255),
    color text DEFAULT '#1068bf'::text,
    total_opened_issue_weight integer DEFAULT 0 NOT NULL,
    total_closed_issue_weight integer DEFAULT 0 NOT NULL,
    total_opened_issue_count integer DEFAULT 0 NOT NULL,
    total_closed_issue_count integer DEFAULT 0 NOT NULL,
    issue_id bigint,
    imported smallint DEFAULT 0 NOT NULL,
    imported_from smallint DEFAULT 0 NOT NULL,
    work_item_parent_link_id bigint,
    CONSTRAINT check_450724d1bb CHECK ((issue_id IS NOT NULL)),
    CONSTRAINT check_ca608c40b3 CHECK ((char_length(color) <= 7)),
    CONSTRAINT check_fcfb4a93ff CHECK ((lock_version IS NOT NULL))
);

CREATE SEQUENCE epics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epics_id_seq OWNED BY epics.id;

CREATE TABLE error_tracking_client_keys (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    active boolean DEFAULT true NOT NULL,
    public_key text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_840b719790 CHECK ((char_length(public_key) <= 255))
);

CREATE SEQUENCE error_tracking_client_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE error_tracking_client_keys_id_seq OWNED BY error_tracking_client_keys.id;

CREATE TABLE error_tracking_error_events (
    id bigint NOT NULL,
    error_id bigint NOT NULL,
    description text NOT NULL,
    environment text,
    level text,
    occurred_at timestamp with time zone NOT NULL,
    payload jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    CONSTRAINT check_4af348772e CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_92ecc3077b CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_c67d5b8007 CHECK ((char_length(level) <= 255)),
    CONSTRAINT check_f4b52474ad CHECK ((char_length(environment) <= 255))
);

CREATE SEQUENCE error_tracking_error_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE error_tracking_error_events_id_seq OWNED BY error_tracking_error_events.id;

CREATE TABLE error_tracking_errors (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    actor text NOT NULL,
    first_seen_at timestamp with time zone DEFAULT now() NOT NULL,
    last_seen_at timestamp with time zone DEFAULT now() NOT NULL,
    platform text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    events_count bigint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_18a758e537 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b5cb4d3888 CHECK ((char_length(actor) <= 255)),
    CONSTRAINT check_c739788b12 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_fe99886883 CHECK ((char_length(platform) <= 255))
);

CREATE SEQUENCE error_tracking_errors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE error_tracking_errors_id_seq OWNED BY error_tracking_errors.id;

CREATE TABLE events (
    project_id bigint,
    author_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    action smallint NOT NULL,
    target_type character varying,
    group_id bigint,
    fingerprint bytea,
    id bigint NOT NULL,
    target_id bigint,
    imported_from smallint DEFAULT 0 NOT NULL,
    personal_namespace_id bigint,
    CONSTRAINT check_97e06e05ad CHECK ((octet_length(fingerprint) <= 128)),
    CONSTRAINT check_events_sharding_key_is_not_null CHECK (((group_id IS NOT NULL) OR (project_id IS NOT NULL) OR (personal_namespace_id IS NOT NULL)))
);

CREATE SEQUENCE events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE events_id_seq OWNED BY events.id;

CREATE TABLE evidences (
    id bigint NOT NULL,
    release_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    summary_sha bytea,
    summary jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_32e833c325 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE evidences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE evidences_id_seq OWNED BY evidences.id;

CREATE TABLE excluded_merge_requests (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL
);

CREATE SEQUENCE excluded_merge_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE excluded_merge_requests_id_seq OWNED BY excluded_merge_requests.id;

CREATE TABLE external_approval_rules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_url text NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_1c64b53ea5 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b634ca168d CHECK ((char_length(external_url) <= 255))
);

CREATE SEQUENCE external_approval_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_approval_rules_id_seq OWNED BY external_approval_rules.id;

CREATE TABLE external_pull_requests (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    pull_request_iid integer NOT NULL,
    status smallint NOT NULL,
    source_branch character varying(255) NOT NULL,
    target_branch character varying(255) NOT NULL,
    source_repository character varying(255) NOT NULL,
    target_repository character varying(255) NOT NULL,
    source_sha bytea NOT NULL,
    target_sha bytea NOT NULL
);

CREATE SEQUENCE external_pull_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_pull_requests_id_seq OWNED BY external_pull_requests.id;

CREATE TABLE external_status_checks (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_url text NOT NULL,
    name text NOT NULL,
    encrypted_shared_secret bytea,
    encrypted_shared_secret_iv bytea,
    CONSTRAINT check_7e3b9eb41a CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_ae0dec3f61 CHECK ((char_length(external_url) <= 255))
);

CREATE SEQUENCE external_status_checks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_status_checks_id_seq OWNED BY external_status_checks.id;

CREATE TABLE external_status_checks_protected_branches (
    id bigint NOT NULL,
    external_status_check_id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_94f27f1988 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE external_status_checks_protected_branches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_status_checks_protected_branches_id_seq OWNED BY external_status_checks_protected_branches.id;

CREATE TABLE feature_gates (
    id bigint NOT NULL,
    feature_key character varying NOT NULL,
    key character varying NOT NULL,
    value character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE feature_gates_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE feature_gates_id_seq OWNED BY feature_gates.id;

CREATE TABLE features (
    id bigint NOT NULL,
    key character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE features_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE features_id_seq OWNED BY features.id;

CREATE TABLE fork_network_members (
    id bigint NOT NULL,
    fork_network_id bigint NOT NULL,
    project_id bigint NOT NULL,
    forked_from_project_id bigint
);

CREATE SEQUENCE fork_network_members_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE fork_network_members_id_seq OWNED BY fork_network_members.id;

CREATE TABLE fork_networks (
    id bigint NOT NULL,
    root_project_id bigint,
    deleted_root_project_name character varying,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE fork_networks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE fork_networks_id_seq OWNED BY fork_networks.id;

CREATE TABLE geo_cache_invalidation_events (
    id bigint NOT NULL,
    key character varying NOT NULL
);

CREATE SEQUENCE geo_cache_invalidation_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_cache_invalidation_events_id_seq OWNED BY geo_cache_invalidation_events.id;

CREATE TABLE geo_event_log (
    id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    cache_invalidation_event_id bigint,
    geo_event_id bigint
);

CREATE SEQUENCE geo_event_log_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_event_log_id_seq OWNED BY geo_event_log.id;

CREATE TABLE geo_events (
    id bigint NOT NULL,
    replicable_name character varying(255) NOT NULL,
    event_name character varying(255) NOT NULL,
    payload jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE geo_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_events_id_seq OWNED BY geo_events.id;

CREATE TABLE geo_node_namespace_links (
    id bigint NOT NULL,
    geo_node_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE geo_node_namespace_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_node_namespace_links_id_seq OWNED BY geo_node_namespace_links.id;

CREATE TABLE geo_node_organization_links (
    id bigint NOT NULL,
    geo_node_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE geo_node_organization_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_node_organization_links_id_seq OWNED BY geo_node_organization_links.id;

CREATE TABLE geo_node_statuses (
    id bigint NOT NULL,
    geo_node_id bigint NOT NULL,
    db_replication_lag_seconds integer,
    last_event_id bigint,
    last_event_date timestamp without time zone,
    cursor_last_event_id bigint,
    cursor_last_event_date timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    last_successful_status_check_at timestamp without time zone,
    status_message character varying,
    replication_slots_count integer,
    replication_slots_used_count integer,
    replication_slots_max_retained_wal_bytes bigint,
    version character varying,
    revision character varying,
    storage_configuration_digest bytea,
    projects_count integer,
    status jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE geo_node_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_node_statuses_id_seq OWNED BY geo_node_statuses.id;

CREATE TABLE geo_nodes (
    id bigint NOT NULL,
    "primary" boolean DEFAULT false NOT NULL,
    oauth_application_id bigint,
    enabled boolean DEFAULT true NOT NULL,
    access_key character varying,
    encrypted_secret_access_key character varying,
    encrypted_secret_access_key_iv character varying,
    clone_url_prefix character varying,
    files_max_capacity integer DEFAULT 10 NOT NULL,
    repos_max_capacity integer DEFAULT 10 NOT NULL,
    url character varying NOT NULL,
    selective_sync_type character varying,
    selective_sync_shards text,
    verification_max_capacity integer DEFAULT 10 NOT NULL,
    minimum_reverification_interval integer DEFAULT 90 NOT NULL,
    internal_url character varying,
    name character varying NOT NULL,
    container_repositories_max_capacity integer DEFAULT 2 NOT NULL,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    sync_object_storage boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE geo_nodes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_nodes_id_seq OWNED BY geo_nodes.id;

CREATE TABLE ghost_user_migrations (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    initiator_user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    hard_delete boolean DEFAULT false NOT NULL,
    consume_after timestamp with time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE ghost_user_migrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ghost_user_migrations_id_seq OWNED BY ghost_user_migrations.id;

CREATE TABLE gitlab_subscription_histories (
    id bigint NOT NULL,
    gitlab_subscription_created_at timestamp with time zone,
    gitlab_subscription_updated_at timestamp with time zone,
    start_date date,
    end_date date,
    trial_ends_on date,
    namespace_id bigint,
    hosted_plan_id bigint,
    max_seats_used integer,
    seats integer,
    trial boolean,
    change_type smallint,
    gitlab_subscription_id bigint NOT NULL,
    created_at timestamp with time zone,
    trial_starts_on date,
    auto_renew boolean,
    trial_extension_type smallint,
    seats_in_use integer,
    hosted_plan_name_uid smallint,
    CONSTRAINT check_6d5f27a106 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE gitlab_subscription_histories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gitlab_subscription_histories_id_seq OWNED BY gitlab_subscription_histories.id;

CREATE TABLE gitlab_subscriptions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    end_date date,
    trial_ends_on date,
    namespace_id bigint,
    hosted_plan_id bigint,
    max_seats_used integer DEFAULT 0,
    seats integer DEFAULT 0,
    trial boolean DEFAULT false,
    trial_starts_on date,
    auto_renew boolean,
    seats_in_use integer DEFAULT 0 NOT NULL,
    seats_owed integer DEFAULT 0 NOT NULL,
    trial_extension_type smallint,
    max_seats_used_changed_at timestamp with time zone,
    last_seat_refresh_at timestamp with time zone,
    hosted_plan_name_uid smallint,
    CONSTRAINT check_77fea3f0e7 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE gitlab_subscriptions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gitlab_subscriptions_id_seq OWNED BY gitlab_subscriptions.id;

CREATE TABLE gpg_key_subkeys (
    id bigint NOT NULL,
    gpg_key_id bigint NOT NULL,
    keyid bytea,
    fingerprint bytea,
    user_id bigint
);

CREATE SEQUENCE gpg_key_subkeys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gpg_key_subkeys_id_seq OWNED BY gpg_key_subkeys.id;

CREATE TABLE gpg_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint,
    primary_keyid bytea,
    fingerprint bytea,
    key text,
    externally_verified boolean DEFAULT false NOT NULL,
    externally_verified_at timestamp with time zone,
    CONSTRAINT check_db8a6cf29e CHECK ((user_id IS NOT NULL))
);

CREATE SEQUENCE gpg_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gpg_keys_id_seq OWNED BY gpg_keys.id;

CREATE TABLE gpg_signatures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    gpg_key_id bigint,
    commit_sha bytea,
    gpg_key_primary_keyid bytea,
    gpg_key_user_name text,
    gpg_key_user_email text,
    verification_status smallint DEFAULT 0 NOT NULL,
    gpg_key_subkey_id bigint,
    committer_email text,
    CONSTRAINT check_0ce2b9f622 CHECK ((char_length(committer_email) <= 255)),
    CONSTRAINT check_271c7cad6d CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE gpg_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gpg_signatures_id_seq OWNED BY gpg_signatures.id;

CREATE TABLE grafana_integrations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_token character varying(255) NOT NULL,
    encrypted_token_iv character varying(255) NOT NULL,
    grafana_url character varying(1024) NOT NULL,
    enabled boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE grafana_integrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE grafana_integrations_id_seq OWNED BY grafana_integrations.id;

CREATE TABLE granular_scopes (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    namespace_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    permissions jsonb DEFAULT '[]'::jsonb NOT NULL,
    access smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_permissions_is_array CHECK ((jsonb_typeof(permissions) = 'array'::text))
);

CREATE SEQUENCE granular_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE granular_scopes_id_seq OWNED BY granular_scopes.id;

CREATE TABLE group_crm_settings (
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT true NOT NULL,
    source_group_id bigint
);

CREATE SEQUENCE group_crm_settings_group_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_crm_settings_group_id_seq OWNED BY group_crm_settings.group_id;

CREATE TABLE group_custom_attributes (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    key character varying NOT NULL,
    value character varying NOT NULL
);

CREATE SEQUENCE group_custom_attributes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_custom_attributes_id_seq OWNED BY group_custom_attributes.id;

CREATE TABLE group_deletion_schedules (
    group_id bigint NOT NULL,
    user_id bigint NOT NULL,
    marked_for_deletion_on date NOT NULL
);

CREATE TABLE group_deploy_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    deploy_token_id bigint NOT NULL
);

CREATE SEQUENCE group_deploy_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_deploy_tokens_id_seq OWNED BY group_deploy_tokens.id;

CREATE TABLE group_features (
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    wiki_access_level smallint DEFAULT 20 NOT NULL
);

CREATE TABLE group_group_links (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    shared_group_id bigint NOT NULL,
    shared_with_group_id bigint NOT NULL,
    expires_at date,
    group_access smallint DEFAULT 30 NOT NULL,
    member_role_id bigint
);

CREATE SEQUENCE group_group_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_group_links_id_seq OWNED BY group_group_links.id;

CREATE TABLE group_import_states (
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    jid text,
    last_error text,
    user_id bigint,
    CONSTRAINT check_87b58f6b30 CHECK ((char_length(last_error) <= 255)),
    CONSTRAINT check_96558fff96 CHECK ((char_length(jid) <= 100))
);

CREATE SEQUENCE group_import_states_group_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_import_states_group_id_seq OWNED BY group_import_states.group_id;

CREATE TABLE group_merge_request_approval_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    allow_author_approval boolean DEFAULT false NOT NULL,
    allow_committer_approval boolean DEFAULT false NOT NULL,
    allow_overrides_to_approver_list_per_merge_request boolean DEFAULT false NOT NULL,
    retain_approvals_on_push boolean DEFAULT false NOT NULL,
    require_password_to_approve boolean DEFAULT false NOT NULL,
    require_saml_auth_to_approve boolean DEFAULT false NOT NULL,
    require_reauthentication_to_approve boolean DEFAULT false NOT NULL
);

CREATE TABLE group_push_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    max_file_size integer DEFAULT 0 NOT NULL,
    member_check boolean DEFAULT false NOT NULL,
    prevent_secrets boolean DEFAULT false NOT NULL,
    commit_committer_name_check boolean DEFAULT false NOT NULL,
    deny_delete_tag boolean,
    reject_unsigned_commits boolean,
    commit_committer_check boolean,
    reject_non_dco_commits boolean,
    commit_message_regex text,
    branch_name_regex text,
    commit_message_negative_regex text,
    author_email_regex text,
    file_name_regex text,
    CONSTRAINT check_0bba2c16da CHECK ((char_length(commit_message_negative_regex) <= 2047)),
    CONSTRAINT check_41c1a11ab8 CHECK ((char_length(file_name_regex) <= 511)),
    CONSTRAINT check_6f0da85c6c CHECK ((char_length(commit_message_regex) <= 511)),
    CONSTRAINT check_710cf4213a CHECK ((char_length(author_email_regex) <= 511)),
    CONSTRAINT check_b02376d0ad CHECK ((char_length(branch_name_regex) <= 511))
);

CREATE SEQUENCE group_push_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_push_rules_id_seq OWNED BY group_push_rules.id;

CREATE TABLE group_repository_storage_moves (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    source_storage_name text NOT NULL,
    destination_storage_name text NOT NULL,
    error_message text,
    CONSTRAINT check_266d0cf596 CHECK ((char_length(error_message) <= 256)),
    CONSTRAINT group_repository_storage_moves_destination_storage_name CHECK ((char_length(destination_storage_name) <= 255)),
    CONSTRAINT group_repository_storage_moves_source_storage_name CHECK ((char_length(source_storage_name) <= 255))
);

CREATE SEQUENCE group_repository_storage_moves_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_repository_storage_moves_id_seq OWNED BY group_repository_storage_moves.id;

CREATE TABLE group_saved_replies (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    content text NOT NULL,
    CONSTRAINT check_13510378d3 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_4a96378d43 CHECK ((char_length(content) <= 10000))
);

CREATE SEQUENCE group_saved_replies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_saved_replies_id_seq OWNED BY group_saved_replies.id;

CREATE TABLE group_scim_auth_access_tokens (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    token_encrypted bytea NOT NULL
);

CREATE SEQUENCE group_scim_auth_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_scim_auth_access_tokens_id_seq OWNED BY group_scim_auth_access_tokens.id;

CREATE TABLE group_scim_identities (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT false,
    extern_uid text NOT NULL,
    CONSTRAINT check_53de3ba272 CHECK ((char_length(extern_uid) <= 255))
);

CREATE SEQUENCE group_scim_identities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_scim_identities_id_seq OWNED BY group_scim_identities.id;

CREATE TABLE group_secrets_managers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint,
    status smallint DEFAULT 0 NOT NULL,
    group_path text,
    CONSTRAINT check_f88f28737e CHECK ((char_length(group_path) <= 64))
);

CREATE SEQUENCE group_secrets_managers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_secrets_managers_id_seq OWNED BY group_secrets_managers.id;

CREATE TABLE group_security_exclusions (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    scanner smallint NOT NULL,
    type smallint NOT NULL,
    active boolean DEFAULT true NOT NULL,
    description text,
    value text NOT NULL,
    CONSTRAINT check_12e9b59302 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_45d9a8e422 CHECK ((char_length(value) <= 255))
);

CREATE SEQUENCE group_security_exclusions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_security_exclusions_id_seq OWNED BY group_security_exclusions.id;

CREATE TABLE group_ssh_certificates (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    fingerprint bytea NOT NULL,
    title text NOT NULL,
    key text NOT NULL,
    CONSTRAINT check_480296705f CHECK ((char_length(title) <= 256)),
    CONSTRAINT check_6c1d920167 CHECK ((char_length(key) <= 524288))
);

CREATE SEQUENCE group_ssh_certificates_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_ssh_certificates_id_seq OWNED BY group_ssh_certificates.id;

CREATE TABLE group_type_ci_runner_machines (
    id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    contacted_at timestamp with time zone,
    creation_state smallint DEFAULT 0 NOT NULL,
    executor_type smallint,
    runner_type smallint NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL,
    system_xid text NOT NULL,
    platform text,
    architecture text,
    revision text,
    ip_address text,
    version text,
    runtime_features jsonb DEFAULT '{}'::jsonb NOT NULL,
    organization_id bigint,
    labels jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_3d8736b3af CHECK ((char_length(system_xid) <= 64)),
    CONSTRAINT check_5bad2a6944 CHECK ((char_length(revision) <= 255)),
    CONSTRAINT check_7dc4eee8a5 CHECK ((char_length(version) <= 2048)),
    CONSTRAINT check_b1e456641b CHECK ((char_length(ip_address) <= 1024)),
    CONSTRAINT check_c788f4b18a CHECK ((char_length(platform) <= 255)),
    CONSTRAINT check_f3d25ab844 CHECK ((char_length(architecture) <= 255)),
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NOT NULL))
);

CREATE TABLE group_type_ci_runners (
    id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    contacted_at timestamp with time zone,
    token_expires_at timestamp with time zone,
    public_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    private_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    access_level integer DEFAULT 0 NOT NULL,
    maximum_timeout integer,
    runner_type smallint NOT NULL,
    registration_type smallint DEFAULT 0 NOT NULL,
    creation_state smallint DEFAULT 0 NOT NULL,
    active boolean DEFAULT true NOT NULL,
    run_untagged boolean DEFAULT true NOT NULL,
    locked boolean DEFAULT false NOT NULL,
    name text,
    token_encrypted text,
    description text,
    maintainer_note text,
    allowed_plans text[] DEFAULT '{}'::text[] NOT NULL,
    allowed_plan_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    organization_id bigint,
    allowed_plan_name_uids smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    CONSTRAINT check_030ad0773d CHECK ((char_length(token_encrypted) <= 512)),
    CONSTRAINT check_1f8618ab23 CHECK ((char_length(name) <= 256)),
    CONSTRAINT check_24b281f5bf CHECK ((char_length(maintainer_note) <= 1024)),
    CONSTRAINT check_5db8ae9d30 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NOT NULL))
);

CREATE TABLE group_wiki_repositories (
    shard_id bigint NOT NULL,
    group_id bigint NOT NULL,
    disk_path text NOT NULL,
    CONSTRAINT check_07f1c81806 CHECK ((char_length(disk_path) <= 80))
);

CREATE TABLE group_wiki_repository_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    group_wiki_repository_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    group_id bigint,
    CONSTRAINT check_14d288436d CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE group_wiki_repository_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_wiki_repository_states_id_seq OWNED BY group_wiki_repository_states.id;

CREATE TABLE routes (
    id bigint NOT NULL,
    source_id bigint NOT NULL,
    source_type character varying NOT NULL,
    path character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    name character varying,
    namespace_id bigint,
    CONSTRAINT check_af84c6c93f CHECK ((namespace_id IS NOT NULL))
);

CREATE TABLE shards (
    id bigint NOT NULL,
    name character varying NOT NULL
);

CREATE VIEW group_wikis_routes_view AS
 SELECT gr.group_id,
    sh.name AS repository_storage,
    gr.disk_path,
    r.path AS path_with_namespace,
    r.name AS name_with_namespace
   FROM ((group_wiki_repositories gr
     JOIN routes r ON (((r.source_id = gr.group_id) AND ((r.source_type)::text = 'Namespace'::text))))
     JOIN shards sh ON ((gr.shard_id = sh.id)));

CREATE SEQUENCE groups_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE groups_visits_id_seq OWNED BY groups_visits.id;

CREATE TABLE historical_data (
    id bigint NOT NULL,
    date date,
    active_user_count integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    recorded_at timestamp with time zone,
    CONSTRAINT check_640e8cf66c CHECK ((recorded_at IS NOT NULL))
);

CREATE SEQUENCE historical_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE historical_data_id_seq OWNED BY historical_data.id;

CREATE TABLE identities (
    id bigint NOT NULL,
    extern_uid character varying,
    provider character varying,
    user_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    secondary_extern_uid character varying,
    saml_provider_id bigint,
    trusted_extern_uid boolean DEFAULT true,
    CONSTRAINT check_e6693ca8db CHECK ((user_id IS NOT NULL))
);

CREATE SEQUENCE identities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE identities_id_seq OWNED BY identities.id;

CREATE TABLE import_export_upload_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_5c88ee613c CHECK ((num_nonnulls(namespace_id, project_id) = 1)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE import_export_uploads (
    id bigint NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    import_file text,
    export_file text,
    group_id bigint,
    remote_import_url text,
    user_id bigint,
    CONSTRAINT check_58f0d37481 CHECK ((char_length(remote_import_url) <= 2048)),
    CONSTRAINT check_e54579866d CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE import_export_uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_export_uploads_id_seq OWNED BY import_export_uploads.id;

CREATE TABLE import_failures (
    id bigint NOT NULL,
    relation_index integer,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    relation_key character varying(64),
    exception_class character varying(128),
    correlation_id_value character varying(128),
    exception_message character varying(255),
    retry_count integer,
    group_id bigint,
    source character varying(128),
    external_identifiers jsonb DEFAULT '{}'::jsonb NOT NULL,
    user_id bigint,
    organization_id bigint,
    CONSTRAINT check_df3c29175a CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1))
);

CREATE SEQUENCE import_failures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_failures_id_seq OWNED BY import_failures.id;

CREATE TABLE import_offline_configurations (
    id bigint NOT NULL,
    offline_export_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    provider smallint NOT NULL,
    bucket text NOT NULL,
    export_prefix text NOT NULL,
    object_storage_credentials jsonb NOT NULL,
    CONSTRAINT check_94d334d71c CHECK ((char_length(bucket) <= 256)),
    CONSTRAINT check_f28fa120fe CHECK ((char_length(export_prefix) <= 255))
);

CREATE SEQUENCE import_offline_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_offline_configurations_id_seq OWNED BY import_offline_configurations.id;

CREATE TABLE import_offline_exports (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    has_failures boolean DEFAULT false NOT NULL,
    source_hostname text NOT NULL,
    CONSTRAINT check_dcd47fbc18 CHECK ((char_length(source_hostname) <= 255))
);

CREATE SEQUENCE import_offline_exports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_offline_exports_id_seq OWNED BY import_offline_exports.id;

CREATE TABLE import_placeholder_memberships (
    id bigint NOT NULL,
    source_user_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    group_id bigint,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    expires_at date,
    access_level smallint NOT NULL
);

CREATE SEQUENCE import_placeholder_memberships_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_placeholder_memberships_id_seq OWNED BY import_placeholder_memberships.id;

CREATE TABLE import_placeholder_user_details (
    id bigint NOT NULL,
    placeholder_user_id bigint NOT NULL,
    namespace_id bigint,
    deletion_attempts integer DEFAULT 0 NOT NULL,
    organization_id bigint NOT NULL,
    last_deletion_attempt_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE import_placeholder_user_details_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_placeholder_user_details_id_seq OWNED BY import_placeholder_user_details.id;

CREATE TABLE import_source_user_placeholder_references (
    id bigint NOT NULL,
    source_user_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    numeric_key bigint,
    created_at timestamp with time zone NOT NULL,
    model text NOT NULL,
    user_reference_column text NOT NULL,
    composite_key jsonb,
    alias_version smallint NOT NULL,
    CONSTRAINT check_782140eb9d CHECK ((char_length(user_reference_column) <= 50)),
    CONSTRAINT check_d17bd9dd4d CHECK ((char_length(model) <= 150))
);

CREATE SEQUENCE import_source_user_placeholder_references_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_source_user_placeholder_references_id_seq OWNED BY import_source_user_placeholder_references.id;

CREATE TABLE import_source_users (
    id bigint NOT NULL,
    placeholder_user_id bigint,
    reassign_to_user_id bigint,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    source_username text,
    source_name text,
    source_user_identifier text NOT NULL,
    source_hostname text NOT NULL,
    import_type text NOT NULL,
    reassigned_by_user_id bigint,
    reassignment_error text,
    reassignment_token text,
    CONSTRAINT check_05708218cd CHECK ((char_length(reassignment_error) <= 255)),
    CONSTRAINT check_0d7295a307 CHECK ((char_length(import_type) <= 255)),
    CONSTRAINT check_199c28ec54 CHECK ((char_length(source_username) <= 255)),
    CONSTRAINT check_562655155f CHECK ((char_length(source_name) <= 255)),
    CONSTRAINT check_cc9d4093b5 CHECK ((char_length(source_user_identifier) <= 255)),
    CONSTRAINT check_cd2edb9334 CHECK ((char_length(reassignment_token) <= 32)),
    CONSTRAINT check_e2039840c5 CHECK ((char_length(source_hostname) <= 255))
);

CREATE SEQUENCE import_source_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_source_users_id_seq OWNED BY import_source_users.id;

CREATE TABLE incident_management_escalation_policies (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    description text,
    CONSTRAINT check_510b2a5258 CHECK ((char_length(description) <= 160)),
    CONSTRAINT check_9a26365850 CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE incident_management_escalation_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_escalation_policies_id_seq OWNED BY incident_management_escalation_policies.id;

CREATE TABLE incident_management_escalation_rules (
    id bigint NOT NULL,
    policy_id bigint NOT NULL,
    oncall_schedule_id bigint,
    status smallint NOT NULL,
    elapsed_time_seconds integer NOT NULL,
    is_removed boolean DEFAULT false NOT NULL,
    user_id bigint,
    project_id bigint,
    CONSTRAINT check_a54b79b2fa CHECK ((project_id IS NOT NULL)),
    CONSTRAINT escalation_rules_one_of_oncall_schedule_or_user CHECK ((num_nonnulls(oncall_schedule_id, user_id) = 1))
);

CREATE SEQUENCE incident_management_escalation_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_escalation_rules_id_seq OWNED BY incident_management_escalation_rules.id;

CREATE TABLE incident_management_issuable_escalation_statuses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_id bigint NOT NULL,
    policy_id bigint,
    escalations_started_at timestamp with time zone,
    resolved_at timestamp with time zone,
    status smallint DEFAULT 0 NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_ad48232311 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE incident_management_issuable_escalation_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_issuable_escalation_statuses_id_seq OWNED BY incident_management_issuable_escalation_statuses.id;

CREATE TABLE incident_management_oncall_participants (
    id bigint NOT NULL,
    oncall_rotation_id bigint NOT NULL,
    user_id bigint NOT NULL,
    color_palette smallint NOT NULL,
    color_weight smallint NOT NULL,
    is_removed boolean DEFAULT false NOT NULL,
    project_id bigint,
    CONSTRAINT check_d53b689825 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE incident_management_oncall_participants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_participants_id_seq OWNED BY incident_management_oncall_participants.id;

CREATE TABLE incident_management_oncall_rotations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    oncall_schedule_id bigint NOT NULL,
    length integer NOT NULL,
    length_unit smallint NOT NULL,
    starts_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    ends_at timestamp with time zone,
    active_period_start time without time zone,
    active_period_end time without time zone,
    project_id bigint,
    CONSTRAINT check_28c39f8a0c CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_5209fb5d02 CHECK ((char_length(name) <= 200))
);

CREATE SEQUENCE incident_management_oncall_rotations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_rotations_id_seq OWNED BY incident_management_oncall_rotations.id;

CREATE TABLE incident_management_oncall_schedules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    iid integer NOT NULL,
    name text NOT NULL,
    description text,
    timezone text,
    CONSTRAINT check_7ed1fd5aa7 CHECK ((char_length(description) <= 1000)),
    CONSTRAINT check_cc77cbb103 CHECK ((char_length(timezone) <= 100)),
    CONSTRAINT check_e6ef43a664 CHECK ((char_length(name) <= 200))
);

CREATE SEQUENCE incident_management_oncall_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_schedules_id_seq OWNED BY incident_management_oncall_schedules.id;

CREATE TABLE incident_management_oncall_shifts (
    id bigint NOT NULL,
    rotation_id bigint NOT NULL,
    participant_id bigint NOT NULL,
    starts_at timestamp with time zone NOT NULL,
    ends_at timestamp with time zone NOT NULL,
    project_id bigint,
    CONSTRAINT check_a37955f387 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE incident_management_oncall_shifts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_shifts_id_seq OWNED BY incident_management_oncall_shifts.id;

CREATE SEQUENCE incident_management_pending_alert_escalations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_pending_alert_escalations_id_seq OWNED BY incident_management_pending_alert_escalations.id;

CREATE SEQUENCE incident_management_pending_issue_escalations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_pending_issue_escalations_id_seq OWNED BY incident_management_pending_issue_escalations.id;

CREATE TABLE incident_management_timeline_event_tag_links (
    id bigint NOT NULL,
    timeline_event_id bigint NOT NULL,
    timeline_event_tag_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    project_id bigint,
    CONSTRAINT check_e693cb4516 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE incident_management_timeline_event_tag_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_timeline_event_tag_links_id_seq OWNED BY incident_management_timeline_event_tag_links.id;

CREATE TABLE incident_management_timeline_event_tags (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_8717184e2c CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE incident_management_timeline_event_tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_timeline_event_tags_id_seq OWNED BY incident_management_timeline_event_tags.id;

CREATE TABLE incident_management_timeline_events (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    occurred_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    author_id bigint,
    issue_id bigint NOT NULL,
    updated_by_user_id bigint,
    promoted_from_note_id bigint,
    cached_markdown_version integer,
    editable boolean DEFAULT false NOT NULL,
    note text NOT NULL,
    note_html text NOT NULL,
    action text NOT NULL,
    CONSTRAINT check_18fd072206 CHECK ((char_length(action) <= 128)),
    CONSTRAINT check_3875ed0aac CHECK ((char_length(note) <= 10000)),
    CONSTRAINT check_94a235d6a4 CHECK ((char_length(note_html) <= 10000))
);

CREATE SEQUENCE incident_management_timeline_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_timeline_events_id_seq OWNED BY incident_management_timeline_events.id;

CREATE TABLE index_statuses (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    indexed_at timestamp without time zone,
    note text,
    last_commit character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    last_wiki_commit bytea,
    wiki_indexed_at timestamp with time zone
);

CREATE SEQUENCE index_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE index_statuses_id_seq OWNED BY index_statuses.id;

CREATE TABLE insights (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE SEQUENCE insights_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE insights_id_seq OWNED BY insights.id;

CREATE TABLE instance_audit_events_streaming_headers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    instance_external_audit_event_destination_id bigint NOT NULL,
    key text NOT NULL,
    value text NOT NULL,
    active boolean DEFAULT true NOT NULL,
    CONSTRAINT check_d52adbbabb CHECK ((char_length(value) <= 2000)),
    CONSTRAINT check_e92010d531 CHECK ((char_length(key) <= 255))
);

CREATE SEQUENCE instance_audit_events_streaming_headers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE instance_audit_events_streaming_headers_id_seq OWNED BY instance_audit_events_streaming_headers.id;

CREATE TABLE instance_model_selection_feature_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    feature smallint NOT NULL,
    offered_model_ref text,
    offered_model_name text,
    CONSTRAINT check_2d921a9d8a CHECK ((char_length(offered_model_ref) <= 255)),
    CONSTRAINT check_6159907afe CHECK ((char_length(offered_model_name) <= 255))
);

CREATE SEQUENCE instance_model_selection_feature_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE instance_model_selection_feature_settings_id_seq OWNED BY instance_model_selection_feature_settings.id;

CREATE TABLE instance_type_ci_runner_machines (
    id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    contacted_at timestamp with time zone,
    creation_state smallint DEFAULT 0 NOT NULL,
    executor_type smallint,
    runner_type smallint NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL,
    system_xid text NOT NULL,
    platform text,
    architecture text,
    revision text,
    ip_address text,
    version text,
    runtime_features jsonb DEFAULT '{}'::jsonb NOT NULL,
    organization_id bigint,
    labels jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_3d8736b3af CHECK ((char_length(system_xid) <= 64)),
    CONSTRAINT check_5bad2a6944 CHECK ((char_length(revision) <= 255)),
    CONSTRAINT check_7dc4eee8a5 CHECK ((char_length(version) <= 2048)),
    CONSTRAINT check_b1e456641b CHECK ((char_length(ip_address) <= 1024)),
    CONSTRAINT check_c788f4b18a CHECK ((char_length(platform) <= 255)),
    CONSTRAINT check_f3d25ab844 CHECK ((char_length(architecture) <= 255)),
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NULL))
);

CREATE TABLE instance_type_ci_runners (
    id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    contacted_at timestamp with time zone,
    token_expires_at timestamp with time zone,
    public_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    private_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    access_level integer DEFAULT 0 NOT NULL,
    maximum_timeout integer,
    runner_type smallint NOT NULL,
    registration_type smallint DEFAULT 0 NOT NULL,
    creation_state smallint DEFAULT 0 NOT NULL,
    active boolean DEFAULT true NOT NULL,
    run_untagged boolean DEFAULT true NOT NULL,
    locked boolean DEFAULT false NOT NULL,
    name text,
    token_encrypted text,
    description text,
    maintainer_note text,
    allowed_plans text[] DEFAULT '{}'::text[] NOT NULL,
    allowed_plan_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    organization_id bigint,
    allowed_plan_name_uids smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    CONSTRAINT check_030ad0773d CHECK ((char_length(token_encrypted) <= 512)),
    CONSTRAINT check_1f8618ab23 CHECK ((char_length(name) <= 256)),
    CONSTRAINT check_24b281f5bf CHECK ((char_length(maintainer_note) <= 1024)),
    CONSTRAINT check_5db8ae9d30 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NULL))
);

CREATE TABLE integrations (
    id bigint NOT NULL,
    project_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    active boolean DEFAULT false NOT NULL,
    push_events boolean DEFAULT true,
    issues_events boolean DEFAULT true,
    merge_requests_events boolean DEFAULT true,
    tag_push_events boolean DEFAULT true,
    note_events boolean DEFAULT true NOT NULL,
    category character varying DEFAULT 'common'::character varying NOT NULL,
    wiki_page_events boolean DEFAULT true,
    pipeline_events boolean DEFAULT false NOT NULL,
    confidential_issues_events boolean DEFAULT true NOT NULL,
    commit_events boolean DEFAULT true NOT NULL,
    job_events boolean DEFAULT false NOT NULL,
    confidential_note_events boolean DEFAULT true,
    deployment_events boolean DEFAULT false NOT NULL,
    comment_on_event_enabled boolean DEFAULT true NOT NULL,
    instance boolean DEFAULT false NOT NULL,
    comment_detail smallint,
    inherit_from_id bigint,
    alert_events boolean,
    group_id bigint,
    type_new text,
    vulnerability_events boolean DEFAULT false NOT NULL,
    archive_trace_events boolean DEFAULT false NOT NULL,
    encrypted_properties bytea,
    encrypted_properties_iv bytea,
    incident_events boolean DEFAULT false NOT NULL,
    group_mention_events boolean DEFAULT false NOT NULL,
    group_confidential_mention_events boolean DEFAULT false NOT NULL,
    organization_id bigint,
    CONSTRAINT check_2aae034509 CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1)),
    CONSTRAINT check_a948a0aa7e CHECK ((char_length(type_new) <= 255))
);

CREATE SEQUENCE integrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE integrations_id_seq OWNED BY integrations.id;

CREATE TABLE internal_ids (
    id bigint NOT NULL,
    project_id bigint,
    usage integer NOT NULL,
    last_value integer NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_5ecc6454b1 CHECK ((num_nonnulls(namespace_id, project_id) = 1))
);

CREATE SEQUENCE internal_ids_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE internal_ids_id_seq OWNED BY internal_ids.id;

CREATE TABLE ip_restrictions (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    range character varying NOT NULL
);

CREATE SEQUENCE ip_restrictions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ip_restrictions_id_seq OWNED BY ip_restrictions.id;

CREATE TABLE issuable_metric_image_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_5bfdc76f5c CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE issuable_metric_images (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint,
    file text NOT NULL,
    url text,
    url_text text,
    namespace_id bigint,
    CONSTRAINT check_10213e2b5c CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_3bc6d47661 CHECK ((char_length(url_text) <= 128)),
    CONSTRAINT check_5b3011e234 CHECK ((char_length(url) <= 255)),
    CONSTRAINT check_7ed527062f CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE issuable_metric_images_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_metric_images_id_seq OWNED BY issuable_metric_images.id;

CREATE TABLE issuable_resource_links (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    link_text text,
    link text NOT NULL,
    link_type smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    is_unique boolean,
    namespace_id bigint,
    CONSTRAINT check_67be6729db CHECK ((char_length(link) <= 2200)),
    CONSTRAINT check_897e487714 CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_b137147e0b CHECK ((char_length(link_text) <= 255))
);

CREATE SEQUENCE issuable_resource_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_resource_links_id_seq OWNED BY issuable_resource_links.id;

CREATE TABLE issuable_severities (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    severity smallint DEFAULT 0 NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_34d8321a84 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issuable_severities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_severities_id_seq OWNED BY issuable_severities.id;

CREATE TABLE issuable_slas (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    due_at timestamp with time zone NOT NULL,
    label_applied boolean DEFAULT false NOT NULL,
    issuable_closed boolean DEFAULT false NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_1ae7689c41 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issuable_slas_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_slas_id_seq OWNED BY issuable_slas.id;

CREATE TABLE issue_assignees (
    user_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_d88fe18cfa CHECK ((namespace_id IS NOT NULL))
);

CREATE TABLE issue_assignment_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    action smallint DEFAULT 1 NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_b1ee75f25d CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issue_assignment_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_assignment_events_id_seq OWNED BY issue_assignment_events.id;

CREATE TABLE issue_customer_relations_contacts (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    contact_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_9fd68b6ded CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issue_customer_relations_contacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_customer_relations_contacts_id_seq OWNED BY issue_customer_relations_contacts.id;

CREATE TABLE issue_email_participants (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    email text NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_2c321d408d CHECK ((char_length(email) <= 255)),
    CONSTRAINT check_9d8a1ecc85 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issue_email_participants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_email_participants_id_seq OWNED BY issue_email_participants.id;

CREATE TABLE issue_emails (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    email_message_id text NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_0c81c7ea61 CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_5abf3e6aea CHECK ((char_length(email_message_id) <= 1000))
);

CREATE SEQUENCE issue_emails_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_emails_id_seq OWNED BY issue_emails.id;

CREATE TABLE issue_links (
    id bigint NOT NULL,
    source_id bigint NOT NULL,
    target_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    link_type smallint DEFAULT 0 NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_c32f659c75 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issue_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_links_id_seq OWNED BY issue_links.id;

CREATE TABLE issue_metrics (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    first_mentioned_in_commit_at timestamp without time zone,
    first_associated_with_milestone_at timestamp without time zone,
    first_added_to_board_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_ed784787ee CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issue_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_metrics_id_seq OWNED BY issue_metrics.id;

CREATE TABLE issue_tracker_data (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_project_url character varying,
    encrypted_project_url_iv character varying,
    encrypted_issues_url character varying,
    encrypted_issues_url_iv character varying,
    encrypted_new_issue_url character varying,
    encrypted_new_issue_url_iv character varying,
    integration_id bigint,
    project_id bigint,
    group_id bigint,
    organization_id bigint,
    CONSTRAINT check_f02a3f53bf CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1))
);

CREATE SEQUENCE issue_tracker_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_tracker_data_id_seq OWNED BY issue_tracker_data.id;

CREATE TABLE issue_user_mentions (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    note_id bigint,
    namespace_id bigint,
    CONSTRAINT check_34b2c10b9c CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE issue_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_user_mentions_id_seq OWNED BY issue_user_mentions.id;

CREATE TABLE issues (
    id bigint NOT NULL,
    title character varying,
    author_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    description text,
    milestone_id bigint,
    iid integer,
    updated_by_id bigint,
    weight integer,
    confidential boolean DEFAULT false NOT NULL,
    due_date date,
    moved_to_id bigint,
    lock_version integer DEFAULT 0,
    title_html text,
    description_html text,
    time_estimate integer DEFAULT 0,
    relative_position integer,
    service_desk_reply_to character varying,
    cached_markdown_version integer,
    last_edited_at timestamp without time zone,
    last_edited_by_id bigint,
    discussion_locked boolean,
    closed_at timestamp with time zone,
    closed_by_id bigint,
    state_id smallint DEFAULT 1 NOT NULL,
    duplicated_to_id bigint,
    promoted_to_epic_id bigint,
    health_status smallint,
    sprint_id bigint,
    blocking_issues_count integer DEFAULT 0 NOT NULL,
    upvotes_count integer DEFAULT 0 NOT NULL,
    work_item_type_id bigint,
    namespace_id bigint,
    start_date date,
    imported_from smallint DEFAULT 0 NOT NULL,
    author_id_convert_to_bigint bigint,
    closed_by_id_convert_to_bigint bigint,
    duplicated_to_id_convert_to_bigint bigint,
    id_convert_to_bigint bigint DEFAULT 0 NOT NULL,
    last_edited_by_id_convert_to_bigint bigint,
    milestone_id_convert_to_bigint bigint,
    moved_to_id_convert_to_bigint bigint,
    project_id_convert_to_bigint bigint,
    promoted_to_epic_id_convert_to_bigint bigint,
    updated_by_id_convert_to_bigint bigint,
    namespace_traversal_ids bigint[] DEFAULT '{}'::bigint[],
    CONSTRAINT check_2addf801cd CHECK ((work_item_type_id IS NOT NULL)),
    CONSTRAINT check_c33362cd43 CHECK ((namespace_id IS NOT NULL)),
    CONSTRAINT check_fba63f706d CHECK ((lock_version IS NOT NULL))
);

CREATE SEQUENCE issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issues_id_seq OWNED BY issues.id;

CREATE TABLE iterations_cadences (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    duration_in_weeks integer,
    iterations_in_advance integer,
    active boolean DEFAULT true NOT NULL,
    automatic boolean DEFAULT true NOT NULL,
    title text NOT NULL,
    roll_over boolean DEFAULT false NOT NULL,
    description text,
    next_run_date date,
    CONSTRAINT check_5c5d2b44bd CHECK ((char_length(description) <= 5000)),
    CONSTRAINT check_fedff82d3b CHECK ((char_length(title) <= 255))
);

CREATE SEQUENCE iterations_cadences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE iterations_cadences_id_seq OWNED BY iterations_cadences.id;

CREATE TABLE jira_connect_installations (
    id bigint NOT NULL,
    client_key character varying,
    encrypted_shared_secret character varying,
    encrypted_shared_secret_iv character varying,
    base_url character varying,
    instance_url text,
    organization_id bigint,
    display_url text,
    CONSTRAINT check_4c6abed669 CHECK ((char_length(instance_url) <= 255)),
    CONSTRAINT check_dc0d039821 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_fb61e0d5f7 CHECK ((char_length(display_url) <= 255))
);

CREATE SEQUENCE jira_connect_installations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_connect_installations_id_seq OWNED BY jira_connect_installations.id;

CREATE TABLE jira_connect_subscriptions (
    id bigint NOT NULL,
    jira_connect_installation_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE jira_connect_subscriptions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_connect_subscriptions_id_seq OWNED BY jira_connect_subscriptions.id;

CREATE TABLE jira_imports (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint,
    label_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    finished_at timestamp with time zone,
    jira_project_xid bigint NOT NULL,
    total_issue_count integer DEFAULT 0 NOT NULL,
    imported_issues_count integer DEFAULT 0 NOT NULL,
    failed_to_import_count integer DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    jid character varying(255),
    jira_project_key character varying(255) NOT NULL,
    jira_project_name character varying(255) NOT NULL,
    scheduled_at timestamp with time zone,
    error_message text,
    CONSTRAINT check_9ed451c5b1 CHECK ((char_length(error_message) <= 1000))
);

CREATE SEQUENCE jira_imports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_imports_id_seq OWNED BY jira_imports.id;

CREATE TABLE jira_tracker_data (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_url character varying,
    encrypted_url_iv character varying,
    encrypted_api_url character varying,
    encrypted_api_url_iv character varying,
    encrypted_username character varying,
    encrypted_username_iv character varying,
    encrypted_password character varying,
    encrypted_password_iv character varying,
    jira_issue_transition_id character varying,
    project_key text,
    issues_enabled boolean DEFAULT false NOT NULL,
    deployment_type smallint DEFAULT 0 NOT NULL,
    vulnerabilities_issuetype text,
    vulnerabilities_enabled boolean DEFAULT false NOT NULL,
    jira_issue_transition_automatic boolean DEFAULT false NOT NULL,
    integration_id bigint,
    jira_issue_prefix text,
    jira_issue_regex text,
    jira_auth_type smallint DEFAULT 0 NOT NULL,
    project_keys text[] DEFAULT '{}'::text[] NOT NULL,
    customize_jira_issue_enabled boolean DEFAULT false,
    project_id bigint,
    group_id bigint,
    organization_id bigint,
    jira_check_enabled boolean DEFAULT false NOT NULL,
    jira_assignee_check_enabled boolean DEFAULT false NOT NULL,
    jira_status_check_enabled boolean DEFAULT false NOT NULL,
    jira_exists_check_enabled boolean DEFAULT false NOT NULL,
    jira_allowed_statuses_string text,
    CONSTRAINT check_0bf84b76e9 CHECK ((char_length(vulnerabilities_issuetype) <= 255)),
    CONSTRAINT check_214cf6a48b CHECK ((char_length(project_key) <= 255)),
    CONSTRAINT check_4cc5bbc801 CHECK ((char_length(jira_issue_prefix) <= 255)),
    CONSTRAINT check_9863a0a5fd CHECK ((char_length(jira_issue_regex) <= 255)),
    CONSTRAINT check_b5ab881f50 CHECK ((char_length(jira_allowed_statuses_string) <= 1024)),
    CONSTRAINT check_eca1fbd6bd CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1))
);

CREATE SEQUENCE jira_tracker_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_tracker_data_id_seq OWNED BY jira_tracker_data.id;

CREATE TABLE job_environments (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    environment_id bigint NOT NULL,
    ci_pipeline_id bigint NOT NULL,
    ci_job_id bigint NOT NULL,
    deployment_id bigint,
    expanded_environment_name text NOT NULL,
    options jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_1580b8d3c4 CHECK ((char_length(expanded_environment_name) <= 255))
);

CREATE SEQUENCE job_environments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE job_environments_id_seq OWNED BY job_environments.id;

CREATE TABLE keys (
    id bigint NOT NULL,
    user_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    key text,
    title character varying,
    type character varying,
    fingerprint character varying,
    public boolean DEFAULT false NOT NULL,
    last_used_at timestamp without time zone,
    fingerprint_sha256 bytea,
    expires_at timestamp with time zone,
    expiry_notification_delivered_at timestamp with time zone,
    before_expiry_notification_delivered_at timestamp with time zone,
    usage_type smallint DEFAULT 0 NOT NULL,
    organization_id bigint
);

CREATE SEQUENCE keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE keys_id_seq OWNED BY keys.id;

CREATE TABLE knowledge_graph_enabled_namespaces (
    id bigint NOT NULL,
    root_namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE knowledge_graph_enabled_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE knowledge_graph_enabled_namespaces_id_seq OWNED BY knowledge_graph_enabled_namespaces.id;

CREATE TABLE label_links (
    id bigint NOT NULL,
    label_id bigint,
    target_id bigint,
    target_type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    namespace_id bigint,
    CONSTRAINT check_3853a1e8ed CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE label_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE label_links_id_seq OWNED BY label_links.id;

CREATE TABLE label_priorities (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    label_id bigint NOT NULL,
    priority integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE label_priorities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE label_priorities_id_seq OWNED BY label_priorities.id;

CREATE TABLE labels (
    id bigint NOT NULL,
    title character varying,
    color character varying,
    project_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    template boolean DEFAULT false,
    description character varying,
    description_html text,
    type character varying,
    group_id bigint,
    cached_markdown_version integer,
    lock_on_merge boolean DEFAULT false NOT NULL,
    archived boolean DEFAULT false NOT NULL,
    organization_id bigint
);

CREATE SEQUENCE labels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE labels_id_seq OWNED BY labels.id;

CREATE TABLE ldap_admin_role_links (
    id bigint NOT NULL,
    member_role_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    provider text NOT NULL,
    cn text,
    filter text,
    sync_status smallint DEFAULT 0 NOT NULL,
    sync_started_at timestamp with time zone,
    sync_ended_at timestamp with time zone,
    last_successful_sync_at timestamp with time zone,
    sync_error text,
    organization_id bigint,
    CONSTRAINT check_044d783383 CHECK ((char_length(sync_error) <= 255)),
    CONSTRAINT check_7f4c5b8292 CHECK ((char_length(filter) <= 255)),
    CONSTRAINT check_db3fe65cb5 CHECK ((char_length(cn) <= 255)),
    CONSTRAINT check_ee9eb87da0 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_f2efc15b43 CHECK ((char_length(provider) <= 255))
);

CREATE SEQUENCE ldap_admin_role_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ldap_admin_role_links_id_seq OWNED BY ldap_admin_role_links.id;

CREATE TABLE ldap_group_links (
    id bigint NOT NULL,
    cn character varying,
    group_access integer NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    provider character varying,
    filter character varying,
    member_role_id bigint
);

CREATE SEQUENCE ldap_group_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ldap_group_links_id_seq OWNED BY ldap_group_links.id;

CREATE TABLE lfs_file_locks (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    path character varying(511)
);

CREATE SEQUENCE lfs_file_locks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_file_locks_id_seq OWNED BY lfs_file_locks.id;

CREATE TABLE lfs_object_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    lfs_object_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_efe45a8ab3 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE lfs_object_states_lfs_object_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_object_states_lfs_object_id_seq OWNED BY lfs_object_states.lfs_object_id;

CREATE TABLE lfs_objects (
    id bigint NOT NULL,
    oid character varying NOT NULL,
    size bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    file character varying,
    file_store integer DEFAULT 1,
    CONSTRAINT check_eecfc5717d CHECK ((file_store IS NOT NULL))
);

CREATE SEQUENCE lfs_objects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_objects_id_seq OWNED BY lfs_objects.id;

CREATE TABLE lfs_objects_projects (
    id bigint NOT NULL,
    lfs_object_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    repository_type smallint,
    oid text,
    CONSTRAINT check_76ef4585ad CHECK ((char_length(oid) <= 255))
);

CREATE SEQUENCE lfs_objects_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_objects_projects_id_seq OWNED BY lfs_objects_projects.id;

CREATE TABLE licenses (
    id bigint NOT NULL,
    data text NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    cloud boolean DEFAULT false,
    last_synced_at timestamp with time zone
);

CREATE SEQUENCE licenses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE licenses_id_seq OWNED BY licenses.id;

CREATE TABLE list_user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    list_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    collapsed boolean
);

CREATE SEQUENCE list_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE list_user_preferences_id_seq OWNED BY list_user_preferences.id;

CREATE TABLE lists (
    id bigint NOT NULL,
    board_id bigint NOT NULL,
    label_id bigint,
    list_type integer DEFAULT 1 NOT NULL,
    "position" integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    user_id bigint,
    milestone_id bigint,
    max_issue_count integer DEFAULT 0 NOT NULL,
    max_issue_weight integer DEFAULT 0 NOT NULL,
    limit_metric character varying(20),
    iteration_id bigint,
    group_id bigint,
    project_id bigint,
    custom_status_id bigint,
    system_defined_status_identifier smallint,
    CONSTRAINT check_5ed14cb08c CHECK (((list_type <> 6) OR (num_nonnulls(system_defined_status_identifier, custom_status_id) = 1))),
    CONSTRAINT check_6dadb82d36 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lists_id_seq OWNED BY lists.id;

CREATE SEQUENCE loose_foreign_keys_deleted_records_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE loose_foreign_keys_deleted_records_id_seq OWNED BY loose_foreign_keys_deleted_records.id;

CREATE TABLE member_approvals (
    id bigint NOT NULL,
    reviewed_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    member_id bigint,
    member_namespace_id bigint NOT NULL,
    requested_by_id bigint,
    reviewed_by_id bigint,
    new_access_level integer NOT NULL,
    old_access_level integer,
    status smallint DEFAULT 0 NOT NULL,
    user_id bigint NOT NULL,
    member_role_id bigint,
    metadata jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE member_approvals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE member_approvals_id_seq OWNED BY member_approvals.id;

CREATE TABLE member_roles (
    id bigint NOT NULL,
    namespace_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    base_access_level integer,
    name text DEFAULT 'Custom'::text NOT NULL,
    description text,
    occupies_seat boolean DEFAULT false NOT NULL,
    permissions jsonb DEFAULT '{}'::jsonb NOT NULL,
    organization_id bigint,
    CONSTRAINT check_4364846f58 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_9907916995 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_ae96d7c575 CHECK ((num_nonnulls(namespace_id, organization_id) = 1))
);

CREATE SEQUENCE member_roles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE member_roles_id_seq OWNED BY member_roles.id;

CREATE TABLE members (
    id bigint NOT NULL,
    access_level integer NOT NULL,
    source_id bigint NOT NULL,
    source_type character varying NOT NULL,
    user_id bigint,
    notification_level integer NOT NULL,
    type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    created_by_id bigint,
    invite_email character varying,
    invite_token character varying,
    invite_accepted_at timestamp without time zone,
    requested_at timestamp without time zone,
    expires_at date,
    ldap boolean DEFAULT false NOT NULL,
    override boolean DEFAULT false NOT NULL,
    state smallint DEFAULT 0,
    invite_email_success boolean DEFAULT true NOT NULL,
    member_namespace_id bigint,
    member_role_id bigint,
    expiry_notified_at timestamp with time zone,
    request_accepted_at timestamp with time zone,
    CONSTRAINT check_508774aac0 CHECK ((member_namespace_id IS NOT NULL))
);

CREATE TABLE members_deletion_schedules (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    user_id bigint NOT NULL,
    scheduled_by_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE members_deletion_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE members_deletion_schedules_id_seq OWNED BY members_deletion_schedules.id;

CREATE SEQUENCE members_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE members_id_seq OWNED BY members.id;

CREATE TABLE merge_request_approval_metrics (
    merge_request_id bigint NOT NULL,
    last_approved_at timestamp with time zone NOT NULL,
    target_project_id bigint NOT NULL
);

CREATE TABLE merge_request_assignees (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    created_at timestamp with time zone,
    project_id bigint,
    CONSTRAINT check_1442f79624 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_assignees_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_assignees_id_seq OWNED BY merge_request_assignees.id;

CREATE TABLE merge_request_assignment_events (
    id bigint NOT NULL,
    user_id bigint,
    merge_request_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    action smallint DEFAULT 1 NOT NULL,
    project_id bigint,
    CONSTRAINT check_31395542a4 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_assignment_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_assignment_events_id_seq OWNED BY merge_request_assignment_events.id;

CREATE TABLE merge_request_blocks (
    id bigint NOT NULL,
    blocking_merge_request_id bigint NOT NULL,
    blocked_merge_request_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    CONSTRAINT check_f8034ca45e CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_blocks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_blocks_id_seq OWNED BY merge_request_blocks.id;

CREATE TABLE merge_request_cleanup_schedules (
    merge_request_id bigint NOT NULL,
    scheduled_at timestamp with time zone NOT NULL,
    completed_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    failed_count integer DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_8ac5179c82 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_cleanup_schedules_merge_request_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_cleanup_schedules_merge_request_id_seq OWNED BY merge_request_cleanup_schedules.merge_request_id;

CREATE SEQUENCE merge_request_commits_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_commits_metadata_id_seq OWNED BY merge_request_commits_metadata.id;

CREATE TABLE merge_request_context_commit_diff_files (
    sha bytea NOT NULL,
    relative_order integer NOT NULL,
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    too_large boolean NOT NULL,
    a_mode character varying(255) NOT NULL,
    b_mode character varying(255) NOT NULL,
    new_path text NOT NULL,
    old_path text NOT NULL,
    diff text,
    "binary" boolean,
    merge_request_context_commit_id bigint NOT NULL,
    generated boolean,
    encoded_file_path boolean DEFAULT false NOT NULL,
    project_id bigint
);

CREATE TABLE merge_request_context_commits (
    id bigint NOT NULL,
    authored_date timestamp with time zone,
    committed_date timestamp with time zone,
    relative_order integer NOT NULL,
    sha bytea NOT NULL,
    author_name text,
    author_email text,
    committer_name text,
    committer_email text,
    message text,
    merge_request_id bigint,
    trailers jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_1dc6b5f2ac CHECK ((merge_request_id IS NOT NULL)),
    CONSTRAINT check_777e62d390 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_context_commits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_context_commits_id_seq OWNED BY merge_request_context_commits.id;

CREATE TABLE merge_request_diff_commit_users (
    id bigint NOT NULL,
    name text,
    email text,
    organization_id bigint NOT NULL,
    CONSTRAINT check_147358fc42 CHECK ((char_length(name) <= 512)),
    CONSTRAINT check_f5fa206cf7 CHECK ((char_length(email) <= 512)),
    CONSTRAINT merge_request_diff_commit_users_name_or_email_existence CHECK (((COALESCE(name, ''::text) <> ''::text) OR (COALESCE(email, ''::text) <> ''::text)))
);

CREATE SEQUENCE merge_request_diff_commit_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_diff_commit_users_id_seq OWNED BY merge_request_diff_commit_users.id;

CREATE TABLE merge_request_diff_commits (
    authored_date timestamp without time zone,
    committed_date timestamp without time zone,
    merge_request_diff_id bigint NOT NULL,
    relative_order integer NOT NULL,
    sha bytea,
    message text,
    trailers jsonb DEFAULT '{}'::jsonb,
    commit_author_id bigint,
    committer_id bigint,
    merge_request_commits_metadata_id bigint,
    project_id bigint
);

CREATE TABLE merge_request_diff_details (
    merge_request_diff_id bigint NOT NULL,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    project_id bigint,
    CONSTRAINT check_81429e3622 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE merge_request_diff_details_merge_request_diff_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_diff_details_merge_request_diff_id_seq OWNED BY merge_request_diff_details.merge_request_diff_id;

CREATE TABLE merge_request_diff_files (
    merge_request_diff_id bigint NOT NULL,
    relative_order integer NOT NULL,
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    too_large boolean NOT NULL,
    a_mode character varying NOT NULL,
    b_mode character varying NOT NULL,
    new_path text,
    old_path text NOT NULL,
    diff text,
    "binary" boolean,
    external_diff_offset integer,
    external_diff_size integer,
    generated boolean,
    encoded_file_path boolean DEFAULT false NOT NULL,
    project_id bigint
);

CREATE TABLE merge_request_diffs (
    id bigint NOT NULL,
    state character varying,
    merge_request_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    base_commit_sha character varying,
    real_size character varying,
    head_commit_sha character varying,
    start_commit_sha character varying,
    commits_count integer,
    external_diff character varying,
    external_diff_store integer DEFAULT 1,
    stored_externally boolean,
    files_count smallint,
    sorted boolean DEFAULT false NOT NULL,
    diff_type smallint DEFAULT 1 NOT NULL,
    patch_id_sha bytea,
    project_id bigint,
    CONSTRAINT check_11c5f029ad CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_93ee616ac9 CHECK ((external_diff_store IS NOT NULL))
);

CREATE SEQUENCE merge_request_diffs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_diffs_id_seq OWNED BY merge_request_diffs.id;

CREATE TABLE merge_request_merge_schedules (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    merge_after timestamp with time zone,
    project_id bigint NOT NULL
);

CREATE SEQUENCE merge_request_merge_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_merge_schedules_id_seq OWNED BY merge_request_merge_schedules.id;

CREATE TABLE merge_request_metrics (
    merge_request_id bigint NOT NULL,
    latest_build_started_at timestamp without time zone,
    latest_build_finished_at timestamp without time zone,
    first_deployed_to_production_at timestamp without time zone,
    merged_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    merged_by_id bigint,
    latest_closed_by_id bigint,
    latest_closed_at timestamp with time zone,
    first_comment_at timestamp with time zone,
    first_commit_at timestamp with time zone,
    last_commit_at timestamp with time zone,
    diff_size integer,
    modified_paths_size integer,
    commits_count integer,
    first_approved_at timestamp with time zone,
    first_reassigned_at timestamp with time zone,
    added_lines integer,
    removed_lines integer,
    target_project_id bigint,
    id bigint NOT NULL,
    first_contribution boolean DEFAULT false NOT NULL,
    pipeline_id bigint,
    reviewer_first_assigned_at timestamp with time zone,
    CONSTRAINT check_e03d0900bf CHECK ((target_project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_metrics_id_seq OWNED BY merge_request_metrics.id;

CREATE TABLE merge_request_predictions (
    merge_request_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    suggested_reviewers jsonb DEFAULT '{}'::jsonb NOT NULL,
    accepted_reviewers jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_d25609c35e CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_predictions_merge_request_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_predictions_merge_request_id_seq OWNED BY merge_request_predictions.merge_request_id;

CREATE TABLE merge_request_requested_changes (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE merge_request_requested_changes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_requested_changes_id_seq OWNED BY merge_request_requested_changes.id;

CREATE TABLE merge_request_reviewers (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_fb72c99774 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_reviewers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_reviewers_id_seq OWNED BY merge_request_reviewers.id;

CREATE TABLE merge_request_user_mentions (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    note_id bigint,
    project_id bigint,
    CONSTRAINT check_0f5d7f30e4 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_user_mentions_id_seq OWNED BY merge_request_user_mentions.id;

CREATE TABLE merge_requests (
    id bigint NOT NULL,
    target_branch character varying NOT NULL,
    source_branch character varying NOT NULL,
    source_project_id bigint,
    author_id bigint,
    assignee_id bigint,
    title character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    milestone_id bigint,
    merge_status character varying DEFAULT 'unchecked'::character varying NOT NULL,
    target_project_id bigint NOT NULL,
    iid integer,
    description text,
    updated_by_id bigint,
    merge_error text,
    merge_params text,
    merge_when_pipeline_succeeds boolean DEFAULT false NOT NULL,
    merge_user_id bigint,
    merge_commit_sha character varying,
    approvals_before_merge integer,
    rebase_commit_sha character varying,
    in_progress_merge_commit_sha character varying,
    lock_version integer DEFAULT 0,
    title_html text,
    description_html text,
    time_estimate integer DEFAULT 0,
    squash boolean DEFAULT false NOT NULL,
    cached_markdown_version integer,
    last_edited_at timestamp without time zone,
    last_edited_by_id bigint,
    merge_jid character varying,
    discussion_locked boolean,
    latest_merge_request_diff_id bigint,
    allow_maintainer_to_push boolean DEFAULT true,
    state_id smallint DEFAULT 1 NOT NULL,
    rebase_jid character varying,
    squash_commit_sha bytea,
    merge_ref_sha bytea,
    draft boolean DEFAULT false NOT NULL,
    prepared_at timestamp with time zone,
    merged_commit_sha bytea,
    override_requested_changes boolean DEFAULT false NOT NULL,
    head_pipeline_id bigint,
    imported_from smallint DEFAULT 0 NOT NULL,
    retargeted boolean DEFAULT false NOT NULL,
    CONSTRAINT check_970d272570 CHECK ((lock_version IS NOT NULL))
);

CREATE TABLE merge_requests_approval_rules (
    id bigint NOT NULL,
    name text NOT NULL,
    approvals_required integer DEFAULT 0 NOT NULL,
    rule_type smallint DEFAULT 0 NOT NULL,
    origin smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    group_id bigint,
    source_rule_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_ba7b03c61a CHECK ((num_nonnulls(group_id, project_id) = 1)),
    CONSTRAINT check_c7c36145b7 CHECK ((char_length(name) <= 255))
);

CREATE TABLE merge_requests_approval_rules_approver_groups (
    id bigint NOT NULL,
    approval_rule_id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE merge_requests_approval_rules_approver_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_approval_rules_approver_groups_id_seq OWNED BY merge_requests_approval_rules_approver_groups.id;

CREATE TABLE merge_requests_approval_rules_approver_users (
    id bigint NOT NULL,
    approval_rule_id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint,
    group_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_ccdbd0e37e CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE merge_requests_approval_rules_approver_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_approval_rules_approver_users_id_seq OWNED BY merge_requests_approval_rules_approver_users.id;

CREATE TABLE merge_requests_approval_rules_groups (
    id bigint NOT NULL,
    approval_rule_id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE merge_requests_approval_rules_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_approval_rules_groups_id_seq OWNED BY merge_requests_approval_rules_groups.id;

CREATE SEQUENCE merge_requests_approval_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_approval_rules_id_seq OWNED BY merge_requests_approval_rules.id;

CREATE TABLE merge_requests_approval_rules_merge_requests (
    id bigint NOT NULL,
    approval_rule_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE merge_requests_approval_rules_merge_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_approval_rules_merge_requests_id_seq OWNED BY merge_requests_approval_rules_merge_requests.id;

CREATE TABLE merge_requests_approval_rules_projects (
    id bigint NOT NULL,
    approval_rule_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE merge_requests_approval_rules_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_approval_rules_projects_id_seq OWNED BY merge_requests_approval_rules_projects.id;

CREATE TABLE merge_requests_closing_issues (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    from_mr_description boolean DEFAULT true NOT NULL,
    project_id bigint,
    CONSTRAINT check_8532dd8dc4 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE merge_requests_closing_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_closing_issues_id_seq OWNED BY merge_requests_closing_issues.id;

CREATE TABLE merge_requests_compliance_violations (
    id bigint NOT NULL,
    violating_user_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    reason smallint NOT NULL,
    severity_level smallint DEFAULT 0 NOT NULL,
    merged_at timestamp with time zone,
    target_project_id bigint,
    title text,
    target_branch text,
    CONSTRAINT check_860e317e6f CHECK ((target_project_id IS NOT NULL))
);

CREATE SEQUENCE merge_requests_compliance_violations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_compliance_violations_id_seq OWNED BY merge_requests_compliance_violations.id;

CREATE SEQUENCE merge_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_id_seq OWNED BY merge_requests.id;

CREATE TABLE merge_trains (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    target_project_id bigint NOT NULL,
    target_branch text NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    merged_at timestamp with time zone,
    duration integer,
    pipeline_id bigint
);

CREATE SEQUENCE merge_trains_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_trains_id_seq OWNED BY merge_trains.id;

CREATE TABLE milestone_releases (
    milestone_id bigint NOT NULL,
    release_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_8141b5b804 CHECK ((project_id IS NOT NULL))
);

CREATE TABLE milestones (
    id bigint NOT NULL,
    title character varying NOT NULL,
    project_id bigint,
    description text,
    due_date date,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    state character varying,
    iid integer,
    title_html text,
    description_html text,
    start_date date,
    cached_markdown_version integer,
    group_id bigint,
    lock_version integer DEFAULT 0 NOT NULL,
    CONSTRAINT check_08e9c27987 CHECK ((num_nonnulls(group_id, project_id) = 1))
);

CREATE SEQUENCE milestones_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE milestones_id_seq OWNED BY milestones.id;

CREATE TABLE ml_candidate_metadata (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    candidate_id bigint NOT NULL,
    name text NOT NULL,
    value text NOT NULL,
    project_id bigint,
    CONSTRAINT check_6b38a286a5 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_9453f4a8e9 CHECK ((char_length(value) <= 5000)),
    CONSTRAINT check_b964e2ac27 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ml_candidate_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_candidate_metadata_id_seq OWNED BY ml_candidate_metadata.id;

CREATE TABLE ml_candidate_metrics (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    candidate_id bigint,
    value double precision,
    step integer,
    is_nan bytea,
    name text NOT NULL,
    tracked_at bigint,
    project_id bigint,
    CONSTRAINT check_3bb4a3fbd9 CHECK ((char_length(name) <= 250)),
    CONSTRAINT check_d7dfd3de26 CHECK ((candidate_id IS NOT NULL)),
    CONSTRAINT check_e0012be52e CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ml_candidate_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_candidate_metrics_id_seq OWNED BY ml_candidate_metrics.id;

CREATE TABLE ml_candidate_params (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    candidate_id bigint,
    name text NOT NULL,
    value text NOT NULL,
    project_id bigint,
    CONSTRAINT check_093034d049 CHECK ((char_length(name) <= 250)),
    CONSTRAINT check_28a3c29e43 CHECK ((char_length(value) <= 250)),
    CONSTRAINT check_7a0505ca91 CHECK ((candidate_id IS NOT NULL)),
    CONSTRAINT check_b42534522f CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ml_candidate_params_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_candidate_params_id_seq OWNED BY ml_candidate_params.id;

CREATE TABLE ml_candidates (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    experiment_id bigint NOT NULL,
    user_id bigint,
    start_time bigint,
    end_time bigint,
    status smallint DEFAULT 0 NOT NULL,
    name text,
    package_id bigint,
    eid uuid,
    project_id bigint,
    internal_id bigint,
    ci_build_id bigint,
    model_version_id bigint,
    CONSTRAINT check_01584ca6db CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_25e6c65051 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_cd160587d4 CHECK ((eid IS NOT NULL))
);

CREATE SEQUENCE ml_candidates_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_candidates_id_seq OWNED BY ml_candidates.id;

CREATE TABLE ml_experiment_metadata (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    experiment_id bigint NOT NULL,
    name text NOT NULL,
    value text NOT NULL,
    project_id bigint,
    CONSTRAINT check_112fe5002d CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_a91c633d68 CHECK ((char_length(value) <= 5000)),
    CONSTRAINT check_ca9b8315ef CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ml_experiment_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_experiment_metadata_id_seq OWNED BY ml_experiment_metadata.id;

CREATE TABLE ml_experiments (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    iid bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint,
    name text NOT NULL,
    deleted_on timestamp with time zone,
    model_id bigint,
    CONSTRAINT check_ee07a0be2c CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE ml_experiments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_experiments_id_seq OWNED BY ml_experiments.id;

CREATE TABLE ml_model_metadata (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    model_id bigint NOT NULL,
    name text NOT NULL,
    value text NOT NULL,
    project_id bigint,
    CONSTRAINT check_26d3322153 CHECK ((char_length(value) <= 5000)),
    CONSTRAINT check_36240c80a7 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_9a8615c7cc CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE ml_model_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_model_metadata_id_seq OWNED BY ml_model_metadata.id;

CREATE TABLE ml_model_version_metadata (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    model_version_id bigint NOT NULL,
    name text NOT NULL,
    value text NOT NULL,
    CONSTRAINT check_09a0e5cb5b CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_21c444e039 CHECK ((char_length(value) <= 5000))
);

CREATE SEQUENCE ml_model_version_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_model_version_metadata_id_seq OWNED BY ml_model_version_metadata.id;

CREATE TABLE ml_model_versions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    model_id bigint NOT NULL,
    package_id bigint,
    version text NOT NULL,
    description text,
    semver_major integer,
    semver_minor integer,
    semver_patch integer,
    semver_prerelease text,
    cached_markdown_version integer,
    description_html text,
    CONSTRAINT check_246f5048b5 CHECK ((char_length(semver_prerelease) <= 255)),
    CONSTRAINT check_28b2d892c8 CHECK ((char_length(version) <= 255)),
    CONSTRAINT check_4d50116294 CHECK ((char_length(description_html) <= 50000)),
    CONSTRAINT check_f1545d8a9e CHECK ((char_length(description) <= 10000))
);

CREATE SEQUENCE ml_model_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_model_versions_id_seq OWNED BY ml_model_versions.id;

CREATE TABLE ml_models (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    description text,
    user_id bigint,
    cached_markdown_version integer,
    description_html text,
    CONSTRAINT check_1fd2cc7d93 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_51a38acdaa CHECK ((char_length(description_html) <= 50000)),
    CONSTRAINT check_f8df2fefc5 CHECK ((char_length(description) <= 10000))
);

CREATE SEQUENCE ml_models_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ml_models_id_seq OWNED BY ml_models.id;

CREATE TABLE namespace_admin_notes (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    note text,
    CONSTRAINT check_e9d2e71b5d CHECK ((char_length(note) <= 1000))
);

CREATE SEQUENCE namespace_admin_notes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_admin_notes_id_seq OWNED BY namespace_admin_notes.id;

CREATE TABLE namespace_aggregation_schedules (
    namespace_id bigint NOT NULL
);

CREATE TABLE namespace_ai_settings (
    namespace_id bigint NOT NULL,
    duo_workflow_mcp_enabled boolean DEFAULT false NOT NULL,
    foundational_agents_default_enabled boolean DEFAULT true,
    minimum_access_level_execute smallint,
    minimum_access_level_manage smallint,
    minimum_access_level_enable_on_projects smallint,
    minimum_access_level_execute_async smallint,
    feature_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    prompt_injection_protection_level smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_namespace_ai_settings_feature_settings_is_hash CHECK ((jsonb_typeof(feature_settings) = 'object'::text))
);

CREATE TABLE namespace_bans (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE namespace_bans_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_bans_id_seq OWNED BY namespace_bans.id;

CREATE TABLE namespace_ci_cd_settings (
    namespace_id bigint NOT NULL,
    allow_stale_runner_pruning boolean DEFAULT false NOT NULL
);

CREATE TABLE namespace_cluster_agent_mappings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    cluster_agent_id bigint NOT NULL,
    creator_id bigint
);

CREATE SEQUENCE namespace_cluster_agent_mappings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_cluster_agent_mappings_id_seq OWNED BY namespace_cluster_agent_mappings.id;

CREATE TABLE namespace_commit_emails (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    email_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE namespace_commit_emails_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_commit_emails_id_seq OWNED BY namespace_commit_emails.id;

CREATE TABLE namespace_details (
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    cached_markdown_version integer,
    description text,
    description_html text,
    creator_id bigint,
    deleted_at timestamp with time zone,
    state_metadata jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_namespace_details_state_metadata_is_hash CHECK ((jsonb_typeof(state_metadata) = 'object'::text))
);

CREATE TABLE namespace_foundational_agent_statuses (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    reference text NOT NULL,
    enabled boolean NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_6c51d38425 CHECK ((char_length(reference) <= 255))
);

CREATE SEQUENCE namespace_foundational_agent_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_foundational_agent_statuses_id_seq OWNED BY namespace_foundational_agent_statuses.id;

CREATE TABLE namespace_import_users (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE namespace_import_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_import_users_id_seq OWNED BY namespace_import_users.id;

CREATE TABLE namespace_isolations (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    isolated boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE namespace_isolations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_isolations_id_seq OWNED BY namespace_isolations.id;

CREATE TABLE namespace_ldap_settings (
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    sync_last_start_at timestamp with time zone,
    sync_last_update_at timestamp with time zone,
    sync_last_successful_at timestamp with time zone,
    sync_status smallint DEFAULT 0 NOT NULL,
    sync_error text,
    CONSTRAINT check_51a03d26b6 CHECK ((char_length(sync_error) <= 255))
);

CREATE TABLE namespace_limits (
    additional_purchased_storage_size bigint DEFAULT 0 NOT NULL,
    additional_purchased_storage_ends_on date,
    namespace_id bigint NOT NULL,
    pre_enforcement_notification_at timestamp with time zone,
    first_enforced_at timestamp with time zone,
    last_enforced_at timestamp with time zone,
    last_seat_all_used_seats_notification_at timestamp with time zone,
    max_number_of_vulnerabilities_per_project integer
);

CREATE TABLE namespace_package_settings (
    namespace_id bigint NOT NULL,
    maven_duplicates_allowed boolean DEFAULT true NOT NULL,
    maven_duplicate_exception_regex text DEFAULT ''::text NOT NULL,
    generic_duplicates_allowed boolean DEFAULT true NOT NULL,
    generic_duplicate_exception_regex text DEFAULT ''::text NOT NULL,
    maven_package_requests_forwarding boolean,
    lock_maven_package_requests_forwarding boolean DEFAULT false NOT NULL,
    pypi_package_requests_forwarding boolean,
    lock_pypi_package_requests_forwarding boolean DEFAULT false NOT NULL,
    npm_package_requests_forwarding boolean,
    lock_npm_package_requests_forwarding boolean DEFAULT false NOT NULL,
    nuget_duplicates_allowed boolean DEFAULT true NOT NULL,
    nuget_duplicate_exception_regex text DEFAULT ''::text NOT NULL,
    nuget_symbol_server_enabled boolean DEFAULT false NOT NULL,
    terraform_module_duplicates_allowed boolean DEFAULT false NOT NULL,
    terraform_module_duplicate_exception_regex text DEFAULT ''::text NOT NULL,
    audit_events_enabled boolean DEFAULT false NOT NULL,
    CONSTRAINT check_31340211b1 CHECK ((char_length(generic_duplicate_exception_regex) <= 255)),
    CONSTRAINT check_d63274b2b6 CHECK ((char_length(maven_duplicate_exception_regex) <= 255)),
    CONSTRAINT check_eedcf85c48 CHECK ((char_length(nuget_duplicate_exception_regex) <= 255)),
    CONSTRAINT check_f10503f1ad CHECK ((char_length(terraform_module_duplicate_exception_regex) <= 255))
);

CREATE TABLE namespace_root_storage_statistics (
    namespace_id bigint NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    repository_size bigint DEFAULT 0 NOT NULL,
    lfs_objects_size bigint DEFAULT 0 NOT NULL,
    wiki_size bigint DEFAULT 0 NOT NULL,
    build_artifacts_size bigint DEFAULT 0 NOT NULL,
    storage_size bigint DEFAULT 0 NOT NULL,
    packages_size bigint DEFAULT 0 NOT NULL,
    snippets_size bigint DEFAULT 0 NOT NULL,
    pipeline_artifacts_size bigint DEFAULT 0 NOT NULL,
    uploads_size bigint DEFAULT 0 NOT NULL,
    dependency_proxy_size bigint DEFAULT 0 NOT NULL,
    notification_level smallint DEFAULT 100 NOT NULL,
    container_registry_size bigint DEFAULT 0 NOT NULL,
    registry_size_estimated boolean DEFAULT false NOT NULL,
    public_forks_storage_size bigint DEFAULT 0 NOT NULL,
    internal_forks_storage_size bigint DEFAULT 0 NOT NULL,
    private_forks_storage_size bigint DEFAULT 0 NOT NULL
);

CREATE TABLE namespace_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    prevent_forking_outside_group boolean DEFAULT false NOT NULL,
    allow_mfa_for_subgroups boolean DEFAULT true NOT NULL,
    default_branch_name text,
    repository_read_only boolean DEFAULT false NOT NULL,
    resource_access_token_creation_allowed boolean DEFAULT true NOT NULL,
    prevent_sharing_groups_outside_hierarchy boolean DEFAULT false NOT NULL,
    new_user_signups_cap integer,
    setup_for_company boolean,
    jobs_to_be_done smallint,
    runner_token_expiration_interval integer,
    subgroup_runner_token_expiration_interval integer,
    project_runner_token_expiration_interval integer,
    show_diff_preview_in_email boolean DEFAULT true NOT NULL,
    enabled_git_access_protocol smallint DEFAULT 0 NOT NULL,
    unique_project_download_limit smallint DEFAULT 0 NOT NULL,
    unique_project_download_limit_interval_in_seconds integer DEFAULT 0 NOT NULL,
    unique_project_download_limit_allowlist text[] DEFAULT '{}'::text[] NOT NULL,
    auto_ban_user_on_excessive_projects_download boolean DEFAULT false NOT NULL,
    only_allow_merge_if_pipeline_succeeds boolean DEFAULT false NOT NULL,
    allow_merge_on_skipped_pipeline boolean DEFAULT false NOT NULL,
    only_allow_merge_if_all_discussions_are_resolved boolean DEFAULT false NOT NULL,
    default_compliance_framework_id bigint,
    runner_registration_enabled boolean DEFAULT true,
    allow_runner_registration_token boolean DEFAULT true NOT NULL,
    unique_project_download_limit_alertlist integer[] DEFAULT '{}'::integer[] NOT NULL,
    emails_enabled boolean DEFAULT true NOT NULL,
    experiment_features_enabled boolean DEFAULT false NOT NULL,
    default_branch_protection_defaults jsonb DEFAULT '{}'::jsonb NOT NULL,
    service_access_tokens_expiration_enforced boolean DEFAULT true NOT NULL,
    product_analytics_enabled boolean DEFAULT false NOT NULL,
    allow_merge_without_pipeline boolean DEFAULT false NOT NULL,
    enforce_ssh_certificates boolean DEFAULT false NOT NULL,
    math_rendering_limits_enabled boolean,
    lock_math_rendering_limits_enabled boolean DEFAULT false NOT NULL,
    duo_features_enabled boolean,
    lock_duo_features_enabled boolean DEFAULT false NOT NULL,
    disable_personal_access_tokens boolean DEFAULT false NOT NULL,
    early_access_program_participant boolean DEFAULT false NOT NULL,
    remove_dormant_members boolean DEFAULT false NOT NULL,
    remove_dormant_members_period integer DEFAULT 90 NOT NULL,
    early_access_program_joined_by_id bigint,
    seat_control smallint DEFAULT 0 NOT NULL,
    last_dormant_member_review_at timestamp with time zone,
    enterprise_users_extensions_marketplace_opt_in_status smallint DEFAULT 0 NOT NULL,
    spp_repository_pipeline_access boolean,
    lock_spp_repository_pipeline_access boolean DEFAULT false NOT NULL,
    archived boolean DEFAULT false NOT NULL,
    token_expiry_notify_inherited boolean DEFAULT true NOT NULL,
    resource_access_token_notify_inherited boolean,
    lock_resource_access_token_notify_inherited boolean DEFAULT false NOT NULL,
    pipeline_variables_default_role smallint DEFAULT 2 NOT NULL,
    force_pages_access_control boolean DEFAULT false NOT NULL,
    extended_grat_expiry_webhooks_execute boolean DEFAULT false NOT NULL,
    jwt_ci_cd_job_token_enabled boolean DEFAULT false NOT NULL,
    jwt_ci_cd_job_token_opted_out boolean DEFAULT false NOT NULL,
    require_dpop_for_manage_api_endpoints boolean DEFAULT false NOT NULL,
    job_token_policies_enabled boolean DEFAULT false NOT NULL,
    security_policies jsonb DEFAULT '{}'::jsonb NOT NULL,
    duo_nano_features_enabled boolean,
    model_prompt_cache_enabled boolean,
    lock_model_prompt_cache_enabled boolean DEFAULT false NOT NULL,
    disable_invite_members boolean DEFAULT false NOT NULL,
    web_based_commit_signing_enabled boolean,
    lock_web_based_commit_signing_enabled boolean DEFAULT false NOT NULL,
    allow_enterprise_bypass_placeholder_confirmation boolean DEFAULT false NOT NULL,
    enterprise_bypass_expires_at timestamp with time zone,
    hide_email_on_profile boolean DEFAULT false NOT NULL,
    allow_personal_snippets boolean DEFAULT true NOT NULL,
    auto_duo_code_review_enabled boolean,
    lock_auto_duo_code_review_enabled boolean DEFAULT false NOT NULL,
    step_up_auth_required_oauth_provider text,
    duo_remote_flows_enabled boolean,
    lock_duo_remote_flows_enabled boolean DEFAULT false NOT NULL,
    duo_agent_platform_request_count integer DEFAULT 0 NOT NULL,
    disable_ssh_keys boolean DEFAULT false NOT NULL,
    duo_foundational_flows_enabled boolean,
    lock_duo_foundational_flows_enabled boolean DEFAULT false NOT NULL,
    duo_sast_fp_detection_enabled boolean,
    lock_duo_sast_fp_detection_enabled boolean DEFAULT false NOT NULL,
    usage_billing jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_0ba93c78c7 CHECK ((char_length(default_branch_name) <= 255)),
    CONSTRAINT check_d9644d516f CHECK ((char_length(step_up_auth_required_oauth_provider) <= 255)),
    CONSTRAINT check_namespace_settings_security_policies_is_hash CHECK ((jsonb_typeof(security_policies) = 'object'::text)),
    CONSTRAINT check_namespace_settings_usage_billing_is_hash CHECK ((jsonb_typeof(usage_billing) = 'object'::text)),
    CONSTRAINT namespace_settings_unique_project_download_limit_alertlist_size CHECK ((cardinality(unique_project_download_limit_alertlist) <= 100)),
    CONSTRAINT namespace_settings_unique_project_download_limit_allowlist_size CHECK ((cardinality(unique_project_download_limit_allowlist) <= 100))
);

CREATE TABLE namespace_statistics (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    shared_runners_seconds integer DEFAULT 0 NOT NULL,
    shared_runners_seconds_last_reset timestamp without time zone,
    storage_size bigint DEFAULT 0 NOT NULL,
    wiki_size bigint DEFAULT 0 NOT NULL,
    dependency_proxy_size bigint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE namespace_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_statistics_id_seq OWNED BY namespace_statistics.id;

CREATE TABLE namespace_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64)),
    CONSTRAINT check_dff00b5115 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespaces_id_seq OWNED BY namespaces.id;

CREATE TABLE namespaces_storage_limit_exclusions (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    reason text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_81640b2ee2 CHECK ((char_length(reason) <= 255))
);

CREATE SEQUENCE namespaces_storage_limit_exclusions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespaces_storage_limit_exclusions_id_seq OWNED BY namespaces_storage_limit_exclusions.id;

CREATE TABLE namespaces_sync_events (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE namespaces_sync_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespaces_sync_events_id_seq OWNED BY namespaces_sync_events.id;

CREATE TABLE non_sql_service_pings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    payload jsonb NOT NULL,
    organization_id bigint NOT NULL,
    metadata jsonb
);

CREATE SEQUENCE non_sql_service_pings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE non_sql_service_pings_id_seq OWNED BY non_sql_service_pings.id;

CREATE TABLE note_diff_files (
    id bigint NOT NULL,
    diff text NOT NULL,
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    a_mode character varying NOT NULL,
    b_mode character varying NOT NULL,
    new_path text NOT NULL,
    old_path text NOT NULL,
    diff_note_id bigint NOT NULL,
    namespace_id bigint
);

CREATE SEQUENCE note_diff_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE note_diff_files_id_seq OWNED BY note_diff_files.id;

CREATE TABLE note_metadata (
    note_id bigint NOT NULL,
    email_participant text,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    namespace_id bigint,
    CONSTRAINT check_40aa5ff1c6 CHECK ((char_length(email_participant) <= 255))
);

CREATE SEQUENCE note_metadata_note_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE note_metadata_note_id_seq OWNED BY note_metadata.note_id;

CREATE TABLE notes (
    note text,
    noteable_type character varying,
    author_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id bigint,
    line_code character varying,
    commit_id character varying,
    noteable_id bigint,
    system boolean DEFAULT false NOT NULL,
    st_diff text,
    updated_by_id bigint,
    type character varying,
    "position" text,
    original_position text,
    resolved_at timestamp without time zone,
    resolved_by_id bigint,
    discussion_id character varying,
    note_html text,
    cached_markdown_version integer,
    change_position text,
    resolved_by_push boolean,
    review_id bigint,
    confidential boolean,
    last_edited_at timestamp with time zone,
    internal boolean DEFAULT false NOT NULL,
    id bigint NOT NULL,
    namespace_id bigint,
    imported_from smallint DEFAULT 0 NOT NULL,
    organization_id bigint,
    CONSTRAINT check_1244cbd7d0 CHECK ((noteable_type IS NOT NULL)),
    CONSTRAINT check_82f260979e CHECK ((num_nonnulls(namespace_id, organization_id, project_id) >= 1))
);

CREATE SEQUENCE notes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE notes_id_seq OWNED BY notes.id;

CREATE TABLE notification_settings (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    source_id bigint,
    source_type character varying,
    level integer DEFAULT 0 NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    new_note boolean,
    new_issue boolean,
    reopen_issue boolean,
    close_issue boolean,
    reassign_issue boolean,
    new_merge_request boolean,
    reopen_merge_request boolean,
    close_merge_request boolean,
    reassign_merge_request boolean,
    merge_merge_request boolean,
    failed_pipeline boolean,
    success_pipeline boolean,
    push_to_merge_request boolean,
    issue_due boolean,
    new_epic boolean,
    notification_email character varying,
    fixed_pipeline boolean,
    new_release boolean,
    moved_project boolean DEFAULT true NOT NULL,
    change_reviewer_merge_request boolean,
    merge_when_pipeline_succeeds boolean DEFAULT false NOT NULL,
    approver boolean DEFAULT false NOT NULL,
    service_account_failed_pipeline boolean DEFAULT false NOT NULL,
    service_account_success_pipeline boolean DEFAULT false NOT NULL,
    service_account_fixed_pipeline boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE notification_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE notification_settings_id_seq OWNED BY notification_settings.id;

CREATE TABLE oauth_access_grants (
    id bigint NOT NULL,
    resource_owner_id bigint NOT NULL,
    application_id bigint NOT NULL,
    token character varying NOT NULL,
    expires_in integer NOT NULL,
    redirect_uri text NOT NULL,
    created_at timestamp without time zone NOT NULL,
    revoked_at timestamp without time zone,
    scopes character varying,
    code_challenge text,
    code_challenge_method text,
    organization_id bigint NOT NULL,
    CONSTRAINT oauth_access_grants_code_challenge CHECK ((char_length(code_challenge) <= 128)),
    CONSTRAINT oauth_access_grants_code_challenge_method CHECK ((char_length(code_challenge_method) <= 5))
);

CREATE SEQUENCE oauth_access_grants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_access_grants_id_seq OWNED BY oauth_access_grants.id;

CREATE TABLE oauth_access_token_archived_records (
    id bigint NOT NULL,
    resource_owner_id bigint,
    application_id bigint,
    token character varying NOT NULL,
    refresh_token character varying,
    expires_in integer DEFAULT 7200 NOT NULL,
    revoked_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    scopes character varying,
    organization_id bigint NOT NULL,
    archived_at timestamp with time zone DEFAULT CURRENT_TIMESTAMP NOT NULL
);

CREATE TABLE oauth_access_tokens (
    id bigint NOT NULL,
    resource_owner_id bigint,
    application_id bigint,
    token character varying NOT NULL,
    refresh_token character varying,
    expires_in integer DEFAULT 7200,
    revoked_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    scopes character varying,
    organization_id bigint NOT NULL,
    CONSTRAINT check_70f294ef54 CHECK ((expires_in IS NOT NULL))
);

CREATE SEQUENCE oauth_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_access_tokens_id_seq OWNED BY oauth_access_tokens.id;

CREATE TABLE oauth_applications (
    id bigint NOT NULL,
    name character varying NOT NULL,
    uid character varying NOT NULL,
    secret character varying NOT NULL,
    redirect_uri text NOT NULL,
    scopes text DEFAULT ''::text NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    owner_id bigint,
    owner_type character varying,
    trusted boolean DEFAULT false NOT NULL,
    confidential boolean DEFAULT true NOT NULL,
    expire_access_tokens boolean DEFAULT false NOT NULL,
    ropc_enabled boolean DEFAULT true NOT NULL,
    dynamic boolean DEFAULT false NOT NULL,
    organization_id bigint,
    CONSTRAINT check_75750847b8 CHECK ((char_length(scopes) <= 2048))
);

CREATE SEQUENCE oauth_applications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_applications_id_seq OWNED BY oauth_applications.id;

CREATE TABLE oauth_device_grants (
    id bigint NOT NULL,
    resource_owner_id bigint,
    application_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    last_polling_at timestamp with time zone,
    expires_in integer NOT NULL,
    scopes text DEFAULT ''::text NOT NULL,
    device_code text NOT NULL,
    user_code text,
    organization_id bigint NOT NULL,
    CONSTRAINT check_a6271f2c07 CHECK ((char_length(device_code) <= 255)),
    CONSTRAINT check_b0459113c7 CHECK ((char_length(scopes) <= 255)),
    CONSTRAINT check_b36370c6df CHECK ((char_length(user_code) <= 255))
);

CREATE SEQUENCE oauth_device_grants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_device_grants_id_seq OWNED BY oauth_device_grants.id;

CREATE TABLE oauth_openid_requests (
    id bigint NOT NULL,
    access_grant_id bigint NOT NULL,
    nonce character varying NOT NULL,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE oauth_openid_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_openid_requests_id_seq OWNED BY oauth_openid_requests.id;

CREATE TABLE observability_group_o11y_settings (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    o11y_service_url text NOT NULL,
    o11y_service_user_email text NOT NULL,
    o11y_service_password jsonb NOT NULL,
    o11y_service_post_message_encryption_key jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_470a188df1 CHECK ((char_length(o11y_service_user_email) <= 255)),
    CONSTRAINT check_9a69bf69bb CHECK ((char_length(o11y_service_url) <= 255))
);

CREATE SEQUENCE observability_group_o11y_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE observability_group_o11y_settings_id_seq OWNED BY observability_group_o11y_settings.id;

CREATE TABLE observability_logs_issues_connections (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    project_id bigint NOT NULL,
    log_timestamp timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    severity_number smallint NOT NULL,
    service_name text NOT NULL,
    trace_identifier text NOT NULL,
    log_fingerprint text NOT NULL,
    CONSTRAINT check_80945187b6 CHECK ((char_length(trace_identifier) <= 128)),
    CONSTRAINT check_ab38275544 CHECK ((char_length(log_fingerprint) <= 128)),
    CONSTRAINT check_d86a20d56b CHECK ((char_length(service_name) <= 500))
);

CREATE SEQUENCE observability_logs_issues_connections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE observability_logs_issues_connections_id_seq OWNED BY observability_logs_issues_connections.id;

CREATE TABLE observability_metrics_issues_connections (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    metric_type smallint NOT NULL,
    metric_name text NOT NULL,
    project_id bigint,
    CONSTRAINT check_3c743c1262 CHECK ((char_length(metric_name) <= 500))
);

CREATE SEQUENCE observability_metrics_issues_connections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE observability_metrics_issues_connections_id_seq OWNED BY observability_metrics_issues_connections.id;

CREATE TABLE observability_traces_issues_connections (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    trace_identifier text NOT NULL,
    CONSTRAINT check_5b51f9ea15 CHECK ((char_length(trace_identifier) <= 128))
);

CREATE SEQUENCE observability_traces_issues_connections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE observability_traces_issues_connections_id_seq OWNED BY observability_traces_issues_connections.id;

CREATE TABLE onboarding_progresses (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    merge_request_created_at timestamp with time zone,
    pipeline_created_at timestamp with time zone,
    user_added_at timestamp with time zone,
    trial_started_at timestamp with time zone,
    required_mr_approvals_enabled_at timestamp with time zone,
    code_owners_enabled_at timestamp with time zone,
    issue_created_at timestamp with time zone,
    secure_dependency_scanning_run_at timestamp with time zone,
    secure_dast_run_at timestamp with time zone,
    license_scanning_run_at timestamp with time zone,
    code_added_at timestamp with time zone,
    ended_at timestamp with time zone,
    duo_seat_assigned_at timestamp with time zone
);

CREATE SEQUENCE onboarding_progresses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE onboarding_progresses_id_seq OWNED BY onboarding_progresses.id;

CREATE TABLE operations_feature_flags (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    active boolean NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    description text,
    iid integer NOT NULL,
    version smallint DEFAULT 1 NOT NULL
);

CREATE TABLE operations_feature_flags_clients (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    token_encrypted character varying,
    last_feature_flag_updated_at timestamp with time zone
);

CREATE SEQUENCE operations_feature_flags_clients_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flags_clients_id_seq OWNED BY operations_feature_flags_clients.id;

CREATE SEQUENCE operations_feature_flags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flags_id_seq OWNED BY operations_feature_flags.id;

CREATE TABLE operations_feature_flags_issues (
    id bigint NOT NULL,
    feature_flag_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_0e57762955 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE operations_feature_flags_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flags_issues_id_seq OWNED BY operations_feature_flags_issues.id;

CREATE TABLE operations_scopes (
    id bigint NOT NULL,
    strategy_id bigint NOT NULL,
    environment_scope character varying(255) NOT NULL,
    project_id bigint,
    CONSTRAINT check_722a570b84 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE operations_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_scopes_id_seq OWNED BY operations_scopes.id;

CREATE TABLE operations_strategies (
    id bigint NOT NULL,
    feature_flag_id bigint NOT NULL,
    name character varying(255) NOT NULL,
    parameters jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_85b486853f CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE operations_strategies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_strategies_id_seq OWNED BY operations_strategies.id;

CREATE TABLE operations_strategies_user_lists (
    id bigint NOT NULL,
    strategy_id bigint NOT NULL,
    user_list_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_be4b61e4d3 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE operations_strategies_user_lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_strategies_user_lists_id_seq OWNED BY operations_strategies_user_lists.id;

CREATE TABLE operations_user_lists (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    iid integer NOT NULL,
    name character varying(255) NOT NULL,
    user_xids text DEFAULT ''::text NOT NULL
);

CREATE SEQUENCE operations_user_lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_user_lists_id_seq OWNED BY operations_user_lists.id;

CREATE TABLE organization_cluster_agent_mappings (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    cluster_agent_id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE organization_cluster_agent_mappings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organization_cluster_agent_mappings_id_seq OWNED BY organization_cluster_agent_mappings.id;

CREATE TABLE organization_detail_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_8c86d5dff9 CHECK ((organization_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE organization_details (
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cached_markdown_version integer,
    description text,
    description_html text,
    avatar text,
    CONSTRAINT check_71dfb7807f CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_9fbd483b51 CHECK ((char_length(avatar) <= 255))
);

CREATE TABLE organization_foundational_agent_statuses (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    reference text NOT NULL,
    enabled boolean NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_81fef9b9a6 CHECK ((char_length(reference) <= 255))
);

CREATE SEQUENCE organization_foundational_agent_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organization_foundational_agent_statuses_id_seq OWNED BY organization_foundational_agent_statuses.id;

CREATE TABLE organization_isolations (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    isolated boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE organization_isolations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organization_isolations_id_seq OWNED BY organization_isolations.id;

CREATE TABLE organization_push_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint NOT NULL,
    max_file_size integer DEFAULT 0 NOT NULL,
    member_check boolean DEFAULT false NOT NULL,
    prevent_secrets boolean DEFAULT false NOT NULL,
    commit_committer_name_check boolean DEFAULT false NOT NULL,
    deny_delete_tag boolean,
    reject_unsigned_commits boolean,
    commit_committer_check boolean,
    reject_non_dco_commits boolean,
    commit_message_regex text,
    branch_name_regex text,
    commit_message_negative_regex text,
    author_email_regex text,
    file_name_regex text,
    CONSTRAINT author_email_regex_size_constraint CHECK ((char_length(author_email_regex) <= 511)),
    CONSTRAINT branch_name_regex_size_constraint CHECK ((char_length(branch_name_regex) <= 511)),
    CONSTRAINT commit_message_negative_regex_size_constraint CHECK ((char_length(commit_message_negative_regex) <= 2047)),
    CONSTRAINT commit_message_regex_size_constraint CHECK ((char_length(commit_message_regex) <= 511)),
    CONSTRAINT file_name_regex_size_constraint CHECK ((char_length(file_name_regex) <= 511))
);

CREATE SEQUENCE organization_push_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organization_push_rules_id_seq OWNED BY organization_push_rules.id;

CREATE TABLE organization_settings (
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    settings jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE TABLE organization_user_details (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    user_id bigint NOT NULL,
    username text NOT NULL,
    display_name text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_470dbccf9b CHECK ((char_length(display_name) <= 510)),
    CONSTRAINT check_dc5e9cf6f2 CHECK ((char_length(username) <= 510))
);

CREATE SEQUENCE organization_user_details_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organization_user_details_id_seq OWNED BY organization_user_details.id;

CREATE TABLE organization_users (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    access_level smallint DEFAULT 10 NOT NULL
);

CREATE SEQUENCE organization_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organization_users_id_seq OWNED BY organization_users.id;

CREATE TABLE organizations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text DEFAULT ''::text NOT NULL,
    path text NOT NULL,
    visibility_level smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_0b4296b5ea CHECK ((char_length(path) <= 255)),
    CONSTRAINT check_d130d769e0 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE organizations_id_seq
    START WITH 1000
    INCREMENT BY 1
    MINVALUE 1000
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE organizations_id_seq OWNED BY organizations.id;

CREATE SEQUENCE p_ai_active_context_code_enabled_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ai_active_context_code_enabled_namespaces_id_seq OWNED BY p_ai_active_context_code_enabled_namespaces.id;

CREATE SEQUENCE p_ai_active_context_code_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ai_active_context_code_repositories_id_seq OWNED BY p_ai_active_context_code_repositories.id;

CREATE SEQUENCE p_batched_git_ref_updates_deletions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_batched_git_ref_updates_deletions_id_seq OWNED BY p_batched_git_ref_updates_deletions.id;

CREATE SEQUENCE p_catalog_resource_sync_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_catalog_resource_sync_events_id_seq OWNED BY p_catalog_resource_sync_events.id;

CREATE SEQUENCE p_ci_build_tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_build_tags_id_seq OWNED BY p_ci_build_tags.id;

CREATE SEQUENCE p_ci_builds_execution_configs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_builds_execution_configs_id_seq OWNED BY p_ci_builds_execution_configs.id;

CREATE SEQUENCE p_ci_job_annotations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_job_annotations_id_seq OWNED BY p_ci_job_annotations.id;

CREATE SEQUENCE p_ci_job_definitions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_job_definitions_id_seq OWNED BY p_ci_job_definitions.id;

CREATE SEQUENCE p_ci_job_inputs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_job_inputs_id_seq OWNED BY p_ci_job_inputs.id;

CREATE SEQUENCE p_ci_job_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_job_messages_id_seq OWNED BY p_ci_job_messages.id;

CREATE SEQUENCE p_ci_workload_variable_inclusions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_workload_variable_inclusions_id_seq OWNED BY p_ci_workload_variable_inclusions.id;

CREATE SEQUENCE p_ci_workloads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_ci_workloads_id_seq OWNED BY p_ci_workloads.id;

CREATE SEQUENCE p_duo_workflows_checkpoints_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_duo_workflows_checkpoints_id_seq OWNED BY p_duo_workflows_checkpoints.id;

CREATE SEQUENCE p_generated_ref_commits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_generated_ref_commits_id_seq OWNED BY p_generated_ref_commits.id;

CREATE SEQUENCE p_sent_notifications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE p_sent_notifications_id_seq OWNED BY p_sent_notifications.id;

CREATE TABLE packages_build_infos (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    pipeline_id bigint,
    project_id bigint,
    CONSTRAINT check_d979c653e1 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_build_infos_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_build_infos_id_seq OWNED BY packages_build_infos.id;

CREATE TABLE packages_cargo_metadata (
    package_id bigint NOT NULL,
    index_content jsonb,
    project_id bigint NOT NULL,
    normalized_name text,
    normalized_version text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_245ce00e05 CHECK ((char_length(normalized_name) <= 64)),
    CONSTRAINT check_de6f67d97b CHECK ((char_length(normalized_version) <= 255))
);

CREATE TABLE packages_cleanup_policies (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    next_run_at timestamp with time zone,
    keep_n_duplicated_package_files text DEFAULT 'all'::text NOT NULL,
    CONSTRAINT check_e53f35ab7b CHECK ((char_length(keep_n_duplicated_package_files) <= 255))
);

CREATE TABLE packages_composer_metadata (
    package_id bigint NOT NULL,
    target_sha bytea NOT NULL,
    composer_json jsonb DEFAULT '{}'::jsonb NOT NULL,
    version_cache_sha bytea,
    project_id bigint,
    CONSTRAINT check_250f62a87a CHECK ((project_id IS NOT NULL))
);

CREATE TABLE packages_packages (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    version character varying,
    package_type smallint NOT NULL,
    creator_id bigint,
    status smallint DEFAULT 0 NOT NULL,
    last_downloaded_at timestamp with time zone,
    status_message text
);

CREATE SEQUENCE packages_packages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_packages_id_seq OWNED BY packages_packages.id;

CREATE TABLE packages_composer_packages (
    id bigint DEFAULT nextval('packages_packages_id_seq'::regclass) NOT NULL,
    project_id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    last_downloaded_at timestamp with time zone,
    status smallint DEFAULT 0 NOT NULL,
    name text NOT NULL,
    version text,
    target_sha bytea,
    version_cache_sha bytea,
    status_message text,
    composer_json jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_2168170eb8 CHECK ((char_length(status_message) <= 255))
);

CREATE TABLE packages_conan_file_metadata (
    id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    conan_file_type smallint NOT NULL,
    recipe_revision_id bigint,
    package_revision_id bigint,
    package_reference_id bigint,
    project_id bigint,
    CONSTRAINT check_7a4526796d CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_conan_file_metadata_ref_null_for_recipe_files CHECK ((NOT ((conan_file_type = 1) AND (package_reference_id IS NOT NULL))))
);

CREATE SEQUENCE packages_conan_file_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_file_metadata_id_seq OWNED BY packages_conan_file_metadata.id;

CREATE TABLE packages_conan_metadata (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_username character varying(255) NOT NULL,
    package_channel character varying(255) NOT NULL,
    project_id bigint,
    CONSTRAINT check_9cda5a20a8 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_conan_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_metadata_id_seq OWNED BY packages_conan_metadata.id;

CREATE TABLE packages_conan_package_references (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    project_id bigint NOT NULL,
    recipe_revision_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    reference bytea NOT NULL,
    info jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT chk_conan_references_info_length CHECK ((char_length((info)::text) <= 20000))
);

CREATE SEQUENCE packages_conan_package_references_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_package_references_id_seq OWNED BY packages_conan_package_references.id;

CREATE TABLE packages_conan_package_revisions (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    project_id bigint NOT NULL,
    package_reference_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    revision bytea NOT NULL,
    status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE packages_conan_package_revisions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_package_revisions_id_seq OWNED BY packages_conan_package_revisions.id;

CREATE TABLE packages_conan_recipe_revisions (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    revision bytea NOT NULL,
    status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE packages_conan_recipe_revisions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_recipe_revisions_id_seq OWNED BY packages_conan_recipe_revisions.id;

CREATE TABLE packages_debian_file_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_file_id bigint NOT NULL,
    file_type smallint NOT NULL,
    component text,
    architecture text,
    fields jsonb,
    project_id bigint,
    CONSTRAINT check_2ebedda4b6 CHECK ((char_length(component) <= 255)),
    CONSTRAINT check_58297dfb13 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_e6e1fffcca CHECK ((char_length(architecture) <= 255))
);

CREATE TABLE packages_debian_group_architectures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    group_id bigint,
    CONSTRAINT check_7a087a5b9f CHECK ((group_id IS NOT NULL)),
    CONSTRAINT check_ddb220164a CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE packages_debian_group_architectures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_architectures_id_seq OWNED BY packages_debian_group_architectures.id;

CREATE TABLE packages_debian_group_component_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_id bigint NOT NULL,
    architecture_id bigint,
    size integer NOT NULL,
    file_type smallint NOT NULL,
    compression_type smallint,
    file_store smallint DEFAULT 1 NOT NULL,
    file text NOT NULL,
    file_sha256 bytea NOT NULL,
    group_id bigint,
    CONSTRAINT check_6d7454a717 CHECK ((group_id IS NOT NULL)),
    CONSTRAINT check_839e1685bc CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_debian_group_component_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_component_files_id_seq OWNED BY packages_debian_group_component_files.id;

CREATE TABLE packages_debian_group_components (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    group_id bigint,
    CONSTRAINT check_a9bc7d85be CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_bb77e71a15 CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE packages_debian_group_components_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_components_id_seq OWNED BY packages_debian_group_components.id;

CREATE TABLE packages_debian_group_distribution_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    encrypted_private_key text NOT NULL,
    encrypted_private_key_iv text NOT NULL,
    encrypted_passphrase text NOT NULL,
    encrypted_passphrase_iv text NOT NULL,
    public_key text NOT NULL,
    fingerprint text NOT NULL,
    group_id bigint,
    CONSTRAINT check_008dba9ceb CHECK ((group_id IS NOT NULL)),
    CONSTRAINT check_bc95dc3fbe CHECK ((char_length(fingerprint) <= 255)),
    CONSTRAINT check_f708183491 CHECK ((char_length(public_key) <= 524288))
);

CREATE SEQUENCE packages_debian_group_distribution_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_distribution_keys_id_seq OWNED BY packages_debian_group_distribution_keys.id;

CREATE TABLE packages_debian_group_distributions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    creator_id bigint,
    valid_time_duration_seconds integer,
    file_store smallint DEFAULT 1 NOT NULL,
    automatic boolean DEFAULT true NOT NULL,
    automatic_upgrades boolean DEFAULT false NOT NULL,
    codename text NOT NULL,
    suite text,
    origin text,
    label text,
    version text,
    description text,
    file text,
    file_signature text,
    signed_file text,
    signed_file_store smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_0007e0bf61 CHECK ((char_length(signed_file) <= 255)),
    CONSTRAINT check_310ac457b8 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_3d6f87fc31 CHECK ((char_length(file_signature) <= 4096)),
    CONSTRAINT check_3fdadf4a0c CHECK ((char_length(version) <= 255)),
    CONSTRAINT check_590e18405a CHECK ((char_length(codename) <= 255)),
    CONSTRAINT check_b057cd840a CHECK ((char_length(origin) <= 255)),
    CONSTRAINT check_be5ed8d307 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_d3244bfc0b CHECK ((char_length(label) <= 255)),
    CONSTRAINT check_e7c928a24b CHECK ((char_length(suite) <= 255))
);

CREATE SEQUENCE packages_debian_group_distributions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_distributions_id_seq OWNED BY packages_debian_group_distributions.id;

CREATE TABLE packages_debian_project_architectures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    project_id bigint,
    CONSTRAINT check_9c2e1c99d8 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e187c75bb4 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_debian_project_architectures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_architectures_id_seq OWNED BY packages_debian_project_architectures.id;

CREATE TABLE packages_debian_project_component_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_id bigint NOT NULL,
    architecture_id bigint,
    size integer NOT NULL,
    file_type smallint NOT NULL,
    compression_type smallint,
    file_store smallint DEFAULT 1 NOT NULL,
    file text NOT NULL,
    file_sha256 bytea NOT NULL,
    project_id bigint,
    CONSTRAINT check_4eafc9503d CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_e5af03fa2d CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_debian_project_component_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_component_files_id_seq OWNED BY packages_debian_project_component_files.id;

CREATE TABLE packages_debian_project_components (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    project_id bigint,
    CONSTRAINT check_517559f298 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_6c727037a7 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_debian_project_components_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_components_id_seq OWNED BY packages_debian_project_components.id;

CREATE TABLE packages_debian_project_distribution_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    encrypted_private_key text NOT NULL,
    encrypted_private_key_iv text NOT NULL,
    encrypted_passphrase text NOT NULL,
    encrypted_passphrase_iv text NOT NULL,
    public_key text NOT NULL,
    fingerprint text NOT NULL,
    project_id bigint,
    CONSTRAINT check_9e8a5eef0a CHECK ((char_length(fingerprint) <= 255)),
    CONSTRAINT check_c2a4dc05d5 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_d188f6547f CHECK ((char_length(public_key) <= 524288))
);

CREATE SEQUENCE packages_debian_project_distribution_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_distribution_keys_id_seq OWNED BY packages_debian_project_distribution_keys.id;

CREATE TABLE packages_debian_project_distributions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    creator_id bigint,
    valid_time_duration_seconds integer,
    file_store smallint DEFAULT 1 NOT NULL,
    automatic boolean DEFAULT true NOT NULL,
    automatic_upgrades boolean DEFAULT false NOT NULL,
    codename text NOT NULL,
    suite text,
    origin text,
    label text,
    version text,
    description text,
    file text,
    file_signature text,
    signed_file text,
    signed_file_store smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_6177ccd4a6 CHECK ((char_length(origin) <= 255)),
    CONSTRAINT check_6f6b55a4c4 CHECK ((char_length(label) <= 255)),
    CONSTRAINT check_834dabadb6 CHECK ((char_length(codename) <= 255)),
    CONSTRAINT check_96965792c2 CHECK ((char_length(version) <= 255)),
    CONSTRAINT check_9e5e22b7ff CHECK ((char_length(signed_file) <= 255)),
    CONSTRAINT check_a56ae58a17 CHECK ((char_length(suite) <= 255)),
    CONSTRAINT check_a5a2ac6af2 CHECK ((char_length(file_signature) <= 4096)),
    CONSTRAINT check_b93154339f CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_cb4ac9599e CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_debian_project_distributions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_distributions_id_seq OWNED BY packages_debian_project_distributions.id;

CREATE TABLE packages_debian_publications (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    distribution_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_30a36cda06 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_debian_publications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_publications_id_seq OWNED BY packages_debian_publications.id;

CREATE TABLE packages_dependencies (
    id bigint NOT NULL,
    name character varying(255) NOT NULL,
    version_pattern character varying(255) NOT NULL,
    project_id bigint,
    CONSTRAINT check_83faf1f5e7 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_dependencies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_dependencies_id_seq OWNED BY packages_dependencies.id;

CREATE TABLE packages_dependency_links (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    dependency_id bigint NOT NULL,
    dependency_type smallint NOT NULL,
    project_id bigint,
    CONSTRAINT check_dea82eaa8e CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_dependency_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_dependency_links_id_seq OWNED BY packages_dependency_links.id;

CREATE TABLE packages_helm_file_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_file_id bigint NOT NULL,
    channel text NOT NULL,
    metadata jsonb,
    project_id bigint,
    CONSTRAINT check_06e8d100af CHECK ((char_length(channel) <= 255)),
    CONSTRAINT check_109d878e47 CHECK ((project_id IS NOT NULL))
);

CREATE TABLE packages_helm_metadata_caches (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    last_downloaded_at timestamp with time zone,
    project_id bigint NOT NULL,
    size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    file_store integer DEFAULT 1,
    channel text NOT NULL,
    file text NOT NULL,
    object_storage_key text NOT NULL,
    CONSTRAINT check_1ad8e76464 CHECK ((char_length(object_storage_key) <= 255)),
    CONSTRAINT check_471469b475 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_9b1333efe0 CHECK ((char_length(channel) <= 255))
);

CREATE SEQUENCE packages_helm_metadata_caches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_helm_metadata_caches_id_seq OWNED BY packages_helm_metadata_caches.id;

CREATE TABLE packages_maven_metadata (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    app_group character varying NOT NULL,
    app_name character varying NOT NULL,
    app_version character varying,
    path character varying(512) NOT NULL,
    project_id bigint,
    CONSTRAINT check_bf287ce98c CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_maven_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_maven_metadata_id_seq OWNED BY packages_maven_metadata.id;

CREATE TABLE packages_npm_metadata (
    package_id bigint NOT NULL,
    package_json jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    CONSTRAINT check_8d2e047947 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT chk_rails_e5cbc301ae CHECK ((char_length((package_json)::text) < 20000))
);

CREATE TABLE packages_npm_metadata_caches (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    last_downloaded_at timestamp with time zone,
    project_id bigint,
    file_store integer DEFAULT 1,
    size integer NOT NULL,
    file text NOT NULL,
    package_name text NOT NULL,
    object_storage_key text NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_57aa07a4b2 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_734454a615 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_f97c15aa60 CHECK ((char_length(object_storage_key) <= 255))
);

CREATE SEQUENCE packages_npm_metadata_caches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_npm_metadata_caches_id_seq OWNED BY packages_npm_metadata_caches.id;

CREATE TABLE packages_nuget_dependency_link_metadata (
    dependency_link_id bigint NOT NULL,
    target_framework text NOT NULL,
    project_id bigint,
    CONSTRAINT check_1c3e07cfff CHECK ((project_id IS NOT NULL)),
    CONSTRAINT packages_nuget_dependency_link_metadata_target_framework_constr CHECK ((char_length(target_framework) <= 255))
);

CREATE TABLE packages_nuget_metadata (
    package_id bigint NOT NULL,
    license_url text,
    project_url text,
    icon_url text,
    authors text,
    description text,
    normalized_version text,
    project_id bigint,
    CONSTRAINT check_6b272cad10 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_9973c0cc33 CHECK ((char_length(normalized_version) <= 255)),
    CONSTRAINT check_d39a5fe9ee CHECK ((char_length(description) <= 4000)),
    CONSTRAINT check_e2fc129ebd CHECK ((char_length(authors) <= 255)),
    CONSTRAINT packages_nuget_metadata_icon_url_constraint CHECK ((char_length(icon_url) <= 255)),
    CONSTRAINT packages_nuget_metadata_license_url_constraint CHECK ((char_length(license_url) <= 255)),
    CONSTRAINT packages_nuget_metadata_project_url_constraint CHECK ((char_length(project_url) <= 255))
);

CREATE TABLE packages_nuget_symbol_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    packages_nuget_symbol_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_e1484aadc9 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE packages_nuget_symbol_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_nuget_symbol_states_id_seq OWNED BY packages_nuget_symbol_states.id;

CREATE TABLE packages_nuget_symbols (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_id bigint,
    size integer NOT NULL,
    file_store smallint DEFAULT 1,
    file text NOT NULL,
    file_path text NOT NULL,
    signature text NOT NULL,
    object_storage_key text NOT NULL,
    file_sha256 bytea,
    status smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_0e93ca58b7 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_28b82b08fa CHECK ((char_length(object_storage_key) <= 255)),
    CONSTRAINT check_30b0ef2ca2 CHECK ((char_length(file_path) <= 255)),
    CONSTRAINT check_7a67b0fc8b CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_8dc7152679 CHECK ((char_length(signature) <= 255))
);

CREATE SEQUENCE packages_nuget_symbols_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_nuget_symbols_id_seq OWNED BY packages_nuget_symbols.id;

CREATE TABLE packages_package_file_build_infos (
    id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    pipeline_id bigint,
    project_id bigint,
    CONSTRAINT check_102fc16781 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_package_file_build_infos_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_package_file_build_infos_id_seq OWNED BY packages_package_file_build_infos.id;

CREATE TABLE packages_package_file_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    package_file_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_975cbbb43b CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE packages_package_file_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_package_file_states_id_seq OWNED BY packages_package_file_states.id;

CREATE TABLE packages_package_files (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    size bigint,
    file_store integer DEFAULT 1,
    file_md5 bytea,
    file_sha1 bytea,
    file_name character varying NOT NULL,
    file text NOT NULL,
    file_sha256 bytea,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_failure character varying(255),
    verification_retry_count integer,
    verification_checksum bytea,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    status smallint DEFAULT 0 NOT NULL,
    file_final_path text,
    project_id bigint,
    CONSTRAINT check_0f29938b18 CHECK ((char_length(file_final_path) <= 1024)),
    CONSTRAINT check_43773f06dc CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_4c5e6bb0b3 CHECK ((file_store IS NOT NULL))
);

CREATE SEQUENCE packages_package_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_package_files_id_seq OWNED BY packages_package_files.id;

CREATE TABLE packages_protection_rules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_type smallint NOT NULL,
    package_name_pattern text NOT NULL,
    minimum_access_level_for_push smallint,
    minimum_access_level_for_delete smallint,
    pattern text,
    pattern_type smallint DEFAULT 0 NOT NULL,
    target_field smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_520a0596a3 CHECK ((num_nonnulls(minimum_access_level_for_delete, minimum_access_level_for_push) > 0)),
    CONSTRAINT check_96c7dcb821 CHECK ((char_length(pattern) <= 255)),
    CONSTRAINT check_d2d75d206d CHECK ((char_length(package_name_pattern) <= 255))
);

CREATE SEQUENCE packages_protection_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_protection_rules_id_seq OWNED BY packages_protection_rules.id;

CREATE TABLE packages_pypi_file_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_file_id bigint NOT NULL,
    project_id bigint NOT NULL,
    required_python text DEFAULT ''::text NOT NULL,
    CONSTRAINT check_a5d85e7cff CHECK ((char_length(required_python) <= 255))
);

CREATE TABLE packages_pypi_metadata (
    package_id bigint NOT NULL,
    required_python text DEFAULT ''::text,
    metadata_version text,
    summary text,
    keywords text,
    author_email text,
    description text,
    description_content_type text,
    project_id bigint,
    CONSTRAINT check_0d9aed55b2 CHECK ((required_python IS NOT NULL)),
    CONSTRAINT check_222e4f5b58 CHECK ((char_length(keywords) <= 1024)),
    CONSTRAINT check_2d3ed32225 CHECK ((char_length(metadata_version) <= 16)),
    CONSTRAINT check_379019d5da CHECK ((char_length(required_python) <= 255)),
    CONSTRAINT check_65d8dbbd9f CHECK ((char_length(author_email) <= 2048)),
    CONSTRAINT check_76afb6d4f3 CHECK ((char_length(summary) <= 255)),
    CONSTRAINT check_77e2d63abb CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_80308aa9bd CHECK ((char_length(description) <= 4000)),
    CONSTRAINT check_b1f32be96c CHECK ((char_length(description_content_type) <= 128))
);

CREATE TABLE packages_rpm_metadata (
    package_id bigint NOT NULL,
    release text DEFAULT '1'::text NOT NULL,
    summary text DEFAULT ''::text NOT NULL,
    description text DEFAULT ''::text NOT NULL,
    arch text DEFAULT ''::text NOT NULL,
    license text,
    url text,
    epoch integer DEFAULT 0 NOT NULL,
    project_id bigint,
    CONSTRAINT check_3798bae3d6 CHECK ((char_length(arch) <= 255)),
    CONSTRAINT check_4506c26fc1 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_5d29ba59ac CHECK ((char_length(description) <= 5000)),
    CONSTRAINT check_6e8cbd536d CHECK ((char_length(url) <= 1000)),
    CONSTRAINT check_845ba4d7d0 CHECK ((char_length(license) <= 1000)),
    CONSTRAINT check_b010bf4870 CHECK ((char_length(summary) <= 1000)),
    CONSTRAINT check_c3e2fc2e89 CHECK ((char_length(release) <= 128))
);

CREATE TABLE packages_rpm_repository_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    file_store integer DEFAULT 1,
    status smallint DEFAULT 0 NOT NULL,
    size integer,
    file_md5 bytea,
    file_sha1 bytea,
    file_sha256 bytea,
    file text NOT NULL,
    file_name text NOT NULL,
    CONSTRAINT check_a9fef187f5 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_b6b721b275 CHECK ((char_length(file_name) <= 255))
);

CREATE SEQUENCE packages_rpm_repository_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_rpm_repository_files_id_seq OWNED BY packages_rpm_repository_files.id;

CREATE TABLE packages_rubygems_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_id bigint NOT NULL,
    authors text,
    files text,
    summary text,
    description text,
    email text,
    homepage text,
    licenses text,
    metadata text,
    author text,
    bindir text,
    cert_chain text,
    executables text,
    extensions text,
    extra_rdoc_files text,
    platform text,
    post_install_message text,
    rdoc_options text,
    require_paths text,
    required_ruby_version text,
    required_rubygems_version text,
    requirements text,
    rubygems_version text,
    signing_key text,
    project_id bigint,
    CONSTRAINT check_0154a18c82 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_22814c771b CHECK ((char_length(email) <= 255)),
    CONSTRAINT check_242293030e CHECK ((char_length(extensions) <= 255)),
    CONSTRAINT check_27619a7922 CHECK ((char_length(rubygems_version) <= 255)),
    CONSTRAINT check_3d1b6f3a39 CHECK ((char_length(post_install_message) <= 255)),
    CONSTRAINT check_545f7606f9 CHECK ((char_length(required_rubygems_version) <= 255)),
    CONSTRAINT check_5988451714 CHECK ((char_length(executables) <= 255)),
    CONSTRAINT check_5f9c84ea17 CHECK ((char_length(platform) <= 255)),
    CONSTRAINT check_64f1cecf05 CHECK ((char_length(requirements) <= 255)),
    CONSTRAINT check_6ac7043c50 CHECK ((char_length(extra_rdoc_files) <= 255)),
    CONSTRAINT check_6ff3abe325 CHECK ((char_length(cert_chain) <= 255)),
    CONSTRAINT check_7cb01436df CHECK ((char_length(licenses) <= 255)),
    CONSTRAINT check_8be21d92e7 CHECK ((char_length(summary) <= 1024)),
    CONSTRAINT check_946cb96acb CHECK ((char_length(homepage) <= 255)),
    CONSTRAINT check_9824fc9efc CHECK ((char_length(bindir) <= 255)),
    CONSTRAINT check_994b68eb64 CHECK ((char_length(authors) <= 255)),
    CONSTRAINT check_9d42fa48ae CHECK ((char_length(signing_key) <= 255)),
    CONSTRAINT check_b0f4f8c853 CHECK ((char_length(files) <= 255)),
    CONSTRAINT check_b7b296b420 CHECK ((char_length(author) <= 255)),
    CONSTRAINT check_bf16b21a47 CHECK ((char_length(rdoc_options) <= 255)),
    CONSTRAINT check_ca641a3354 CHECK ((char_length(required_ruby_version) <= 255)),
    CONSTRAINT check_e7009bea46 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_ea02f4800f CHECK ((char_length(metadata) <= 30000)),
    CONSTRAINT check_f76bad1a9a CHECK ((char_length(require_paths) <= 255))
);

CREATE TABLE packages_tags (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    name character varying(255) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    CONSTRAINT check_91b8472153 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE packages_tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_tags_id_seq OWNED BY packages_tags.id;

CREATE TABLE packages_terraform_module_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_id bigint NOT NULL,
    project_id bigint NOT NULL,
    fields jsonb NOT NULL,
    semver_major integer,
    semver_minor integer,
    semver_prerelease text,
    semver_patch bigint,
    CONSTRAINT check_46aa6c883a CHECK ((char_length(semver_prerelease) <= 255)),
    CONSTRAINT chk_rails_49f7b485ae CHECK ((char_length((fields)::text) <= 10485760))
);

CREATE TABLE pages_deployment_states (
    pages_deployment_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    project_id bigint,
    CONSTRAINT check_15217e8c3a CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE pages_deployment_states_pages_deployment_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_deployment_states_pages_deployment_id_seq OWNED BY pages_deployment_states.pages_deployment_id;

CREATE TABLE pages_deployments (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    ci_build_id bigint,
    file_store smallint NOT NULL,
    file text NOT NULL,
    file_count integer NOT NULL,
    file_sha256 bytea NOT NULL,
    size bigint,
    root_directory text DEFAULT 'public'::text,
    path_prefix text,
    build_ref text,
    deleted_at timestamp with time zone,
    upload_ready boolean DEFAULT false,
    expires_at timestamp with time zone,
    CONSTRAINT check_4d04b8dc9a CHECK ((char_length(path_prefix) <= 128)),
    CONSTRAINT check_5f9132a958 CHECK ((size IS NOT NULL)),
    CONSTRAINT check_7e938c810a CHECK ((char_length(root_directory) <= 255)),
    CONSTRAINT check_b44e900e5c CHECK ((char_length(build_ref) <= 512)),
    CONSTRAINT check_f0fe8032dd CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE pages_deployments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_deployments_id_seq OWNED BY pages_deployments.id;

CREATE TABLE pages_domain_acme_orders (
    id bigint NOT NULL,
    pages_domain_id bigint NOT NULL,
    expires_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    url character varying NOT NULL,
    challenge_token character varying NOT NULL,
    challenge_file_content text NOT NULL,
    encrypted_private_key text NOT NULL,
    encrypted_private_key_iv text NOT NULL,
    project_id bigint,
    CONSTRAINT check_07cb634b65 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE pages_domain_acme_orders_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_domain_acme_orders_id_seq OWNED BY pages_domain_acme_orders.id;

CREATE TABLE pages_domains (
    id bigint NOT NULL,
    project_id bigint,
    certificate text,
    encrypted_key text,
    encrypted_key_iv character varying,
    encrypted_key_salt character varying,
    domain character varying,
    verified_at timestamp with time zone,
    verification_code character varying NOT NULL,
    enabled_until timestamp with time zone,
    remove_at timestamp with time zone,
    auto_ssl_enabled boolean DEFAULT false NOT NULL,
    certificate_valid_not_before timestamp with time zone,
    certificate_valid_not_after timestamp with time zone,
    certificate_source smallint DEFAULT 0 NOT NULL,
    wildcard boolean DEFAULT false NOT NULL,
    usage smallint DEFAULT 0 NOT NULL,
    scope smallint DEFAULT 2 NOT NULL,
    auto_ssl_failed boolean DEFAULT false NOT NULL,
    CONSTRAINT check_790fbb64fa CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE pages_domains_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_domains_id_seq OWNED BY pages_domains.id;

CREATE TABLE path_locks (
    id bigint NOT NULL,
    path character varying NOT NULL,
    project_id bigint,
    user_id bigint,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    CONSTRAINT check_e1de2eb0f1 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE path_locks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE path_locks_id_seq OWNED BY path_locks.id;

CREATE TABLE personal_access_token_granular_scopes (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    personal_access_token_id bigint NOT NULL,
    granular_scope_id bigint NOT NULL
);

CREATE SEQUENCE personal_access_token_granular_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE personal_access_token_granular_scopes_id_seq OWNED BY personal_access_token_granular_scopes.id;

CREATE TABLE personal_access_token_last_used_ips (
    id bigint NOT NULL,
    personal_access_token_id bigint NOT NULL,
    organization_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    ip_address inet
);

CREATE SEQUENCE personal_access_token_last_used_ips_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE personal_access_token_last_used_ips_id_seq OWNED BY personal_access_token_last_used_ips.id;

CREATE TABLE personal_access_tokens (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    name character varying NOT NULL,
    revoked boolean DEFAULT false,
    expires_at date,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    scopes character varying DEFAULT '--- []
'::character varying NOT NULL,
    impersonation boolean DEFAULT false NOT NULL,
    token_digest character varying,
    expire_notification_delivered boolean DEFAULT false NOT NULL,
    last_used_at timestamp with time zone,
    after_expiry_notification_delivered boolean DEFAULT false NOT NULL,
    previous_personal_access_token_id bigint,
    organization_id bigint NOT NULL,
    seven_days_notification_sent_at timestamp with time zone,
    thirty_days_notification_sent_at timestamp with time zone,
    sixty_days_notification_sent_at timestamp with time zone,
    description text,
    group_id bigint,
    user_type smallint,
    granular boolean DEFAULT false NOT NULL,
    CONSTRAINT check_6d2ddc9355 CHECK ((char_length(description) <= 255))
);

CREATE SEQUENCE personal_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE personal_access_tokens_id_seq OWNED BY personal_access_tokens.id;

CREATE TABLE snippet_repositories (
    snippet_id bigint NOT NULL,
    shard_id bigint,
    disk_path character varying(80) NOT NULL,
    verification_retry_count smallint,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_checksum bytea,
    verification_failure text,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    snippet_project_id bigint,
    snippet_organization_id bigint,
    CONSTRAINT check_0d1a38de4c CHECK ((num_nonnulls(snippet_organization_id, snippet_project_id) = 1)),
    CONSTRAINT snippet_repositories_verification_failure_text_limit CHECK ((char_length(verification_failure) <= 255))
);

CREATE TABLE snippets (
    id bigint NOT NULL,
    title character varying,
    content text,
    author_id bigint NOT NULL,
    project_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    file_name character varying,
    type character varying,
    visibility_level integer DEFAULT 0 NOT NULL,
    title_html text,
    content_html text,
    cached_markdown_version integer,
    description text,
    description_html text,
    encrypted_secret_token character varying(255),
    encrypted_secret_token_iv character varying(255),
    secret boolean DEFAULT false NOT NULL,
    repository_read_only boolean DEFAULT false NOT NULL,
    imported_from smallint DEFAULT 0 NOT NULL,
    organization_id bigint,
    CONSTRAINT check_82c1d40fab CHECK ((num_nonnulls(organization_id, project_id) = 1))
);

CREATE VIEW personal_snippets_view AS
 SELECT sn.id,
    sh.name AS repository_storage,
    sr.disk_path
   FROM ((snippets sn
     JOIN snippet_repositories sr ON (((sn.id = sr.snippet_id) AND ((sn.type)::text = 'PersonalSnippet'::text))))
     JOIN shards sh ON ((sr.shard_id = sh.id)));

CREATE TABLE pipl_users (
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    initial_email_sent_at timestamp with time zone,
    last_access_from_pipl_country_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL
);

CREATE TABLE plan_limits (
    id bigint NOT NULL,
    plan_id bigint NOT NULL,
    ci_pipeline_size integer DEFAULT 0 NOT NULL,
    ci_active_jobs integer DEFAULT 0 NOT NULL,
    project_hooks integer DEFAULT 100 NOT NULL,
    group_hooks integer DEFAULT 50 NOT NULL,
    ci_project_subscriptions integer DEFAULT 2 NOT NULL,
    ci_pipeline_schedules integer DEFAULT 10 NOT NULL,
    offset_pagination_limit integer DEFAULT 50000 NOT NULL,
    ci_instance_level_variables integer DEFAULT 25 NOT NULL,
    storage_size_limit integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_lsif integer DEFAULT 200 NOT NULL,
    ci_max_artifact_size_archive integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_metadata integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_trace integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_junit integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_sast integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_dependency_scanning integer DEFAULT 350 NOT NULL,
    ci_max_artifact_size_container_scanning integer DEFAULT 150 NOT NULL,
    ci_max_artifact_size_dast integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_codequality integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_license_management integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_license_scanning integer DEFAULT 100 NOT NULL,
    ci_max_artifact_size_performance integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_metrics integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_metrics_referee integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_network_referee integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_dotenv integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_cobertura integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_terraform integer DEFAULT 5 NOT NULL,
    ci_max_artifact_size_accessibility integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_cluster_applications integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_secret_detection integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_requirements integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_coverage_fuzzing integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_browser_performance integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_load_performance integer DEFAULT 0 NOT NULL,
    ci_needs_size_limit integer DEFAULT 50 NOT NULL,
    conan_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    maven_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    npm_max_file_size bigint DEFAULT 524288000 NOT NULL,
    nuget_max_file_size bigint DEFAULT 524288000 NOT NULL,
    pypi_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    generic_packages_max_file_size bigint DEFAULT '5368709120'::bigint NOT NULL,
    golang_max_file_size bigint DEFAULT 104857600 NOT NULL,
    debian_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    project_feature_flags integer DEFAULT 200 NOT NULL,
    ci_max_artifact_size_api_fuzzing integer DEFAULT 0 NOT NULL,
    ci_pipeline_deployments integer DEFAULT 500 NOT NULL,
    pull_mirror_interval_seconds integer DEFAULT 300 NOT NULL,
    daily_invites integer DEFAULT 0 NOT NULL,
    rubygems_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    terraform_module_max_file_size bigint DEFAULT 1073741824 NOT NULL,
    helm_max_file_size bigint DEFAULT 5242880 NOT NULL,
    ci_registered_group_runners integer DEFAULT 1000 NOT NULL,
    ci_registered_project_runners integer DEFAULT 1000 NOT NULL,
    ci_daily_pipeline_schedule_triggers integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_running_container_scanning integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_cluster_image_scanning integer DEFAULT 0 NOT NULL,
    ci_jobs_trace_size_limit integer DEFAULT 100 NOT NULL,
    pages_file_entries integer DEFAULT 200000 NOT NULL,
    dast_profile_schedules integer DEFAULT 1 NOT NULL,
    external_audit_event_destinations integer DEFAULT 5 NOT NULL,
    dotenv_variables integer DEFAULT 20 NOT NULL,
    dotenv_size integer DEFAULT 5120 NOT NULL,
    pipeline_triggers integer DEFAULT 25000 NOT NULL,
    project_ci_secure_files integer DEFAULT 100 NOT NULL,
    repository_size bigint,
    security_policy_scan_execution_schedules integer DEFAULT 0 NOT NULL,
    web_hook_calls_mid integer DEFAULT 0 NOT NULL,
    web_hook_calls_low integer DEFAULT 0 NOT NULL,
    project_ci_variables integer DEFAULT 8000 NOT NULL,
    group_ci_variables integer DEFAULT 30000 NOT NULL,
    ci_max_artifact_size_cyclonedx integer DEFAULT 1 NOT NULL,
    rpm_max_file_size bigint DEFAULT '5368709120'::bigint NOT NULL,
    ci_max_artifact_size_requirements_v2 integer DEFAULT 0 NOT NULL,
    pipeline_hierarchy_size integer DEFAULT 1000 NOT NULL,
    enforcement_limit integer DEFAULT 0 NOT NULL,
    notification_limit integer DEFAULT 0 NOT NULL,
    dashboard_limit_enabled_at timestamp with time zone,
    web_hook_calls integer DEFAULT 0 NOT NULL,
    project_access_token_limit integer DEFAULT 0 NOT NULL,
    google_cloud_logging_configurations integer DEFAULT 5 NOT NULL,
    ml_model_max_file_size bigint DEFAULT '10737418240'::bigint NOT NULL,
    limits_history jsonb DEFAULT '{}'::jsonb NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    ci_max_artifact_size_annotations integer DEFAULT 0 NOT NULL,
    ci_job_annotations_size integer DEFAULT 81920 NOT NULL,
    ci_job_annotations_num integer DEFAULT 20 NOT NULL,
    file_size_limit_mb double precision DEFAULT 100.0 NOT NULL,
    audit_events_amazon_s3_configurations integer DEFAULT 5 NOT NULL,
    ci_max_artifact_size_repository_xray bigint DEFAULT 1073741824 NOT NULL,
    active_versioned_pages_deployments_limit_by_namespace integer DEFAULT 1000 NOT NULL,
    ci_max_artifact_size_jacoco bigint DEFAULT 0 NOT NULL,
    import_placeholder_user_limit_tier_1 integer DEFAULT 0 NOT NULL,
    import_placeholder_user_limit_tier_2 integer DEFAULT 0 NOT NULL,
    import_placeholder_user_limit_tier_3 integer DEFAULT 0 NOT NULL,
    import_placeholder_user_limit_tier_4 integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_slsa_provenance_statement bigint DEFAULT 0 NOT NULL,
    cargo_max_file_size bigint DEFAULT '5368709120'::bigint NOT NULL,
    ci_max_artifact_size_scip integer DEFAULT 200 NOT NULL,
    plan_name_uid smallint
);

CREATE SEQUENCE plan_limits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE plan_limits_id_seq OWNED BY plan_limits.id;

CREATE TABLE plans (
    id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    name character varying,
    title character varying,
    plan_name_uid smallint,
    CONSTRAINT check_226f07be4f CHECK ((plan_name_uid IS NOT NULL))
);

CREATE SEQUENCE plans_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE plans_id_seq OWNED BY plans.id;

CREATE TABLE pm_advisories (
    id bigint NOT NULL,
    advisory_xid text NOT NULL,
    published_date date NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    source_xid smallint NOT NULL,
    title text,
    description text,
    cvss_v2 text,
    cvss_v3 text,
    urls text[] DEFAULT '{}'::text[],
    identifiers jsonb NOT NULL,
    cve text,
    CONSTRAINT check_152def3868 CHECK ((char_length(cvss_v2) <= 128)),
    CONSTRAINT check_19cbd06439 CHECK ((char_length(advisory_xid) <= 36)),
    CONSTRAINT check_b1c980b212 CHECK ((char_length(cve) <= 24)),
    CONSTRAINT check_bed97fa77a CHECK ((char_length(cvss_v3) <= 128)),
    CONSTRAINT check_e4bfd3ffbf CHECK ((char_length(title) <= 256)),
    CONSTRAINT check_fee880f7aa CHECK ((char_length(description) <= 8192)),
    CONSTRAINT chk_rails_e73af9de76 CHECK ((cardinality(urls) <= 20))
);

CREATE SEQUENCE pm_advisories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_advisories_id_seq OWNED BY pm_advisories.id;

CREATE TABLE pm_affected_packages (
    id bigint NOT NULL,
    pm_advisory_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    purl_type smallint NOT NULL,
    package_name text NOT NULL,
    distro_version text DEFAULT ''::text NOT NULL,
    solution text,
    affected_range text NOT NULL,
    fixed_versions text[] DEFAULT '{}'::text[],
    overridden_advisory_fields jsonb DEFAULT '{}'::jsonb NOT NULL,
    versions jsonb DEFAULT '[]'::jsonb NOT NULL,
    CONSTRAINT check_5dd528a2be CHECK ((char_length(package_name) <= 256)),
    CONSTRAINT check_80dea16c7b CHECK ((char_length(affected_range) <= 512)),
    CONSTRAINT check_d1d4646298 CHECK ((char_length(solution) <= 2048)),
    CONSTRAINT check_ec4c8efb5e CHECK ((char_length(distro_version) <= 256)),
    CONSTRAINT chk_rails_a0f80d74e0 CHECK ((cardinality(fixed_versions) <= 10))
);

CREATE SEQUENCE pm_affected_packages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_affected_packages_id_seq OWNED BY pm_affected_packages.id;

CREATE TABLE pm_checkpoints (
    sequence integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    purl_type smallint NOT NULL,
    chunk smallint NOT NULL,
    data_type smallint DEFAULT 1 NOT NULL,
    version_format smallint DEFAULT 1 NOT NULL,
    id bigint NOT NULL
);

CREATE SEQUENCE pm_checkpoints_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_checkpoints_id_seq OWNED BY pm_checkpoints.id;

CREATE TABLE pm_cve_enrichment (
    id bigint NOT NULL,
    epss_score double precision NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cve text NOT NULL,
    is_known_exploit boolean DEFAULT false NOT NULL,
    CONSTRAINT check_16651e3ffb CHECK ((char_length(cve) <= 24))
);

CREATE SEQUENCE pm_cve_enrichment_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_cve_enrichment_id_seq OWNED BY pm_cve_enrichment.id;

CREATE TABLE pm_licenses (
    id bigint NOT NULL,
    spdx_identifier text NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_c1eb81d1ba CHECK ((char_length(spdx_identifier) <= 50))
);

CREATE SEQUENCE pm_licenses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_licenses_id_seq OWNED BY pm_licenses.id;

CREATE TABLE pm_package_version_licenses (
    pm_package_version_id bigint NOT NULL,
    pm_license_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    id bigint NOT NULL
);

CREATE SEQUENCE pm_package_version_licenses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_package_version_licenses_id_seq OWNED BY pm_package_version_licenses.id;

CREATE TABLE pm_package_versions (
    id bigint NOT NULL,
    pm_package_id bigint NOT NULL,
    version text NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_2d8a88cfcc CHECK ((char_length(version) <= 255))
);

CREATE SEQUENCE pm_package_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_package_versions_id_seq OWNED BY pm_package_versions.id;

CREATE TABLE pm_packages (
    id bigint NOT NULL,
    purl_type smallint NOT NULL,
    name text NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    licenses jsonb,
    CONSTRAINT check_3a3aedb8ba CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE pm_packages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pm_packages_id_seq OWNED BY pm_packages.id;

CREATE TABLE pool_repositories (
    id bigint NOT NULL,
    shard_id bigint NOT NULL,
    disk_path character varying,
    state character varying,
    source_project_id bigint,
    organization_id bigint
);

CREATE SEQUENCE pool_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pool_repositories_id_seq OWNED BY pool_repositories.id;

CREATE TABLE postgres_async_foreign_key_validations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    table_name text NOT NULL,
    last_error text,
    attempts integer DEFAULT 0 NOT NULL,
    constraint_type smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_536a40afbf CHECK ((char_length(last_error) <= 10000)),
    CONSTRAINT check_74fb7c8e57 CHECK ((char_length(name) <= 63)),
    CONSTRAINT check_cd435d6301 CHECK ((char_length(table_name) <= 63))
);

CREATE SEQUENCE postgres_async_foreign_key_validations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_async_foreign_key_validations_id_seq OWNED BY postgres_async_foreign_key_validations.id;

CREATE TABLE postgres_async_indexes (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    definition text NOT NULL,
    table_name text NOT NULL,
    attempts integer DEFAULT 0 NOT NULL,
    last_error text,
    CONSTRAINT check_083b21157b CHECK ((char_length(definition) <= 2048)),
    CONSTRAINT check_45dc23c315 CHECK ((char_length(last_error) <= 10000)),
    CONSTRAINT check_b732c6cd1d CHECK ((char_length(name) <= 63)),
    CONSTRAINT check_schema_and_name_length CHECK ((char_length(table_name) <= 127))
);

CREATE SEQUENCE postgres_async_indexes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_async_indexes_id_seq OWNED BY postgres_async_indexes.id;

CREATE VIEW postgres_autovacuum_activity AS
 WITH processes AS (
         SELECT postgres_pg_stat_activity_autovacuum.query,
            postgres_pg_stat_activity_autovacuum.query_start,
            regexp_matches(postgres_pg_stat_activity_autovacuum.query, '^autovacuum: VACUUM (\w+)\.(\w+)'::text) AS matches,
                CASE
                    WHEN (postgres_pg_stat_activity_autovacuum.query ~~* '%wraparound)'::text) THEN true
                    ELSE false
                END AS wraparound_prevention
           FROM postgres_pg_stat_activity_autovacuum() postgres_pg_stat_activity_autovacuum(query, query_start)
          WHERE (postgres_pg_stat_activity_autovacuum.query ~* '^autovacuum: VACUUM \w+\.\w+'::text)
        )
 SELECT ((matches[1] || '.'::text) || matches[2]) AS table_identifier,
    matches[1] AS schema,
    matches[2] AS "table",
    query_start AS vacuum_start,
    wraparound_prevention
   FROM processes;

COMMENT ON VIEW postgres_autovacuum_activity IS 'Contains information about PostgreSQL backends currently performing autovacuum operations on the tables indicated here.';

CREATE VIEW postgres_constraints AS
 SELECT pg_constraint.oid,
    pg_constraint.conname AS name,
    pg_constraint.contype AS constraint_type,
    pg_constraint.convalidated AS constraint_valid,
    ( SELECT array_agg(pg_attribute.attname ORDER BY attnums.ordering) AS array_agg
           FROM (unnest(pg_constraint.conkey) WITH ORDINALITY attnums(attnum, ordering)
             JOIN pg_attribute ON (((pg_attribute.attnum = attnums.attnum) AND (pg_attribute.attrelid = pg_class.oid))))) AS column_names,
    (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text) AS table_identifier,
    NULLIF(pg_constraint.conparentid, (0)::oid) AS parent_constraint_oid,
    pg_get_constraintdef(pg_constraint.oid) AS definition
   FROM ((pg_constraint
     JOIN pg_class ON ((pg_constraint.conrelid = pg_class.oid)))
     JOIN pg_namespace ON ((pg_class.relnamespace = pg_namespace.oid)));

CREATE VIEW postgres_foreign_keys AS
 SELECT pg_constraint.oid,
    pg_constraint.conname AS name,
    (((constrained_namespace.nspname)::text || '.'::text) || (constrained_table.relname)::text) AS constrained_table_identifier,
    (((referenced_namespace.nspname)::text || '.'::text) || (referenced_table.relname)::text) AS referenced_table_identifier,
    (constrained_table.relname)::text AS constrained_table_name,
    (referenced_table.relname)::text AS referenced_table_name,
    constrained_cols.constrained_columns,
    referenced_cols.referenced_columns,
    pg_constraint.confdeltype AS on_delete_action,
    pg_constraint.confupdtype AS on_update_action,
    (pg_constraint.coninhcount > 0) AS is_inherited,
    pg_constraint.convalidated AS is_valid,
    partitioned_parent_oids.parent_oid
   FROM (((((((pg_constraint
     JOIN pg_class constrained_table ON ((constrained_table.oid = pg_constraint.conrelid)))
     JOIN pg_class referenced_table ON ((referenced_table.oid = pg_constraint.confrelid)))
     JOIN pg_namespace constrained_namespace ON ((constrained_table.relnamespace = constrained_namespace.oid)))
     JOIN pg_namespace referenced_namespace ON ((referenced_table.relnamespace = referenced_namespace.oid)))
     CROSS JOIN LATERAL ( SELECT array_agg(pg_attribute.attname ORDER BY conkey.idx) AS array_agg
           FROM (unnest(pg_constraint.conkey) WITH ORDINALITY conkey(attnum, idx)
             JOIN pg_attribute ON (((pg_attribute.attnum = conkey.attnum) AND (pg_attribute.attrelid = constrained_table.oid))))) constrained_cols(constrained_columns))
     CROSS JOIN LATERAL ( SELECT array_agg(pg_attribute.attname ORDER BY confkey.idx) AS array_agg
           FROM (unnest(pg_constraint.confkey) WITH ORDINALITY confkey(attnum, idx)
             JOIN pg_attribute ON (((pg_attribute.attnum = confkey.attnum) AND (pg_attribute.attrelid = referenced_table.oid))))) referenced_cols(referenced_columns))
     LEFT JOIN LATERAL ( SELECT pg_depend.refobjid AS parent_oid
           FROM pg_depend
          WHERE ((pg_depend.objid = pg_constraint.oid) AND (pg_depend.deptype = 'P'::"char") AND (pg_depend.refobjid IN ( SELECT pg_constraint_1.oid
                   FROM pg_constraint pg_constraint_1
                  WHERE (pg_constraint_1.contype = 'f'::"char"))))
         LIMIT 1) partitioned_parent_oids(parent_oid) ON (true))
  WHERE (pg_constraint.contype = 'f'::"char");

CREATE VIEW postgres_index_bloat_estimates AS
 SELECT (((nspname)::text || '.'::text) || (idxname)::text) AS identifier,
    (
        CASE
            WHEN ((relpages)::double precision > est_pages_ff) THEN ((bs)::double precision * ((relpages)::double precision - est_pages_ff))
            ELSE (0)::double precision
        END)::bigint AS bloat_size_bytes
   FROM ( SELECT COALESCE(((1)::double precision + ceil((rows_hdr_pdg_stats.reltuples / floor((((((rows_hdr_pdg_stats.bs - (rows_hdr_pdg_stats.pageopqdata)::numeric) - (rows_hdr_pdg_stats.pagehdr)::numeric) * (rows_hdr_pdg_stats.fillfactor)::numeric))::double precision / ((100)::double precision * (((4)::numeric + rows_hdr_pdg_stats.nulldatahdrwidth))::double precision)))))), (0)::double precision) AS est_pages_ff,
            rows_hdr_pdg_stats.bs,
            rows_hdr_pdg_stats.nspname,
            rows_hdr_pdg_stats.tblname,
            rows_hdr_pdg_stats.idxname,
            rows_hdr_pdg_stats.relpages,
            rows_hdr_pdg_stats.is_na
           FROM ( SELECT rows_data_stats.maxalign,
                    rows_data_stats.bs,
                    rows_data_stats.nspname,
                    rows_data_stats.tblname,
                    rows_data_stats.idxname,
                    rows_data_stats.reltuples,
                    rows_data_stats.relpages,
                    rows_data_stats.idxoid,
                    rows_data_stats.fillfactor,
                    (((((((rows_data_stats.index_tuple_hdr_bm + rows_data_stats.maxalign) -
                        CASE
                            WHEN ((rows_data_stats.index_tuple_hdr_bm % rows_data_stats.maxalign) = 0) THEN rows_data_stats.maxalign
                            ELSE (rows_data_stats.index_tuple_hdr_bm % rows_data_stats.maxalign)
                        END))::double precision + rows_data_stats.nulldatawidth) + (rows_data_stats.maxalign)::double precision) - (
                        CASE
                            WHEN (rows_data_stats.nulldatawidth = (0)::double precision) THEN 0
                            WHEN (((rows_data_stats.nulldatawidth)::integer % rows_data_stats.maxalign) = 0) THEN rows_data_stats.maxalign
                            ELSE ((rows_data_stats.nulldatawidth)::integer % rows_data_stats.maxalign)
                        END)::double precision))::numeric AS nulldatahdrwidth,
                    rows_data_stats.pagehdr,
                    rows_data_stats.pageopqdata,
                    rows_data_stats.is_na
                   FROM ( SELECT n.nspname,
                            i.tblname,
                            i.idxname,
                            i.reltuples,
                            i.relpages,
                            i.idxoid,
                            i.fillfactor,
                            (current_setting('block_size'::text))::numeric AS bs,
                                CASE
                                    WHEN ((version() ~ 'mingw32'::text) OR (version() ~ '64-bit|x86_64|ppc64|ia64|amd64'::text)) THEN 8
                                    ELSE 4
                                END AS maxalign,
                            24 AS pagehdr,
                            16 AS pageopqdata,
                                CASE
                                    WHEN (max(COALESCE(s.null_frac, (0)::real)) = (0)::double precision) THEN 2
                                    ELSE (2 + (((32 + 8) - 1) / 8))
                                END AS index_tuple_hdr_bm,
                            sum((((1)::double precision - COALESCE(s.null_frac, (0)::real)) * (COALESCE(s.avg_width, 1024))::double precision)) AS nulldatawidth,
                            (max(
                                CASE
                                    WHEN (i.atttypid = ('name'::regtype)::oid) THEN 1
                                    ELSE 0
                                END) > 0) AS is_na
                           FROM ((( SELECT ct.relname AS tblname,
                                    ct.relnamespace,
                                    ic.idxname,
                                    ic.attpos,
                                    ic.indkey,
                                    ic.indkey[ic.attpos] AS indkey,
                                    ic.reltuples,
                                    ic.relpages,
                                    ic.tbloid,
                                    ic.idxoid,
                                    ic.fillfactor,
                                    COALESCE(a1.attnum, a2.attnum) AS attnum,
                                    COALESCE(a1.attname, a2.attname) AS attname,
                                    COALESCE(a1.atttypid, a2.atttypid) AS atttypid,
CASE
 WHEN (a1.attnum IS NULL) THEN ic.idxname
 ELSE ct.relname
END AS attrelname
                                   FROM (((( SELECT idx_data.idxname,
    idx_data.reltuples,
    idx_data.relpages,
    idx_data.tbloid,
    idx_data.idxoid,
    idx_data.fillfactor,
    idx_data.indkey,
    generate_series(1, (idx_data.indnatts)::integer) AS attpos
   FROM ( SELECT ci.relname AS idxname,
      ci.reltuples,
      ci.relpages,
      i_1.indrelid AS tbloid,
      i_1.indexrelid AS idxoid,
      COALESCE((("substring"(array_to_string(ci.reloptions, ' '::text), 'fillfactor=([0-9]+)'::text))::smallint)::integer, 90) AS fillfactor,
      i_1.indnatts,
      (string_to_array(textin(int2vectorout(i_1.indkey)), ' '::text))::integer[] AS indkey
     FROM (pg_index i_1
       JOIN pg_class ci ON ((ci.oid = i_1.indexrelid)))
    WHERE ((ci.relam = ( SELECT pg_am.oid
       FROM pg_am
      WHERE (pg_am.amname = 'btree'::name))) AND (ci.relpages > 0))) idx_data) ic
                                     JOIN pg_class ct ON ((ct.oid = ic.tbloid)))
                                     LEFT JOIN pg_attribute a1 ON (((ic.indkey[ic.attpos] <> 0) AND (a1.attrelid = ic.tbloid) AND (a1.attnum = ic.indkey[ic.attpos]))))
                                     LEFT JOIN pg_attribute a2 ON (((ic.indkey[ic.attpos] = 0) AND (a2.attrelid = ic.idxoid) AND (a2.attnum = ic.attpos))))) i(tblname, relnamespace, idxname, attpos, indkey, indkey_1, reltuples, relpages, tbloid, idxoid, fillfactor, attnum, attname, atttypid, attrelname)
                             JOIN pg_namespace n ON ((n.oid = i.relnamespace)))
                             JOIN pg_stats s ON (((s.schemaname = n.nspname) AND (s.tablename = i.attrelname) AND (s.attname = i.attname))))
                          GROUP BY n.nspname, i.tblname, i.idxname, i.reltuples, i.relpages, i.idxoid, i.fillfactor, (current_setting('block_size'::text))::numeric,
                                CASE
                                    WHEN ((version() ~ 'mingw32'::text) OR (version() ~ '64-bit|x86_64|ppc64|ia64|amd64'::text)) THEN 8
                                    ELSE 4
                                END, 24::integer, 16::integer) rows_data_stats) rows_hdr_pdg_stats) relation_stats
  WHERE ((nspname = ANY (ARRAY["current_schema"(), 'gitlab_partitions_dynamic'::name, 'gitlab_partitions_static'::name])) AND (NOT is_na))
  ORDER BY nspname, tblname, idxname;

CREATE VIEW postgres_indexes AS
 SELECT (((pg_namespace.nspname)::text || '.'::text) || (i.relname)::text) AS identifier,
    pg_index.indexrelid,
    pg_namespace.nspname AS schema,
    i.relname AS name,
    pg_indexes.tablename,
    a.amname AS type,
    pg_index.indisunique AS "unique",
    pg_index.indisvalid AS valid_index,
    i.relispartition AS partitioned,
    pg_index.indisexclusion AS exclusion,
    (pg_index.indexprs IS NOT NULL) AS expression,
    (pg_index.indpred IS NOT NULL) AS partial,
    pg_indexes.indexdef AS definition,
    pg_relation_size((i.oid)::regclass) AS ondisk_size_bytes
   FROM ((((pg_index
     JOIN pg_class i ON ((i.oid = pg_index.indexrelid)))
     JOIN pg_namespace ON ((i.relnamespace = pg_namespace.oid)))
     JOIN pg_indexes ON (((i.relname = pg_indexes.indexname) AND (pg_namespace.nspname = pg_indexes.schemaname))))
     JOIN pg_am a ON ((i.relam = a.oid)))
  WHERE ((pg_namespace.nspname <> 'pg_catalog'::name) AND (pg_namespace.nspname = ANY (ARRAY["current_schema"(), 'gitlab_partitions_dynamic'::name, 'gitlab_partitions_static'::name])));

CREATE VIEW postgres_partitioned_tables AS
 SELECT (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text) AS identifier,
    pg_class.oid,
    pg_namespace.nspname AS schema,
    pg_class.relname AS name,
        CASE partitioned_tables.partstrat
            WHEN 'l'::"char" THEN 'list'::text
            WHEN 'r'::"char" THEN 'range'::text
            WHEN 'h'::"char" THEN 'hash'::text
            ELSE NULL::text
        END AS strategy,
    array_agg(pg_attribute.attname) AS key_columns
   FROM (((( SELECT pg_partitioned_table.partrelid,
            pg_partitioned_table.partstrat,
            unnest(pg_partitioned_table.partattrs) AS column_position
           FROM pg_partitioned_table) partitioned_tables
     JOIN pg_class ON ((partitioned_tables.partrelid = pg_class.oid)))
     JOIN pg_namespace ON ((pg_class.relnamespace = pg_namespace.oid)))
     JOIN pg_attribute ON (((pg_attribute.attrelid = pg_class.oid) AND (pg_attribute.attnum = partitioned_tables.column_position))))
  WHERE (pg_namespace.nspname = "current_schema"())
  GROUP BY (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text), pg_class.oid, pg_namespace.nspname, pg_class.relname,
        CASE partitioned_tables.partstrat
            WHEN 'l'::"char" THEN 'list'::text
            WHEN 'r'::"char" THEN 'range'::text
            WHEN 'h'::"char" THEN 'hash'::text
            ELSE NULL::text
        END;

CREATE VIEW postgres_partitions AS
 SELECT (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text) AS identifier,
    pg_class.oid,
    pg_namespace.nspname AS schema,
    pg_class.relname AS name,
    (((parent_namespace.nspname)::text || '.'::text) || (parent_class.relname)::text) AS parent_identifier,
    pg_get_expr(pg_class.relpartbound, pg_inherits.inhrelid) AS condition
   FROM ((((pg_class
     JOIN pg_namespace ON ((pg_namespace.oid = pg_class.relnamespace)))
     JOIN pg_inherits ON ((pg_class.oid = pg_inherits.inhrelid)))
     JOIN pg_class parent_class ON ((pg_inherits.inhparent = parent_class.oid)))
     JOIN pg_namespace parent_namespace ON ((parent_class.relnamespace = parent_namespace.oid)))
  WHERE (pg_class.relispartition AND (pg_namespace.nspname = ANY (ARRAY["current_schema"(), 'gitlab_partitions_dynamic'::name, 'gitlab_partitions_static'::name])));

CREATE TABLE postgres_reindex_actions (
    id bigint NOT NULL,
    action_start timestamp with time zone NOT NULL,
    action_end timestamp with time zone,
    ondisk_size_bytes_start bigint NOT NULL,
    ondisk_size_bytes_end bigint,
    state smallint DEFAULT 0 NOT NULL,
    index_identifier text NOT NULL,
    bloat_estimate_bytes_start bigint,
    CONSTRAINT check_f12527622c CHECK ((char_length(index_identifier) <= 255))
);

CREATE SEQUENCE postgres_reindex_actions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_reindex_actions_id_seq OWNED BY postgres_reindex_actions.id;

CREATE TABLE postgres_reindex_queued_actions (
    id bigint NOT NULL,
    index_identifier text NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_e4b01395c0 CHECK ((char_length(index_identifier) <= 255))
);

CREATE SEQUENCE postgres_reindex_queued_actions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_reindex_queued_actions_id_seq OWNED BY postgres_reindex_queued_actions.id;

CREATE VIEW postgres_sequences AS
 SELECT seq_pg_class.relname AS seq_name,
    dep_pg_class.relname AS table_name,
    pg_attribute.attname AS col_name,
    pg_sequence.seqmax AS seq_max,
    pg_sequence.seqmin AS seq_min,
    pg_sequence.seqstart AS seq_start,
    pg_sequence_last_value((pg_sequence.seqrelid)::regclass) AS last_value
   FROM ((((pg_class seq_pg_class
     JOIN pg_sequence ON ((seq_pg_class.oid = pg_sequence.seqrelid)))
     LEFT JOIN pg_depend ON (((seq_pg_class.oid = pg_depend.objid) AND (pg_depend.classid = ('pg_class'::regclass)::oid) AND (pg_depend.refclassid = ('pg_class'::regclass)::oid))))
     LEFT JOIN pg_class dep_pg_class ON ((pg_depend.refobjid = dep_pg_class.oid)))
     LEFT JOIN pg_attribute ON (((dep_pg_class.oid = pg_attribute.attrelid) AND (pg_depend.refobjsubid = pg_attribute.attnum))))
  WHERE (seq_pg_class.relkind = 'S'::"char");

CREATE VIEW postgres_table_sizes AS
 SELECT (((schemaname)::text || '.'::text) || (relname)::text) AS identifier,
    schemaname AS schema_name,
    relname AS table_name,
    pg_size_pretty(total_bytes) AS total_size,
    pg_size_pretty(table_bytes) AS table_size,
    pg_size_pretty(index_bytes) AS index_size,
    pg_size_pretty(toast_bytes) AS toast_size,
    pg_size_pretty((((total_bytes - table_bytes) - index_bytes) - toast_bytes)) AS auxiliary_size,
    total_bytes AS size_in_bytes
   FROM ( SELECT pg_stat_user_tables.schemaname,
            pg_stat_user_tables.relname,
            pg_total_relation_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass) AS total_bytes,
            pg_relation_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass) AS table_bytes,
            pg_indexes_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass) AS index_bytes,
            ((pg_total_relation_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass) - pg_relation_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass)) - pg_indexes_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass)) AS toast_bytes
           FROM pg_stat_user_tables
          WHERE (pg_total_relation_size((((quote_ident((pg_stat_user_tables.schemaname)::text) || '.'::text) || quote_ident((pg_stat_user_tables.relname)::text)))::regclass) IS NOT NULL)) t
  ORDER BY total_bytes DESC;

CREATE TABLE programming_languages (
    id bigint NOT NULL,
    name character varying NOT NULL,
    color character varying NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE programming_languages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE programming_languages_id_seq OWNED BY programming_languages.id;

CREATE TABLE project_access_tokens (
    personal_access_token_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE TABLE project_alerting_settings (
    project_id bigint NOT NULL,
    encrypted_token character varying NOT NULL,
    encrypted_token_iv character varying NOT NULL
);

CREATE TABLE project_aliases (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name character varying NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_aliases_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_aliases_id_seq OWNED BY project_aliases.id;

CREATE TABLE project_authorizations (
    user_id bigint NOT NULL,
    project_id bigint NOT NULL,
    access_level integer NOT NULL,
    is_unique boolean
);

CREATE TABLE project_authorizations_for_migration (
    user_id bigint NOT NULL,
    project_id bigint NOT NULL,
    access_level smallint NOT NULL
);

CREATE TABLE project_auto_devops (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean,
    deploy_strategy integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_auto_devops_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_auto_devops_id_seq OWNED BY project_auto_devops.id;

CREATE TABLE project_build_artifacts_size_refreshes (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    last_job_artifact_id bigint,
    state smallint DEFAULT 1 NOT NULL,
    refresh_started_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    last_job_artifact_id_on_refresh_start bigint DEFAULT 0
);

CREATE SEQUENCE project_build_artifacts_size_refreshes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_build_artifacts_size_refreshes_id_seq OWNED BY project_build_artifacts_size_refreshes.id;

CREATE TABLE project_ci_cd_settings (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    group_runners_enabled boolean DEFAULT true NOT NULL,
    merge_pipelines_enabled boolean,
    default_git_depth integer,
    forward_deployment_enabled boolean,
    merge_trains_enabled boolean DEFAULT false,
    auto_rollback_enabled boolean DEFAULT false NOT NULL,
    keep_latest_artifact boolean DEFAULT true NOT NULL,
    job_token_scope_enabled boolean DEFAULT false NOT NULL,
    runner_token_expiration_interval integer,
    separated_caches boolean DEFAULT true NOT NULL,
    allow_fork_pipelines_to_run_in_parent_project boolean DEFAULT true NOT NULL,
    inbound_job_token_scope_enabled boolean DEFAULT true NOT NULL,
    forward_deployment_rollback_allowed boolean DEFAULT true NOT NULL,
    merge_trains_skip_train_allowed boolean DEFAULT false NOT NULL,
    restrict_pipeline_cancellation_role smallint DEFAULT 0 NOT NULL,
    pipeline_variables_minimum_override_role smallint DEFAULT 3 NOT NULL,
    push_repository_for_job_token_allowed boolean DEFAULT false NOT NULL,
    id_token_sub_claim_components character varying[] DEFAULT '{project_path,ref_type,ref}'::character varying[] NOT NULL,
    delete_pipelines_in_seconds integer,
    allow_composite_identities_to_run_pipelines boolean DEFAULT false NOT NULL,
    display_pipeline_variables boolean DEFAULT false NOT NULL,
    resource_group_default_process_mode smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_ci_cd_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_ci_cd_settings_id_seq OWNED BY project_ci_cd_settings.id;

CREATE TABLE project_ci_feature_usages (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    feature smallint NOT NULL,
    default_branch boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE project_ci_feature_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_ci_feature_usages_id_seq OWNED BY project_ci_feature_usages.id;

CREATE TABLE project_compliance_framework_settings (
    project_id bigint NOT NULL,
    framework_id bigint,
    id bigint NOT NULL,
    created_at timestamp with time zone,
    CONSTRAINT check_d348de9e2d CHECK ((framework_id IS NOT NULL))
);

CREATE SEQUENCE project_compliance_framework_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_compliance_framework_settings_id_seq OWNED BY project_compliance_framework_settings.id;

CREATE TABLE project_compliance_standards_adherence (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    namespace_id bigint,
    status smallint NOT NULL,
    check_name smallint NOT NULL,
    standard smallint NOT NULL
);

CREATE SEQUENCE project_compliance_standards_adherence_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_compliance_standards_adherence_id_seq OWNED BY project_compliance_standards_adherence.id;

CREATE TABLE project_compliance_violations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    project_id bigint NOT NULL,
    audit_event_id bigint NOT NULL,
    compliance_requirements_control_id bigint NOT NULL,
    status smallint NOT NULL,
    audit_event_table_name smallint NOT NULL
);

CREATE SEQUENCE project_compliance_violations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_compliance_violations_id_seq OWNED BY project_compliance_violations.id;

CREATE TABLE project_compliance_violations_issues (
    id bigint NOT NULL,
    project_compliance_violation_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE SEQUENCE project_compliance_violations_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_compliance_violations_issues_id_seq OWNED BY project_compliance_violations_issues.id;

CREATE TABLE project_control_compliance_statuses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    compliance_requirements_control_id bigint NOT NULL,
    project_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    compliance_requirement_id bigint NOT NULL,
    status smallint NOT NULL,
    requirement_status_id bigint
);

CREATE SEQUENCE project_control_compliance_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_control_compliance_statuses_id_seq OWNED BY project_control_compliance_statuses.id;

CREATE TABLE project_custom_attributes (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    key character varying NOT NULL,
    value character varying NOT NULL
);

CREATE SEQUENCE project_custom_attributes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_custom_attributes_id_seq OWNED BY project_custom_attributes.id;

CREATE TABLE project_daily_statistics_archived (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    fetch_count integer NOT NULL,
    date date
);

CREATE SEQUENCE project_daily_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_daily_statistics_id_seq OWNED BY project_daily_statistics.id;

CREATE TABLE project_data_transfers (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    repository_egress bigint DEFAULT 0 NOT NULL,
    artifacts_egress bigint DEFAULT 0 NOT NULL,
    packages_egress bigint DEFAULT 0 NOT NULL,
    registry_egress bigint DEFAULT 0 NOT NULL,
    date date NOT NULL,
    created_at timestamp with time zone NOT NULL,
    CONSTRAINT project_data_transfers_project_year_month_constraint CHECK ((date = date_trunc('month'::text, (date)::timestamp with time zone)))
);

CREATE SEQUENCE project_data_transfers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_data_transfers_id_seq OWNED BY project_data_transfers.id;

CREATE TABLE project_deploy_tokens (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    deploy_token_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_deploy_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_deploy_tokens_id_seq OWNED BY project_deploy_tokens.id;

CREATE VIEW project_design_management_routes_view AS
 SELECT p.id,
    p.repository_storage,
    r.path AS path_with_namespace,
    r.name AS name_with_namespace
   FROM ((design_management_repositories dr
     JOIN projects p ON ((dr.project_id = p.id)))
     JOIN routes r ON (((p.id = r.source_id) AND ((r.source_type)::text = 'Project'::text))));

CREATE TABLE project_error_tracking_settings (
    project_id bigint NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    api_url character varying,
    encrypted_token character varying,
    encrypted_token_iv character varying,
    project_name character varying,
    organization_name character varying,
    integrated boolean DEFAULT true NOT NULL,
    sentry_project_id bigint
);

CREATE TABLE project_export_jobs (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    jid character varying(100) NOT NULL,
    user_id bigint,
    exported_by_admin boolean DEFAULT false
);

CREATE SEQUENCE project_export_jobs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_export_jobs_id_seq OWNED BY project_export_jobs.id;

CREATE TABLE project_feature_usages (
    project_id bigint NOT NULL,
    jira_dvcs_cloud_last_sync_at timestamp without time zone,
    jira_dvcs_server_last_sync_at timestamp without time zone
);

CREATE TABLE project_features (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    merge_requests_access_level integer,
    issues_access_level integer,
    wiki_access_level integer,
    snippets_access_level integer DEFAULT 20 NOT NULL,
    builds_access_level integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    repository_access_level integer DEFAULT 20 NOT NULL,
    pages_access_level integer NOT NULL,
    forking_access_level integer,
    metrics_dashboard_access_level integer,
    requirements_access_level integer DEFAULT 20 NOT NULL,
    operations_access_level integer DEFAULT 20 NOT NULL,
    analytics_access_level integer DEFAULT 20 NOT NULL,
    security_and_compliance_access_level integer DEFAULT 10 NOT NULL,
    container_registry_access_level integer DEFAULT 0 NOT NULL,
    package_registry_access_level integer DEFAULT 0 NOT NULL,
    monitor_access_level integer DEFAULT 20 NOT NULL,
    infrastructure_access_level integer DEFAULT 20 NOT NULL,
    feature_flags_access_level integer DEFAULT 20 NOT NULL,
    environments_access_level integer DEFAULT 20 NOT NULL,
    releases_access_level integer DEFAULT 20 NOT NULL,
    model_experiments_access_level integer DEFAULT 20 NOT NULL,
    model_registry_access_level integer DEFAULT 20 NOT NULL
);

CREATE SEQUENCE project_features_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_features_id_seq OWNED BY project_features.id;

CREATE TABLE project_group_links (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    group_access integer DEFAULT 30 NOT NULL,
    expires_at date,
    member_role_id bigint
);

CREATE SEQUENCE project_group_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_group_links_id_seq OWNED BY project_group_links.id;

CREATE TABLE project_import_data (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    data text,
    encrypted_credentials text,
    encrypted_credentials_iv character varying,
    encrypted_credentials_salt character varying
);

CREATE SEQUENCE project_import_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_import_data_id_seq OWNED BY project_import_data.id;

CREATE TABLE project_import_export_relation_export_upload_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_3802d7ee83 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE project_incident_management_settings (
    project_id bigint NOT NULL,
    create_issue boolean DEFAULT false NOT NULL,
    send_email boolean DEFAULT false NOT NULL,
    issue_template_key text,
    pagerduty_active boolean DEFAULT false NOT NULL,
    encrypted_pagerduty_token bytea,
    encrypted_pagerduty_token_iv bytea,
    auto_close_incident boolean DEFAULT true NOT NULL,
    sla_timer boolean DEFAULT false,
    sla_timer_minutes integer,
    CONSTRAINT pagerduty_token_iv_length_constraint CHECK ((octet_length(encrypted_pagerduty_token_iv) <= 12)),
    CONSTRAINT pagerduty_token_length_constraint CHECK ((octet_length(encrypted_pagerduty_token) <= 255))
);

CREATE TABLE project_metrics_settings (
    project_id bigint NOT NULL,
    external_dashboard_url character varying,
    dashboard_timezone smallint DEFAULT 0 NOT NULL
);

CREATE TABLE project_mirror_data (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    retry_count integer DEFAULT 0 NOT NULL,
    last_update_started_at timestamp without time zone,
    last_update_scheduled_at timestamp without time zone,
    next_execution_timestamp timestamp without time zone,
    status character varying,
    jid character varying,
    last_error text,
    last_update_at timestamp with time zone,
    last_successful_update_at timestamp with time zone,
    correlation_id_value character varying(128),
    checksums jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE project_mirror_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_mirror_data_id_seq OWNED BY project_mirror_data.id;

CREATE TABLE project_pages_metadata (
    project_id bigint NOT NULL,
    onboarding_complete boolean DEFAULT false NOT NULL
);

CREATE TABLE project_push_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    max_file_size integer DEFAULT 0 NOT NULL,
    member_check boolean DEFAULT false NOT NULL,
    prevent_secrets boolean DEFAULT false NOT NULL,
    commit_committer_name_check boolean DEFAULT false NOT NULL,
    deny_delete_tag boolean,
    reject_unsigned_commits boolean,
    commit_committer_check boolean,
    reject_non_dco_commits boolean,
    commit_message_regex text,
    branch_name_regex text,
    commit_message_negative_regex text,
    author_email_regex text,
    file_name_regex text,
    CONSTRAINT check_02434ac6ef CHECK ((char_length(author_email_regex) <= 511)),
    CONSTRAINT check_295b330afd CHECK ((char_length(commit_message_regex) <= 511)),
    CONSTRAINT check_5423d3cb99 CHECK ((char_length(commit_message_negative_regex) <= 2047)),
    CONSTRAINT check_b99be14656 CHECK ((char_length(file_name_regex) <= 511)),
    CONSTRAINT check_c0e6f89741 CHECK ((char_length(branch_name_regex) <= 511))
);

CREATE SEQUENCE project_push_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_push_rules_id_seq OWNED BY project_push_rules.id;

CREATE TABLE project_relation_export_uploads (
    id bigint NOT NULL,
    project_relation_export_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    export_file text NOT NULL,
    project_id bigint,
    CONSTRAINT check_d8ee243e9e CHECK ((char_length(export_file) <= 255)),
    CONSTRAINT check_f8d6cd1562 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE project_relation_export_uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_relation_export_uploads_id_seq OWNED BY project_relation_export_uploads.id;

CREATE TABLE project_relation_exports (
    id bigint NOT NULL,
    project_export_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relation text NOT NULL,
    jid text,
    export_error text,
    project_id bigint,
    CONSTRAINT check_15e644d856 CHECK ((char_length(jid) <= 255)),
    CONSTRAINT check_4b5880b795 CHECK ((char_length(relation) <= 255)),
    CONSTRAINT check_dbd1cf73d0 CHECK ((char_length(export_error) <= 300)),
    CONSTRAINT check_f461e3537f CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE project_relation_exports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_relation_exports_id_seq OWNED BY project_relation_exports.id;

CREATE TABLE project_repositories (
    id bigint NOT NULL,
    shard_id bigint NOT NULL,
    disk_path character varying NOT NULL,
    project_id bigint NOT NULL,
    object_format smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_repositories_id_seq OWNED BY project_repositories.id;

CREATE TABLE project_repository_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    project_repository_id bigint NOT NULL,
    project_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_443803a648 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE project_repository_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_repository_states_id_seq OWNED BY project_repository_states.id;

CREATE TABLE project_repository_storage_moves (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    source_storage_name text NOT NULL,
    destination_storage_name text NOT NULL,
    error_message text,
    CONSTRAINT check_85854380db CHECK ((char_length(error_message) <= 256)),
    CONSTRAINT project_repository_storage_moves_destination_storage_name CHECK ((char_length(destination_storage_name) <= 255)),
    CONSTRAINT project_repository_storage_moves_source_storage_name CHECK ((char_length(source_storage_name) <= 255))
);

CREATE SEQUENCE project_repository_storage_moves_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_repository_storage_moves_id_seq OWNED BY project_repository_storage_moves.id;

CREATE TABLE project_requirement_compliance_statuses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    compliance_requirement_id bigint NOT NULL,
    compliance_framework_id bigint NOT NULL,
    pass_count integer DEFAULT 0 NOT NULL,
    fail_count integer DEFAULT 0 NOT NULL,
    pending_count integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_requirement_compliance_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_requirement_compliance_statuses_id_seq OWNED BY project_requirement_compliance_statuses.id;

CREATE VIEW project_routes_view AS
 SELECT p.id,
    p.repository_storage,
    r.path AS path_with_namespace,
    r.name AS name_with_namespace
   FROM (projects p
     JOIN routes r ON (((p.id = r.source_id) AND ((r.source_type)::text = 'Project'::text))));

CREATE TABLE project_saved_replies (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    content text NOT NULL,
    CONSTRAINT check_5569cfc14e CHECK ((char_length(content) <= 10000)),
    CONSTRAINT check_a3993908da CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE project_saved_replies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_saved_replies_id_seq OWNED BY project_saved_replies.id;

CREATE TABLE project_secrets_managers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    status smallint DEFAULT 0 NOT NULL,
    namespace_path text,
    project_path text,
    CONSTRAINT check_36f529abba CHECK ((char_length(project_path) <= 64)),
    CONSTRAINT check_bbdf5d083f CHECK ((char_length(namespace_path) <= 64))
);

CREATE SEQUENCE project_secrets_managers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_secrets_managers_id_seq OWNED BY project_secrets_managers.id;

CREATE TABLE project_security_exclusions (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    scanner smallint NOT NULL,
    type smallint NOT NULL,
    active boolean DEFAULT true NOT NULL,
    description text,
    value text NOT NULL,
    CONSTRAINT check_3c70ee8804 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_3e918b71ed CHECK ((char_length(value) <= 255))
);

CREATE SEQUENCE project_security_exclusions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_security_exclusions_id_seq OWNED BY project_security_exclusions.id;

CREATE TABLE project_security_settings (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    auto_fix_container_scanning boolean DEFAULT true NOT NULL,
    auto_fix_dast boolean DEFAULT true NOT NULL,
    auto_fix_dependency_scanning boolean DEFAULT true NOT NULL,
    auto_fix_sast boolean DEFAULT true NOT NULL,
    continuous_vulnerability_scans_enabled boolean DEFAULT false NOT NULL,
    container_scanning_for_registry_enabled boolean DEFAULT false NOT NULL,
    secret_push_protection_enabled boolean DEFAULT false,
    validity_checks_enabled boolean DEFAULT false NOT NULL,
    license_configuration_source smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_20a23efdb6 CHECK ((secret_push_protection_enabled IS NOT NULL))
);

CREATE SEQUENCE project_security_settings_project_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_security_settings_project_id_seq OWNED BY project_security_settings.project_id;

CREATE TABLE project_security_statistics (
    project_id bigint NOT NULL,
    vulnerability_count integer DEFAULT 0 NOT NULL
);

CREATE TABLE project_settings (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    push_rule_id bigint,
    show_default_award_emojis boolean DEFAULT true,
    allow_merge_on_skipped_pipeline boolean,
    squash_option smallint DEFAULT 3,
    has_confluence boolean DEFAULT false NOT NULL,
    has_vulnerabilities boolean DEFAULT false NOT NULL,
    prevent_merge_without_jira_issue boolean DEFAULT false NOT NULL,
    cve_id_request_enabled boolean DEFAULT true NOT NULL,
    mr_default_target_self boolean DEFAULT false NOT NULL,
    previous_default_branch text,
    warn_about_potentially_unwanted_characters boolean DEFAULT true NOT NULL,
    merge_commit_template text,
    has_shimo boolean DEFAULT false NOT NULL,
    squash_commit_template text,
    legacy_open_source_license_available boolean DEFAULT true NOT NULL,
    target_platforms character varying[] DEFAULT '{}'::character varying[] NOT NULL,
    enforce_auth_checks_on_uploads boolean DEFAULT true NOT NULL,
    selective_code_owner_removals boolean DEFAULT false NOT NULL,
    issue_branch_template text,
    show_diff_preview_in_email boolean DEFAULT true NOT NULL,
    suggested_reviewers_enabled boolean DEFAULT false NOT NULL,
    only_allow_merge_if_all_status_checks_passed boolean DEFAULT false NOT NULL,
    mirror_branch_regex text,
    allow_pipeline_trigger_approve_deployment boolean DEFAULT false NOT NULL,
    emails_enabled boolean DEFAULT true NOT NULL,
    pages_unique_domain_enabled boolean DEFAULT false NOT NULL,
    pages_unique_domain text,
    runner_registration_enabled boolean DEFAULT true,
    product_analytics_instrumentation_key text,
    product_analytics_data_collector_host text,
    cube_api_base_url text,
    encrypted_cube_api_key bytea,
    encrypted_cube_api_key_iv bytea,
    encrypted_product_analytics_configurator_connection_string bytea,
    encrypted_product_analytics_configurator_connection_string_iv bytea,
    pages_multiple_versions_enabled boolean DEFAULT false NOT NULL,
    allow_merge_without_pipeline boolean DEFAULT false NOT NULL,
    duo_features_enabled boolean DEFAULT true NOT NULL,
    require_reauthentication_to_approve boolean,
    observability_alerts_enabled boolean DEFAULT true NOT NULL,
    spp_repository_pipeline_access boolean DEFAULT true,
    max_number_of_vulnerabilities integer,
    pages_primary_domain text,
    extended_prat_expiry_webhooks_execute boolean DEFAULT false NOT NULL,
    merge_request_title_regex text,
    protect_merge_request_pipelines boolean DEFAULT true NOT NULL,
    auto_duo_code_review_enabled boolean,
    model_prompt_cache_enabled boolean,
    web_based_commit_signing_enabled boolean DEFAULT false NOT NULL,
    duo_context_exclusion_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    merge_request_title_regex_description text,
    duo_remote_flows_enabled boolean,
    duo_foundational_flows_enabled boolean,
    duo_sast_fp_detection_enabled boolean DEFAULT false NOT NULL,
    CONSTRAINT check_1a30456322 CHECK ((char_length(pages_unique_domain) <= 63)),
    CONSTRAINT check_237486989c CHECK ((char_length(merge_request_title_regex_description) <= 255)),
    CONSTRAINT check_3a03e7557a CHECK ((char_length(previous_default_branch) <= 4096)),
    CONSTRAINT check_3ca5cbffe6 CHECK ((char_length(issue_branch_template) <= 255)),
    CONSTRAINT check_4b142e71f3 CHECK ((char_length(product_analytics_data_collector_host) <= 255)),
    CONSTRAINT check_67292e4b99 CHECK ((char_length(mirror_branch_regex) <= 255)),
    CONSTRAINT check_999e5f0aaa CHECK ((char_length(pages_primary_domain) <= 255)),
    CONSTRAINT check_acb7fad2f9 CHECK ((char_length(product_analytics_instrumentation_key) <= 255)),
    CONSTRAINT check_b09644994b CHECK ((char_length(squash_commit_template) <= 500)),
    CONSTRAINT check_bde223416c CHECK ((show_default_award_emojis IS NOT NULL)),
    CONSTRAINT check_eaf7cfb6a7 CHECK ((char_length(merge_commit_template) <= 500)),
    CONSTRAINT check_ee0d751d5c CHECK ((char_length(merge_request_title_regex) <= 255)),
    CONSTRAINT check_f9df7bcee2 CHECK ((char_length(cube_api_base_url) <= 512))
);

CREATE VIEW project_snippets_routes_view AS
 SELECT sn.id,
    sh.name AS repository_storage,
    sr.disk_path,
    r.path AS path_with_namespace,
    r.name AS name_with_namespace
   FROM (((snippets sn
     JOIN snippet_repositories sr ON (((sn.id = sr.snippet_id) AND ((sn.type)::text = 'ProjectSnippet'::text))))
     JOIN shards sh ON ((sr.shard_id = sh.id)))
     JOIN routes r ON (((r.source_id = sn.project_id) AND ((r.source_type)::text = 'Project'::text))));

CREATE TABLE project_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    project_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_0d5a9e7bde CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE project_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_states_id_seq OWNED BY project_states.id;

CREATE TABLE project_statistics (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    commit_count bigint DEFAULT 0 NOT NULL,
    storage_size bigint DEFAULT 0 NOT NULL,
    repository_size bigint DEFAULT 0 NOT NULL,
    lfs_objects_size bigint DEFAULT 0 NOT NULL,
    build_artifacts_size bigint DEFAULT 0 NOT NULL,
    shared_runners_seconds bigint DEFAULT 0 NOT NULL,
    shared_runners_seconds_last_reset timestamp without time zone,
    packages_size bigint DEFAULT 0 NOT NULL,
    wiki_size bigint,
    snippets_size bigint,
    pipeline_artifacts_size bigint DEFAULT 0 NOT NULL,
    uploads_size bigint DEFAULT 0 NOT NULL,
    container_registry_size bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    root_namespace_id bigint
);

CREATE SEQUENCE project_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_statistics_id_seq OWNED BY project_statistics.id;

CREATE TABLE project_to_security_attributes (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    security_attribute_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL
);

CREATE SEQUENCE project_to_security_attributes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_to_security_attributes_id_seq OWNED BY project_to_security_attributes.id;

CREATE TABLE project_topic_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64)),
    CONSTRAINT check_d6cd4bbec3 CHECK ((organization_id IS NOT NULL))
);

CREATE TABLE project_topics (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    topic_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_topics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_topics_id_seq OWNED BY project_topics.id;

CREATE TABLE project_type_ci_runner_machines (
    id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    contacted_at timestamp with time zone,
    creation_state smallint DEFAULT 0 NOT NULL,
    executor_type smallint,
    runner_type smallint NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL,
    system_xid text NOT NULL,
    platform text,
    architecture text,
    revision text,
    ip_address text,
    version text,
    runtime_features jsonb DEFAULT '{}'::jsonb NOT NULL,
    organization_id bigint,
    labels jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_3d8736b3af CHECK ((char_length(system_xid) <= 64)),
    CONSTRAINT check_5bad2a6944 CHECK ((char_length(revision) <= 255)),
    CONSTRAINT check_7dc4eee8a5 CHECK ((char_length(version) <= 2048)),
    CONSTRAINT check_b1e456641b CHECK ((char_length(ip_address) <= 1024)),
    CONSTRAINT check_c788f4b18a CHECK ((char_length(platform) <= 255)),
    CONSTRAINT check_f3d25ab844 CHECK ((char_length(architecture) <= 255)),
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NOT NULL))
);

CREATE TABLE project_type_ci_runners (
    id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    contacted_at timestamp with time zone,
    token_expires_at timestamp with time zone,
    public_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    private_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    access_level integer DEFAULT 0 NOT NULL,
    maximum_timeout integer,
    runner_type smallint NOT NULL,
    registration_type smallint DEFAULT 0 NOT NULL,
    creation_state smallint DEFAULT 0 NOT NULL,
    active boolean DEFAULT true NOT NULL,
    run_untagged boolean DEFAULT true NOT NULL,
    locked boolean DEFAULT false NOT NULL,
    name text,
    token_encrypted text,
    description text,
    maintainer_note text,
    allowed_plans text[] DEFAULT '{}'::text[] NOT NULL,
    allowed_plan_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    organization_id bigint,
    allowed_plan_name_uids smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    CONSTRAINT check_030ad0773d CHECK ((char_length(token_encrypted) <= 512)),
    CONSTRAINT check_1f8618ab23 CHECK ((char_length(name) <= 256)),
    CONSTRAINT check_24b281f5bf CHECK ((char_length(maintainer_note) <= 1024)),
    CONSTRAINT check_5db8ae9d30 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_organization_id_nullness CHECK ((organization_id IS NOT NULL))
);

CREATE TABLE project_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64)),
    CONSTRAINT check_d5b2a170a1 CHECK ((project_id IS NOT NULL))
);

CREATE TABLE project_wiki_repositories (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_wiki_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_wiki_repositories_id_seq OWNED BY project_wiki_repositories.id;

CREATE TABLE projects_branch_rules_merge_request_approval_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    protected_branch_id bigint NOT NULL,
    project_id bigint NOT NULL,
    prevent_author_approval boolean DEFAULT false NOT NULL,
    prevent_committer_approval boolean DEFAULT false NOT NULL,
    prevent_editing_approval_rules boolean DEFAULT false NOT NULL,
    require_reauthentication_to_approve boolean DEFAULT false NOT NULL,
    approval_removals smallint DEFAULT 1 NOT NULL
);

CREATE SEQUENCE projects_branch_rules_merge_request_approval_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_branch_rules_merge_request_approval_settings_id_seq OWNED BY projects_branch_rules_merge_request_approval_settings.id;

CREATE TABLE projects_branch_rules_squash_options (
    id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    squash_option smallint DEFAULT 3 NOT NULL
);

CREATE SEQUENCE projects_branch_rules_squash_options_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_branch_rules_squash_options_id_seq OWNED BY projects_branch_rules_squash_options.id;

CREATE SEQUENCE projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_id_seq OWNED BY projects.id;

CREATE TABLE projects_sync_events (
    id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE SEQUENCE projects_sync_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_sync_events_id_seq OWNED BY projects_sync_events.id;

CREATE SEQUENCE projects_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_visits_id_seq OWNED BY projects_visits.id;

CREATE TABLE projects_with_pipeline_variables (
    project_id bigint NOT NULL
);

CREATE TABLE protected_branch_merge_access_levels (
    id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    access_level integer DEFAULT 40,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    user_id bigint,
    group_id bigint,
    protected_branch_project_id bigint,
    protected_branch_namespace_id bigint,
    CONSTRAINT check_66e95f5ee9 CHECK ((num_nonnulls(protected_branch_namespace_id, protected_branch_project_id) = 1))
);

CREATE SEQUENCE protected_branch_merge_access_levels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branch_merge_access_levels_id_seq OWNED BY protected_branch_merge_access_levels.id;

CREATE TABLE protected_branch_push_access_levels (
    id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    access_level integer DEFAULT 40,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    user_id bigint,
    group_id bigint,
    deploy_key_id bigint,
    protected_branch_project_id bigint,
    protected_branch_namespace_id bigint,
    CONSTRAINT check_2b64375289 CHECK ((num_nonnulls(protected_branch_namespace_id, protected_branch_project_id) = 1))
);

CREATE SEQUENCE protected_branch_push_access_levels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branch_push_access_levels_id_seq OWNED BY protected_branch_push_access_levels.id;

CREATE TABLE protected_branch_unprotect_access_levels (
    id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    access_level integer DEFAULT 40,
    user_id bigint,
    group_id bigint,
    protected_branch_project_id bigint,
    protected_branch_namespace_id bigint,
    CONSTRAINT check_a5a558921b CHECK ((num_nonnulls(protected_branch_namespace_id, protected_branch_project_id) = 1))
);

CREATE SEQUENCE protected_branch_unprotect_access_levels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branch_unprotect_access_levels_id_seq OWNED BY protected_branch_unprotect_access_levels.id;

CREATE TABLE protected_branches (
    id bigint NOT NULL,
    project_id bigint,
    name character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    code_owner_approval_required boolean DEFAULT false NOT NULL,
    allow_force_push boolean DEFAULT false NOT NULL,
    namespace_id bigint,
    CONSTRAINT protected_branches_project_id_namespace_id_any_not_null CHECK (((project_id IS NULL) <> (namespace_id IS NULL)))
);

CREATE SEQUENCE protected_branches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branches_id_seq OWNED BY protected_branches.id;

CREATE TABLE protected_environment_approval_rules (
    id bigint NOT NULL,
    protected_environment_id bigint NOT NULL,
    user_id bigint,
    group_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    access_level smallint,
    required_approvals smallint NOT NULL,
    group_inheritance_type smallint DEFAULT 0 NOT NULL,
    protected_environment_project_id bigint,
    protected_environment_group_id bigint,
    CONSTRAINT check_e853acbde6 CHECK ((num_nonnulls(protected_environment_group_id, protected_environment_project_id) = 1)),
    CONSTRAINT chk_rails_bed75249bc CHECK ((((access_level IS NOT NULL) AND (group_id IS NULL) AND (user_id IS NULL)) OR ((user_id IS NOT NULL) AND (access_level IS NULL) AND (group_id IS NULL)) OR ((group_id IS NOT NULL) AND (user_id IS NULL) AND (access_level IS NULL)))),
    CONSTRAINT chk_rails_cfa90ae3b5 CHECK ((required_approvals > 0))
);

CREATE SEQUENCE protected_environment_approval_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_environment_approval_rules_id_seq OWNED BY protected_environment_approval_rules.id;

CREATE TABLE protected_environment_deploy_access_levels (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    access_level integer,
    protected_environment_id bigint NOT NULL,
    user_id bigint,
    group_id bigint,
    group_inheritance_type smallint DEFAULT 0 NOT NULL,
    protected_environment_project_id bigint,
    protected_environment_group_id bigint,
    CONSTRAINT check_cee712b465 CHECK ((num_nonnulls(protected_environment_group_id, protected_environment_project_id) = 1)),
    CONSTRAINT check_deploy_access_levels_user_group_access_level_any_not_null CHECK ((num_nonnulls(user_id, group_id, access_level) = 1))
);

CREATE SEQUENCE protected_environment_deploy_access_levels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_environment_deploy_access_levels_id_seq OWNED BY protected_environment_deploy_access_levels.id;

CREATE TABLE protected_environments (
    id bigint NOT NULL,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    group_id bigint,
    required_approval_count integer DEFAULT 0 NOT NULL,
    CONSTRAINT protected_environments_project_or_group_existence CHECK (((project_id IS NULL) <> (group_id IS NULL))),
    CONSTRAINT protected_environments_required_approval_count_positive CHECK ((required_approval_count >= 0))
);

CREATE SEQUENCE protected_environments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_environments_id_seq OWNED BY protected_environments.id;

CREATE TABLE protected_tag_create_access_levels (
    id bigint NOT NULL,
    protected_tag_id bigint NOT NULL,
    access_level integer DEFAULT 40,
    user_id bigint,
    group_id bigint,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    deploy_key_id bigint,
    project_id bigint,
    CONSTRAINT check_e56dc4b33a CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE protected_tag_create_access_levels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_tag_create_access_levels_id_seq OWNED BY protected_tag_create_access_levels.id;

CREATE TABLE protected_tags (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE protected_tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_tags_id_seq OWNED BY protected_tags.id;

CREATE TABLE push_event_payloads (
    commit_count bigint NOT NULL,
    action smallint NOT NULL,
    ref_type smallint NOT NULL,
    commit_from bytea,
    commit_to bytea,
    ref text,
    commit_title character varying(70),
    ref_count integer,
    event_id bigint NOT NULL,
    project_id bigint,
    CONSTRAINT check_37c617d07d CHECK ((project_id IS NOT NULL))
);

CREATE TABLE push_rules (
    id bigint NOT NULL,
    commit_message_regex character varying,
    deny_delete_tag boolean,
    project_id bigint,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    author_email_regex character varying,
    member_check boolean DEFAULT false NOT NULL,
    file_name_regex character varying,
    is_sample boolean DEFAULT false,
    max_file_size integer DEFAULT 0 NOT NULL,
    prevent_secrets boolean DEFAULT false NOT NULL,
    branch_name_regex character varying,
    reject_unsigned_commits boolean,
    commit_committer_check boolean,
    regexp_uses_re2 boolean DEFAULT true,
    commit_message_negative_regex character varying,
    reject_non_dco_commits boolean,
    commit_committer_name_check boolean DEFAULT false NOT NULL,
    organization_id bigint,
    CONSTRAINT author_email_regex_size_constraint CHECK ((char_length((author_email_regex)::text) <= 511)),
    CONSTRAINT branch_name_regex_size_constraint CHECK ((char_length((branch_name_regex)::text) <= 511)),
    CONSTRAINT commit_message_negative_regex_size_constraint CHECK ((char_length((commit_message_negative_regex)::text) <= 2047)),
    CONSTRAINT commit_message_regex_size_constraint CHECK ((char_length((commit_message_regex)::text) <= 511)),
    CONSTRAINT file_name_regex_size_constraint CHECK ((char_length((file_name_regex)::text) <= 511))
);

CREATE SEQUENCE push_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE push_rules_id_seq OWNED BY push_rules.id;

CREATE TABLE queries_service_pings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    payload jsonb NOT NULL,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE queries_service_pings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE queries_service_pings_id_seq OWNED BY queries_service_pings.id;

CREATE TABLE raw_usage_data (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    sent_at timestamp with time zone,
    payload jsonb NOT NULL,
    version_usage_data_id_value bigint,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE raw_usage_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE raw_usage_data_id_seq OWNED BY raw_usage_data.id;

CREATE TABLE redirect_routes (
    id bigint NOT NULL,
    source_id bigint NOT NULL,
    source_type character varying NOT NULL,
    path character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_e82ff70482 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE redirect_routes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE redirect_routes_id_seq OWNED BY redirect_routes.id;

CREATE TABLE related_epic_links (
    id bigint NOT NULL,
    source_id bigint NOT NULL,
    target_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    link_type smallint DEFAULT 0 NOT NULL,
    group_id bigint,
    issue_link_id bigint,
    CONSTRAINT check_9c7bbef67d CHECK ((group_id IS NOT NULL))
);

CREATE SEQUENCE related_epic_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE related_epic_links_id_seq OWNED BY related_epic_links.id;

CREATE TABLE relation_import_trackers (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relation smallint NOT NULL,
    status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE relation_import_trackers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE relation_import_trackers_id_seq OWNED BY relation_import_trackers.id;

CREATE TABLE release_links (
    id bigint NOT NULL,
    release_id bigint NOT NULL,
    url character varying NOT NULL,
    name character varying NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    filepath character varying(128),
    link_type smallint DEFAULT 0,
    project_id bigint,
    CONSTRAINT check_959e7fdd89 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE release_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE release_links_id_seq OWNED BY release_links.id;

CREATE TABLE releases (
    id bigint NOT NULL,
    tag character varying,
    description text,
    project_id bigint,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    description_html text,
    cached_markdown_version integer,
    author_id bigint,
    name character varying,
    sha character varying,
    released_at timestamp with time zone NOT NULL,
    release_published_at timestamp with time zone,
    CONSTRAINT check_6bb9ce4925 CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE releases_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE releases_id_seq OWNED BY releases.id;

CREATE TABLE remote_mirrors (
    id bigint NOT NULL,
    project_id bigint,
    url character varying,
    enabled boolean DEFAULT false,
    update_status character varying,
    last_update_at timestamp without time zone,
    last_successful_update_at timestamp without time zone,
    last_error character varying,
    encrypted_credentials text,
    encrypted_credentials_iv character varying,
    encrypted_credentials_salt character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    last_update_started_at timestamp without time zone,
    only_protected_branches boolean DEFAULT false NOT NULL,
    remote_name character varying,
    error_notification_sent boolean,
    keep_divergent_refs boolean,
    mirror_branch_regex text,
    CONSTRAINT check_7547191afa CHECK ((project_id IS NOT NULL)),
    CONSTRAINT check_aa6b497785 CHECK ((char_length(mirror_branch_regex) <= 255))
);

CREATE SEQUENCE remote_mirrors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE remote_mirrors_id_seq OWNED BY remote_mirrors.id;

CREATE TABLE repository_languages (
    project_id bigint NOT NULL,
    programming_language_id bigint NOT NULL,
    share double precision NOT NULL
);

CREATE TABLE required_code_owners_sections (
    id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    name text NOT NULL,
    protected_branch_project_id bigint,
    protected_branch_namespace_id bigint,
    CONSTRAINT check_e58d53741e CHECK ((char_length(name) <= 1024)),
    CONSTRAINT check_e7c067043a CHECK ((num_nonnulls(protected_branch_namespace_id, protected_branch_project_id) = 1))
);

CREATE SEQUENCE required_code_owners_sections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE required_code_owners_sections_id_seq OWNED BY required_code_owners_sections.id;

CREATE TABLE requirements (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    iid integer NOT NULL,
    issue_id bigint,
    CONSTRAINT check_requirement_issue_not_null CHECK ((issue_id IS NOT NULL))
);

CREATE SEQUENCE requirements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE requirements_id_seq OWNED BY requirements.id;

CREATE TABLE requirements_management_test_reports (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    author_id bigint,
    state smallint NOT NULL,
    build_id bigint,
    issue_id bigint,
    uses_legacy_iid boolean DEFAULT true NOT NULL,
    project_id bigint,
    CONSTRAINT check_715b56da9a CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE requirements_management_test_reports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE requirements_management_test_reports_id_seq OWNED BY requirements_management_test_reports.id;

CREATE TABLE resource_iteration_events (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    issue_id bigint,
    merge_request_id bigint,
    iteration_id bigint,
    created_at timestamp with time zone NOT NULL,
    action smallint NOT NULL,
    automated boolean DEFAULT false NOT NULL,
    triggered_by_id bigint,
    namespace_id bigint NOT NULL,
    CONSTRAINT check_52cee5f824 CHECK ((iteration_id IS NOT NULL))
);

CREATE SEQUENCE resource_iteration_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_iteration_events_id_seq OWNED BY resource_iteration_events.id;

CREATE TABLE resource_label_events (
    id bigint NOT NULL,
    action integer NOT NULL,
    issue_id bigint,
    merge_request_id bigint,
    epic_id bigint,
    label_id bigint,
    user_id bigint,
    created_at timestamp with time zone NOT NULL,
    reference text,
    imported_from smallint DEFAULT 0 NOT NULL,
    namespace_id bigint NOT NULL,
    CONSTRAINT check_614704e750 CHECK ((num_nonnulls(epic_id, issue_id, merge_request_id) = 1))
);

CREATE SEQUENCE resource_label_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_label_events_id_seq OWNED BY resource_label_events.id;

CREATE TABLE resource_link_events (
    id bigint NOT NULL,
    action smallint NOT NULL,
    user_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    child_work_item_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    system_note_metadata_id bigint,
    namespace_id bigint,
    CONSTRAINT check_47e459b05e CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE resource_link_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_link_events_id_seq OWNED BY resource_link_events.id;

CREATE TABLE resource_milestone_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint,
    merge_request_id bigint,
    milestone_id bigint,
    action smallint NOT NULL,
    state smallint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    imported_from smallint DEFAULT 0 NOT NULL,
    namespace_id bigint NOT NULL,
    CONSTRAINT check_fa0260b82e CHECK ((num_nonnulls(issue_id, merge_request_id) = 1))
);

CREATE SEQUENCE resource_milestone_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_milestone_events_id_seq OWNED BY resource_milestone_events.id;

CREATE TABLE resource_state_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint,
    merge_request_id bigint,
    created_at timestamp with time zone NOT NULL,
    state smallint NOT NULL,
    epic_id bigint,
    source_commit text,
    close_after_error_tracking_resolve boolean DEFAULT false NOT NULL,
    close_auto_resolve_prometheus_alert boolean DEFAULT false NOT NULL,
    source_merge_request_id bigint,
    imported_from smallint DEFAULT 0 NOT NULL,
    namespace_id bigint NOT NULL,
    CONSTRAINT check_465d337634 CHECK ((num_nonnulls(epic_id, issue_id, merge_request_id) = 1)),
    CONSTRAINT check_f0bcfaa3a2 CHECK ((char_length(source_commit) <= 40))
);

CREATE SEQUENCE resource_state_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_state_events_id_seq OWNED BY resource_state_events.id;

CREATE TABLE resource_weight_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint NOT NULL,
    weight integer,
    created_at timestamp with time zone NOT NULL,
    previous_weight integer,
    namespace_id bigint,
    CONSTRAINT check_30317d1ce0 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE resource_weight_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_weight_events_id_seq OWNED BY resource_weight_events.id;

CREATE TABLE reviews (
    id bigint NOT NULL,
    author_id bigint,
    merge_request_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE reviews_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE reviews_id_seq OWNED BY reviews.id;

CREATE SEQUENCE routes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE routes_id_seq OWNED BY routes.id;

CREATE TABLE saml_group_links (
    id bigint NOT NULL,
    access_level smallint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    saml_group_name text NOT NULL,
    member_role_id bigint,
    assign_duo_seats boolean DEFAULT false NOT NULL,
    scim_group_uid uuid,
    provider text,
    CONSTRAINT check_1b3fc49d1e CHECK ((char_length(saml_group_name) <= 255)),
    CONSTRAINT check_59e993f34e CHECK ((char_length(provider) <= 255))
);

CREATE SEQUENCE saml_group_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE saml_group_links_id_seq OWNED BY saml_group_links.id;

CREATE TABLE saml_providers (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    enabled boolean NOT NULL,
    certificate_fingerprint character varying NOT NULL,
    sso_url character varying NOT NULL,
    enforced_sso boolean DEFAULT false NOT NULL,
    enforced_group_managed_accounts boolean DEFAULT false NOT NULL,
    prohibited_outer_forks boolean DEFAULT true NOT NULL,
    default_membership_role smallint DEFAULT 10 NOT NULL,
    git_check_enforced boolean DEFAULT false NOT NULL,
    member_role_id bigint,
    disable_password_authentication_for_enterprise_users boolean DEFAULT false
);

CREATE SEQUENCE saml_providers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE saml_providers_id_seq OWNED BY saml_providers.id;

CREATE TABLE saved_replies (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    content text NOT NULL,
    CONSTRAINT check_0cb57dc22a CHECK ((char_length(content) <= 10000)),
    CONSTRAINT check_2eb3366d7f CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE saved_replies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE saved_replies_id_seq OWNED BY saved_replies.id;

CREATE TABLE saved_views (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    created_by_id bigint,
    version integer NOT NULL,
    private boolean DEFAULT true NOT NULL,
    name text NOT NULL,
    description text,
    sort smallint,
    filter_data jsonb,
    display_settings jsonb,
    lock_version integer DEFAULT 0 NOT NULL,
    CONSTRAINT check_61a6c07bf6 CHECK ((char_length(name) <= 140)),
    CONSTRAINT check_d27167623c CHECK ((char_length(description) <= 140))
);

CREATE SEQUENCE saved_views_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE saved_views_id_seq OWNED BY saved_views.id;

CREATE TABLE sbom_component_versions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_id bigint NOT NULL,
    version text NOT NULL,
    source_package_name text,
    organization_id bigint NOT NULL,
    CONSTRAINT check_39636b9a8a CHECK ((char_length(source_package_name) <= 255)),
    CONSTRAINT check_e71cad08d3 CHECK ((char_length(version) <= 255))
);

CREATE SEQUENCE sbom_component_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_component_versions_id_seq OWNED BY sbom_component_versions.id;

CREATE TABLE sbom_components (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_type smallint NOT NULL,
    name text NOT NULL,
    purl_type smallint,
    organization_id bigint NOT NULL,
    CONSTRAINT check_91a8f6ad53 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE sbom_components_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_components_id_seq OWNED BY sbom_components.id;

CREATE TABLE sbom_graph_paths (
    id bigint NOT NULL,
    ancestor_id bigint NOT NULL,
    descendant_id bigint NOT NULL,
    project_id bigint NOT NULL,
    path_length integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    top_level_ancestor boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE sbom_graph_paths_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_graph_paths_id_seq OWNED BY sbom_graph_paths.id;

CREATE TABLE sbom_occurrence_refs (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    sbom_occurrence_id bigint NOT NULL,
    security_project_tracked_context_id bigint NOT NULL,
    commit_sha bytea NOT NULL,
    pipeline_id bigint
);

CREATE SEQUENCE sbom_occurrence_refs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_occurrence_refs_id_seq OWNED BY sbom_occurrence_refs.id;

CREATE TABLE sbom_occurrences (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_version_id bigint,
    project_id bigint NOT NULL,
    pipeline_id bigint,
    source_id bigint,
    commit_sha bytea NOT NULL,
    component_id bigint NOT NULL,
    uuid uuid NOT NULL,
    package_manager text,
    component_name text,
    input_file_path text,
    licenses jsonb DEFAULT '[]'::jsonb,
    highest_severity smallint,
    vulnerability_count integer DEFAULT 0 NOT NULL,
    source_package_id bigint,
    archived boolean DEFAULT false NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    ancestors jsonb DEFAULT '[]'::jsonb NOT NULL,
    reachability smallint DEFAULT 0,
    partition_id bigint DEFAULT 1,
    CONSTRAINT check_3f2d2c7ffc CHECK ((char_length(package_manager) <= 255)),
    CONSTRAINT check_9b29021fa8 CHECK ((char_length(component_name) <= 255)),
    CONSTRAINT check_e6b8437cfe CHECK ((char_length(input_file_path) <= 1024))
);

CREATE SEQUENCE sbom_occurrences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_occurrences_id_seq OWNED BY sbom_occurrences.id;

CREATE TABLE sbom_occurrences_vulnerabilities (
    id bigint NOT NULL,
    sbom_occurrence_id bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint,
    vulnerability_occurrence_id bigint,
    CONSTRAINT check_a02e48df9c CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE sbom_occurrences_vulnerabilities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_occurrences_vulnerabilities_id_seq OWNED BY sbom_occurrences_vulnerabilities.id;

CREATE TABLE sbom_source_packages (
    id bigint NOT NULL,
    name text NOT NULL,
    purl_type smallint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    organization_id bigint NOT NULL,
    CONSTRAINT check_8fba79abed CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE sbom_source_packages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_source_packages_id_seq OWNED BY sbom_source_packages.id;

CREATE TABLE sbom_sources (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    source_type smallint NOT NULL,
    source jsonb DEFAULT '{}'::jsonb NOT NULL,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE sbom_sources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_sources_id_seq OWNED BY sbom_sources.id;

CREATE TABLE sbom_vulnerability_scans (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    build_id bigint NOT NULL,
    status smallint DEFAULT 0,
    sbom_file_store smallint DEFAULT 1,
    result_file_store smallint DEFAULT 1,
    sbom_file text,
    sbom_file_final_path text,
    result_file text,
    error_message text,
    CONSTRAINT check_0225eb20d7 CHECK ((char_length(error_message) <= 1024)),
    CONSTRAINT check_08ddfcbe95 CHECK ((char_length(sbom_file_final_path) <= 1024)),
    CONSTRAINT check_31a6970d96 CHECK ((char_length(result_file) <= 255)),
    CONSTRAINT check_6c95e56fd1 CHECK ((char_length(sbom_file) <= 255))
);

CREATE SEQUENCE sbom_vulnerability_scans_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sbom_vulnerability_scans_id_seq OWNED BY sbom_vulnerability_scans.id;

CREATE TABLE scan_execution_policy_rules (
    id bigint NOT NULL,
    security_policy_id bigint NOT NULL,
    security_policy_management_project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    rule_index smallint NOT NULL,
    type smallint NOT NULL,
    content jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE scan_execution_policy_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scan_execution_policy_rules_id_seq OWNED BY scan_execution_policy_rules.id;

CREATE TABLE scan_result_policies (
    id bigint NOT NULL,
    security_orchestration_policy_configuration_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    orchestration_policy_idx smallint NOT NULL,
    license_states text[] DEFAULT '{}'::text[],
    match_on_inclusion boolean,
    role_approvers integer[] DEFAULT '{}'::integer[],
    age_value integer,
    age_operator smallint,
    age_interval smallint,
    vulnerability_attributes jsonb DEFAULT '{}'::jsonb,
    project_id bigint,
    rule_idx smallint,
    project_approval_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    commits smallint,
    send_bot_message jsonb DEFAULT '{}'::jsonb NOT NULL,
    fallback_behavior jsonb DEFAULT '{}'::jsonb NOT NULL,
    policy_tuning jsonb DEFAULT '{}'::jsonb NOT NULL,
    action_idx smallint DEFAULT 0 NOT NULL,
    custom_roles bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    licenses jsonb DEFAULT '{}'::jsonb NOT NULL,
    namespace_id bigint,
    approval_policy_rule_id bigint,
    CONSTRAINT age_value_null_or_positive CHECK (((age_value IS NULL) OR (age_value >= 0))),
    CONSTRAINT check_2c600018b6 CHECK ((num_nonnulls(namespace_id, project_id) >= 1)),
    CONSTRAINT check_scan_result_policies_rule_idx_positive CHECK (((rule_idx IS NULL) OR (rule_idx >= 0))),
    CONSTRAINT custom_roles_array_check CHECK ((array_position(custom_roles, NULL::bigint) IS NULL))
);

CREATE SEQUENCE scan_result_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scan_result_policies_id_seq OWNED BY scan_result_policies.id;

CREATE TABLE scan_result_policy_violations (
    id bigint NOT NULL,
    scan_result_policy_id bigint,
    merge_request_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    violation_data jsonb,
    approval_policy_rule_id bigint,
    status smallint DEFAULT 1 NOT NULL,
    CONSTRAINT chk_policy_violations_rule_id_or_policy_id_not_null CHECK (((approval_policy_rule_id IS NOT NULL) OR (scan_result_policy_id IS NOT NULL)))
);

CREATE SEQUENCE scan_result_policy_violations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scan_result_policy_violations_id_seq OWNED BY scan_result_policy_violations.id;

CREATE TABLE schema_migrations (
    version character varying NOT NULL,
    finished_at timestamp with time zone DEFAULT now()
);

CREATE TABLE scim_group_memberships (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    scim_group_uid uuid NOT NULL
);

CREATE SEQUENCE scim_group_memberships_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scim_group_memberships_id_seq OWNED BY scim_group_memberships.id;

CREATE TABLE scim_identities (
    id bigint NOT NULL,
    group_id bigint,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT false,
    extern_uid character varying(255) NOT NULL
);

CREATE SEQUENCE scim_identities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scim_identities_id_seq OWNED BY scim_identities.id;

CREATE TABLE scim_oauth_access_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint,
    token_encrypted character varying NOT NULL,
    organization_id bigint,
    CONSTRAINT check_77e6840638 CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE scim_oauth_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scim_oauth_access_tokens_id_seq OWNED BY scim_oauth_access_tokens.id;

CREATE TABLE secret_detection_token_statuses (
    vulnerability_occurrence_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    last_verified_at timestamp with time zone
);

CREATE TABLE secret_rotation_infos (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    secret_name text NOT NULL,
    secret_metadata_version integer NOT NULL,
    rotation_interval_days integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    next_reminder_at timestamp with time zone NOT NULL,
    last_reminder_at timestamp with time zone,
    CONSTRAINT check_c0fb39dc52 CHECK ((char_length(secret_name) <= 255))
);

CREATE SEQUENCE secret_rotation_infos_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE secret_rotation_infos_id_seq OWNED BY secret_rotation_infos.id;

CREATE TABLE secrets_management_recovery_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT false NOT NULL,
    key jsonb NOT NULL
);

CREATE SEQUENCE secrets_management_recovery_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE secrets_management_recovery_keys_id_seq OWNED BY secrets_management_recovery_keys.id;

CREATE TABLE security_attributes (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    security_category_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    editable_state smallint DEFAULT 0 NOT NULL,
    name text NOT NULL,
    description text,
    color text NOT NULL,
    template_type smallint,
    deleted_at timestamp with time zone,
    CONSTRAINT check_219cd2b143 CHECK ((char_length(color) <= 7)),
    CONSTRAINT check_518516df75 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_5f6fd50ef3 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE security_attributes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_attributes_id_seq OWNED BY security_attributes.id;

CREATE TABLE security_categories (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    editable_state smallint DEFAULT 0 NOT NULL,
    template_type smallint,
    multiple_selection boolean DEFAULT false NOT NULL,
    name text NOT NULL,
    description text,
    deleted_at timestamp with time zone,
    CONSTRAINT check_6a761c4c9f CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_d643dfc44b CHECK ((char_length(description) <= 255))
);

CREATE SEQUENCE security_categories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_categories_id_seq OWNED BY security_categories.id;

CREATE TABLE security_finding_enrichments (
    id bigint NOT NULL,
    finding_uuid uuid NOT NULL,
    project_id bigint NOT NULL,
    cve_enrichment_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cve text NOT NULL,
    CONSTRAINT check_1f198c362f CHECK ((char_length(cve) <= 24))
);

CREATE SEQUENCE security_finding_enrichments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_finding_enrichments_id_seq OWNED BY security_finding_enrichments.id;

CREATE TABLE security_finding_token_statuses (
    security_finding_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    raw_source_code_extract text,
    last_verified_at timestamp with time zone,
    CONSTRAINT raw_source_code_extract_not_longer_than_2048 CHECK ((char_length(raw_source_code_extract) <= 2048))
);

CREATE SEQUENCE security_findings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_findings_id_seq OWNED BY security_findings.id;

CREATE TABLE security_inventory_filters (
    id bigint NOT NULL,
    archived boolean DEFAULT false NOT NULL,
    sast smallint DEFAULT 0 NOT NULL,
    sast_advanced smallint DEFAULT 0 NOT NULL,
    sast_iac smallint DEFAULT 0 NOT NULL,
    dast smallint DEFAULT 0 NOT NULL,
    dependency_scanning smallint DEFAULT 0 NOT NULL,
    coverage_fuzzing smallint DEFAULT 0 NOT NULL,
    api_fuzzing smallint DEFAULT 0 NOT NULL,
    cluster_image_scanning smallint DEFAULT 0 NOT NULL,
    secret_detection_secret_push_protection smallint DEFAULT 0 NOT NULL,
    container_scanning_for_registry smallint DEFAULT 0 NOT NULL,
    secret_detection_pipeline_based smallint DEFAULT 0 NOT NULL,
    container_scanning_pipeline_based smallint DEFAULT 0 NOT NULL,
    secret_detection smallint DEFAULT 0 NOT NULL,
    container_scanning smallint DEFAULT 0 NOT NULL,
    total integer DEFAULT 0 NOT NULL,
    critical integer DEFAULT 0 NOT NULL,
    high integer DEFAULT 0 NOT NULL,
    medium integer DEFAULT 0 NOT NULL,
    low integer DEFAULT 0 NOT NULL,
    info integer DEFAULT 0 NOT NULL,
    unknown integer DEFAULT 0 NOT NULL,
    project_id bigint NOT NULL,
    traversal_ids bigint[] DEFAULT '{}'::bigint[] NOT NULL,
    project_name text NOT NULL,
    CONSTRAINT check_aeacee81ba CHECK ((char_length(project_name) <= 255))
);

CREATE SEQUENCE security_inventory_filters_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_inventory_filters_id_seq OWNED BY security_inventory_filters.id;

CREATE TABLE security_orchestration_policy_configurations (
    id bigint NOT NULL,
    project_id bigint,
    security_policy_management_project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    configured_at timestamp with time zone,
    namespace_id bigint,
    experiments jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT cop_configs_project_or_namespace_existence CHECK (((project_id IS NULL) <> (namespace_id IS NULL)))
);

COMMENT ON TABLE security_orchestration_policy_configurations IS '{"owner":"group::container security","description":"Configuration used to store relationship between project and security policy repository"}';

CREATE SEQUENCE security_orchestration_policy_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_orchestration_policy_configurations_id_seq OWNED BY security_orchestration_policy_configurations.id;

CREATE TABLE security_orchestration_policy_rule_schedules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    next_run_at timestamp with time zone,
    security_orchestration_policy_configuration_id bigint NOT NULL,
    user_id bigint,
    policy_index integer NOT NULL,
    cron text NOT NULL,
    rule_index integer DEFAULT 0 NOT NULL,
    project_id bigint,
    namespace_id bigint,
    policy_type smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_915825a76e CHECK ((char_length(cron) <= 255)),
    CONSTRAINT check_b2b0883c5c CHECK ((num_nonnulls(namespace_id, project_id) = 1))
);

COMMENT ON TABLE security_orchestration_policy_rule_schedules IS '{"owner":"group::container security","description":"Schedules used to store relationship between project and security policy repository"}';

CREATE SEQUENCE security_orchestration_policy_rule_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_orchestration_policy_rule_schedules_id_seq OWNED BY security_orchestration_policy_rule_schedules.id;

CREATE TABLE security_pipeline_execution_policy_config_links (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    security_policy_id bigint NOT NULL
);

CREATE SEQUENCE security_pipeline_execution_policy_config_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_pipeline_execution_policy_config_links_id_seq OWNED BY security_pipeline_execution_policy_config_links.id;

CREATE TABLE security_pipeline_execution_project_schedules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    next_run_at timestamp with time zone NOT NULL,
    security_policy_id bigint NOT NULL,
    project_id bigint NOT NULL,
    time_window_seconds integer NOT NULL,
    cron text NOT NULL,
    cron_timezone text NOT NULL,
    snoozed_until timestamp with time zone,
    CONSTRAINT check_b93315bfbb CHECK ((char_length(cron_timezone) <= 255)),
    CONSTRAINT check_bbbe4b1b8d CHECK ((char_length(cron) <= 128)),
    CONSTRAINT check_c440017377 CHECK ((time_window_seconds > 0))
);

CREATE SEQUENCE security_pipeline_execution_project_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_pipeline_execution_project_schedules_id_seq OWNED BY security_pipeline_execution_project_schedules.id;

CREATE TABLE security_policies (
    id bigint NOT NULL,
    security_orchestration_policy_configuration_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    policy_index smallint NOT NULL,
    type smallint NOT NULL,
    enabled boolean DEFAULT true NOT NULL,
    name text NOT NULL,
    description text,
    checksum text NOT NULL,
    scope jsonb DEFAULT '{}'::jsonb NOT NULL,
    security_policy_management_project_id bigint NOT NULL,
    content jsonb DEFAULT '{}'::jsonb NOT NULL,
    metadata jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_3fa0f29e4b CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_966e08b242 CHECK ((char_length(checksum) <= 255)),
    CONSTRAINT check_99c8e08928 CHECK ((char_length(description) <= 1000000))
);

CREATE SEQUENCE security_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_policies_id_seq OWNED BY security_policies.id;

CREATE TABLE security_policy_dismissals (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    security_policy_id bigint,
    user_id bigint,
    security_findings_uuids text[] DEFAULT '{}'::text[],
    dismissal_types smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    comment text,
    status smallint DEFAULT 0 NOT NULL,
    licenses jsonb DEFAULT '{}'::jsonb NOT NULL,
    license_occurrence_uuids text[] DEFAULT '{}'::text[] NOT NULL,
    CONSTRAINT check_654ff06528 CHECK ((char_length(comment) <= 255)),
    CONSTRAINT check_88beed9dc9 CHECK ((security_findings_uuids IS NOT NULL))
);

CREATE SEQUENCE security_policy_dismissals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_policy_dismissals_id_seq OWNED BY security_policy_dismissals.id;

CREATE TABLE security_policy_project_links (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    security_policy_id bigint NOT NULL
);

CREATE SEQUENCE security_policy_project_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_policy_project_links_id_seq OWNED BY security_policy_project_links.id;

CREATE TABLE security_policy_requirements (
    id bigint NOT NULL,
    compliance_framework_security_policy_id bigint NOT NULL,
    compliance_requirement_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE security_policy_requirements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_policy_requirements_id_seq OWNED BY security_policy_requirements.id;

CREATE TABLE security_policy_settings (
    id bigint NOT NULL,
    csp_namespace_id bigint,
    organization_id bigint NOT NULL,
    csp_namespace_locked_until timestamp with time zone
);

CREATE SEQUENCE security_policy_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_policy_settings_id_seq OWNED BY security_policy_settings.id;

CREATE TABLE security_project_tracked_contexts (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    context_type smallint DEFAULT 1 NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    is_default boolean DEFAULT false NOT NULL,
    context_name text NOT NULL,
    CONSTRAINT check_032d33c1cc CHECK ((char_length(context_name) <= 1024))
);

CREATE SEQUENCE security_project_tracked_contexts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_project_tracked_contexts_id_seq OWNED BY security_project_tracked_contexts.id;

CREATE TABLE security_scan_profile_triggers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    security_scan_profile_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    trigger_type smallint NOT NULL
);

CREATE SEQUENCE security_scan_profile_triggers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_scan_profile_triggers_id_seq OWNED BY security_scan_profile_triggers.id;

CREATE TABLE security_scan_profiles (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    scan_type smallint NOT NULL,
    gitlab_recommended boolean DEFAULT false NOT NULL,
    name text NOT NULL,
    description text,
    CONSTRAINT check_58c7066495 CHECK ((char_length(description) <= 2047)),
    CONSTRAINT check_f1e9f004bb CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE security_scan_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_scan_profiles_id_seq OWNED BY security_scan_profiles.id;

CREATE TABLE security_scan_profiles_projects (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    security_scan_profile_id bigint NOT NULL
);

CREATE SEQUENCE security_scan_profiles_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_scan_profiles_projects_id_seq OWNED BY security_scan_profiles_projects.id;

CREATE TABLE security_scans (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    build_id bigint NOT NULL,
    scan_type smallint NOT NULL,
    info jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    pipeline_id bigint,
    latest boolean DEFAULT true NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    findings_partition_number integer DEFAULT 1 NOT NULL
);

CREATE SEQUENCE security_scans_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_scans_id_seq OWNED BY security_scans.id;

CREATE TABLE security_training_providers (
    id bigint NOT NULL,
    name text NOT NULL,
    description text,
    url text NOT NULL,
    logo_url text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_544b3dc935 CHECK ((char_length(url) <= 512)),
    CONSTRAINT check_6fe222f071 CHECK ((char_length(logo_url) <= 512)),
    CONSTRAINT check_a8ff21ced5 CHECK ((char_length(description) <= 512)),
    CONSTRAINT check_dae433eed6 CHECK ((char_length(name) <= 256))
);

CREATE SEQUENCE security_training_providers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_training_providers_id_seq OWNED BY security_training_providers.id;

CREATE TABLE security_trainings (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    provider_id bigint,
    is_primary boolean DEFAULT false NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    training_provider_id bigint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE security_trainings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_trainings_id_seq OWNED BY security_trainings.id;

CREATE TABLE sentry_issues (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    sentry_issue_identifier bigint NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_7c50ed861c CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE sentry_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sentry_issues_id_seq OWNED BY sentry_issues.id;

CREATE TABLE service_access_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_token bytea NOT NULL,
    encrypted_token_iv bytea NOT NULL,
    expires_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE service_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE service_access_tokens_id_seq OWNED BY service_access_tokens.id;

CREATE TABLE service_desk_custom_email_credentials (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    smtp_port integer,
    smtp_address text,
    encrypted_smtp_username bytea,
    encrypted_smtp_username_iv bytea,
    encrypted_smtp_password bytea,
    encrypted_smtp_password_iv bytea,
    smtp_authentication smallint,
    CONSTRAINT check_6dd11e956a CHECK ((char_length(smtp_address) <= 255))
);

CREATE TABLE service_desk_custom_email_verifications (
    project_id bigint NOT NULL,
    triggerer_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    triggered_at timestamp with time zone,
    state smallint DEFAULT 0 NOT NULL,
    error smallint,
    encrypted_token bytea,
    encrypted_token_iv bytea
);

CREATE TABLE service_desk_settings (
    project_id bigint NOT NULL,
    issue_template_key character varying(255),
    outgoing_name character varying(255),
    project_key character varying(255),
    file_template_project_id bigint,
    custom_email_enabled boolean DEFAULT false NOT NULL,
    custom_email text,
    service_desk_enabled boolean DEFAULT true NOT NULL,
    add_external_participants_from_cc boolean DEFAULT false NOT NULL,
    reopen_issue_on_external_participant_note boolean DEFAULT false NOT NULL,
    tickets_confidential_by_default boolean DEFAULT true NOT NULL,
    CONSTRAINT check_57a79552e1 CHECK ((char_length(custom_email) <= 255))
);

CREATE SEQUENCE shards_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE shards_id_seq OWNED BY shards.id;

CREATE TABLE slack_api_scopes (
    id bigint NOT NULL,
    name text NOT NULL,
    organization_id bigint,
    CONSTRAINT check_738678187a CHECK ((char_length(name) <= 100))
);

CREATE SEQUENCE slack_api_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE slack_api_scopes_id_seq OWNED BY slack_api_scopes.id;

CREATE TABLE slack_integrations (
    id bigint NOT NULL,
    team_id character varying NOT NULL,
    team_name character varying NOT NULL,
    alias character varying NOT NULL,
    user_id character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    bot_user_id text,
    encrypted_bot_access_token bytea,
    encrypted_bot_access_token_iv bytea,
    integration_id bigint,
    organization_id bigint,
    group_id bigint,
    project_id bigint,
    CONSTRAINT check_bc553aea8a CHECK ((char_length(bot_user_id) <= 255)),
    CONSTRAINT check_c9ca9ae80d CHECK ((integration_id IS NOT NULL))
);

CREATE SEQUENCE slack_integrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE slack_integrations_id_seq OWNED BY slack_integrations.id;

CREATE TABLE slack_integrations_scopes (
    id bigint NOT NULL,
    slack_api_scope_id bigint NOT NULL,
    slack_integration_id bigint NOT NULL,
    project_id bigint,
    group_id bigint,
    organization_id bigint
);

CREATE TABLE slack_integrations_scopes_archived (
    id bigint NOT NULL,
    slack_api_scope_id bigint NOT NULL,
    slack_integration_id bigint NOT NULL,
    project_id bigint,
    group_id bigint,
    organization_id bigint,
    archived_at timestamp with time zone DEFAULT CURRENT_TIMESTAMP NOT NULL
);

COMMENT ON TABLE slack_integrations_scopes_archived IS 'Temporary table for storing duplicate slack_integrations_scopes records during sharding key backfill. Stores duplicate/conflicting records with archival timestamp. TODO: Drop after BBM completion and verification.';

CREATE SEQUENCE slack_integrations_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE slack_integrations_scopes_id_seq OWNED BY slack_integrations_scopes.id;

CREATE TABLE slsa_attestations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    build_id bigint,
    status smallint DEFAULT 0 NOT NULL,
    expire_at timestamp with time zone,
    predicate_kind smallint DEFAULT 0 NOT NULL,
    predicate_type text NOT NULL,
    subject_digest text NOT NULL,
    file text,
    file_store smallint DEFAULT 1,
    iid integer,
    predicate_file text,
    predicate_file_store smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_3575e9121e CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_dec11b603a CHECK ((char_length(subject_digest) <= 255)),
    CONSTRAINT check_ea0d61030d CHECK ((char_length(predicate_type) <= 255)),
    CONSTRAINT check_ed09edea1d CHECK ((char_length(predicate_file) <= 1024))
);

CREATE SEQUENCE slsa_attestations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE slsa_attestations_id_seq OWNED BY slsa_attestations.id;

CREATE TABLE smartcard_identities (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    subject character varying NOT NULL,
    issuer character varying NOT NULL
);

CREATE SEQUENCE smartcard_identities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE smartcard_identities_id_seq OWNED BY smartcard_identities.id;

CREATE TABLE snippet_repository_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    snippet_repository_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_0dabaefb7f CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE snippet_repository_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippet_repository_states_id_seq OWNED BY snippet_repository_states.id;

CREATE TABLE snippet_repository_storage_moves (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    snippet_id bigint NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    source_storage_name text NOT NULL,
    destination_storage_name text NOT NULL,
    error_message text,
    snippet_project_id bigint,
    snippet_organization_id bigint,
    CONSTRAINT check_4b127f0a5d CHECK ((num_nonnulls(snippet_organization_id, snippet_project_id) = 1)),
    CONSTRAINT check_a42ab83060 CHECK ((char_length(error_message) <= 256)),
    CONSTRAINT snippet_repository_storage_moves_destination_storage_name CHECK ((char_length(destination_storage_name) <= 255)),
    CONSTRAINT snippet_repository_storage_moves_source_storage_name CHECK ((char_length(source_storage_name) <= 255))
);

CREATE SEQUENCE snippet_repository_storage_moves_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippet_repository_storage_moves_id_seq OWNED BY snippet_repository_storage_moves.id;

CREATE TABLE snippet_statistics (
    snippet_id bigint NOT NULL,
    repository_size bigint DEFAULT 0 NOT NULL,
    file_count bigint DEFAULT 0 NOT NULL,
    commit_count bigint DEFAULT 0 NOT NULL,
    snippet_project_id bigint,
    snippet_organization_id bigint,
    CONSTRAINT check_4240d2eb99 CHECK ((num_nonnulls(snippet_organization_id, snippet_project_id) = 1))
);

CREATE TABLE snippet_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64)),
    CONSTRAINT check_f1b1b6e769 CHECK ((organization_id IS NOT NULL))
);

CREATE TABLE snippet_user_mentions (
    id bigint NOT NULL,
    snippet_id bigint NOT NULL,
    mentioned_users_ids bigint[],
    mentioned_projects_ids bigint[],
    mentioned_groups_ids bigint[],
    note_id bigint,
    snippet_project_id bigint,
    snippet_organization_id bigint,
    CONSTRAINT check_25b8666c20 CHECK ((num_nonnulls(snippet_organization_id, snippet_project_id) = 1))
);

CREATE SEQUENCE snippet_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippet_user_mentions_id_seq OWNED BY snippet_user_mentions.id;

CREATE SEQUENCE snippets_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippets_id_seq OWNED BY snippets.id;

CREATE TABLE software_license_policies (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    classification integer DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    scan_result_policy_id bigint,
    custom_software_license_id bigint,
    approval_policy_rule_id bigint,
    software_license_spdx_identifier text,
    CONSTRAINT check_6cb3facbb3 CHECK ((num_nonnulls(custom_software_license_id, software_license_spdx_identifier) = 1)),
    CONSTRAINT check_986c4e5c59 CHECK ((char_length(software_license_spdx_identifier) <= 255))
);

CREATE SEQUENCE software_license_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE software_license_policies_id_seq OWNED BY software_license_policies.id;

CREATE TABLE spam_logs (
    id bigint NOT NULL,
    user_id bigint,
    source_ip character varying,
    user_agent character varying,
    via_api boolean,
    noteable_type character varying,
    title character varying,
    description text,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    submitted_as_ham boolean DEFAULT false NOT NULL,
    recaptcha_verified boolean DEFAULT false NOT NULL,
    target_id bigint,
    organization_id bigint,
    CONSTRAINT check_56d1d910ee CHECK ((user_id IS NOT NULL))
);

CREATE SEQUENCE spam_logs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE spam_logs_id_seq OWNED BY spam_logs.id;

CREATE TABLE sprints (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    due_date date,
    group_id bigint,
    iid integer NOT NULL,
    cached_markdown_version integer,
    title text,
    title_html text,
    description text,
    description_html text,
    state_enum smallint DEFAULT 1 NOT NULL,
    iterations_cadence_id bigint,
    sequence integer,
    CONSTRAINT check_73910a3b6c CHECK ((group_id IS NOT NULL)),
    CONSTRAINT sprints_title CHECK ((char_length(title) <= 255))
);

CREATE SEQUENCE sprints_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sprints_id_seq OWNED BY sprints.id;

CREATE TABLE ssh_signatures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    key_id bigint,
    verification_status smallint DEFAULT 0 NOT NULL,
    commit_sha bytea NOT NULL,
    user_id bigint,
    key_fingerprint_sha256 bytea,
    committer_email text,
    CONSTRAINT check_73776e38f9 CHECK ((char_length(committer_email) <= 255))
);

CREATE SEQUENCE ssh_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ssh_signatures_id_seq OWNED BY ssh_signatures.id;

CREATE TABLE status_check_responses (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    external_approval_rule_id bigint,
    sha bytea NOT NULL,
    external_status_check_id bigint NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    retried_at timestamp with time zone,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    project_id bigint,
    CONSTRAINT check_29114cce9c CHECK ((project_id IS NOT NULL))
);

CREATE SEQUENCE status_check_responses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE status_check_responses_id_seq OWNED BY status_check_responses.id;

CREATE TABLE status_page_published_incidents (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_id bigint NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_8ab6fb2f34 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE status_page_published_incidents_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE status_page_published_incidents_id_seq OWNED BY status_page_published_incidents.id;

CREATE TABLE status_page_settings (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    aws_s3_bucket_name character varying(63) NOT NULL,
    aws_region character varying(255) NOT NULL,
    aws_access_key character varying(255) NOT NULL,
    encrypted_aws_secret_key character varying(255) NOT NULL,
    encrypted_aws_secret_key_iv character varying(255) NOT NULL,
    status_page_url text,
    CONSTRAINT check_75a79cd992 CHECK ((char_length(status_page_url) <= 1024))
);

CREATE SEQUENCE status_page_settings_project_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE status_page_settings_project_id_seq OWNED BY status_page_settings.project_id;

CREATE TABLE subscription_add_on_purchases (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    subscription_add_on_id bigint NOT NULL,
    namespace_id bigint,
    quantity integer NOT NULL,
    expires_on date NOT NULL,
    purchase_xid text NOT NULL,
    last_assigned_users_refreshed_at timestamp with time zone,
    trial boolean DEFAULT false NOT NULL,
    started_at date,
    organization_id bigint NOT NULL,
    CONSTRAINT check_3313c4d200 CHECK ((char_length(purchase_xid) <= 255)),
    CONSTRAINT check_d79ce199b3 CHECK ((started_at IS NOT NULL))
);

CREATE SEQUENCE subscription_add_on_purchases_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscription_add_on_purchases_id_seq OWNED BY subscription_add_on_purchases.id;

CREATE TABLE subscription_add_ons (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name smallint NOT NULL,
    description text NOT NULL,
    CONSTRAINT check_4c39d15ada CHECK ((char_length(description) <= 512))
);

CREATE SEQUENCE subscription_add_ons_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscription_add_ons_id_seq OWNED BY subscription_add_ons.id;

CREATE TABLE subscription_seat_assignments (
    id bigint NOT NULL,
    namespace_id bigint,
    user_id bigint NOT NULL,
    last_activity_on timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint NOT NULL,
    seat_type smallint
);

CREATE SEQUENCE subscription_seat_assignments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscription_seat_assignments_id_seq OWNED BY subscription_seat_assignments.id;

CREATE TABLE subscription_user_add_on_assignment_versions (
    id bigint NOT NULL,
    organization_id bigint NOT NULL,
    item_id bigint,
    purchase_id bigint,
    user_id bigint,
    created_at timestamp with time zone,
    item_type text NOT NULL,
    event text NOT NULL,
    namespace_path text,
    add_on_name text,
    whodunnit text,
    object jsonb,
    CONSTRAINT check_211bad6d65 CHECK ((char_length(item_type) <= 255)),
    CONSTRAINT check_34ca72be24 CHECK ((char_length(event) <= 255)),
    CONSTRAINT check_839913a25d CHECK ((char_length(namespace_path) <= 255)),
    CONSTRAINT check_9ceaa5668c CHECK ((char_length(add_on_name) <= 255)),
    CONSTRAINT check_e185bf0c82 CHECK ((char_length(whodunnit) <= 255))
);

CREATE SEQUENCE subscription_user_add_on_assignment_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscription_user_add_on_assignment_versions_id_seq OWNED BY subscription_user_add_on_assignment_versions.id;

CREATE TABLE subscription_user_add_on_assignments (
    id bigint NOT NULL,
    add_on_purchase_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint,
    CONSTRAINT check_7d21f9cebf CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE subscription_user_add_on_assignments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscription_user_add_on_assignments_id_seq OWNED BY subscription_user_add_on_assignments.id;

CREATE TABLE subscriptions (
    id bigint NOT NULL,
    user_id bigint,
    subscribable_id bigint,
    subscribable_type character varying,
    subscribed boolean,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id bigint,
    CONSTRAINT check_285574a00a CHECK ((user_id IS NOT NULL))
);

CREATE SEQUENCE subscriptions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscriptions_id_seq OWNED BY subscriptions.id;

CREATE TABLE suggestions (
    id bigint NOT NULL,
    relative_order smallint NOT NULL,
    applied boolean DEFAULT false NOT NULL,
    commit_id character varying,
    from_content text NOT NULL,
    to_content text NOT NULL,
    lines_above integer DEFAULT 0 NOT NULL,
    lines_below integer DEFAULT 0 NOT NULL,
    outdated boolean DEFAULT false NOT NULL,
    note_id bigint NOT NULL,
    namespace_id bigint
);

CREATE SEQUENCE suggestions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE suggestions_id_seq OWNED BY suggestions.id;

CREATE TABLE supply_chain_attestation_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    supply_chain_attestation_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_ef9d3c1760 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE supply_chain_attestation_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE supply_chain_attestation_states_id_seq OWNED BY supply_chain_attestation_states.id;

CREATE TABLE system_access_group_microsoft_applications (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    tenant_xid text NOT NULL,
    client_xid text NOT NULL,
    login_endpoint text DEFAULT 'https://login.microsoftonline.com'::text NOT NULL,
    graph_endpoint text DEFAULT 'https://graph.microsoft.com'::text NOT NULL,
    encrypted_client_secret bytea NOT NULL,
    encrypted_client_secret_iv bytea NOT NULL,
    CONSTRAINT check_027535e932 CHECK ((char_length(graph_endpoint) <= 255)),
    CONSTRAINT check_350406e92e CHECK ((char_length(login_endpoint) <= 255)),
    CONSTRAINT check_92ce93bc07 CHECK ((char_length(tenant_xid) <= 255)),
    CONSTRAINT check_f4c8cf8195 CHECK ((char_length(client_xid) <= 255))
);

CREATE SEQUENCE system_access_group_microsoft_applications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE system_access_group_microsoft_applications_id_seq OWNED BY system_access_group_microsoft_applications.id;

CREATE TABLE system_access_group_microsoft_graph_access_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    system_access_group_microsoft_application_id bigint,
    group_id bigint NOT NULL,
    expires_in integer NOT NULL,
    encrypted_token bytea NOT NULL,
    encrypted_token_iv bytea NOT NULL
);

CREATE SEQUENCE system_access_group_microsoft_graph_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE system_access_group_microsoft_graph_access_tokens_id_seq OWNED BY system_access_group_microsoft_graph_access_tokens.id;

CREATE TABLE system_access_microsoft_applications (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint,
    enabled boolean DEFAULT false NOT NULL,
    tenant_xid text NOT NULL,
    client_xid text NOT NULL,
    login_endpoint text DEFAULT 'https://login.microsoftonline.com'::text NOT NULL,
    graph_endpoint text DEFAULT 'https://graph.microsoft.com'::text NOT NULL,
    encrypted_client_secret bytea NOT NULL,
    encrypted_client_secret_iv bytea NOT NULL,
    CONSTRAINT check_042f6b21aa CHECK ((char_length(login_endpoint) <= 255)),
    CONSTRAINT check_1e8b2d405f CHECK ((char_length(tenant_xid) <= 255)),
    CONSTRAINT check_339c3ffca8 CHECK ((char_length(graph_endpoint) <= 255)),
    CONSTRAINT check_ee72fb5459 CHECK ((char_length(client_xid) <= 255))
);

CREATE SEQUENCE system_access_microsoft_applications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE system_access_microsoft_applications_id_seq OWNED BY system_access_microsoft_applications.id;

CREATE TABLE system_access_microsoft_graph_access_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    system_access_microsoft_application_id bigint,
    expires_in integer NOT NULL,
    encrypted_token bytea NOT NULL,
    encrypted_token_iv bytea NOT NULL
);

CREATE SEQUENCE system_access_microsoft_graph_access_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE system_access_microsoft_graph_access_tokens_id_seq OWNED BY system_access_microsoft_graph_access_tokens.id;

CREATE TABLE system_note_metadata (
    commit_count integer,
    action character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    description_version_id bigint,
    note_id bigint NOT NULL,
    id bigint NOT NULL,
    namespace_id bigint
);

CREATE SEQUENCE system_note_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE system_note_metadata_id_seq OWNED BY system_note_metadata.id;

CREATE TABLE tag_gpg_signatures (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    gpg_key_id bigint,
    gpg_key_subkey_id bigint,
    verification_status smallint DEFAULT 0 NOT NULL,
    object_name bytea NOT NULL,
    gpg_key_primary_keyid bytea NOT NULL,
    gpg_key_user_name text,
    gpg_key_user_email text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_ade3afeeaf CHECK ((char_length(gpg_key_user_name) <= 255)),
    CONSTRAINT check_afd515f6c7 CHECK ((char_length(gpg_key_user_email) <= 255))
);

CREATE SEQUENCE tag_gpg_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE tag_gpg_signatures_id_seq OWNED BY tag_gpg_signatures.id;

CREATE TABLE tag_ssh_signatures (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    key_id bigint,
    verification_status smallint DEFAULT 0 NOT NULL,
    object_name bytea NOT NULL,
    key_fingerprint_sha256 bytea,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE tag_ssh_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE tag_ssh_signatures_id_seq OWNED BY tag_ssh_signatures.id;

CREATE TABLE tag_x509_signatures (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    x509_certificate_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    verification_status smallint DEFAULT 0 NOT NULL,
    object_name bytea NOT NULL
);

CREATE SEQUENCE tag_x509_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE tag_x509_signatures_id_seq OWNED BY tag_x509_signatures.id;

CREATE TABLE tags (
    id bigint NOT NULL,
    name character varying,
    taggings_count integer DEFAULT 0
);

CREATE SEQUENCE tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE tags_id_seq OWNED BY tags.id;

CREATE TABLE target_branch_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    target_branch text NOT NULL,
    CONSTRAINT check_3a0b12cf8c CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE target_branch_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE target_branch_rules_id_seq OWNED BY target_branch_rules.id;

CREATE TABLE targeted_message_dismissals (
    id bigint NOT NULL,
    targeted_message_id bigint NOT NULL,
    user_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE targeted_message_dismissals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE targeted_message_dismissals_id_seq OWNED BY targeted_message_dismissals.id;

CREATE TABLE targeted_message_namespaces (
    id bigint NOT NULL,
    targeted_message_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE targeted_message_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE targeted_message_namespaces_id_seq OWNED BY targeted_message_namespaces.id;

CREATE TABLE targeted_messages (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    target_type smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE targeted_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE targeted_messages_id_seq OWNED BY targeted_messages.id;

CREATE TABLE term_agreements (
    id bigint NOT NULL,
    term_id bigint NOT NULL,
    user_id bigint NOT NULL,
    accepted boolean DEFAULT false NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE term_agreements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE term_agreements_id_seq OWNED BY term_agreements.id;

CREATE TABLE terraform_state_version_states (
    id bigint NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    terraform_state_version_id bigint NOT NULL,
    project_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint DEFAULT 0 NOT NULL,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_dc0c9c9162 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE terraform_state_version_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE terraform_state_version_states_id_seq OWNED BY terraform_state_version_states.id;

CREATE TABLE terraform_state_versions (
    id bigint NOT NULL,
    terraform_state_id bigint NOT NULL,
    created_by_user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    version integer NOT NULL,
    file_store smallint NOT NULL,
    file text NOT NULL,
    verification_retry_count smallint,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_checksum bytea,
    verification_failure text,
    ci_build_id bigint,
    verification_started_at timestamp with time zone,
    verification_state smallint DEFAULT 0 NOT NULL,
    project_id bigint,
    is_encrypted boolean DEFAULT true NOT NULL,
    CONSTRAINT check_0824bb7bbd CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_84142902f6 CHECK ((project_id IS NOT NULL)),
    CONSTRAINT tf_state_versions_verification_failure_text_limit CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE terraform_state_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE terraform_state_versions_id_seq OWNED BY terraform_state_versions.id;

CREATE TABLE terraform_states (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint,
    file character varying(255),
    lock_xid character varying(255),
    locked_at timestamp with time zone,
    locked_by_user_id bigint,
    uuid character varying(32) NOT NULL,
    name character varying(255) NOT NULL,
    versioning_enabled boolean DEFAULT true NOT NULL,
    deleted_at timestamp with time zone,
    activerecord_lock_version integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE terraform_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE terraform_states_id_seq OWNED BY terraform_states.id;

CREATE TABLE timelog_categories (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    billing_rate numeric(18,4) DEFAULT 0.0,
    billable boolean DEFAULT false NOT NULL,
    name text NOT NULL,
    description text,
    color text DEFAULT '#6699cc'::text NOT NULL,
    CONSTRAINT check_37ad5f23d7 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_4ba862ba3e CHECK ((char_length(color) <= 7)),
    CONSTRAINT check_c4b8aec13a CHECK ((char_length(description) <= 1024))
);

CREATE SEQUENCE timelog_categories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE timelog_categories_id_seq OWNED BY timelog_categories.id;

CREATE TABLE timelogs (
    id bigint NOT NULL,
    time_spent integer NOT NULL,
    user_id bigint,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    issue_id bigint,
    merge_request_id bigint,
    spent_at timestamp without time zone DEFAULT now(),
    project_id bigint,
    summary text,
    note_id bigint,
    timelog_category_id bigint,
    namespace_id bigint NOT NULL,
    CONSTRAINT check_271d321699 CHECK ((char_length(summary) <= 255)),
    CONSTRAINT check_3b0344e3b6 CHECK ((num_nonnulls(issue_id, merge_request_id) = 1))
);

CREATE SEQUENCE timelogs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE timelogs_id_seq OWNED BY timelogs.id;

CREATE TABLE todos (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint,
    target_id bigint,
    target_type character varying NOT NULL,
    author_id bigint NOT NULL,
    action integer NOT NULL,
    state character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    commit_id character varying,
    group_id bigint,
    resolved_by_action smallint,
    note_id bigint,
    snoozed_until timestamp with time zone,
    organization_id bigint,
    CONSTRAINT check_3c13ed1c7a CHECK ((num_nonnulls(group_id, organization_id, project_id) = 1))
);

CREATE SEQUENCE todos_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE todos_id_seq OWNED BY todos.id;

CREATE TABLE topics (
    id bigint NOT NULL,
    name text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    avatar text,
    description text,
    total_projects_count bigint DEFAULT 0 NOT NULL,
    non_private_projects_count bigint DEFAULT 0 NOT NULL,
    title text,
    slug text,
    organization_id bigint NOT NULL,
    cached_markdown_version integer DEFAULT 0 NOT NULL,
    description_html text,
    CONSTRAINT check_0eda72aeb0 CHECK ((char_length(slug) <= 255)),
    CONSTRAINT check_223b50f9be CHECK ((char_length(title) <= 255)),
    CONSTRAINT check_26753fb43a CHECK ((char_length(avatar) <= 255)),
    CONSTRAINT check_5d1a07c8c8 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_7a90d4c757 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_7c7a7b2c84 CHECK ((char_length(description_html) <= 50000))
);

CREATE SEQUENCE topics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE topics_id_seq OWNED BY topics.id;

CREATE TABLE trending_projects (
    id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE SEQUENCE trending_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE trending_projects_id_seq OWNED BY trending_projects.id;

CREATE TABLE upcoming_reconciliations (
    id bigint NOT NULL,
    namespace_id bigint,
    next_reconciliation_date date NOT NULL,
    display_alert_from date NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    organization_id bigint NOT NULL
);

CREATE SEQUENCE upcoming_reconciliations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE upcoming_reconciliations_id_seq OWNED BY upcoming_reconciliations.id;

CREATE TABLE upload_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    upload_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_7396dc8591 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE upload_states_upload_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE upload_states_upload_id_seq OWNED BY upload_states.upload_id;

CREATE TABLE uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    path character varying(511) NOT NULL,
    checksum character varying(64),
    model_id bigint,
    model_type character varying,
    uploader character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    store integer DEFAULT 1,
    mount_point character varying,
    secret character varying,
    version integer DEFAULT 1 NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    CONSTRAINT check_5e9547379c CHECK ((store IS NOT NULL))
);

CREATE SEQUENCE uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE uploads_id_seq OWNED BY uploads.id;

CREATE TABLE user_achievements (
    id bigint NOT NULL,
    achievement_id bigint NOT NULL,
    user_id bigint NOT NULL,
    awarded_by_user_id bigint,
    revoked_by_user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    revoked_at timestamp with time zone,
    priority integer,
    namespace_id bigint,
    show_on_profile boolean DEFAULT true NOT NULL,
    CONSTRAINT check_2236a10887 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE user_achievements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_achievements_id_seq OWNED BY user_achievements.id;

CREATE TABLE user_admin_roles (
    user_id bigint NOT NULL,
    admin_role_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    ldap boolean DEFAULT false NOT NULL
);

CREATE TABLE user_agent_details (
    id bigint NOT NULL,
    user_agent character varying NOT NULL,
    ip_address character varying NOT NULL,
    subject_id bigint NOT NULL,
    subject_type character varying NOT NULL,
    submitted boolean DEFAULT false NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    organization_id bigint
);

CREATE SEQUENCE user_agent_details_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_agent_details_id_seq OWNED BY user_agent_details.id;

CREATE TABLE user_broadcast_message_dismissals (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    broadcast_message_id bigint NOT NULL,
    expires_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE user_broadcast_message_dismissals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_broadcast_message_dismissals_id_seq OWNED BY user_broadcast_message_dismissals.id;

CREATE TABLE user_callouts (
    id bigint NOT NULL,
    feature_name integer NOT NULL,
    user_id bigint NOT NULL,
    dismissed_at timestamp with time zone
);

CREATE SEQUENCE user_callouts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_callouts_id_seq OWNED BY user_callouts.id;

CREATE TABLE user_credit_card_validations (
    user_id bigint NOT NULL,
    credit_card_validated_at timestamp with time zone NOT NULL,
    last_digits_hash text,
    holder_name_hash text,
    expiration_date_hash text,
    network_hash text,
    zuora_payment_method_xid text,
    stripe_setup_intent_xid text,
    stripe_payment_method_xid text,
    stripe_card_fingerprint text,
    CONSTRAINT check_126615a57d CHECK ((char_length(stripe_payment_method_xid) <= 255)),
    CONSTRAINT check_209503e313 CHECK ((char_length(stripe_card_fingerprint) <= 255)),
    CONSTRAINT check_5d9e69ede5 CHECK ((char_length(stripe_setup_intent_xid) <= 255)),
    CONSTRAINT check_7721e1961a CHECK ((char_length(network_hash) <= 44)),
    CONSTRAINT check_83f1e2ace3 CHECK ((char_length(expiration_date_hash) <= 44)),
    CONSTRAINT check_9a15d14e37 CHECK ((char_length(zuora_payment_method_xid) <= 50)),
    CONSTRAINT check_aca7c2607c CHECK ((char_length(holder_name_hash) <= 44)),
    CONSTRAINT check_f5c35b1a6e CHECK ((char_length(last_digits_hash) <= 44))
);

CREATE TABLE user_custom_attributes (
    id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    user_id bigint NOT NULL,
    key character varying NOT NULL,
    value character varying NOT NULL
);

CREATE SEQUENCE user_custom_attributes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_custom_attributes_id_seq OWNED BY user_custom_attributes.id;

CREATE TABLE user_details (
    user_id bigint NOT NULL,
    job_title character varying(200) DEFAULT ''::character varying NOT NULL,
    bio character varying(255) DEFAULT ''::character varying NOT NULL,
    webauthn_xid text,
    provisioned_by_group_id bigint,
    pronouns text,
    pronunciation text,
    phone text,
    linkedin text DEFAULT ''::text NOT NULL,
    twitter text DEFAULT ''::text NOT NULL,
    skype text DEFAULT ''::text NOT NULL,
    website_url text DEFAULT ''::text NOT NULL,
    location text DEFAULT ''::text NOT NULL,
    organization text DEFAULT ''::text NOT NULL,
    password_last_changed_at timestamp with time zone DEFAULT now() NOT NULL,
    discord text DEFAULT ''::text NOT NULL,
    enterprise_group_id bigint,
    enterprise_group_associated_at timestamp with time zone,
    email_reset_offered_at timestamp with time zone,
    mastodon text DEFAULT ''::text NOT NULL,
    project_authorizations_recalculated_at timestamp with time zone DEFAULT '2010-01-01 00:00:00+00'::timestamp with time zone NOT NULL,
    onboarding_status jsonb DEFAULT '{}'::jsonb NOT NULL,
    bluesky text DEFAULT ''::text NOT NULL,
    bot_namespace_id bigint,
    orcid text DEFAULT ''::text NOT NULL,
    github text DEFAULT ''::text NOT NULL,
    email_otp text,
    email_otp_last_sent_to text,
    email_otp_last_sent_at timestamp with time zone,
    email_otp_required_after timestamp with time zone,
    company text DEFAULT ''::text NOT NULL,
    CONSTRAINT check_18a53381cd CHECK ((char_length(bluesky) <= 256)),
    CONSTRAINT check_245664af82 CHECK ((char_length(webauthn_xid) <= 100)),
    CONSTRAINT check_444573ee52 CHECK ((char_length(skype) <= 500)),
    CONSTRAINT check_466a25be35 CHECK ((char_length(twitter) <= 500)),
    CONSTRAINT check_4925cf9fd2 CHECK ((char_length(email_otp_last_sent_to) <= 511)),
    CONSTRAINT check_4ef1de1a15 CHECK ((char_length(discord) <= 500)),
    CONSTRAINT check_7b246dad73 CHECK ((char_length(organization) <= 500)),
    CONSTRAINT check_7d6489f8f3 CHECK ((char_length(linkedin) <= 500)),
    CONSTRAINT check_7fe2044093 CHECK ((char_length(website_url) <= 500)),
    CONSTRAINT check_8a7fcf8a60 CHECK ((char_length(location) <= 500)),
    CONSTRAINT check_99b0365865 CHECK ((char_length(orcid) <= 256)),
    CONSTRAINT check_a73b398c60 CHECK ((char_length(phone) <= 50)),
    CONSTRAINT check_bbe110f371 CHECK ((char_length(github) <= 500)),
    CONSTRAINT check_ec514a06ad CHECK ((char_length(email_otp) <= 64)),
    CONSTRAINT check_eeeaf8d4f0 CHECK ((char_length(pronouns) <= 50)),
    CONSTRAINT check_f1a8a05b9a CHECK ((char_length(mastodon) <= 500)),
    CONSTRAINT check_f932ed37db CHECK ((char_length(pronunciation) <= 255))
);

COMMENT ON COLUMN user_details.phone IS 'JiHu-specific column';

COMMENT ON COLUMN user_details.password_last_changed_at IS 'JiHu-specific column';

COMMENT ON COLUMN user_details.email_otp IS 'SHA256 hash (64 hex characters)';

CREATE SEQUENCE user_details_user_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_details_user_id_seq OWNED BY user_details.user_id;

CREATE TABLE user_follow_users (
    follower_id bigint NOT NULL,
    followee_id bigint NOT NULL
);

CREATE TABLE user_group_callouts (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    group_id bigint NOT NULL,
    feature_name smallint NOT NULL,
    dismissed_at timestamp with time zone
);

CREATE SEQUENCE user_group_callouts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_group_callouts_id_seq OWNED BY user_group_callouts.id;

CREATE TABLE user_group_member_roles (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    group_id bigint NOT NULL,
    shared_with_group_id bigint,
    member_role_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE user_group_member_roles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_group_member_roles_id_seq OWNED BY user_group_member_roles.id;

CREATE TABLE user_highest_roles (
    user_id bigint NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    highest_access_level integer
);

CREATE TABLE user_member_roles (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    member_role_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    ldap boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE user_member_roles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_member_roles_id_seq OWNED BY user_member_roles.id;

CREATE TABLE user_namespace_callouts (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    dismissed_at timestamp with time zone,
    feature_name smallint NOT NULL
);

CREATE SEQUENCE user_namespace_callouts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_namespace_callouts_id_seq OWNED BY user_namespace_callouts.id;

CREATE TABLE user_permission_export_upload_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_824c448850 CHECK ((uploaded_by_user_id IS NOT NULL)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64))
);

CREATE TABLE user_permission_export_uploads (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    file_store integer,
    status smallint DEFAULT 0 NOT NULL,
    file text,
    CONSTRAINT check_1956806648 CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE user_permission_export_uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_permission_export_uploads_id_seq OWNED BY user_permission_export_uploads.id;

CREATE TABLE user_phone_number_validations (
    user_id bigint NOT NULL,
    validated_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    international_dial_code smallint NOT NULL,
    verification_attempts smallint DEFAULT 0 NOT NULL,
    risk_score smallint DEFAULT 0 NOT NULL,
    country text NOT NULL,
    phone_number text NOT NULL,
    telesign_reference_xid text,
    sms_sent_at timestamp with time zone,
    sms_send_count smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_193736da9f CHECK ((char_length(country) <= 3)),
    CONSTRAINT check_d2f31fc815 CHECK ((char_length(phone_number) <= 12)),
    CONSTRAINT check_d7af4d3eb5 CHECK ((char_length(telesign_reference_xid) <= 255))
);

CREATE TABLE user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    issue_notes_filter smallint DEFAULT 0 NOT NULL,
    merge_request_notes_filter smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    epics_sort character varying,
    roadmap_epics_state integer,
    epic_notes_filter smallint DEFAULT 0 NOT NULL,
    issues_sort character varying,
    merge_requests_sort character varying,
    roadmaps_sort character varying,
    first_day_of_week integer,
    timezone character varying,
    time_display_relative boolean DEFAULT true,
    projects_sort character varying(64),
    show_whitespace_in_diffs boolean DEFAULT true NOT NULL,
    sourcegraph_enabled boolean,
    setup_for_company boolean,
    render_whitespace_in_code boolean DEFAULT false,
    tab_width smallint DEFAULT 8,
    view_diffs_file_by_file boolean DEFAULT false NOT NULL,
    gitpod_enabled boolean DEFAULT false NOT NULL,
    markdown_surround_selection boolean DEFAULT true NOT NULL,
    diffs_deletion_color text,
    diffs_addition_color text,
    markdown_automatic_lists boolean DEFAULT true NOT NULL,
    use_new_navigation boolean,
    achievements_enabled boolean DEFAULT true NOT NULL,
    pinned_nav_items jsonb DEFAULT '{}'::jsonb NOT NULL,
    pass_user_identities_to_ci_jwt boolean DEFAULT false NOT NULL,
    enabled_following boolean DEFAULT true NOT NULL,
    visibility_pipeline_id_type smallint DEFAULT 0 NOT NULL,
    project_shortcut_buttons boolean DEFAULT true NOT NULL,
    enabled_zoekt boolean DEFAULT true NOT NULL,
    keyboard_shortcuts_enabled boolean DEFAULT true NOT NULL,
    time_display_format smallint DEFAULT 0 NOT NULL,
    early_access_program_participant boolean DEFAULT false NOT NULL,
    early_access_program_tracking boolean DEFAULT false NOT NULL,
    extensions_marketplace_opt_in_status smallint DEFAULT 0 NOT NULL,
    organization_groups_projects_sort text,
    organization_groups_projects_display smallint DEFAULT 1 NOT NULL,
    dpop_enabled boolean DEFAULT false NOT NULL,
    use_work_items_view boolean DEFAULT false NOT NULL,
    text_editor_type smallint DEFAULT 2 NOT NULL,
    merge_request_dashboard_list_type smallint DEFAULT 0 NOT NULL,
    extensions_marketplace_opt_in_url text,
    dark_color_scheme_id smallint,
    work_items_display_settings jsonb DEFAULT '{}'::jsonb NOT NULL,
    default_duo_add_on_assignment_id bigint,
    markdown_maintain_indentation boolean DEFAULT false NOT NULL,
    project_studio_enabled boolean DEFAULT false NOT NULL,
    merge_request_dashboard_show_drafts boolean DEFAULT true NOT NULL,
    duo_default_namespace_id bigint,
    policy_advanced_editor boolean DEFAULT false NOT NULL,
    early_access_studio_participant boolean DEFAULT false NOT NULL,
    new_ui_enabled boolean,
    CONSTRAINT check_1d670edc68 CHECK ((time_display_relative IS NOT NULL)),
    CONSTRAINT check_89bf269f41 CHECK ((char_length(diffs_deletion_color) <= 7)),
    CONSTRAINT check_9b50d9f942 CHECK ((char_length(extensions_marketplace_opt_in_url) <= 512)),
    CONSTRAINT check_b1306f8875 CHECK ((char_length(organization_groups_projects_sort) <= 64)),
    CONSTRAINT check_b22446f91a CHECK ((render_whitespace_in_code IS NOT NULL)),
    CONSTRAINT check_d07ccd35f7 CHECK ((char_length(diffs_addition_color) <= 7)),
    CONSTRAINT check_d3248b1b9c CHECK ((tab_width IS NOT NULL))
);

CREATE SEQUENCE user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_preferences_id_seq OWNED BY user_preferences.id;

CREATE TABLE user_project_callouts (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint NOT NULL,
    feature_name smallint NOT NULL,
    dismissed_at timestamp with time zone
);

CREATE SEQUENCE user_project_callouts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_project_callouts_id_seq OWNED BY user_project_callouts.id;

CREATE TABLE user_project_member_roles (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint NOT NULL,
    shared_with_group_id bigint,
    member_role_id bigint NOT NULL
);

CREATE SEQUENCE user_project_member_roles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_project_member_roles_id_seq OWNED BY user_project_member_roles.id;

CREATE TABLE user_saved_views (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    saved_view_id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    relative_position integer
);

CREATE SEQUENCE user_saved_views_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_saved_views_id_seq OWNED BY user_saved_views.id;

CREATE TABLE user_statuses (
    user_id bigint NOT NULL,
    cached_markdown_version integer,
    emoji character varying DEFAULT 'speech_balloon'::character varying NOT NULL,
    message character varying(100),
    message_html character varying,
    availability smallint DEFAULT 0 NOT NULL,
    clear_status_at timestamp with time zone
);

CREATE TABLE user_synced_attributes_metadata (
    id bigint NOT NULL,
    name_synced boolean DEFAULT false,
    email_synced boolean DEFAULT false,
    location_synced boolean DEFAULT false,
    user_id bigint NOT NULL,
    provider character varying,
    organization_synced boolean DEFAULT false,
    job_title_synced boolean DEFAULT false
);

CREATE SEQUENCE user_synced_attributes_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_synced_attributes_metadata_id_seq OWNED BY user_synced_attributes_metadata.id;

CREATE TABLE user_uploads (
    id bigint NOT NULL,
    size bigint NOT NULL,
    model_id bigint NOT NULL,
    uploaded_by_user_id bigint,
    organization_id bigint,
    namespace_id bigint,
    project_id bigint,
    created_at timestamp without time zone,
    store integer DEFAULT 1 NOT NULL,
    version integer DEFAULT 1,
    path text NOT NULL,
    checksum text,
    model_type text NOT NULL,
    uploader text NOT NULL,
    mount_point text,
    secret text,
    CONSTRAINT check_2849dedce7 CHECK ((char_length(path) <= 511)),
    CONSTRAINT check_b888b1df14 CHECK ((char_length(checksum) <= 64)),
    CONSTRAINT check_fa16e4c6ff CHECK ((organization_id IS NOT NULL))
);

CREATE SEQUENCE users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_id_seq OWNED BY users.id;

CREATE TABLE users_ops_dashboard_projects (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE SEQUENCE users_ops_dashboard_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_ops_dashboard_projects_id_seq OWNED BY users_ops_dashboard_projects.id;

CREATE TABLE users_security_dashboard_projects (
    user_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE TABLE users_star_projects (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE users_star_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_star_projects_id_seq OWNED BY users_star_projects.id;

CREATE TABLE users_statistics (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    without_groups_and_projects integer DEFAULT 0 NOT NULL,
    with_highest_role_guest integer DEFAULT 0 NOT NULL,
    with_highest_role_reporter integer DEFAULT 0 NOT NULL,
    with_highest_role_developer integer DEFAULT 0 NOT NULL,
    with_highest_role_maintainer integer DEFAULT 0 NOT NULL,
    with_highest_role_owner integer DEFAULT 0 NOT NULL,
    bots integer DEFAULT 0 NOT NULL,
    blocked integer DEFAULT 0 NOT NULL,
    with_highest_role_minimal_access integer DEFAULT 0 NOT NULL,
    with_highest_role_guest_with_custom_role integer DEFAULT 0 NOT NULL,
    with_highest_role_planner integer DEFAULT 0 NOT NULL,
    with_highest_role_security_manager integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE users_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_statistics_id_seq OWNED BY users_statistics.id;

CREATE TABLE value_stream_dashboard_aggregations (
    namespace_id bigint NOT NULL,
    last_run_at timestamp with time zone,
    enabled boolean DEFAULT true NOT NULL
);

CREATE SEQUENCE value_stream_dashboard_counts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE value_stream_dashboard_counts_id_seq OWNED BY value_stream_dashboard_counts.id;

CREATE TABLE virtual_registries_cleanup_policies (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    next_run_at timestamp with time zone,
    last_run_at timestamp with time zone,
    last_run_deleted_size bigint DEFAULT 0,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    keep_n_days_after_download integer DEFAULT 30 NOT NULL,
    last_run_deleted_entries_count integer DEFAULT 0,
    status smallint DEFAULT 0 NOT NULL,
    cadence smallint DEFAULT 7 NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    notify_on_success boolean DEFAULT false NOT NULL,
    notify_on_failure boolean DEFAULT false NOT NULL,
    failure_message text,
    last_run_detailed_metrics jsonb DEFAULT '{}'::jsonb,
    CONSTRAINT check_a29ff8b379 CHECK ((char_length(failure_message) <= 255)),
    CONSTRAINT chk_rails_8c9cfeda41 CHECK ((last_run_deleted_size >= 0)),
    CONSTRAINT chk_rails_d543512016 CHECK ((cadence = ANY (ARRAY[1, 7, 14, 30, 90]))),
    CONSTRAINT chk_rails_f01e0249cb CHECK ((keep_n_days_after_download > 0)),
    CONSTRAINT chk_rails_f0b5a893ae CHECK ((last_run_deleted_entries_count >= 0))
);

CREATE SEQUENCE virtual_registries_cleanup_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE virtual_registries_cleanup_policies_id_seq OWNED BY virtual_registries_cleanup_policies.id;

CREATE TABLE virtual_registries_container_registries (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    description text,
    CONSTRAINT check_68bb0189e2 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_e721e79322 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE virtual_registries_container_registries_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE virtual_registries_container_registries_id_seq OWNED BY virtual_registries_container_registries.id;

CREATE TABLE virtual_registries_container_registry_upstreams (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    registry_id bigint NOT NULL,
    upstream_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    "position" smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_918a83b32b CHECK (((1 <= "position") AND ("position" <= 5)))
);

CREATE SEQUENCE virtual_registries_container_registry_upstreams_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE virtual_registries_container_registry_upstreams_id_seq OWNED BY virtual_registries_container_registry_upstreams.id;

CREATE TABLE virtual_registries_container_upstreams (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cache_validity_hours smallint DEFAULT 24 NOT NULL,
    username jsonb,
    password jsonb,
    url text NOT NULL,
    name text NOT NULL,
    description