﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Redshift {
namespace Model {

/**
 * <p>Describes cluster attributes that are in a pending state. A change to one or
 * more the attributes was requested and is in progress or will be
 * applied.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/PendingModifiedValues">AWS
 * API Reference</a></p>
 */
class PendingModifiedValues {
 public:
  AWS_REDSHIFT_API PendingModifiedValues() = default;
  AWS_REDSHIFT_API PendingModifiedValues(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_REDSHIFT_API PendingModifiedValues& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The pending or in-progress change of the admin user password for the
   * cluster.</p>
   */
  inline const Aws::String& GetMasterUserPassword() const { return m_masterUserPassword; }
  inline bool MasterUserPasswordHasBeenSet() const { return m_masterUserPasswordHasBeenSet; }
  template <typename MasterUserPasswordT = Aws::String>
  void SetMasterUserPassword(MasterUserPasswordT&& value) {
    m_masterUserPasswordHasBeenSet = true;
    m_masterUserPassword = std::forward<MasterUserPasswordT>(value);
  }
  template <typename MasterUserPasswordT = Aws::String>
  PendingModifiedValues& WithMasterUserPassword(MasterUserPasswordT&& value) {
    SetMasterUserPassword(std::forward<MasterUserPasswordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the cluster's node type.</p>
   */
  inline const Aws::String& GetNodeType() const { return m_nodeType; }
  inline bool NodeTypeHasBeenSet() const { return m_nodeTypeHasBeenSet; }
  template <typename NodeTypeT = Aws::String>
  void SetNodeType(NodeTypeT&& value) {
    m_nodeTypeHasBeenSet = true;
    m_nodeType = std::forward<NodeTypeT>(value);
  }
  template <typename NodeTypeT = Aws::String>
  PendingModifiedValues& WithNodeType(NodeTypeT&& value) {
    SetNodeType(std::forward<NodeTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the number of nodes in the cluster.</p>
   */
  inline int GetNumberOfNodes() const { return m_numberOfNodes; }
  inline bool NumberOfNodesHasBeenSet() const { return m_numberOfNodesHasBeenSet; }
  inline void SetNumberOfNodes(int value) {
    m_numberOfNodesHasBeenSet = true;
    m_numberOfNodes = value;
  }
  inline PendingModifiedValues& WithNumberOfNodes(int value) {
    SetNumberOfNodes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the cluster type.</p>
   */
  inline const Aws::String& GetClusterType() const { return m_clusterType; }
  inline bool ClusterTypeHasBeenSet() const { return m_clusterTypeHasBeenSet; }
  template <typename ClusterTypeT = Aws::String>
  void SetClusterType(ClusterTypeT&& value) {
    m_clusterTypeHasBeenSet = true;
    m_clusterType = std::forward<ClusterTypeT>(value);
  }
  template <typename ClusterTypeT = Aws::String>
  PendingModifiedValues& WithClusterType(ClusterTypeT&& value) {
    SetClusterType(std::forward<ClusterTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the service version.</p>
   */
  inline const Aws::String& GetClusterVersion() const { return m_clusterVersion; }
  inline bool ClusterVersionHasBeenSet() const { return m_clusterVersionHasBeenSet; }
  template <typename ClusterVersionT = Aws::String>
  void SetClusterVersion(ClusterVersionT&& value) {
    m_clusterVersionHasBeenSet = true;
    m_clusterVersion = std::forward<ClusterVersionT>(value);
  }
  template <typename ClusterVersionT = Aws::String>
  PendingModifiedValues& WithClusterVersion(ClusterVersionT&& value) {
    SetClusterVersion(std::forward<ClusterVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the automated snapshot retention
   * period.</p>
   */
  inline int GetAutomatedSnapshotRetentionPeriod() const { return m_automatedSnapshotRetentionPeriod; }
  inline bool AutomatedSnapshotRetentionPeriodHasBeenSet() const { return m_automatedSnapshotRetentionPeriodHasBeenSet; }
  inline void SetAutomatedSnapshotRetentionPeriod(int value) {
    m_automatedSnapshotRetentionPeriodHasBeenSet = true;
    m_automatedSnapshotRetentionPeriod = value;
  }
  inline PendingModifiedValues& WithAutomatedSnapshotRetentionPeriod(int value) {
    SetAutomatedSnapshotRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the new identifier for the cluster.</p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  PendingModifiedValues& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending or in-progress change of the ability to connect to the cluster
   * from the public network.</p>
   */
  inline bool GetPubliclyAccessible() const { return m_publiclyAccessible; }
  inline bool PubliclyAccessibleHasBeenSet() const { return m_publiclyAccessibleHasBeenSet; }
  inline void SetPubliclyAccessible(bool value) {
    m_publiclyAccessibleHasBeenSet = true;
    m_publiclyAccessible = value;
  }
  inline PendingModifiedValues& WithPubliclyAccessible(bool value) {
    SetPubliclyAccessible(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An option that specifies whether to create the cluster with enhanced VPC
   * routing enabled. To create a cluster that uses enhanced VPC routing, the cluster
   * must be in a VPC. For more information, see <a
   * href="https://docs.aws.amazon.com/redshift/latest/mgmt/enhanced-vpc-routing.html">Enhanced
   * VPC Routing</a> in the Amazon Redshift Cluster Management Guide.</p> <p>If this
   * option is <code>true</code>, enhanced VPC routing is enabled. </p> <p>Default:
   * false</p>
   */
  inline bool GetEnhancedVpcRouting() const { return m_enhancedVpcRouting; }
  inline bool EnhancedVpcRoutingHasBeenSet() const { return m_enhancedVpcRoutingHasBeenSet; }
  inline void SetEnhancedVpcRouting(bool value) {
    m_enhancedVpcRoutingHasBeenSet = true;
    m_enhancedVpcRouting = value;
  }
  inline PendingModifiedValues& WithEnhancedVpcRouting(bool value) {
    SetEnhancedVpcRouting(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the maintenance track that the cluster will change to during the
   * next maintenance window.</p>
   */
  inline const Aws::String& GetMaintenanceTrackName() const { return m_maintenanceTrackName; }
  inline bool MaintenanceTrackNameHasBeenSet() const { return m_maintenanceTrackNameHasBeenSet; }
  template <typename MaintenanceTrackNameT = Aws::String>
  void SetMaintenanceTrackName(MaintenanceTrackNameT&& value) {
    m_maintenanceTrackNameHasBeenSet = true;
    m_maintenanceTrackName = std::forward<MaintenanceTrackNameT>(value);
  }
  template <typename MaintenanceTrackNameT = Aws::String>
  PendingModifiedValues& WithMaintenanceTrackName(MaintenanceTrackNameT&& value) {
    SetMaintenanceTrackName(std::forward<MaintenanceTrackNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption type for a cluster. Possible values are: KMS and None. </p>
   */
  inline const Aws::String& GetEncryptionType() const { return m_encryptionType; }
  inline bool EncryptionTypeHasBeenSet() const { return m_encryptionTypeHasBeenSet; }
  template <typename EncryptionTypeT = Aws::String>
  void SetEncryptionType(EncryptionTypeT&& value) {
    m_encryptionTypeHasBeenSet = true;
    m_encryptionType = std::forward<EncryptionTypeT>(value);
  }
  template <typename EncryptionTypeT = Aws::String>
  PendingModifiedValues& WithEncryptionType(EncryptionTypeT&& value) {
    SetEncryptionType(std::forward<EncryptionTypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_masterUserPassword;

  Aws::String m_nodeType;

  int m_numberOfNodes{0};

  Aws::String m_clusterType;

  Aws::String m_clusterVersion;

  int m_automatedSnapshotRetentionPeriod{0};

  Aws::String m_clusterIdentifier;

  bool m_publiclyAccessible{false};

  bool m_enhancedVpcRouting{false};

  Aws::String m_maintenanceTrackName;

  Aws::String m_encryptionType;
  bool m_masterUserPasswordHasBeenSet = false;
  bool m_nodeTypeHasBeenSet = false;
  bool m_numberOfNodesHasBeenSet = false;
  bool m_clusterTypeHasBeenSet = false;
  bool m_clusterVersionHasBeenSet = false;
  bool m_automatedSnapshotRetentionPeriodHasBeenSet = false;
  bool m_clusterIdentifierHasBeenSet = false;
  bool m_publiclyAccessibleHasBeenSet = false;
  bool m_enhancedVpcRoutingHasBeenSet = false;
  bool m_maintenanceTrackNameHasBeenSet = false;
  bool m_encryptionTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
