/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.kafka.server.group.share;

import org.apache.kafka.common.Uuid;

import java.util.List;
import java.util.Objects;

/**
 * This class is used to contain the data for a topic and its partitions in the interface to {@link Persister}.
 *
 * @param <P> The type of {@link PartitionInfoData}
 */
public class TopicData<P extends PartitionInfoData> {
    private final Uuid topicId;
    private final List<P> partitions;

    public TopicData(Uuid topicId, List<P> partitions) {
        this.topicId = topicId;
        this.partitions = partitions;
    }

    public Uuid topicId() {
        return topicId;
    }

    public List<P> partitions() {
        return partitions;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        TopicData<?> topicData = (TopicData<?>) o;
        return Objects.equals(topicId, topicData.topicId) && Objects.equals(partitions, topicData.partitions);
    }

    @Override
    public int hashCode() {
        return Objects.hash(topicId, partitions);
    }

    @Override
    public String toString() {
        return "TopicData(" +
                "topicId=" + topicId + "," +
                "partitions=" + partitions +
                ")";
    }
}
