/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.bulkwriter;

import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.cassandra.bridge.CassandraBridge;
import org.apache.cassandra.bridge.CassandraBridgeFactory;
import org.apache.cassandra.spark.bulkwriter.BroadcastableTableSchema;
import org.apache.cassandra.spark.bulkwriter.SqlToCqlTypeConverter;
import org.apache.cassandra.spark.bulkwriter.TTLOption;
import org.apache.cassandra.spark.bulkwriter.TableInfoProvider;
import org.apache.cassandra.spark.bulkwriter.TimestampOption;
import org.apache.cassandra.spark.bulkwriter.WriteMode;
import org.apache.cassandra.spark.common.schema.ColumnType;
import org.apache.cassandra.spark.data.CqlField;
import org.apache.cassandra.spark.exception.UnsupportedAnalyticsOperationException;
import org.apache.spark.sql.types.StructType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableSchema {
    private static final Logger LOGGER = LoggerFactory.getLogger(TableSchema.class);
    final String createStatement;
    final String modificationStatement;
    final List<String> partitionKeyColumns;
    final List<ColumnType<?>> partitionKeyColumnTypes;
    final List<SqlToCqlTypeConverter.Converter<?>> converters;
    final List<Integer> keyFieldPositions;
    final WriteMode writeMode;
    final TTLOption ttlOption;
    final TimestampOption timestampOption;
    final String lowestCassandraVersion;
    final boolean quoteIdentifiers;

    public TableSchema(StructType dfSchema, TableInfoProvider tableInfo, WriteMode writeMode, TTLOption ttlOption, TimestampOption timestampOption, String lowestCassandraVersion, boolean quoteIdentifiers) {
        this.writeMode = writeMode;
        this.ttlOption = ttlOption;
        this.timestampOption = timestampOption;
        this.lowestCassandraVersion = lowestCassandraVersion;
        this.quoteIdentifiers = quoteIdentifiers;
        this.validateDataFrameCompatibility(dfSchema, tableInfo);
        TableSchema.validateNoSecondaryIndexes(tableInfo);
        this.createStatement = TableSchema.getCreateStatement(tableInfo);
        this.modificationStatement = this.getModificationStatement(dfSchema, tableInfo);
        this.partitionKeyColumns = TableSchema.getPartitionKeyColumnNames(tableInfo);
        this.partitionKeyColumnTypes = TableSchema.getPartitionKeyColumnTypes(tableInfo);
        this.converters = TableSchema.getConverters(dfSchema, tableInfo, ttlOption, timestampOption);
        LOGGER.info("Converters: {}", this.converters);
        this.keyFieldPositions = TableSchema.getKeyFieldPositions(dfSchema, tableInfo.getColumnNames(), this.getRequiredKeyColumns(tableInfo));
    }

    public TableSchema(BroadcastableTableSchema broadcastable) {
        this.createStatement = broadcastable.getCreateStatement();
        this.modificationStatement = broadcastable.getModificationStatement();
        this.partitionKeyColumns = broadcastable.getPartitionKeyColumns();
        this.partitionKeyColumnTypes = broadcastable.getPartitionKeyColumnTypes();
        this.converters = broadcastable.getConverters();
        this.keyFieldPositions = broadcastable.getKeyFieldPositions();
        this.writeMode = broadcastable.getWriteMode();
        this.ttlOption = broadcastable.getTtlOption();
        this.timestampOption = broadcastable.getTimestampOption();
        this.lowestCassandraVersion = broadcastable.getLowestCassandraVersion();
        this.quoteIdentifiers = broadcastable.isQuoteIdentifiers();
    }

    private List<String> getRequiredKeyColumns(TableInfoProvider tableInfo) {
        switch (this.writeMode) {
            case INSERT: {
                return tableInfo.getPrimaryKeyColumnNames();
            }
            case DELETE_PARTITION: {
                return tableInfo.getPartitionKeyColumnNames();
            }
        }
        throw new UnsupportedOperationException("Unknown WriteMode provided");
    }

    private static List<SqlToCqlTypeConverter.Converter<?>> getConverters(StructType dfSchema, TableInfoProvider tableInfo, TTLOption ttlOption, TimestampOption timestampOption) {
        return Arrays.stream(dfSchema.fieldNames()).map(fieldName -> {
            if (fieldName.equals(ttlOption.columnName())) {
                return SqlToCqlTypeConverter.integerConverter();
            }
            if (fieldName.equals(timestampOption.columnName())) {
                return SqlToCqlTypeConverter.microsecondsTimestampConverter();
            }
            CqlField.CqlType cqlType = tableInfo.getColumnType((String)fieldName);
            return SqlToCqlTypeConverter.getConverter(cqlType);
        }).collect(Collectors.toList());
    }

    private static List<ColumnType<?>> getPartitionKeyColumnTypes(TableInfoProvider tableInfo) {
        return tableInfo.getPartitionKeyTypes();
    }

    private static List<String> getPartitionKeyColumnNames(TableInfoProvider tableInfo) {
        return tableInfo.getPartitionKeyColumnNames();
    }

    private static String getCreateStatement(TableInfoProvider tableInfo) {
        String createStatement = tableInfo.getCreateStatement();
        LOGGER.info("CQL create statement for the table {}", (Object)createStatement);
        return createStatement;
    }

    private String getModificationStatement(StructType dfSchema, TableInfoProvider tableInfo) {
        switch (this.writeMode) {
            case INSERT: {
                return this.getInsertStatement(dfSchema, tableInfo, this.ttlOption, this.timestampOption);
            }
            case DELETE_PARTITION: {
                return this.getDeleteStatement(dfSchema, tableInfo);
            }
        }
        throw new UnsupportedOperationException("Unknown WriteMode provided");
    }

    private String getInsertStatement(StructType dfSchema, TableInfoProvider tableInfo, TTLOption ttlOption, TimestampOption timestampOption) {
        CassandraBridge bridge = CassandraBridgeFactory.get(this.lowestCassandraVersion);
        List columnNames = Arrays.stream(dfSchema.fieldNames()).filter(fieldName -> !fieldName.equals(ttlOption.columnName())).filter(fieldName -> !fieldName.equals(timestampOption.columnName())).collect(Collectors.toList());
        StringBuilder stringBuilder = new StringBuilder("INSERT INTO ").append(CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, tableInfo.getKeyspaceName())).append(".").append(CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, tableInfo.getName())).append(columnNames.stream().map(columnName -> CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, columnName)).collect(Collectors.joining(",", " (", ") ")));
        stringBuilder.append("VALUES").append(columnNames.stream().map(columnName -> ":" + CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, columnName)).collect(Collectors.joining(",", " (", ")")));
        if (ttlOption.withTTl() && timestampOption.withTimestamp()) {
            stringBuilder.append(" USING TIMESTAMP ").append(timestampOption.toCQLString(columnName -> CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, columnName))).append(" AND TTL ").append(ttlOption.toCQLString(columnName -> CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, columnName)));
        } else if (timestampOption.withTimestamp()) {
            stringBuilder.append(" USING TIMESTAMP ").append(timestampOption.toCQLString(columnName -> CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, columnName)));
        } else if (ttlOption.withTTl()) {
            stringBuilder.append(" USING TTL ").append(ttlOption.toCQLString(columnName -> CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, columnName)));
        }
        stringBuilder.append(";");
        String insertStatement = stringBuilder.toString();
        LOGGER.info("CQL insert statement for the RDD {}", (Object)insertStatement);
        return insertStatement;
    }

    private String getDeleteStatement(StructType dfSchema, TableInfoProvider tableInfo) {
        CassandraBridge bridge = CassandraBridgeFactory.get(this.lowestCassandraVersion);
        Stream<String> fieldEqualityStatements = Arrays.stream(dfSchema.fieldNames()).map(key -> CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, key) + "=?");
        String deleteStatement = String.format("DELETE FROM %s.%s where %s;", CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, tableInfo.getKeyspaceName()), CassandraBridgeFactory.maybeQuotedIdentifier(bridge, this.quoteIdentifiers, tableInfo.getName()), fieldEqualityStatements.collect(Collectors.joining(" AND ")));
        LOGGER.info("CQL delete statement for the RDD {}", (Object)deleteStatement);
        return deleteStatement;
    }

    private void validateDataFrameCompatibility(StructType dfSchema, TableInfoProvider tableInfo) {
        LinkedHashSet<String> dfFields = new LinkedHashSet<String>();
        Collections.addAll(dfFields, dfSchema.fieldNames());
        this.validatePrimaryKeyColumnsProvided(tableInfo, dfFields);
        switch (this.writeMode) {
            case INSERT: {
                TableSchema.validateDataframeFieldsInTable(tableInfo, dfFields, this.ttlOption, this.timestampOption);
                return;
            }
            case DELETE_PARTITION: {
                this.validateOnlyPartitionKeyColumnsInDataframe(tableInfo, dfFields);
                return;
            }
        }
        LOGGER.warn("Unrecognized write mode {}", (Object)this.writeMode);
    }

    private void validateOnlyPartitionKeyColumnsInDataframe(TableInfoProvider tableInfo, Set<String> dfFields) {
        LinkedHashSet<String> requiredKeyColumns = new LinkedHashSet<String>(this.getRequiredKeyColumns(tableInfo));
        Preconditions.checkArgument((boolean)requiredKeyColumns.equals(dfFields), (Object)String.format("Only partition key columns (%s) are supported in the input Dataframe when WRITE_MODE=DELETE_PARTITION but (%s) columns were provided", String.join((CharSequence)",", requiredKeyColumns), String.join((CharSequence)",", dfFields)));
    }

    private void validatePrimaryKeyColumnsProvided(TableInfoProvider tableInfo, Set<String> dfFields) {
        List<String> requiredKeyColumns = this.getRequiredKeyColumns(tableInfo);
        Preconditions.checkArgument((boolean)dfFields.containsAll(requiredKeyColumns), (Object)("Missing some required key components in DataFrame => " + requiredKeyColumns.stream().filter(column -> !dfFields.contains(column)).collect(Collectors.joining(","))));
    }

    private static void validateDataframeFieldsInTable(TableInfoProvider tableInfo, Set<String> dfFields, TTLOption ttlOption, TimestampOption timestampOption) {
        List unknownFields = dfFields.stream().filter(columnName -> !tableInfo.columnExists((String)columnName)).filter(columnName -> !columnName.equals(ttlOption.columnName())).filter(columnName -> !columnName.equals(timestampOption.columnName())).collect(Collectors.toList());
        Preconditions.checkArgument((boolean)unknownFields.isEmpty(), (Object)("Unknown fields in data frame => " + String.valueOf(unknownFields)));
    }

    static void validateNoSecondaryIndexes(TableInfoProvider tableInfo) {
        if (tableInfo.hasSecondaryIndex()) {
            throw new UnsupportedAnalyticsOperationException("Bulkwriter doesn't support secondary indexes");
        }
    }

    private static List<Integer> getKeyFieldPositions(StructType dfSchema, List<String> columnNames, List<String> keyFieldNames) {
        List<String> dfFieldNames = Arrays.asList(dfSchema.fieldNames());
        return columnNames.stream().filter(keyFieldNames::contains).map(dfFieldNames::indexOf).collect(Collectors.toList());
    }
}

