/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.compress;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import net.jpountz.lz4.LZ4Exception;
import net.jpountz.lz4.LZ4Factory;
import net.jpountz.lz4.LZ4SafeDecompressor;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LZ4Compressor
implements ICompressor {
    private static final Logger logger = LoggerFactory.getLogger(LZ4Compressor.class);
    public static final String LZ4_FAST_COMPRESSOR = "fast";
    public static final String LZ4_HIGH_COMPRESSOR = "high";
    private static final Set<String> VALID_COMPRESSOR_TYPES = new HashSet<String>(Arrays.asList("fast", "high"));
    private static final int DEFAULT_HIGH_COMPRESSION_LEVEL = 9;
    private static final String DEFAULT_LZ4_COMPRESSOR_TYPE = "fast";
    public static final String LZ4_HIGH_COMPRESSION_LEVEL = "lz4_high_compressor_level";
    public static final String LZ4_COMPRESSOR_TYPE = "lz4_compressor_type";
    private static final int INTEGER_BYTES = 4;
    private static final ConcurrentHashMap<Pair<String, Integer>, LZ4Compressor> instances = new ConcurrentHashMap();
    private final net.jpountz.lz4.LZ4Compressor compressor;
    private final LZ4SafeDecompressor decompressor;
    @VisibleForTesting
    final String compressorType;
    @VisibleForTesting
    final Integer compressionLevel;
    private final Set<ICompressor.Uses> recommendedUses;

    public static LZ4Compressor create(Map<String, String> args) throws ConfigurationException {
        Integer compressionLevel;
        String compressorType = LZ4Compressor.validateCompressorType(args.get(LZ4_COMPRESSOR_TYPE));
        Pair<String, Integer> compressorTypeAndLevel = Pair.create(compressorType, compressionLevel = LZ4Compressor.validateCompressionLevel(args.get(LZ4_HIGH_COMPRESSION_LEVEL)));
        LZ4Compressor instance = (LZ4Compressor)instances.get(compressorTypeAndLevel);
        if (instance == null) {
            LZ4Compressor instanceFromMap;
            if (compressorType.equals("fast") && args.get(LZ4_HIGH_COMPRESSION_LEVEL) != null) {
                logger.warn("'{}' parameter is ignored when '{}' is '{}'", new Object[]{LZ4_HIGH_COMPRESSION_LEVEL, LZ4_COMPRESSOR_TYPE, "fast"});
            }
            if (compressorType.equals(LZ4_HIGH_COMPRESSOR)) {
                logger.info("The ZstdCompressor may be preferable to LZ4 in 'high' mode. Zstd will typically compress much faster while achieving better ratio, but it may decompress more slowly,");
            }
            if ((instanceFromMap = instances.putIfAbsent(compressorTypeAndLevel, instance = new LZ4Compressor(compressorType, compressionLevel))) != null) {
                instance = instanceFromMap;
            }
        }
        return instance;
    }

    private LZ4Compressor(String type, Integer compressionLevel) {
        this.compressorType = type;
        this.compressionLevel = compressionLevel;
        LZ4Factory lz4Factory = LZ4Factory.fastestInstance();
        switch (type) {
            case "high": {
                this.compressor = lz4Factory.highCompressor(compressionLevel);
                this.recommendedUses = ImmutableSet.of(ICompressor.Uses.GENERAL);
                break;
            }
            default: {
                this.compressor = lz4Factory.fastCompressor();
                this.recommendedUses = ImmutableSet.copyOf(EnumSet.allOf(ICompressor.Uses.class));
            }
        }
        this.decompressor = lz4Factory.safeDecompressor();
    }

    @Override
    public int initialCompressedBufferLength(int chunkLength) {
        return 4 + this.compressor.maxCompressedLength(chunkLength);
    }

    @Override
    public void compress(ByteBuffer input, ByteBuffer output) throws IOException {
        int len = input.remaining();
        output.put((byte)len);
        output.put((byte)(len >>> 8));
        output.put((byte)(len >>> 16));
        output.put((byte)(len >>> 24));
        try {
            this.compressor.compress(input, output);
        }
        catch (LZ4Exception e) {
            throw new IOException(e);
        }
    }

    @Override
    public int uncompress(byte[] input, int inputOffset, int inputLength, byte[] output, int outputOffset) throws IOException {
        int writtenLength;
        int decompressedLength = input[inputOffset] & 0xFF | (input[inputOffset + 1] & 0xFF) << 8 | (input[inputOffset + 2] & 0xFF) << 16 | (input[inputOffset + 3] & 0xFF) << 24;
        try {
            writtenLength = this.decompressor.decompress(input, inputOffset + 4, inputLength - 4, output, outputOffset, decompressedLength);
        }
        catch (LZ4Exception e) {
            throw new IOException(e);
        }
        if (writtenLength != decompressedLength) {
            throw new IOException("Decompressed lengths mismatch");
        }
        return decompressedLength;
    }

    @Override
    public void uncompress(ByteBuffer input, ByteBuffer output) throws IOException {
        int decompressedLength = input.get() & 0xFF | (input.get() & 0xFF) << 8 | (input.get() & 0xFF) << 16 | (input.get() & 0xFF) << 24;
        try {
            int compressedLength = input.remaining();
            this.decompressor.decompress(input, input.position(), input.remaining(), output, output.position(), decompressedLength);
            input.position(input.position() + compressedLength);
            output.position(output.position() + decompressedLength);
        }
        catch (LZ4Exception e) {
            throw new IOException(e);
        }
        if (input.remaining() > 0) {
            throw new IOException("Compressed lengths mismatch - " + input.remaining() + " bytes remain");
        }
    }

    @Override
    public Set<String> supportedOptions() {
        return new HashSet<String>(Arrays.asList(LZ4_HIGH_COMPRESSION_LEVEL, LZ4_COMPRESSOR_TYPE));
    }

    public static String validateCompressorType(String compressorType) throws ConfigurationException {
        if (compressorType == null) {
            return "fast";
        }
        if (!VALID_COMPRESSOR_TYPES.contains(compressorType)) {
            throw new ConfigurationException(String.format("Invalid compressor type '%s' specified for LZ4 parameter '%s'. Valid options are %s.", compressorType, LZ4_COMPRESSOR_TYPE, VALID_COMPRESSOR_TYPES.toString()));
        }
        return compressorType;
    }

    public static Integer validateCompressionLevel(String compressionLevel) throws ConfigurationException {
        Integer level;
        if (compressionLevel == null) {
            return 9;
        }
        ConfigurationException ex = new ConfigurationException("Invalid value [" + compressionLevel + "] for parameter 'lz4_high_compressor_level'. Value must be between 1 and 17.");
        try {
            level = Integer.valueOf(compressionLevel);
        }
        catch (NumberFormatException e) {
            throw ex;
        }
        if (level < 1 || level > 17) {
            throw ex;
        }
        return level;
    }

    @Override
    public BufferType preferredBufferType() {
        return BufferType.OFF_HEAP;
    }

    @Override
    public boolean supports(BufferType bufferType) {
        return true;
    }

    @Override
    public Set<ICompressor.Uses> recommendedUses() {
        return this.recommendedUses;
    }
}

