"use strict";
/* --------------------------------------------------------------------------------------------
 * Copyright (c) TypeFox and others. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DiagnosticFeature = void 0;
const vscode_1 = require("vscode");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
const client_1 = require("./client");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = {};
    }
    return target[key];
}
var RequestStateKind;
(function (RequestStateKind) {
    RequestStateKind["active"] = "open";
    RequestStateKind["reschedule"] = "reschedule";
    RequestStateKind["outDated"] = "drop";
})(RequestStateKind || (RequestStateKind = {}));
class DiagnosticFeature extends client_1.TextDocumentFeature {
    constructor(client) {
        super(client, vscode_languageserver_protocol_1.Proposed.DiagnosticRequest.type);
    }
    fillClientCapabilities(capabilities) {
        let capability = ensure(ensure(capabilities, 'textDocument'), 'diagnostic');
        capability.dynamicRegistration = true;
    }
    initialize(capabilities, documentSelector) {
        let [id, options] = this.getRegistration(documentSelector, capabilities.diagnosticProvider);
        if (!id || !options) {
            return;
        }
        this.register({ id: id, registerOptions: options });
    }
    registerLanguageProvider(options) {
        const documentSelector = options.documentSelector;
        const collection = vscode_1.languages.createDiagnosticCollection(options.identifier);
        const openDocuments = new Map();
        const provider = {
            provideDiagnostics: (resource, token) => {
                const client = this._client;
                const provideDiagnostics = (resource, token) => {
                    const params = {
                        textDocument: { uri: client.code2ProtocolConverter.asUri(resource) }
                    };
                    return client.sendRequest(vscode_languageserver_protocol_1.Proposed.DiagnosticRequest.type, params, token).then((result) => {
                        if (result === null) {
                            return [];
                        }
                        return client.protocol2CodeConverter.asDiagnostics(result);
                    }, (error) => {
                        return client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.DiagnosticRequest.type, token, error, []);
                    });
                };
                return provideDiagnostics(resource, token);
            }
        };
        const requestStates = new Map();
        const pullDiagnostics = async (resource) => {
            var _a;
            const key = resource.toString();
            const currentState = requestStates.get(key);
            if (currentState !== undefined) {
                if (currentState.state === RequestStateKind.active) {
                    currentState.tokenSource.cancel();
                    requestStates.set(key, { state: RequestStateKind.reschedule, uri: resource });
                }
                // We have a state. Wait until the request returns.
                return;
            }
            const tokenSource = new vscode_1.CancellationTokenSource();
            requestStates.set(key, { state: RequestStateKind.active, uri: resource, tokenSource });
            const diagnostics = (_a = await provider.provideDiagnostics(resource, tokenSource.token)) !== null && _a !== void 0 ? _a : [];
            const afterState = requestStates.get(key);
            if (afterState === undefined) {
                // This shouldn't happen. Log it
                this._client.error(`Lost request state in diagnostic pull model. Clearing diagnostics for ${key}`);
                collection.delete(resource);
                return;
            }
            requestStates.delete(key);
            if (afterState.state === RequestStateKind.outDated) {
                return;
            }
            collection.set(resource, diagnostics);
            if (afterState.state === RequestStateKind.reschedule) {
                pullDiagnostics(resource);
            }
        };
        const openDocumentHandler = () => {
            const added = new Set();
            const current = new Map(openDocuments);
            openDocuments.clear();
            for (const info of vscode_1.window.openEditors) {
                const key = info.resource.toString();
                openDocuments.set(key, info.resource);
                if (!current.has(key)) {
                    added.add(info.resource);
                }
                else {
                    current.delete(key);
                }
            }
            // The once that are still in current are the once that
            // are not valid anymore. So clear the diagnostics
            for (const entry of current.entries()) {
                const key = entry[0];
                const uri = entry[1];
                collection.delete(uri);
                const requestState = requestStates.get(key);
                if (requestState === undefined) {
                    continue;
                }
                // We have a running request. If it is active, cancel it.
                if (requestState.state === RequestStateKind.active) {
                    requestState.tokenSource.cancel();
                }
                // Mark the result as out dated.
                requestStates.set(key, { state: RequestStateKind.outDated, uri: entry[1] });
            }
            for (const item of added) {
                pullDiagnostics(item);
            }
        };
        openDocumentHandler();
        const openEditorsListener = vscode_1.window.onDidChangeOpenEditors(openDocumentHandler);
        const documentChangeListener = vscode_1.workspace.onDidChangeTextDocument((event) => {
            if (event.contentChanges.length === 0) {
                return;
            }
            const document = event.document;
            if (!vscode_1.languages.match(documentSelector, document)) {
                return;
            }
            if (!openDocuments.has(document.uri.toString())) {
                return;
            }
            pullDiagnostics(document.uri);
        });
        const disposable = new vscode_1.Disposable(() => {
            openEditorsListener.dispose();
            documentChangeListener.dispose();
            collection.dispose();
        });
        return [disposable, provider];
    }
}
exports.DiagnosticFeature = DiagnosticFeature;
//# sourceMappingURL=proposed.diagnostic.js.map