#' Time-dependent precision-recall curve (PRC) estimation from right-censored survival data
#'
#' @description {This function empirically estimate the time-dependent precision-recall curve and RUC curve for
#' right censored survival data  using the cumulative sensitivity and dynamic specificity definitions.
#' It also calculates the time-dependent area under precision-recall curve (AUPRC) and the area under the ROC curve (AUC).
#' The function computes standard error and confidence interval of AUPRC and AUC using non-parametric bootstrap approach.
#' }
#'
#' @param Y The numeric vector of event-time or observed time.
#' @param M The numeric vector of marker values.
#' @param censor The censoring indicator, \code{1} if event, \code{0} otherwise.
#' @param t A scalar time point used to calculate the PRC curve.
#' @param cut A grid of cutoff values for  estimation is computed. Default is sequence of \code{151} numbers between \code{0} and \code{1}.
#' @param len The length of the grid points. Default is \code{151}.
#' @param h A scalar value for Beran's weight calculations. The default is the value obtained by using the method of Sheather and Jones (1991).
#' @param ktype A character string specifying the desired kernel needed for Beran weight calculation. The possible options are "\code{normal}", "\code{epanechnikov}", "\code{tricube}", "\code{boxcar}", "\code{triangular}", or "\code{quartic}". The defaults is "\code{gaussian}" kernel density.
#' @param B The number of bootstrap samples to be used for variance estimation. The default is \code{0}, no variance estimation.
#' @param alpha The significance level. The default is \code{0.05}.
#' @param plot The logical parameter to see the ROC curve plot. Default is \code{FALSE}.
#'
#' @return Returns the following items:
#'
#'  \item{TPR }{vector of estimated TPR.}
#'
#'  \item{FPR }{vector of estimated FPR.}
#'
#'  \item{PPV }{vector of estimated PPV.}
#'
#'  \item{AUPRC }{estimated area under the PR curve at a given time horizon \code{t}.}
#'
#'  \item{AUC }{estimated area under the ROC curve at a given time horizon \code{t}.}
#'
#'  \item{APbot }{estimated area under the PR curve for each bootstrap sample at a given time horizon \code{t}.}
#'
#'  \item{dat }{a data frame with two columns:po = positive and M = marker.}
#'
#' @importFrom survidm Beran
#' @importFrom graphics par
#' @importFrom stats bw.SJ quantile sd
#'
#' @examples library(tdPRC);
#'
#' data(mayo);
#'
#' data <- mayo[ ,c( "time","censor","mayoscore5" )] ;
#' t <- 365*6;
#'
#' resu <- tdPRC(Y=data$time, M=data$mayoscore5, censor=data$censor, t=t, cut=NULL,
#'          len=1000, h=0.1, plot=TRUE);
#' resu$AUPRC
#'
#' @references Beyene, K.M., Chen, D.G., and Kifle, Y.G. (2024). A novel nonparametric time‐dependent precision-recall curve estimator for right‐censored survival data. \emph{Biometrical Journal}, 66(3), 2300135.
#' @references Beyene, K.M. and El Ghouch A. (2020). Smoothed time-dependent receiver operating characteristic curve for right censored survival data. \emph{Statistics in Medicine}. 39: 3373-3396.
#' @references Li, L., Greene, T. and Hu, B. (2016).  A simple method to estimate the time-dependent receiver operating characteristic curve and the area under the curve with right censored data, \emph{Statistical Methods in Medical Research}, 27(8): 2264-2278.
#' @export

tdPRC <- function(Y, M, censor, t, cut=NULL, len=1000, h=0.1, ktype="gaussian", B=0, alpha=0.05, plot=FALSE)
{
  if(!is.vector(Y, mode = "numeric") | !is.vector(M, mode = "numeric")| !is.vector(censor, mode = "numeric") | !is.vector(t,  mode = "numeric"))
    warning("Error! all numeric vectors Y, M, censor and t should be specified")
  else{
    Dt <- Csurv(Y=Y, M=M, censor=censor, t=t, h=h, ktype=ktype)$positive;
    or <- order(M);
    M1 <- M[or]; Y1 <- Y[or]; censor1 <- censor[or]; po1 <- Dt[or];
    if(is.null(cut)){
      cut <- c(seq(min(M1), max(M1), length.out=len)[-len]) ;
    }
    den <- NULL; den1 <- NULL; num <- NULL;
    for (z in cut) {
      den <- c( den, sum(po1*as.numeric(M1 > z))) ;
      num <- c( num, sum(as.numeric(M1 > z))) ;
      den1 <- c( den1, sum((1-po1)*as.numeric(M1 > z))) ;
    }
    sumP <- sum(po1);
    nP <- length(po1);
    TPR <- den / sumP;
    FPR <- den1 / (nP-sumP);
    PPV <- den / num ;
    nu2 <- apply(po1*((outer((M1), (M1), "-")>0)+ 0.5*(outer((M1), (M1), "-")==0)), 2, sum);
    nu3 <- apply(((outer((M1), (M1), "-")>0) + 0.5*(outer((M1), (M1), "-")==0)), 2, sum);
    auprcc <- sum(po1*(nu2/nu3), na.rm=TRUE) / sumP ;
    auprc <- data.frame(Estimate = round(auprcc, 4) , sd = NA, LCL = NA, UCL = NA)
    Z <- 1 - cumsum(1 - po1) / (nP-sumP);
    AUC <- 1 - sum(po1 * Z) / sumP;
    auprcb <- 0;
  }
  if (B>0){
    data <- data.frame(Y=Y, M=M, censor=censor)
    auprcb <- NULL
    for (i in 1:B){
      bootsample <- sample(1:nrow(data), nrow(data), replace=TRUE)
      dat1 <- data[bootsample, ]
      Dt1 <- Csurv(Y=dat1$Y, M=dat1$M, censor=dat1$censor, t=t, h=h, ktype=ktype)$positive;
      nu21 <- apply(Dt1*((outer((dat1$M), (dat1$M), "-")>0)+ 0.5*(outer((dat1$M), (dat1$M), "-")==0)), 2, sum);
      nu31 <- apply(((outer((dat1$M), (dat1$M), "-")>0) + 0.5*(outer((dat1$M), (dat1$M), "-")==0)), 2, sum);
      auprcb[i] <- sum(Dt1*(nu21/nu31), na.rm=TRUE) / sum(Dt1);
    }
    SP <- unname(quantile(auprcb, p=c(alpha/2, 1-alpha/2), na.rm = TRUE))
    auprc <- data.frame(Estimate = round(auprcc, 4), sd = round(sd(auprcb, na.rm = TRUE), 4), LCL = round(SP[1], 4), UCL = round(SP[2], 4))
    }
    if(plot=="TRUE"){
      opar <- par(no.readonly=TRUE);
      on.exit(par(opar));
      par(mfrow=c(1,2));
      plot( TPR, PPV,  type="l", lwd=2, col.lab="blue", col="blue", main="PR curve", xlab="Recall", ylab="Precision") ;
      plot( FPR, TPR,  type="l", lwd=2, col.lab="blue", col="blue", main="ROC curve", xlab="FPR", ylab="TPR");
      }

    return(list(TPR=TPR, PPV=PPV, FPR=FPR, AUPRC=auprc, APbot=auprcb, AUC=AUC, dat=data.frame(po=Dt,M=M)))
}
