% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spGLMexact.R
\name{spGLMexact}
\alias{spGLMexact}
\title{Univariate Bayesian spatial generalized linear model}
\usage{
spGLMexact(
  formula,
  data = parent.frame(),
  family,
  coords,
  cor.fn,
  priors,
  spParams,
  boundary = 0.5,
  n.samples,
  loopd = FALSE,
  loopd.method = "exact",
  CV.K = 10,
  loopd.nMC = 500,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{formula}{a symbolic description of the regression model to be fit.
See example below.}

\item{data}{an optional data frame containing the variables in the model.
If not found in \code{data}, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{spGLMexact} is called.}

\item{family}{Specifies the distribution of the response as a member of the
exponential family. Supported options are \code{'poisson'}, \code{'binomial'} and
\code{'binary'}.}

\item{coords}{an \eqn{n \times 2}{n x 2} matrix of the observation
coordinates in \eqn{\mathbb{R}^2} (e.g., easting and northing).}

\item{cor.fn}{a quoted keyword that specifies the correlation function used
to model the spatial dependence structure among the observations. Supported
covariance model key words are: \code{'exponential'} and \code{'matern'}.
See below for details.}

\item{priors}{(optional) a list with each tag corresponding to a
hyperparameter name and containing hyperprior details. Valid tags include
\code{V.beta}, \code{nu.beta}, \code{nu.z} and \code{sigmaSq.xi}. Values of \code{nu.beta} and \code{nu.z}
must be at least 2.1. If not supplied, uses defaults.}

\item{spParams}{fixed values of spatial process parameters.}

\item{boundary}{Specifies the boundary adjustment parameter. Must be a real
number between 0 and 1. Default is 0.5.}

\item{n.samples}{number of posterior samples to be generated.}

\item{loopd}{logical. If \code{loopd=TRUE}, returns leave-one-out predictive
densities, using method as given by \code{loopd.method}. Default is
\code{FALSE}.}

\item{loopd.method}{character. Ignored if \code{loopd=FALSE}. If \code{loopd=TRUE},
valid inputs are \code{'exact'}, \code{'CV'} and \code{'PSIS'}. The option \code{'exact'}
corresponds to exact leave-one-out predictive densities which requires
computation almost equivalent to fitting the model \eqn{n} times. The
options \code{'CV'} and \code{'PSIS'} are faster and they implement \eqn{K}-fold
cross validation and Pareto-smoothed importance sampling to find approximate
leave-one-out predictive densities (Vehtari \emph{et al.} 2017).}

\item{CV.K}{An integer between 10 and 20. Considered only if
\code{loopd.method='CV'}. Default is 10 (as recommended in Vehtari \emph{et. al} 2017).}

\item{loopd.nMC}{Number of Monte Carlo samples to be used to evaluate
leave-one-out predictive densities when \code{loopd.method} is set to either
'exact' or 'CV'.}

\item{verbose}{logical. If \code{verbose = TRUE}, prints model description.}

\item{...}{currently no additional argument.}
}
\value{
An object of class \code{spGLMexact}, which is a list with the
following tags -
\describe{
\item{priors}{details of the priors used, containing the values of the
boundary adjustment parameter (\code{boundary}), the variance parameter of the
fine-scale variation term (\code{simasq.xi}) and others.}
\item{samples}{a list of length 3, containing posterior samples of fixed
effects (\code{beta}), spatial effects (\code{z}) and the fine-scale
variation term (\code{xi}).}
\item{loopd}{If \code{loopd=TRUE}, contains leave-one-out predictive
densities.}
\item{model.params}{Values of the fixed parameters that includes
\code{phi} (spatial decay), \code{nu} (spatial smoothness).}
}
The return object might include additional data that can be used for
subsequent prediction and/or model fit evaluation.
}
\description{
Fits a Bayesian spatial generalized linear model with fixed
values of spatial process parameters and some auxiliary model parameters. The
output contains posterior samples of the fixed effects, spatial random
effects and, if required, finds leave-one-out predictive densities.
}
\details{
With this function, we fit a Bayesian hierarchical spatial
generalized linear model by sampling exactly from the joint posterior
distribution utilizing the generalized conjugate multivariate distribution
theory (Bradley and Clinch 2024). Suppose \eqn{\chi = (s_1, \ldots, s_n)}
denotes the \eqn{n} spatial locations the response \eqn{y} is observed. Let
\eqn{y(s)} be the outcome at location \eqn{s} endowed with a probability law
from the natural exponential family, which we denote by
\deqn{
y(s) \sim \mathrm{EF}(x(s)^\top \beta + z(s); b, \psi)
}
for some positive parameter \eqn{b > 0} and unit log partition function
\eqn{\psi}. We consider the following response models based on the input
supplied to the argument \code{family}.
\describe{
\item{\code{'poisson'}}{It considers point-referenced Poisson responses
\eqn{y(s) \sim \mathrm{Poisson}(e^{x(s)^\top \beta + z(s)})}. Here,
\eqn{b = 1} and \eqn{\psi(t) = e^t}.}
\item{\code{'binomial'}}{It considers point-referenced binomial counts
\eqn{y(s) \sim \mathrm{Binomial}(m(s), \pi(s))} where, \eqn{m(s)} denotes
the total number of trials and probability of success
\eqn{\pi(s) = \mathrm{ilogit}(x(s)^\top \beta + z(s))} at location \eqn{s}.
Here, \eqn{b = m(s)} and \eqn{\psi(t) = \log(1+e^t)}.}
\item{\code{'binary'}}{It considers point-referenced binary data (0 or, 1) i.e.,
\eqn{y(s) \sim \mathrm{Bernoulli}(\pi(s))}, where probability of success
\eqn{\pi(s) = \mathrm{ilogit}(x(s)^\top \beta + z(s))} at location \eqn{s}.
Here, \eqn{b = 1} and \eqn{\psi(t) = \log(1 + e^t)}.}
}
The hierarchical model is given as
\deqn{
\begin{aligned}
y(s_i) &\mid \beta, z, \xi \sim EF(x(s_i)^\top \beta + z(s_i) +
\xi_i - \mu_i; b_i, \psi_y), i = 1, \ldots, n\\
\xi &\mid \beta, z, \sigma^2_\xi, \alpha_\epsilon \sim
\mathrm{GCM_c}(\cdots),\\
\beta &\mid \sigma^2_\beta \sim N(0, \sigma^2_\beta V_\beta), \quad
\sigma^2_\beta \sim \mathrm{IG}(\nu_\beta/2, \nu_\beta/2)\\
z &\mid \sigma^2_z \sim N(0, \sigma^2_z R(\chi; \phi, \nu)), \quad
\sigma^2_z \sim \mathrm{IG}(\nu_z/2, \nu_z/2),
\end{aligned}
}
where \eqn{\mu = (\mu_1, \ldots, \mu_n)^\top} denotes the discrepancy
parameter. We fix the spatial process parameters \eqn{\phi} and \eqn{\nu} and
the hyperparameters \eqn{V_\beta}, \eqn{\nu_\beta}, \eqn{\nu_z} and
\eqn{\sigma^2_\xi}. The term \eqn{\xi} is known as the fine-scale variation
term which is given a conditional generalized conjugate multivariate
distribution as prior. For more details, see Pan \emph{et al.} 2025. Default
values for \eqn{V_\beta}, \eqn{\nu_\beta}, \eqn{\nu_z}, \eqn{\sigma^2_\xi}
are diagonal with each diagonal element 100, 2.1, 2.1 and 0.1 respectively.
}
\examples{
# Example 1: Analyze spatial poisson count data
data(simPoisson)
dat <- simPoisson[1:10, ]
mod1 <- spGLMexact(y ~ x1, data = dat, family = "poisson",
                   coords = as.matrix(dat[, c("s1", "s2")]),
                   cor.fn = "matern",
                   spParams = list(phi = 4, nu = 0.4),
                   n.samples = 100, verbose = TRUE)

# summarize posterior samples
post_beta <- mod1$samples$beta
print(t(apply(post_beta, 1, function(x) quantile(x, c(0.025, 0.5, 0.975)))))

# Example 2: Analyze spatial binomial count data
data(simBinom)
dat <- simBinom[1:10, ]
mod2 <- spGLMexact(cbind(y, n_trials) ~ x1, data = dat, family = "binomial",
                   coords = as.matrix(dat[, c("s1", "s2")]),
                   cor.fn = "matern",
                   spParams = list(phi = 3, nu = 0.4),
                   n.samples = 100, verbose = TRUE)

# summarize posterior samples
post_beta <- mod2$samples$beta
print(t(apply(post_beta, 1, function(x) quantile(x, c(0.025, 0.5, 0.975)))))

# Example 3: Analyze spatial binary data
data(simBinary)
dat <- simBinary[1:10, ]
mod3 <- spGLMexact(y ~ x1, data = dat, family = "binary",
                   coords = as.matrix(dat[, c("s1", "s2")]),
                   cor.fn = "matern",
                   spParams = list(phi = 4, nu = 0.4),
                   n.samples = 100, verbose = TRUE)

# summarize posterior samples
post_beta <- mod3$samples$beta
print(t(apply(post_beta, 1, function(x) quantile(x, c(0.025, 0.5, 0.975)))))
}
\references{
Bradley JR, Clinch M (2024). "Generating Independent Replicates
Directly from the Posterior Distribution for a Class of Spatial Hierarchical
Models." \emph{Journal of Computational and Graphical Statistics}, \strong{0}(0), 1-17.
\doi{10.1080/10618600.2024.2365728}.

Pan S, Zhang L, Bradley JR, Banerjee S (2025). "Bayesian
Inference for Spatial-temporal Non-Gaussian Data Using Predictive Stacking."
\doi{10.48550/arXiv.2406.04655}.

Vehtari A, Gelman A, Gabry J (2017). "Practical Bayesian Model
Evaluation Using Leave-One-out Cross-Validation and WAIC."
\emph{Statistics and Computing}, \strong{27}(5), 1413-1432. ISSN 0960-3174.
\doi{10.1007/s11222-016-9696-4}.
}
\seealso{
\code{\link[=spLMexact]{spLMexact()}}
}
\author{
Soumyakanti Pan \href{mailto:span18@ucla.edu}{span18@ucla.edu}
}
