% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/internalStructure.R
\name{plotAssemblyEffects}
\alias{plotAssemblyEffects}
\title{Plot predictors of assembly processes}
\usage{
plotAssemblyEffects(
  object,
  response = c("sites", "species"),
  pred = NULL,
  cols = c("#A38310", "#B42398", "#20A382"),
  negatives = c("raw", "scale", "floor")
)
}
\arguments{
\item{object}{An \code{sjSDManova} object from the \code{\link{anova.sjSDM}} function.}

\item{response}{whether to use sites or species. Default is sites}

\item{pred}{predictor variable. If \code{NULL}, environment uniqueness, spatial uniqueness, and richness is calculated from the fitted object and used as predictor.}

\item{cols}{Colors for the three assembly processes.}

\item{negatives}{how to handle negative R squareds}
}
\value{
A list with the following components:

\item{env}{A list of summary tables for env, space, and codist R-squared.}
\item{space}{A list of summary tables for env, space, and codist R-squared.}
\item{codist}{A list of summary tables for env, space, and codist R-squared.}
}
\description{
The function plots correlations between assembly processes and predictors or traits
}
\details{
Correlation and plots of the three assembly processes (environment, space, and codist) against environmental and spatial uniqueness and richness. The importance of the three assembly processes is measured by the partial R-squared (shown in the internal structure plots).

Importances are available for species and sites. Custom environmental predictors or traits can be specified. Environmental predictors are plotted against site R-squared and traits are plotted against species R-squared.
Regression lines are estimated by 50\\% quantile regression models.
}
\note{
Defaults for negative values are different than for \link{plot.sjSDMinternalStructure}
}
\examples{
\dontrun{
library(sjSDM)
# simulate community:
community = simulate_SDM(env = 3L, species = 10L, sites = 100L)

Occ <- community$response
Env <- community$env_weights
SP <- data.frame(matrix(rnorm(200, 0, 0.3), 100, 2)) # spatial coordinates


# fit model:
model <- sjSDM(Y = Occ, 
               env = linear(data = Env, formula = ~X1+X2+X3), 
               spatial = linear(data = SP, formula = ~0+X1*X2), 
               family=binomial("probit"),
               verbose = FALSE,
               iter = 20) # increase iter for real analysis

# Calculate ANOVA for env, space, associations, for details see ?anova.sjSDM
an = anova(model, samples = 10, verbose = FALSE) # increase iter for real analysis

# Show anova fractions
plot(an)

# ANOVA tables with different way to handle fractions
summary(an)
summary(an, fractions = "discard")
summary(an, fractions = "proportional")
summary(an, fractions = "equal")

# Internal structure
int = internalStructure(an, fractions = "proportional")

print(int)

plot(int) # default is negative values will be set to 0
plot(int, negatives = "scale") # global rescaling of all values to range 0-1
plot(int, negatives = "raw") # negative values will be discarded

plotAssemblyEffects(int)
plotAssemblyEffects(int, negatives = "floor")
plotAssemblyEffects(int, response = "sites", pred = as.factor(c(rep(1, 50), rep(2, 50))))
plotAssemblyEffects(int, response = "species", pred = runif(10))
plotAssemblyEffects(int, response = "species", pred = as.factor(c(rep(1, 5), rep(2, 5))))
}
}
\references{
Leibold, M. A., Rudolph, F. J., Blanchet, F. G., De Meester, L., Gravel, D., Hartig, F., ... & Chase, J. M. (2022). The internal structure of metacommunities. \emph{Oikos}, 2022(1).
}
