% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sglg.R
\name{sglg}
\alias{sglg}
\title{Fitting semi-parametric generalized log-gamma regression models}
\usage{
sglg(
  formula,
  npc,
  basis,
  data,
  shape = 0.2,
  method,
  alpha0,
  Knot,
  Tolerance = 5e-05,
  Maxiter = 1000,
  format = "complete"
)
}
\arguments{
\item{formula}{a symbolic description of the systematic component of the model to be fitted. See details for further information.}

\item{npc}{a matrix with the nonparametric variables of the systematic part of the model to be fitted. Must be included the names of each variables.}

\item{basis}{a name of the cubic spline basis to be used in the model. Supported basis include deBoor and Gu basis
which are a B-spline basis and a natural cubic spline basis, respectively.}

\item{data}{an optional data frame, list containing the variables in the model.}

\item{shape}{an optional value for the shape parameter of the error distribution of a generalized log-gamma distribution. Default value is 0.2.}

\item{method}{There are two possibles algorithms to estimate the parameters. The default algorithm is 'FS' Fisher-Scoring,
the other option is 'GSFS' an adequate combination between the block matrix version of non-linear Gauss-Seidel algorithm and Fisher-Scoring algorithm.}

\item{alpha0}{is a vector of positive values for the smoothing parameters alpha. Default vector with 1 in each entry.}

\item{Knot}{is a vector of the number of knots in each non-linear component of the model.}

\item{Tolerance}{an optional positive value, which represents the convergence criterion. Default value is 5e-05.}

\item{Maxiter}{an optional positive integer giving the maximal number of iterations for the estimating process. Default value is 1e03.}

\item{format}{an optional string value that indicates if you want a simple or a complete report of the estimating process. Default value is 'complete'.}
}
\value{
mu a vector of parameter estimates associated with the location parameter.

sigma estimate of the scale parameter associated with the model.

lambda estimate of the shape parameter associated with the model.

interval estimate of a 95\% confidence interval for each estimate parameters associated with the model.

Deviance the deviance associated with the model.
}
\description{
\code{sglg} is used to fit a semi-parametric regression model suitable for analysis of data sets in which the response variable is continuous, strictly positive, and asymmetric.
In this setup, the location parameter of the response variable is explicitly modeled by semi-parametric functions, whose nonparametric components may be approximated by
natural cubic splines or cubic P-splines.
}
\examples{
set.seed(1)
rows<- 300
t_beta <- c(0.5,2)
t_sigma <- 0.5
t_lambda <- 1
x1 <- runif(rows,-3,3)
x2 <- rbinom(rows,1,0.5)
X <- cbind(x1,x2)
t <- as.matrix((2*1:rows - 1)/(2*rows))
colnames(t) <- "t"
f_t <- cos(4*pi*t)
error <- rglg(rows,0,1,t_lambda)
y <- X \%*\%t_beta + f_t + t_sigma*error
colnames(y) <- "y"
data <- data.frame(y,X,t)
fit1 <- sglg(y ~ x1 + x2 - 1, npc = t, data = data, basis = "deBoor", alpha0 = 0.1)
logLik(fit1) # -288.1859 time: 90 milliseconds
quantile_sglg(fit1)
fit2 <- sglg(y ~ x1 + x2 - 1, npc=t, data=data, basis = "Gu", alpha0=0.005)
logLik(fit2)
#################################################
# An example with two non-parametric components #
#################################################
set.seed(2)
t_2 <- as.matrix(rnorm(rows, sd = 0.5))
colnames(t_2) <- 't_2'
f_t_2 <- exp(t_2)
error <- rglg(rows,0,1,t_lambda)
y_2 <- X \%*\%t_beta + f_t + f_t_2 + t_sigma*error
colnames(y_2) <- 'y_2'
data2 <- data.frame(y_2,X,t,t_2)
npcs <- cbind(t,t_2)
fit3 <- sglg(y_2 ~ x1 + x2 - 1, npc = npcs, data = data2, alpha0 = c(0.45,0.65))
logLik(fit3)
#############################################################################
}
\references{
Cardozo C.A.,  Paula G., and Vanegas L. (2022). Generalized log-gamma additive partial linear models with P-spline smoothing. Statistical Papers.
}
\author{
Carlos Alberto Cardozo Delgado <cardozorpackages@gmail.com>
}
