#include "scalablebayesm.h"
#include "cstdio"
using namespace arma;

time_t itime;
char buf[100];

void startMcmcTimer() {
  itime = time(NULL);
  Rcout << " MCMC Iteration (est time to end - min) \n";
}

// void infoMcmcTimer(int rep, int R) {
//   time_t ctime = time(NULL);    
//   char buf[32];
//   
//   double timetoend = difftime(ctime, itime) / 60.0 * (R - rep - 1) / (rep+1);
//   sprintf(buf, " %d (%.1f)\n", rep+1, timetoend);
//   Rcout <<  buf;
// }
// 
// void endMcmcTimer() {
//   time_t ctime = time(NULL);
//   char buf[32];
//   
//   sprintf(buf, " Total Time Elapsed: %.2f \n", difftime(ctime, itime) / 60.0);     
//   Rcout << buf;
//   
//   itime = 0;
// }

void infoMcmcTimer(int rep, int R) {
  time_t ctime = time(NULL);
  char buf[64]; // Increased buffer size for safety
  
  double timetoend = difftime(ctime, itime) / 60.0 * (R - rep - 1) / (rep + 1);
  std::snprintf(buf, sizeof(buf), " %d (%.1f)\n", rep + 1, timetoend);
  Rcout << buf;
}

void endMcmcTimer() {
  time_t ctime = time(NULL);
  char buf[64];
  
  std::snprintf(buf, sizeof(buf), " Total Time Elapsed: %.2f \n", difftime(ctime, itime) / 60.0);
  Rcout << buf;
  
  itime = 0; 
}

List rwishart(int const& nu, mat const& V){
  
  // Wayne Taylor 4/7/2015
  
  // Function to draw from Wishart (nu,V) and IW
  
  // W ~ W(nu,V)
  // E[W]=nuV
  
  // WI=W^-1
  // E[WI]=V^-1/(nu-m-1)
  
  // T has sqrt chisqs on diagonal and normals below diagonal
  int m = V.n_rows;
  mat T = zeros(m,m);
  
  for(int i = 0; i < m; i++) {
    T(i,i) = sqrt(rchisq(1,nu-i)[0]); //rchisq returns a vectorized object, so using [0] allows for the conversion to double
  }
  
  for(int j = 0; j < m; j++) {  
    for(int i = j+1; i < m; i++) {    
      T(i,j) = rnorm(1)[0]; //rnorm returns a NumericVector, so using [0] allows for conversion to double
    }}
  
  mat C = trans(T)*chol(V);
  mat CI = solve(trimatu(C),eye(m,m)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  
  // C is the upper triangular root of Wishart therefore, W=C'C
  // this is the LU decomposition Inv(W) = CICI' Note: this is
  // the UL decomp not LU!
  
  // W is Wishart draw, IW is W^-1
  
  return List::create(
    Named("W") = trans(C) * C,
    Named("IW") = CI * trans(CI),
    Named("C") = C,
    Named("CI") = CI);
}

List rmultireg(mat const& Y, mat const& X, mat const& Bbar, mat const& A, int nu, mat const& V) {
  
  // Keunwoo Kim 09/09/2014
  
  // Purpose: draw from posterior for Multivariate Regression Model with natural conjugate prior
  
  // Arguments:
  //  Y is n x m matrix
  //  X is n x k
  //  Bbar is the prior mean of regression coefficients  (k x m)
  //  A is prior precision matrix
  //  nu, V are parameters for prior on Sigma
  
  // Output: list of B, Sigma draws of matrix of coefficients and Sigma matrix
  
  // Model: 
  //  Y=XB+U  cov(u_i) = Sigma
  //  B is k x m matrix of coefficients
  
  // Prior:  
  //  beta|Sigma  ~ N(betabar,Sigma (x) A^-1)
  //  betabar=vec(Bbar)
  //  beta = vec(B) 
  //  Sigma ~ IW(nu,V) or Sigma^-1 ~ W(nu, V^-1)
  
  int n = Y.n_rows;
  int m = Y.n_cols;
  int k = X.n_cols;
  
  //first draw Sigma
  mat RA = chol(A);
  mat W = join_cols(X, RA); //analogous to rbind() in R
  mat Z = join_cols(Y, RA*Bbar);
  // note:  Y,X,A,Bbar must be matrices!
  mat IR = solve(trimatu(chol(trans(W)*W)), eye(k,k)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  // W'W = R'R  &  (W'W)^-1 = IRIR'  -- this is the UL decomp!
  mat Btilde = (IR*trans(IR)) * (trans(W)*Z);
  // IRIR'(W'Z) = (X'X+A)^-1(X'Y + ABbar)
  mat E = Z-W*Btilde;
  mat S = trans(E)*E;
  // E'E
  
  // compute the inverse of V+S
  mat ucholinv = solve(trimatu(chol(V+S)), eye(m,m));
  mat VSinv = ucholinv*trans(ucholinv);
  
  List rwout = rwishart(nu+n, VSinv);
  
  // now draw B given Sigma
  //   note beta ~ N(vec(Btilde),Sigma (x) Covxxa)
  //       Cov=(X'X + A)^-1  = IR t(IR)  
  //       Sigma=CICI'    
  //       therefore, cov(beta)= Omega = CICI' (x) IR IR' = (CI (x) IR) (CI (x) IR)'
  //  so to draw beta we do beta= vec(Btilde) +(CI (x) IR)vec(Z_mk)  
  //  		Z_mk is m x k matrix of N(0,1)
  //	since vec(ABC) = (C' (x) A)vec(B), we have 
  //		B = Btilde + IR Z_mk CI'
  
  mat CI = rwout["CI"]; //there is no need to use as<mat>(rwout["CI"]) since CI is being initiated as a mat in the same line
  mat draw = mat(rnorm(k*m));
  draw.reshape(k,m);
  mat B = Btilde + IR*draw*trans(CI);
  
  return List::create(
    Named("B") = B, 
    Named("Sigma") = rwout["IW"]);
}

List drawCompsFromLabels(mat const& y,  mat const& Bbar, mat const& A, int nu, mat const& V,  int ncomp,vec const& z){
  
  // Wayne Taylor 3/18/2015
  
  // Function to draw the components based on the z labels
  
  vec b, r, mu;
  mat yk, Xk, Ck, sigma, rooti, S, IW, CI;
  List temp, rw, comps(ncomp);
  
  int n = z.n_rows;
  vec nobincomp = zeros<vec>(ncomp);
  
  //Determine the number of observations in each component
  for(int i = 0; i<n; i++) {
    nobincomp[z[i]-1]++; //Note z starts at 1, not 0
  }
  
  //Draw comps
  for(int k = 0; k<ncomp; k++){
    
    if(nobincomp[k] > 0) {
      // If there are observations in this component, draw from the posterior
      
      yk = y.rows(find(z==(k+1))); //Note k starts at 0 and z starts at 1
      Xk = ones(nobincomp[k], 1);
      temp = rmultireg(yk, Xk, Bbar, A, nu, V);
      
      sigma = as<mat>(temp["Sigma"]); //conversion from Rcpp to Armadillo requires explict declaration of variable type using as<>
      rooti = solve(trimatu(chol(sigma)),eye(sigma.n_rows,sigma.n_cols)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
      
      mu = as<vec>(temp["B"]);
      
      comps(k) = List::create(
        Named("mu") = NumericVector(mu.begin(),mu.end()), //converts to a NumericVector, otherwise it will be interpretted as a matrix
        Named("rooti") = rooti
      );
      
    } else {
      // If there are no obervations in this component, draw from the prior
      S = solve(trimatu(chol(V)),eye(V.n_rows,V.n_cols));
      S = S * trans(S); 
      
      rw = rwishart(nu, S);
      
      IW = as<mat>(rw["IW"]);
      CI = as<mat>(rw["CI"]);
      
      rooti = solve(trimatu(chol(IW)),eye(IW.n_rows,IW.n_cols));        
      b = vectorise(Bbar);
      r = rnorm(b.n_rows,0,1);
      
      mu = b + (CI * r) / sqrt(A(0,0));
      
      comps(k) = List::create(
        Named("mu") = NumericVector(mu.begin(),mu.end()), //converts to a NumericVector, otherwise it will be interpretted as a matrix
        Named("rooti") = rooti);
    } 
  }
  
  return(comps);
}

vec drawLabelsFromComps(mat const& y, vec const& p, List comps) {
  
  // Wayne Taylor 3/18/2015
  
  // Function to determine which label is associated with each y value
  
  double logprod;
  vec mu, u;
  mat rooti;
  List compsk;
  
  int n = y.n_rows;
  vec res = zeros<vec>(n);
  int ncomp  = comps.size();
  mat prob(n,ncomp);
  
  for(int k = 0; k<ncomp; k++) {
    compsk = comps[k];
    mu = as<vec>(compsk["mu"]); //conversion from Rcpp to Armadillo requires explict declaration of variable type using as<>
    rooti = as<mat>(compsk["rooti"]);
    
    //Find log of MVN density using matrices
    logprod = log(prod(diagvec(rooti)));
    mat z(y);
    z.each_row() -= trans(mu); //subtracts mu from each row in z
    z = trans(rooti) * trans(z);
    z = -(y.n_cols/2.0) * log(2*M_PI) + logprod - .5 * sum(z % z, 0); // operator % performs element-wise multiplication
    
    prob.col(k) =  trans(z);
  }
  
  prob = exp(prob);
  prob.each_row() %= trans(p); //element-wise multiplication
  
  // Cumulatively add each row and take a uniform draw between 0 and the cumulative sum
  prob = cumsum(prob, 1);
  u = as<vec>(runif(n)) % prob.col(ncomp-1);
  
  // Evaluative each column of "prob" until the uniform draw is less than the cumulative value
  for(int i = 0; i<n; i++) {
    while(u[i] > prob(i, res[i]++));
  }
  
  return(res);
}

vec rdirichlet(vec const& alpha){
  
  // Wayne Taylor 4/7/2015
  
  // Purpose:
  // draw from Dirichlet(alpha)
  
  int dim = alpha.size();
  vec y = zeros<vec>(dim);
  
  for(int i = 0; i<dim; i++) {    
    y[i] = rgamma(1, alpha[i])[0]; //rgamma returns a NumericVector, so adding [0] extracts the first element and treats it as type "double"
  }
  
  return(y/sum(y));
}

vec drawPFromLabels1(vec const& a, vec const& z) {
  
  // Wayne Taylor 9/10/2014
  
  // Function to draw the probabilities based on the label proportions
  
  vec a2 = a;
  int n = z.n_rows;
  
  //Count number of observations in each component
  for(int i = 0; i<n; i++) a2[z[i]-1]++; //z starts at 1, not 0
  
  return rdirichlet(a2);
}

List rmixGibbs1(mat const& y,mat const& Bbar, mat const& A, int nu, mat const& V, vec const& a, vec const& p, vec const& z) {
  
  // Wayne Taylor 9/10/2014
  
  /*
   // Revision History: R. McCulloch 11/04 P. Rossi 3/05 put in
   // backsolve and improved documentation
   // 
   // purpose: do gibbs sampling inference for a mixture of
   // multivariate normals
   // 
   // arguments: y: data, rows are observations, assumed to be iid
   // draws from normal mixture Bbar,A,nu,V: common prior for mean
   // and variance of each normal component
   // 
   // note: Bbar should be a matrix. usually with only one row
   // 
   // beta ~ N(betabar,Sigma (x) A^-1) betabar=vec(Bbar) Sigma ~
   // IW(nu,V) or Sigma^-1 ~ W(nu, V^-1) note: if you want Sigma ~
   // A, use nu big and rwishart(nu,nu(A)^{-1})$IW a: Dirichlet
   // parameters for prior on p p: prior probabilities of normal
   // components z: components indentities for each observation
   // (vector of intergers each in {1,2,...number of components})
   // comps: list, each member is a list comp with ith normal
   // component ~N(comp[[1]],Sigma), Sigma = t(R)%*%R, R^{-1} =
   // comp[[2]] Output: list with elements [[1]=$p, [[2]]=$z, and
   // [[3]]=$comps, with the updated values
   
   */
  
  List comps = drawCompsFromLabels(y, Bbar, A, nu, V, a.size(), z);
  
  vec z2 = drawLabelsFromComps(y, p, comps);
  
  vec p2 = drawPFromLabels1(a, z2);
  
  return List::create(
    Named("p") = p2,
    Named("z") = z2,
    Named("comps") = comps);
}


double llmnl(vec const& beta, vec const& y, mat const& X){
  
  // Wayne Taylor 9/7/2014
  
  // Evaluates log-likelihood for the multinomial logit model
  
  int n = y.size();
  int j = X.n_rows/n;
  mat Xbeta = X*beta;
  
  vec xby = zeros<vec>(n);
  vec denom = zeros<vec>(n);
  
  for(int i = 0; i<n;i++){      
    for(int p=0;p<j;p++) denom[i]=denom[i]+exp(Xbeta[i*j+p]);
    xby[i] = Xbeta[i*j+y[i]-1];
  }
  
  return(sum(xby - log(denom)));
}

double lndMvn(vec const& x, vec const& mu, mat const& rooti){
  
  //Wayne Taylor 9/7/2014
  
  // function to evaluate log of MV Normal density with  mean mu, var Sigma
  // Sigma=t(root)%*%root   (root is upper tri cholesky root)
  // Sigma^-1=rooti%*%t(rooti)   
  // rooti is in the inverse of upper triangular chol root of sigma
  //          note: this is the UL decomp of sigmai not LU!
  //                Sigma=root'root   root=inv(rooti)
  
  vec z = vectorise(trans(rooti)*(x-mu));
  
  return((-(x.size()/2.0)*log(2*M_PI) -.5*(trans(z)*z) + sum(log(diagvec(rooti))))[0]);
}

//Used in rhierMnlDP and rhierMnlRwMixture------------------------------------------------------------------------
mnlMetropOnceOut mnlMetropOnce(arma::vec const& y, arma::mat const& X, arma::vec const& oldbeta, 
                               double oldll,double s, arma::mat const& incroot, 
                               arma::vec const& betabar, arma::mat const& rootpi){ 
  // Wayne Taylor 10/01/2014
  
  // function to execute rw metropolis for the MNL
  // y is n vector with element = 1,...,j indicating which alt chosen
  // X is nj x k matrix of xvalues for each of j alt on each of n occasions
  // RW increments are N(0,s^2*t(inc.root)%*%inc.root)
  // prior on beta is N(betabar,Sigma)  Sigma^-1=rootpi*t(rootpi)
  //  inc.root, rootpi are upper triangular
  //  this means that we are using the UL decomp of Sigma^-1 for prior 
  // oldbeta is the current
  
  
  mnlMetropOnceOut metropout_struct;
  
  double unif;
  vec betadraw, alphaminv;
  
  int stay = 0;
  vec betac = oldbeta + s*trans(incroot)*as<vec>(rnorm(X.n_cols));
  double cll = llmnl(betac,y,X);
  double clpost = cll+lndMvn(betac,betabar,rootpi);
  double ldiff = clpost-oldll-lndMvn(oldbeta,betabar,rootpi);
  //alphaminv << 1 << exp(ldiff);
  
  alphaminv = {1, exp(ldiff)};
  
  double alpha = min(alphaminv);
  
  if(alpha < 1) {
    unif = runif(1)[0]; //runif returns a NumericVector, so using [0] allows for conversion to double
  } else { 
    unif=0;}
  if (unif <= alpha) {
    betadraw = betac;
    oldll = cll;
  } else {
    betadraw = oldbeta;
    stay = 1;
  }
  
  metropout_struct.betadraw = betadraw;
  metropout_struct.stay = stay;  
  metropout_struct.oldll = oldll;
  
  return (metropout_struct);
}

mat drawDelta1(mat const& x,mat const& y,vec const& z,List const& comps,vec const& deltabar,mat const& Ad){
  
  // Wayne Taylor 10/01/2014
  
  // delta = vec(D)
  //  given z and comps (z[i] gives component indicator for the ith observation, 
  //   comps is a list of mu and rooti)
  // y is n x p
  // x is n x k
  // y = xD' + U , rows of U are indep with covs Sigma_i given by z and comps
  
  int p = y.n_cols;
  int k = x.n_cols;
  int ncomp  = comps.length();
  mat xtx = zeros<mat>(k*p,k*p);
  mat xty = zeros<mat>(p,k); //this is the unvecced version, reshaped after the sum
  
  //Create the index vectors, the colAll vectors are equal to span::all but with uvecs (as required by .submat)
  uvec colAlly(p), colAllx(k);
  for(int i = 0; i<p; i++) colAlly(i) = i;
  for(int i = 0; i<k; i++) colAllx(i) = i;
  
  //Loop through the components
  for(int compi = 0; compi<ncomp; compi++){
    
    //Create an index vector ind, to be used like y[ind,]
    uvec ind = find(z == (compi+1));
    
    //If there are observations in this component
    if(ind.size()>0){
      mat yi = y.submat(ind,colAlly);
      mat xi = x.submat(ind,colAllx);
      
      List compsi = comps[compi];
      rowvec mui = as<rowvec>(compsi[0]); //conversion from Rcpp to Armadillo requires explict declaration of variable type using as<>
      mat rootii = trimatu(as<mat>(compsi[1])); //trimatu interprets the matrix as upper triangular
      yi.each_row() -= mui; //subtracts mui from each row of yi
      mat sigi = rootii*trans(rootii);
      xtx = xtx + kron(trans(xi)*xi,sigi);
      xty = xty + (sigi * (trans(yi)*xi));
    }
  }
  xty.reshape(xty.n_rows*xty.n_cols,1);
  
  //vec(t(D)) ~ N(V^{-1}(xty + Ad*deltabar),V^{-1}) where V = (xtx+Ad)
  // compute the inverse of xtx+Ad
  //mat ucholinv = solve(trimatu(chol(xtx+Ad)), eye(k*p,k*p)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  mat ucholinv = solve(chol(xtx+Ad), eye(k*p,k*p)); //Rico: trimatu may cause chol warning that matrix is not symmetric
  mat Vinv = ucholinv*trans(ucholinv);
  
  return(Vinv*(xty+Ad*deltabar) + trans(chol(Vinv))*as<vec>(rnorm(deltabar.size())));
}

mat drawDeltaDP(mat const& x,mat const& y,ivec const& z,std::vector<murooti> const& comps_vector,vec const& deltabar,mat const& Ad){
  
  // Wayne Taylor 2/21/2015
  
  // delta = vec(D)
  //  given z and comps (z[i] gives component indicator for the ith observation, 
  //   comps is a list of mu and rooti)
  // y is n x p
  // x is n x k
  // y = xD' + U , rows of U are indep with covs Sigma_i given by z and comps
  
  int p = y.n_cols;
  int k = x.n_cols;
  int ncomp  = comps_vector.size();
  mat xtx = zeros<mat>(k*p,k*p);
  mat xty = zeros<mat>(p,k); //this is the unvecced version, reshaped after the sum
  
  //Create the index vectors, the colAll vectors are equal to span::all but with uvecs (as required by .submat)
  uvec colAlly(p), colAllx(k);
  for(int i = 0; i<p; i++) colAlly(i) = i;
  for(int i = 0; i<k; i++) colAllx(i) = i;
  
  //Loop through the components
  for(int compi = 0; compi<ncomp; compi++){
    
    //Create an index vector ind, to be used like y[ind,]
    uvec ind = find(z == (compi+1));
    
    //If there are observations in this component
    if(ind.size()>0){
      mat yi = y.submat(ind,colAlly);
      mat xi = x.submat(ind,colAllx);
      
      murooti compsi_struct = comps_vector[compi];
      yi.each_row() -= trans(compsi_struct.mu); //the subtraction operation is repeated on each row of yi
      mat sigi = compsi_struct.rooti*trans(compsi_struct.rooti);
      xtx = xtx + kron(trans(xi)*xi,sigi);
      xty = xty + (sigi * (trans(yi)*xi));
    }
  }
  xty.reshape(xty.n_rows*xty.n_cols,1);
  
  //vec(t(D)) ~ N(V^{-1}(xty + Ad*deltabar),V^{-1}) where V = (xtx+Ad)
  // compute the inverse of xtx+Ad
  mat ucholinv = solve(trimatu(chol(xtx+Ad)), eye(k*p,k*p)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  mat Vinv = ucholinv*trans(ucholinv);
  
  return(Vinv*(xty+Ad*deltabar) + trans(chol(Vinv))*as<vec>(rnorm(deltabar.size())));
}

vec q01(mat const& y, lambda const& lambda_struct){
  
  // Wayne Taylor 2/4/2015
  
  // function to compute a vector of int f(y[i]|theta) p(theta|lambda)dlambda
  // here p(theta|lambda) is G0 the base prior
  
  // implemented for a multivariate normal data density and standard conjugate prior:
  //  theta=list(mu,Sigma)
  //  f(y|theta,eta) is N(mu,Sigma)
  //  lambda=list(mubar,Amu,nu,V)
  //    mu|Sigma ~ N(mubar,Sigma (x) Amu^-1)
  //    Sigma ~ IW(nu,V)
  
  // arguments:
  //  Y is n x k matrix of observations
  //  lambda=list(mubar,Amu,nu,V)
  
  // output:
  //  vector of q01 values for each obs (row of Y)
  
  // p. rossi 12/05
  //  here y is matrix of observations (each row is an obs)
  
  int k = y.n_cols;
  mat R = chol(lambda_struct.V);
  double logdetR = sum(log(R.diag()));
  double lnk1k2, constant;
  mat transy, m, vivi, lnq0v;
  
  if (k > 1) {
    vec km1(k-1); for(int i = 0; i < (k-1); i++) km1[i] = i+1; //vector of 1:k SEE SEQ_ALONG
    lnk1k2 = (k/2.0)*log(2)+log((lambda_struct.nu-k)/2)+lgamma((lambda_struct.nu-k)/2)-lgamma(lambda_struct.nu/2)+sum(log(lambda_struct.nu/2-km1/2));
  } else {
    lnk1k2 = (k/2.0)*log(2)+log((lambda_struct.nu-k)/2)+lgamma((lambda_struct.nu-k)/2)-lgamma(lambda_struct.nu/2);
  }
  
  constant = -(k/2.0)*log(2*M_PI)+(k/2.0)*log(lambda_struct.Amu/(1+lambda_struct.Amu)) + lnk1k2 + lambda_struct.nu*logdetR;
  
  // note: here we are using the fact that |V + S_i | = |R|^2 (1 + v_i'v_i)
  //  where v_i = sqrt(Amu/(1+Amu))*t(R^-1)*(y_i-mubar), R is chol(V)
  //  and S_i = Amu/(1+Amu) * (y_i-mubar)(y_i-mubar)'
  
  transy = trans(y);
  transy.each_col() -= lambda_struct.mubar;
  
  //m = sqrt(lambda_struct.Amu/(1+lambda_struct.Amu))*trans(solve(trimatu(R),eye(y.n_cols,y.n_cols)))*transy; //trimatu interprets the matrix as upper triangular and makes solve more efficient
  m = sqrt(lambda_struct.Amu/(1+lambda_struct.Amu))*trans(solve(R,eye(y.n_cols,y.n_cols)))*transy; //Rico: trimatu may cause chol warning that matrix is not symmetric
  vivi = sum(square(m),0);
  
  lnq0v = constant - ((lambda_struct.nu+1)/2)*(2*logdetR+log(1+vivi));
  
  return(trans(exp(lnq0v)));
}

mat yden(std::vector<murooti> const& thetaStar_vector, mat const& y){
  
  // Wayne Taylor 2/4/2015
  
  // function to compute f(y | theta) 
  // computes f for all values of theta in theta list of lists
  
  // arguments:
  //  thetaStar is a list of lists.  thetaStar[[i]] is a list with components, mu, rooti
  //  y |theta[[i]] ~ N(mu,(rooti %*% t(rooti))^-1)  rooti is inverse of Chol root of Sigma
  
  // output:
  //  length(thetaStar) x n array of values of f(y[j,]|thetaStar[[i]]
  
  int nunique = thetaStar_vector.size();
  int n = y.n_rows;
  int k = y.n_cols;
  mat ydenmat = zeros<mat>(nunique,n);
  
  vec mu;
  mat rooti, transy, quads;
  
  for(int i = 0; i < nunique; i++){
    //now compute vectorized version of lndMvn 
    //compute y_i'RIRI'y_i for all i
    
    mu = thetaStar_vector[i].mu;
    rooti = thetaStar_vector[i].rooti;
    
    transy = trans(y);
    transy.each_col() -= mu; //column-wise subtraction
    
    quads = sum(square(trans(rooti) * transy),0); //same as colSums
    ydenmat(i,span::all) = exp(-(k/2.0)*log(2*M_PI) + sum(log(rooti.diag())) - .5*quads);
  }
  
  return(ydenmat);
}


ivec numcomp(ivec const& indic, int k){
  
  // Wayne Taylor 1/28/2015
  
  //find the number of times each of k integers is in the vector indic
  ivec ncomp(k);
  
  for(int comp = 0; comp < k; comp++){
    ncomp[comp]=sum(indic == (comp+1));
  }
  
  return(ncomp);
}

int rmultinomF(vec const& p){
  
  // Wayne Taylor 1/28/2015
  
  vec csp = cumsum(p);
  double rnd = runif(1)[0]; //runif returns a NumericVector, so using [0] allows for conversion to double
  int res = 0;
  int psize = p.size();
  
  for(int i = 0; i < psize; i++){
    if(rnd > csp[i]) res = res+1;
  }
  
  return(res+1);
}

murooti thetaD1(mat const& y, lambda const& lambda_struct){
  
  // Wayne Taylor 2/4/2015
  
  // function to draw from posterior of theta given data y and base prior G0(lambda)
  
  // here y ~ N(mu,Sigma)
  // theta = list(mu=mu,rooti=chol(Sigma)^-1)
  // mu|Sigma ~ N(mubar,Sigma (x) Amu-1)
  // Sigma ~ IW(nu,V)
  
  // arguments: 
  //  y is n x k matrix of obs
  //  lambda is list(mubar,Amu,nu,V)
  
  // output:
  //  one draw of theta, list(mu,rooti)
  //  Sigma=inv(rooti)%*%t(inv(rooti))
  
  // note: we assume that y is a matrix. if there is only one obs, y is a 1 x k matrix
  mat X = ones<mat>(y.n_rows,1);
  mat A(1,1); A.fill(lambda_struct.Amu);
  List rout = rmultireg(y,X,trans(lambda_struct.mubar),A,lambda_struct.nu,lambda_struct.V);
  murooti out_struct;
  out_struct.mu = as<vec>(rout["B"]); //conversion from Rcpp to Armadillo requires explict declaration of variable type using as<>
  //    out_struct.rooti = solve(chol(trimatu(as<mat>(rout["Sigma"]))),eye(y.n_cols,y.n_cols)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  out_struct.rooti = solve(chol(as<mat>(rout["Sigma"])),eye(y.n_cols,y.n_cols)); //Rico: trimatu causes chol warning that matrix is not symmetric
  return(out_struct);
}

thetaStarIndex thetaStarDraw(ivec indic, std::vector<murooti> thetaStar_vector, mat const& y, mat ydenmat, vec const& q0v, double alpha, 
                             lambda const& lambda_struct, int maxuniq) {
  
  // Wayne Taylor 2/4/2015
  
  // indic is n x 1 vector of indicator of which of thetaStar is assigned to each observation
  // thetaStar is list of the current components (some of which may never be used)
  // y is n x d matrix of observations
  // ydenmat is maxuniq x n matrix to store density evaluations - we assume first 
  // length(Thetastar) rows are filled in with density evals
  // q0v is vector of bayes factors for new component and each observation
  // alpha is DP process tightness prior
  // lambda is list of priors for the base DP process measure
  // maxuniq maximum number of mixture components
  // yden is function to fill out an array
  // thetaD1 is function to draw theta from posterior of theta given y and G0
  
  int n = indic.size();
  ivec ncomp, indicC;
  int k, inc, cntNonzero;
  std::vector<murooti> listofone_vector(1);
  std::vector<murooti> thetaStarC_vector;
  
  //draw theta_i given theta_-i
  for(int i = 0; i<n; i++){
    
    k = thetaStar_vector.size();
    vec probs(k+1);
    probs[k] = q0v[i]*(alpha/(alpha+(n-1)));
    
    //same as to indicmi = indic[-i]
    ivec indicmi = zeros<ivec>(n-1);
    inc = 0;
    for(int j = 0; j<(n-1); j++){
      if(j == i) {inc = inc + 1;}
      indicmi[j] = indic[inc];
      inc = inc+1;
    }
    ncomp = numcomp(indicmi,k);
    
    for(int comp = 0; comp<k; comp++){
      probs[comp] = ydenmat(comp,i)*ncomp[comp]/(alpha+(n-1));
    }
    probs = probs/sum(probs);
    indic[i] = rmultinomF(probs);
    
    if(indic[i] == (k+1)){
      if((k+1) > maxuniq) {
        //cout<<"thetaStarDraw: max number of comps exceeded: k+1 = "<<k+1<<" > "<<maxuniq<<" = maxuniq\n"; 
        Rcout << "thetaStarDraw: max number of comps exceeded: k+1 = "
              << k + 1 << " > " << maxuniq << " = maxuniq\n";
        stop("max number of comps exceeded");
      } else {
        listofone_vector[0] = thetaD1(y(i,span::all),lambda_struct);
        thetaStar_vector.push_back(listofone_vector[0]);
        ydenmat(k,span::all) = yden(listofone_vector,y);
      }}
  }
  //clean out thetaStar of any components which have zero observations associated with them
  //and re-write indic vector 
  k = thetaStar_vector.size();
  indicC = zeros<ivec>(n);
  ncomp = numcomp(indic,k);
  cntNonzero = 0;
  for(int comp = 0; comp<k; comp++){
    if(ncomp[comp] != 0){
      thetaStarC_vector.push_back(thetaStar_vector[comp]);
      cntNonzero=cntNonzero+1;
      for(int i = 0; i<n; i++){if(indic[i] == (comp+1)) indicC[i] = cntNonzero;} //same as indicC(indic==comp) = cntNonzero;
    }
  }
  thetaStarIndex out_struct;
  out_struct.indic = indicC;
  out_struct.thetaStar_vector = thetaStarC_vector;
  return(out_struct);
}

murooti GD1(lambda const& lambda_struct){
  
  // Wayne Taylor 2/4/2015
  
  // function to draw from prior for Multivariate Normal Model
  
  // mu|Sigma ~ N(mubar,Sigma x Amu^-1)
  // Sigma ~ IW(nu,V)
  
  // note: we must insure that mu is a vector to use most efficient lndMvn routine
  
  int k = lambda_struct.mubar.size();
  
  //List Rout = rwishart(lambda_struct.nu,solve(trimatu(lambda_struct.V),eye(k,k))); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  List Rout = rwishart(lambda_struct.nu,solve(lambda_struct.V,eye(k,k))); //Rico: trimatu may cause chol warning that matrix is not symmetric
  mat Sigma = as<mat>(Rout["IW"]); //conversion from Rcpp to Armadillo requires explict declaration of variable type using as<>
  mat root = chol(Sigma);
  mat draws = rnorm(k);
  mat mu = lambda_struct.mubar + (1/sqrt(lambda_struct.Amu))*trans(root)*draws;
  
  murooti out_struct;
  out_struct.mu = mu;
  //out_struct.rooti = solve(trimatu(root),eye(k,k)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  out_struct.rooti = solve(root,eye(k,k)); //Rico: trimatu may cause chol warning that matrix is not symmetric
  return(out_struct);
}

vec seq_rcpp(double from, double to, int len){
  
  // Wayne Taylor 1/28/2015
  
  // Same as R::seq()
  
  vec res(len);
  res[len-1] = to; res[0] = from; //note the order of these two statements is important, when gridsize = 1 res[0] will be rewritten to the correct number
  double increment = (res[len-1]-res[0])/(len-1);
  for(int i = 1; i<(len-1); i++) res[i] = res[i-1] + increment;
  return(res);
}


double alphaD(priorAlpha const& priorAlpha_struct, int Istar, int gridsize){
  
  // Wayne Taylor 2/4/2015
  
  // function to draw alpha using prior, p(alpha)= (1-(alpha-alphamin)/(alphamax-alphamin))**power
  
  //same as seq
  vec alpha = seq_rcpp(priorAlpha_struct.alphamin,priorAlpha_struct.alphamax-.000001,gridsize);
  
  vec lnprob(gridsize);
  for(int i = 0; i<gridsize; i++){
    lnprob[i] = Istar*log(alpha[i]) + lgamma(alpha[i]) - lgamma(priorAlpha_struct.n+alpha[i]) + priorAlpha_struct.power*log(1-(alpha[i]-priorAlpha_struct.alphamin)/(priorAlpha_struct.alphamax-priorAlpha_struct.alphamin));
  }
  
  lnprob = lnprob - median(lnprob);
  vec probs=exp(lnprob);
  probs=probs/sum(probs);
  
  return(alpha(rmultinomF(probs)-1));
}

lambda lambdaD(lambda const& lambda_struct, std::vector<murooti> const& thetaStar_vector, vec const& alim, vec const& nulim, vec const& vlim, int gridsize){
  
  // Wayne Taylor 2/4/2015
  
  // revision history
  //  p. rossi 7/06
  //  vectorized 1/07
  //  changed 2/08 to paramaterize V matrix of IW prior to nu*v*I; then mode of Sigma=nu/(nu+2)vI
  //    this means that we have a reparameterization to v* = nu*v
  
  // function to draw (nu, v, a) using uniform priors
  
  // theta_j=(mu_j,Sigma_j)  mu_j~N(0,Sigma_j/a)  Sigma_j~IW(nu,vI)
  //  recall E[Sigma]= vI/(nu-dim-1)
  
  vec lnprob, probs, rowSumslgammaarg;
  int ind; //placeholder for matrix indexing
  murooti thetaStari_struct; mat rootii; vec mui;
  mat mout, rimu, arg, lgammaarg;
  double sumdiagriri, sumlogdiag, sumquads, adraw, nudraw, vdraw;
  
  murooti thetaStar0_struct = thetaStar_vector[0];
  int d = thetaStar0_struct.mu.size();
  int Istar = thetaStar_vector.size();
  
  vec aseq = seq_rcpp(alim[0],alim[1],gridsize);
  vec nuseq = d-1+exp(seq_rcpp(nulim[0],nulim[1],gridsize)); //log uniform grid
  vec vseq = seq_rcpp(vlim[0],vlim[1],gridsize);
  
  // "brute" force approach would simply loop over the 
  //  "observations" (theta_j) and use log of the appropriate densities.  To vectorize, we
  // notice that the "data" comes via various statistics:
  //  1. sum of log(diag(rooti_j)
  //  2. sum of tr(V%*%rooti_j%*%t(rooti_j)) where V=vI_d
  //  3. quadratic form t(mu_j-0)%*%rooti%*%t(rooti)%*%(mu_j-0)
  // thus, we will compute these first.
  // for documentation purposes, we leave brute force code in comment fields
  
  // extract needed info from thetastar list
  
  //mout has the rootis in form: [t(rooti_1), t(rooti_2), ...,t(rooti_Istar)]
  mout = zeros<mat>(d,Istar*d);
  ind = 0;
  for(int i = 0; i < Istar; i++){
    thetaStari_struct = thetaStar_vector[i];
    rootii = thetaStari_struct.rooti;
    ind = i*d;
    mout.submat(0, ind,d-1,ind+d-1) = trans(rootii);
  }
  sumdiagriri = sum(sum(square(mout),0)); //sum_i trace(rooti_i*trans(rooti_i))
  
  // now get diagonals of rooti
  sumlogdiag = 0.0;
  for(int i = 0; i < Istar; i++){
    ind = i*d;
    for(int j = 0; j < d; j++){
      sumlogdiag = sumlogdiag+log(mout(j,ind+j));
    }
  }
  
  //columns of rimu contain trans(rooti_i)*mu_i
  rimu = zeros<mat>(d,Istar);
  for(int i = 0; i < Istar; i++){
    thetaStari_struct = thetaStar_vector[i];
    mui = thetaStari_struct.mu;
    rootii = thetaStari_struct.rooti;
    rimu(span::all,i) = trans(rootii) * mui;
  }
  sumquads = sum(sum(square(rimu),0));
  
  // draw a  (conditionally indep of nu,v given theta_j)
  lnprob = zeros<vec>(aseq.size());
  // for(i in seq(along=aseq)){
  // for(j in seq(along=thetastar)){
  // lnprob[i]=lnprob[i]+lndMvn(thetastar[[j]]$mu,c(rep(0,d)),thetastar[[j]]$rooti*sqrt(aseq[i]))}
  lnprob = Istar*(-(d/2.0)*log(2*M_PI))-.5*aseq*sumquads+Istar*d*log(sqrt(aseq))+sumlogdiag;
  lnprob = lnprob-max(lnprob) + 200;
  probs = exp(lnprob);
  probs = probs/sum(probs);
  adraw = aseq[rmultinomF(probs)-1];
  
  // draw nu given v
  
  lnprob = zeros<vec>(nuseq.size());
  // for(i in seq(along=nuseq)){
  // for(j in seq(along=thetastar)){
  // Sigma_j=crossprod(backsolve(thetastar[[j]]$rooti,diag(d)))
  // lnprob[i]=lnprob[i]+lndIWishart(nuseq[i],V,Sigma_j)}
  
  //same as arg = (nuseq+1-arg)/2.0;
  arg = zeros<mat>(gridsize,d);
  for(int i = 0; i < d; i++) {
    vec indvec(gridsize);
    indvec.fill(-(i+1)+1);
    arg(span::all,i) = indvec;
  }
  arg.each_col() += nuseq;
  arg = arg/2.0;
  
  lgammaarg = zeros<mat>(gridsize,d);
  for(int i = 0; i < gridsize; i++){
    for(int j = 0; j < d; j++){
      lgammaarg(i,j) = lgamma(arg(i,j));
    }}
  rowSumslgammaarg = sum(lgammaarg,1);
  
  lnprob = zeros<vec>(gridsize);
  for(int i = 0; i<gridsize; i++){
    lnprob[i] = -Istar*log(2)*d/2.0*nuseq[i] - Istar*rowSumslgammaarg[i] + Istar*d*log(sqrt(lambda_struct.V(0,0)))*nuseq[i] + sumlogdiag*nuseq[i];
  }
  
  lnprob = lnprob-max(lnprob)+200;
  probs = exp(lnprob);
  probs = probs/sum(probs);
  nudraw = nuseq[rmultinomF(probs)-1];
  
  // draw v given nu 
  
  lnprob = zeros<vec>(vseq.size());
  // for(i in seq(along=vseq)){
  // V=vseq[i]*diag(d)
  // for(j in seq(along=thetastar)){
  // Sigma_j=crossprod(backsolve(thetastar[[j]]$rooti,diag(d)))
  // lnprob[i]=lnprob[i]+lndIWishart(nudraw,V,Sigma_j)}
  // lnprob=Istar*nudraw*d*log(sqrt(vseq))-.5*sumdiagriri*vseq
  
  lnprob = Istar*nudraw*d*log(sqrt(vseq*nudraw))-.5*sumdiagriri*vseq*nudraw;
  lnprob = lnprob-max(lnprob)+200;
  probs = exp(lnprob);
  probs = probs/sum(probs);
  vdraw = vseq[rmultinomF(probs)-1];
  
  // put back into lambda
  lambda out_struct;
  out_struct.mubar = zeros<vec>(d);
  out_struct.Amu = adraw;
  out_struct.nu = nudraw;
  out_struct.V = nudraw*vdraw*eye(d,d);
  
  return(out_struct);
}

DPOut rDPGibbs1(mat y, lambda lambda_struct, std::vector<murooti> thetaStar_vector, int maxuniq, ivec indic, 
                vec q0v, double alpha, priorAlpha const& priorAlpha_struct, int gridsize, List const& lambda_hyper){
  

  // Wayne Taylor 2/21/2015
  
  //revision history:
  //created from rDPGibbs by Rossi 3/08
  
  //do one draw of DP Gibbs sampler with normal base
  
  //Model:
  //  y_i ~ N(y|thetai)
  //  thetai|G ~ G
  //  G|lambda,alpha ~ DP(G|G0(lambda),alpha)
  
  //Priors:
  //  alpha: starting value
  //  lambda:
  //    G0 ~ N(mubar,Sigma (x) Amu^-1)
  //    mubar=vec(mubar)
  //    Sigma ~ IW(nu,nu*V) V=v*I  note: mode(Sigma)=nu/(nu+2)*v*I
  //    mubar=0
  //    amu is uniform on grid specified by alim
  //    nu is log uniform, nu=d-1+exp(Z) z is uniform on seq defined bvy nulim
  //    v is uniform on sequence specificd by vlim
  
  //  priorAlpha_struct:
  //    alpha ~ (1-(alpha-alphamin)/(alphamax-alphamin))^power
  //    alphamin=exp(digamma(Istarmin)-log(gamma+log(N)))
  //    alphamax=exp(digamma(Istarmax)-log(gamma+log(N)))
  //    gamma= .5772156649015328606
  
  //output:
  //  ind - vector of indicators for which observations are associated with which comp in thetaStar
  //  thetaStar - list of unique normal component parms
  //  lambda  - list of of (a,nu,V)
  //  alpha 
  //  thetaNp1 - one draw from predictive given thetaStar, lambda,alphama
  
  int n = y.n_rows; //n
  int dimy = y.n_cols; //nvar
  int nunique, indsize, indp, probssize;
  vec probs;
  uvec ind;
  mat ydenmat;
  uvec spanall(dimy); for(int i = 0; i<dimy ; i++) spanall[i] = i; //creates a uvec of [0,1,...,dimy-1]
  thetaStarIndex thetaStarDrawOut_struct;
  std::vector<murooti> new_utheta_vector(1), thetaNp1_vector(1);
  murooti thetaNp10_struct, outGD_struct;
  

  for(int rep = 0; rep<1; rep++) { //note we only do one loop!
    q0v = q01(y,lambda_struct);
    nunique = thetaStar_vector.size();
    if(nunique > maxuniq) stop("maximum number of unique thetas exceeded");
    //ydenmat is a length(thetaStar) x n array of density values given f(y[j,] | thetaStar[[i]]
    //  note: due to remix step (below) we must recompute ydenmat each time!
    ydenmat = zeros<mat>(maxuniq,n);
    
    ydenmat(span(0,nunique-1),span::all) = yden(thetaStar_vector,y);
    thetaStarDrawOut_struct = thetaStarDraw(indic, thetaStar_vector, y, ydenmat, q0v, alpha, lambda_struct, maxuniq);
    thetaStar_vector = thetaStarDrawOut_struct.thetaStar_vector;
    indic = thetaStarDrawOut_struct.indic;
    nunique = thetaStar_vector.size();
    
    
    //thetaNp1 and remix
    probs = zeros<vec>(nunique+1);
    for(int j = 0; j < nunique; j++){
      ind = find(indic == (j+1));
      indsize = ind.size();
      probs[j] = indsize/(alpha + n + 0.0);
      new_utheta_vector[0] = thetaD1(y(ind,spanall),lambda_struct);
      thetaStar_vector[j] = new_utheta_vector[0];
    }
    
    probs[nunique] = alpha/(alpha+n+0.0);
    indp = rmultinomF(probs);
    probssize = probs.size();
    if(indp == probssize) {
      outGD_struct = GD1(lambda_struct);
      thetaNp10_struct.mu = outGD_struct.mu;
      thetaNp10_struct.rooti = outGD_struct.rooti;
      thetaNp1_vector[0] = thetaNp10_struct;
    } else {
      outGD_struct = thetaStar_vector[indp-1];
      thetaNp10_struct.mu = outGD_struct.mu;
      thetaNp10_struct.rooti = outGD_struct.rooti;
      thetaNp1_vector[0] = thetaNp10_struct;
    }
    //draw alpha
    alpha = alphaD(priorAlpha_struct,nunique,gridsize);
    //draw lambda
    lambda_struct = lambdaD(lambda_struct,thetaStar_vector,lambda_hyper["alim"],lambda_hyper["nulim"],lambda_hyper["vlim"],gridsize);
  }
  //note indic is the vector of indicators for each obs correspond to which thetaStar
  DPOut out_struct;
  out_struct.thetaStar_vector = thetaStar_vector;
  out_struct.thetaNp1_vector = thetaNp1_vector;
  out_struct.alpha = alpha;
  out_struct.lambda_struct = lambda_struct;
  out_struct.indic = indic;
  return(out_struct);
}

DPOut rHDPGibbs(mat y, lambda lambda_struct, std::vector<murooti> thetaStar_vector, int maxuniq, ivec indic, 
                vec q0v, double alpha, priorAlpha const& priorAlpha_struct, int gridsize, List const& lambda_hyper){
  
  
  // Wayne Taylor 2/21/2015
  
  //revision history:
  //created from rDPGibbs by Rossi 3/08
  
  //do one draw of DP Gibbs sampler with normal base
  
  //Model:
  //  y_i ~ N(y|thetai)
  //  thetai|G ~ G
  //  G|lambda,alpha ~ DP(G|G0(lambda),alpha)
  
  //Priors:
  //  alpha: starting value
  //  lambda:
  //    G0 ~ N(mubar,Sigma (x) Amu^-1)
  //    mubar=vec(mubar)
  //    Sigma ~ IW(nu,nu*V) V=v*I  note: mode(Sigma)=nu/(nu+2)*v*I
  //    mubar=0
  //    amu is uniform on grid specified by alim
  //    nu is log uniform, nu=d-1+exp(Z) z is uniform on seq defined bvy nulim
  //    v is uniform on sequence specificd by vlim
  
  //  priorAlpha_struct:
  //    alpha ~ (1-(alpha-alphamin)/(alphamax-alphamin))^power
  //    alphamin=exp(digamma(Istarmin)-log(gamma+log(N)))
  //    alphamax=exp(digamma(Istarmax)-log(gamma+log(N)))
  //    gamma= .5772156649015328606
  
  //output:
  //  ind - vector of indicators for which observations are associated with which comp in thetaStar
  //  thetaStar - list of unique normal component parms
  //  lambda  - list of of (a,nu,V)
  //  alpha 
  //  thetaNp1 - one draw from predictive given thetaStar, lambda,alphama
  
  int n = y.n_rows; //n
  int dimy = y.n_cols; //nvar
  int nunique, indsize, indp, probssize;
  vec probs;
  uvec ind;
  mat ydenmat;
  uvec spanall(dimy); for(int i = 0; i<dimy ; i++) spanall[i] = i; //creates a uvec of [0,1,...,dimy-1]
  thetaStarIndex thetaStarDrawOut_struct;
  std::vector<murooti> new_utheta_vector(1), thetaNp1_vector(1);
  murooti thetaNp10_struct, outGD_struct;
  
  
  for(int rep = 0; rep<1; rep++) { //note we only do one loop!
    q0v = q01(y,lambda_struct);
    nunique = thetaStar_vector.size();
    if(nunique > maxuniq) stop("maximum number of unique thetas exceeded");
    //ydenmat is a length(thetaStar) x n array of density values given f(y[j,] | thetaStar[[i]]
    //  note: due to remix step (below) we must recompute ydenmat each time!
    ydenmat = zeros<mat>(maxuniq,n);
    
    ydenmat(span(0,nunique-1),span::all) = yden(thetaStar_vector,y);
    thetaStarDrawOut_struct = thetaStarDraw(indic, thetaStar_vector, y, ydenmat, q0v, alpha, lambda_struct, maxuniq);
    thetaStar_vector = thetaStarDrawOut_struct.thetaStar_vector;
    indic = thetaStarDrawOut_struct.indic;
    nunique = thetaStar_vector.size();
    
    
    //thetaNp1 and remix
    probs = zeros<vec>(nunique+1);
    for(int j = 0; j < nunique; j++){
      ind = find(indic == (j+1));
      indsize = ind.size();
      probs[j] = indsize/(alpha + n + 0.0);
      new_utheta_vector[0] = thetaD1(y(ind,spanall),lambda_struct);
      thetaStar_vector[j] = new_utheta_vector[0];
    }
    
    probs[nunique] = alpha/(alpha+n+0.0);
    indp = rmultinomF(probs);
    probssize = probs.size();
    if(indp == probssize) {
      outGD_struct = GD1(lambda_struct);
      thetaNp10_struct.mu = outGD_struct.mu;
      thetaNp10_struct.rooti = outGD_struct.rooti;
      thetaNp1_vector[0] = thetaNp10_struct;
    } else {
      outGD_struct = thetaStar_vector[indp-1];
      thetaNp10_struct.mu = outGD_struct.mu;
      thetaNp10_struct.rooti = outGD_struct.rooti;
      thetaNp1_vector[0] = thetaNp10_struct;
    }
    //draw alpha
    alpha = alphaD(priorAlpha_struct,nunique,gridsize);
    //draw lambda
    lambda_struct = lambdaD(lambda_struct,thetaStar_vector,lambda_hyper["alim"],lambda_hyper["nulim"],lambda_hyper["vlim"],gridsize);
  }
  //note indic is the vector of indicators for each obs correspond to which thetaStar
  DPOut out_struct;
  out_struct.thetaStar_vector = thetaStar_vector;
  out_struct.thetaNp1_vector = thetaNp1_vector;
  out_struct.alpha = alpha;
  out_struct.lambda_struct = lambda_struct;
  out_struct.indic = indic;
  return(out_struct);
}

unireg runiregG1(vec const& y, mat const& X, mat const& XpX, vec const& Xpy, double sigmasq, mat const& A, 
                vec const& Abetabar, int nu, double ssq) {
  
  // Keunwoo Kim 09/16/2014
  
  // Purpose: 
  //  perform one Gibbs iteration for Univ Regression Model
  //  only does one iteration so can be used in rhierLinearModel
  
  // Model:
  //  y = Xbeta + e  e ~N(0,sigmasq)
  //  y is n x 1
  //  X is n x k
  //  beta is k x 1 vector of coefficients
  
  // Prior:  
  //  beta ~ N(betabar,A^-1)
  //  sigmasq ~ (nu*ssq)/chisq_nu
  
  unireg out_struct;
  
  int n = y.size();
  int k = XpX.n_cols;
  
  //first draw beta | sigmasq
  //mat IR = solve(trimatu(chol(XpX/sigmasq+A)), eye(k,k)); //trimatu interprets the matrix as upper triangular and makes solve more efficient
  mat IR = solve(chol(XpX/sigmasq+A), eye(k,k)); //Rico: trimatu may cause chol warning that matrix is not symmetric
  vec btilde = (IR*trans(IR)) * (Xpy/sigmasq + Abetabar);
  vec beta = btilde + IR*vec(rnorm(k));
  
  //now draw sigmasq | beta
  double s = sum(square(y-X*beta));
  sigmasq = (s + nu*ssq)/rchisq(1,nu+n)[0]; //rchisq returns a vectorized object, so using [0] allows for the conversion to double
  out_struct.beta = beta;
  out_struct.sigmasq = sigmasq;  
  
  return (out_struct);
}


//////Added 2023 For IndepMetropParallelLinear
double lllinear(vec const& beta, vec const& y, mat const& X, double const& sigmasq){
  int n = y.n_rows;
  double rooti = 1/sqrt(sigmasq);
  mat rootimat;
  rootimat.eye(n, n) * rooti;
  return(lndMvn(y,X*beta,rootimat));
}


