% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/robustbetareg.R
\name{robustbetareg.control}
\alias{robustbetareg.control}
\title{Auxiliary for Controlling robustbetareg Fitting}
\usage{
robustbetareg.control(
  start = NULL,
  alpha.optimal = TRUE,
  tolerance = 0.001,
  maxit = 5000,
  L = 0.02,
  M = 3,
  ...
)
}
\arguments{
\item{start}{an optional vector with starting values for the parameter estimates.}

\item{alpha.optimal}{a logical value. If \code{TRUE} the tuning constant will
be select via the data-driven algorithm.}

\item{tolerance}{numeric tolerance for convergence.}

\item{maxit}{argument passed to \code{\link{optim}}.}

\item{L}{numeric specifying the threshold for the data-driven algorithm.
Default is \code{L = 0.02}.}

\item{M}{integer specifying the number of grid spacing for the data-driven
algorithm. Default is \code{M = 3}.}

\item{...}{currently not used.}
}
\value{
A list with components named as the arguments.
}
\description{
Several parameters that control fitting of robust beta regression models using
 \code{\link[=robustbetareg]{robustbetareg.}}
}
\details{
The \code{robustbetareg.control} controls the fitting process of
    the robust estimation in beta regression via the LMDPDE, LSMLE, MDPDE, and
    SMLE. The arguments \code{L} and \code{M} are passed to the data-driven
    algorithm for selecting the optimum alpha value; details can be found in
    Ribeiro and Ferrari (2022). Starting values for the parameters associated
    to the mean and precision submodels may be supplied via \code{start}. \cr
    We do not recommend changing the arguments passed to the data-driven algorithm.
}
\examples{
\donttest{
data("Firm")

# Using a robust start value for the parameters associated with the
# mean submodel
# using the robustbase package
# robust regression to obtain a starting value for beta
fit_lm_Rob <- robustbase::lmrob(FIRMCOST ~ SIZELOG + INDCOST,
                                data = Firm)
initials_beta_rob <-  as.numeric(coef(fit_lm_Rob))
etarob <- model.matrix(fit_lm_Rob)\%*\%initials_beta_rob
muhat_Rob <- set.link(link.mu = "logit",
                      link.phi = "log")$linkfun.mu$inv.link(etarob)
T_1_Rob <- 1/set.link(link.mu = "logit",
                      link.phi = "log")$linkfun.mu$d.linkfun(muhat_Rob)
#estimate of variance of y based on robustbase package
sigma2hat_Rob <- ((fit_lm_Rob$scale^2)*(T_1_Rob^2))
#phi estimate from robust method
phihat_Rob <- mean((muhat_Rob*(1-muhat_Rob))/sigma2hat_Rob)
gama1hat_rob <- log(phihat_Rob)
#gamma estimates from robustbase package
initials_gama_rob <-  as.numeric(gama1hat_rob)
#robust starting values for beta and gamma
thetaStart <- c(initials_beta_rob, initials_gama_rob)

fit_LSMLE <- robustbetareg(FIRMCOST ~ SIZELOG + INDCOST,
                           data = Firm,
                           type = "LSMLE", link.phi = "log",
                           control = robustbetareg.control(start = thetaStart))
}
}
\references{
Maluf, Y.S., Ferrari, S.L.P., and Queiroz, F.F. (2022). Robust
   beta regression through the logit transformation. \emph{Metrika}:61–81.\cr \cr
   Ribeiro, K.A.T. and Ferrari, S.L.P.  (2022). Robust estimation in beta regression
   via maximum Lq-likelihood. \emph{Statistical Papers}. DOI: 10.1007/s00362-022-01320-0. \cr \cr
}
\seealso{
\code{\link{robustbetareg}}
}
\author{
Yuri S. Maluf (\email{yurimaluf@gmail.com}),
Francisco F. Queiroz (\email{ffelipeq@outlook.com}) and Silvia L. P. Ferrari.
}
