% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/srt.r
\name{srt}
\alias{srt}
\alias{srt2}
\title{Signed-rank test}
\usage{
srt(
  data,
  formula,
  conf_level = 0,
  conf_method = "inversion",
  n_resamples = 1000L,
  alternative = "two.sided",
  mu = 0,
  distribution = "auto",
  correct = TRUE,
  zero_method = "wilcoxon",
  agg_fun = "error",
  digits_rank = Inf,
  tol_root = 1e-04
)

srt2(
  x,
  y = NULL,
  conf_level = 0,
  conf_method = "inversion",
  n_resamples = 1000L,
  alternative = "two.sided",
  mu = 0,
  distribution = "auto",
  correct = TRUE,
  zero_method = "wilcoxon",
  digits_rank = Inf,
  tol_root = 1e-04
)
}
\arguments{
\item{data}{(data.frame)\cr
The data frame of interest.}

\item{formula}{(formula)\cr
A formula of form:
\describe{
\item{y ~ group | block}{
Use when \code{data} is in tall format.
\code{y} is the numeric outcome, \code{group} is the binary grouping variable, and \code{block} is the subject/item-level variable indicating pairs of observations.
\code{group} will be converted to a factor and the first level will be the reference value.
For example, when \code{levels(data$group) <- c("pre", "post")}, the focal level is 'post', so differences are \code{post - pre}.
Pairs with non-finite values (infinite or missing) are silently dropped.
See \code{agg_fun} for handling duplicate cases of grouping/blocking combinations.
}
\item{y ~ x}{
Use when \code{data} is in wide format.
\code{y} and \code{x} must be numeric vectors.
Differences are calculated as \code{data$y - data$x}.
Pairs with non-finite values (infinite or missing) are silently dropped.
}
\item{ ~ x}{
Use when \code{data$x} represents pre-calculated differences or for the one-sample case.
Values with non-finite values (infinite or missing) are silently dropped.
}
}}

\item{conf_level}{(Scalar numeric: \verb{[0, 1)})\cr
The confidence level.
Typically \code{0.95}.
If \code{0} (default), no confidence interval is calculated.}

\item{conf_method}{(Scalar character: \code{c("inversion", "percentile", "bca")})\cr
The type of confidence interval.
If \code{"inversion"} (default), the bounds are computed by inverting the hypothesis test.
If \code{"percentile"}, the bounds are computed using a percentile bootstrap.
If \code{"bca"}, the bounds are computed using a bias-corrected and accelerated (BCa) bootstrap.}

\item{n_resamples}{(Scalar integer: \code{1000L}; \verb{[10, Inf)})\cr
The number of bootstrap resamples.
Only used if \code{"percentile"} or \code{"bca"} confidence intervals are chosen.}

\item{alternative}{(Scalar character: \code{c("two.sided", "greater", "less")})\cr
The alternative hypothesis.
Must be one of \code{"two.sided"} (default), \code{"greater"}, or \code{"less"}.}

\item{mu}{(Scalar numeric: \code{0}; \verb{(-Inf, Inf)})\cr
Under the null hypothesis, \code{x} or \code{x - y} is assumed to be symmetric around \code{mu}.}

\item{distribution}{(Scalar character: \code{c("auto", "exact", "asymptotic")})\cr
The method used to calculate the p-value.
If \code{"auto"} (default), an appropriate method will automatically be chosen (\code{distribution = "exact"} when n < 50 or \code{distribution = "asymptotic"} otherwise).
If \code{"exact"}, the exact Wilcoxon signed-rank distribution is used.
If \code{"asymptotic"}, the asymptotic normal approximation is used.}

\item{correct}{(Scalar logical: \code{c(TRUE, FALSE)})\cr
Whether or not to apply a continuity correction to the Z-statistic for the asymptotic approximation of the p-value.}

\item{zero_method}{(Scalar character: \code{c("wilcoxon", "pratt")})\cr
The method used to handle differences equal to zero.
If \code{"wilcoxon"} (default), zeros are removed prior to ranking (classic Wilcoxon convention).
If \code{"pratt"}, zeros are retained for ranking but excluded from the signed-rank sum.}

\item{agg_fun}{(Scalar character or function: \code{"error"})\cr
Used for aggregating duplicate cases of grouping/blocking combinations when data is in tall format and \code{formula} has structure \code{y ~ group | block}.
\code{"error"} (default) will return an error if duplicate grouping/blocking combinations are encountered.
Select one of \code{"first"}, \code{"last"}, \code{"sum"}, \code{"mean"}, \code{"median"}, \code{"min"}, or \code{"max"} for built in aggregation handling (each applies \code{na.rm = TRUE}).
Or define your own function.
For example, \code{myfun <- function(x) {as.numeric(quantile(x, 0.75, na.rm = TRUE))}}.}

\item{digits_rank}{(Scalar integer: \code{Inf}; \verb{(0, Inf]})\cr
Controls ranking precision.
If finite, ranks are computed from \link[base:Round]{base::signif}\verb{(abs(diffs), digits_rank)}.
If \code{Inf} (default), ranks are computed from \code{abs(diffs)}.
Smaller values may introduce ties (because they no longer depend on extremely small numeric differences) and thus change averaged ranks and tie counts.}

\item{tol_root}{(Scalar numeric: \code{1e-4}; \verb{(0, Inf)})\cr
For \link[stats:uniroot]{stats::uniroot}\code{(tol=tol_root)} calls when \code{conf_level > 0} and \code{distribution = "asymptotic"}.}

\item{x}{(numeric)\cr
Numeric vector of data.
Values with non-finite values (infinite or missing) are silently dropped.}

\item{y}{(numeric: \code{NULL})\cr
Numeric vector of data or \code{NULL}.
If \code{NULL} (default), a one-sample test is performed using \code{x}.
If numeric, differences are calculated as \code{x - y}.
Pairs with non-finite values (infinite or missing) are silently dropped.}
}
\value{
A list with the following elements:
\tabular{llll}{
Slot \tab Subslot \tab Name \tab Description \cr
1 \tab \tab \code{p_value}            \tab p-value. \cr
2 \tab \tab \code{statistic}    \tab Test statistic. \eqn{W^+} for the exact Wilcoxon signed-rank distribution. \eqn{Z} for the asymptotic normal approximation. \cr
3 \tab \tab \code{pseudomedian} \tab Measure of centrality. \cr
4 \tab \tab \code{lower}        \tab Lower bound of confidence interval for the pseudomedian. \code{NULL} if no CI requested. \cr
5 \tab \tab \code{upper}        \tab Upper bound of confidence interval for the pseudomedian. \code{NULL} if no CI requested. \cr
6 \tab \tab \code{method}       \tab Test method. \cr
7 \tab   \tab \code{info} \tab Additional test information. \cr
7 \tab 1 \tab \code{p_value_method} \tab Method used to calculate the p-value. \cr
7 \tab 2 \tab \code{pseudomedian_method} \tab Method used to calculate the pseudomedian. \cr
7 \tab 3 \tab \code{conf_method} \tab Method used to calculate the confidence interval. \cr
7 \tab 4 \tab \code{conf_level_achieved} \tab Achieved confidence level. \cr
7 \tab 5 \tab \code{n_sample} \tab Number of observations in the original data. \cr
7 \tab 6 \tab \code{n_analytic} \tab Number of observations after removing non-finite values from the original data. \cr
7 \tab 7 \tab \code{n_zeros} \tab Number of zeros among differences in the analytic data set. \cr
7 \tab 8 \tab \code{n_signed} \tab Number of nonzero differences in the analytic data set. \cr
7 \tab 9 \tab \code{n_ties} \tab Number of tied ranks after ranking the absolute differences. \cr
7 \tab 10 \tab \code{data_type} \tab Data type. \cr
7 \tab 11 \tab \code{focal_name} \tab Name of the focal variable (differences are focal - reference). \cr
7 \tab 12 \tab \code{reference_name} \tab Name of the reference variable (differences are focal - reference). \cr
8 \tab   \tab \code{call} \tab A named list of the function's arguments (use \code{as.call()} to convert to a call; \code{call$distribution} may be updated from \code{"exact"} to \code{"asymptotic"}).
}
}
\description{
Performs Wilcoxon's signed-rank test.
}
\details{
The procedure for Wilcoxon's signed-rank test is as follows:
\enumerate{
\item For one-sample data \code{x} or paired samples \code{x} and \code{y}, where \code{mu} is the measure of center under the null hypothesis, define the values used for analysis as \code{(x - mu)} or \code{(x - y - mu)}.
\item Define 'zero' values as \code{(x - mu == 0)} or \code{(x - y - mu == 0)}.
\itemize{
\item \code{zero_method = "wilcoxon"}: Remove values equal to zero.
\item \code{zero_method = "pratt"}: Keep values equal to zero.
}
\item Order the absolute values from smallest to largest.
\item Assign ranks \eqn{1, 2, \dots, n} to the ordered absolute values, using mean rank for ties.
\itemize{
\item \code{zero_method = "pratt"}: remove values equal to zero and their corresponding ranks.
}
\item Calculate the test statistic.

Calculate \eqn{W^+} as the sum of the ranks for positive values.
Let \eqn{r_i} denote the absolute-value rank of the \eqn{i}-th observation after applying the chosen zero handling and ranking precision and \eqn{v_i} denote the values used for analysis.
Then
\deqn{W^+ = \sum_{i : v_i > 0} r_i.}
\eqn{W^+ + W^- = n(n+1)/2}, thus either can be calculated from the other.
If the null hypothesis is true, \eqn{W^+} and \eqn{W^-} are expected to be equal.
\itemize{
\item \code{distribution = "exact"}: Use \eqn{W^+} which takes values between \eqn{0} and \eqn{n(n+1)/2}.
\item \code{distribution = "asymptotic"}: Use the standardized test statistic \eqn{Z=\frac{W^+ - E_0(W^+) - cc}{Var_0(W^+)^{1/2}}} where \eqn{Z \sim N(0, 1)} asymptotically.
\itemize{
\item \code{zero_method = "wilcoxon"}: Under the null hypothesis, the expected mean and variance are
\deqn{
       \begin{aligned}
       E_0(W^+) &= \frac{n(n+1)}{4} \\
       Var_0(W^+) &= \frac{n(n+1)(2n+1)}{24} - \frac{\sum(t^3-t)}{48},
       \end{aligned}
       }
where \eqn{t} is the number of ties for each unique ranked absolute value.
\item \code{zero_method = "pratt"}: Under the null hypothesis, the expected mean and variance are
\deqn{
       \begin{aligned}
       E_0(W^+) &= \frac{n(n+1)}{4} - \frac{n_{zeros}(n_{zeros}+1)}{4} \\
       Var_0(W^+) &= \frac{n(n+1)(2n+1) - n_{zeros}(n_{zeros}+1)(2n_{zeros}+1)}{24} - \frac{\sum(t^3-t)}{48},
       \end{aligned}
       }
where \eqn{t} is the number of ties for each unique ranked absolute value.
\item \code{correct = TRUE}: The continuity correction is defined by:
\deqn{
         cc = \begin{cases}
           0.5 \times \text{sign}(W^+ - E_0(W^+)) & \text{two-sided alternative} \\
           0.5 & \text{greater than alternative} \\
           -0.5 & \text{less than alternative.}
           \end{cases}
       }
\item \code{correct = FALSE}: Set \eqn{cc = 0}.
}
}

Alternatively, \eqn{E_0(W^+)} and \eqn{Var_0(W^+)} can be calculated without need for closed form expressions that include zero correction and/or tie correction.
Consider each rank \eqn{r_i} (with averaged rank for ties) as a random variable \eqn{X_i} that contributes to \eqn{W^+}.
Under the null hypothesis, each rank has 50\% chance of being positive or negative. So \eqn{X_i} can take values
\deqn{
     X_i = \begin{cases}
           r_i & \text{with probability } p = 0.5 \\
           0 & \text{with probability } 1 - p = 0.5.
           \end{cases}
   }
Using \eqn{Var(X_i) = E(X_i^2) - E(X_i)^2} where
\deqn{
   \begin{aligned}
   E(X_i) &= p \cdot r_i + (1 - p) \cdot 0 \\
          &= 0.5r_i \\
   E(X_i^2) &= p \cdot r_i^2 + (1 - p) \cdot 0^2 \\
            &= 0.5r_i^2 \\
   E(X_i)^2 &= (0.5r_i)^2,
   \end{aligned}
   }
it follows that \eqn{Var(X_i) = 0.5r_i^2 - (0.5r_i)^2 = 0.25r_i^2.}
Hence, \eqn{E_0(W^+) = \frac{\sum r_i}{2}} and \eqn{Var_0(W^+) = \frac{\sum r_i^2}{4}.}
\item Calculate the p-value.
\itemize{
\item \code{distribution = "exact"}
\itemize{
\item No zeros among the differences and ranks are tie free
\itemize{
\item Use the Wilcoxon signed-rank distribution as implemented in \code{\link[stats:SignRank]{stats::psignrank()}} to calculate the probability of being as or more extreme than \eqn{W^+.}
}
\item Zeros present and/or ties present
\itemize{
\item Use the Shift-Algorithm from \insertCite{streitberg1984;textual}{rankdifferencetest} as implemented in \code{\link[exactRankTests:dperm]{exactRankTests::pperm()}}.
}
}
\item \code{distribution = "asymptotic"}
\itemize{
\item Use the standard normal distribution as implemented in \code{\link[stats:Normal]{stats::pnorm()}} to calculate the probability of being as or more extreme than \eqn{Z.}
}
}
}
\subsection{Hypotheses and test assumptions}{

The signed-rank test hypotheses are stated as:
\itemize{
\item Null: \code{(x)} or \code{(x - y)} is centered at \code{mu}.
\item Two-sided alternative: \code{(x)} or \code{(x - y)} is not centered at \code{mu}.
\item Greater than alternative: \code{(x)} or \code{(x - y)} is centered at a value greater than \code{mu}.
\item Less than alternative: \code{(x)} or \code{(x - y)} is centered at a value less than \code{mu}.
}

The signed-rank test traditionally assumes the differences are independent with identical, continuous, and symmetric distribution.
However, not all of these assumptions may be required \insertCite{pratt1981}{rankdifferencetest}.
The 'identically distributed' assumption is not required, keeping the level of test as expected for the hypotheses as stated above.
The symmetry assumption is not required when using one-sided alternative hypotheses.
For example:
\itemize{
\item Null: \code{(x)} or \code{(x - y)} is symmetric and centered at \code{mu}.
\item Greater than alternative: \code{(x)} or \code{(x - y)} is stochastically larger than \code{mu}.
\item Less than alternative: \code{(x)} or \code{(x - y)} is stochastically smaller than \code{mu}.
}
}

\subsection{Zero handling}{

\code{zero_method = "pratt"} uses the method by \insertCite{pratt1959;textual}{rankdifferencetest}, which first rank-transforms the absolute values, including zeros, and then removes the ranks corresponding to the zeros.
\code{zero_method = "wilcoxon"} uses the method by \insertCite{wilcoxon1950;textual}{rankdifferencetest}, which first removes the zeros and then rank-transforms the remaining absolute values.
\insertCite{conover1973;textual}{rankdifferencetest} found that when comparing a discrete uniform distribution to a distribution where probabilities linearly increase from left to right, Pratt's method outperforms Wilcoxon's.
When testing a binomial distribution centered at zero to see whether the parameter of each Bernoulli trial is \eqn{\frac{1}{2}}, Wilcoxon's method outperforms Pratt's.
}

\subsection{Pseudomedians}{
\subsection{Hodges-Lehmann estimator}{

The Hodges-Lehmann estimator is the median of all pairwise averages of the sample values.
\deqn{\mathrm{HL} = \mathrm{median} \left\{ \frac{x_i + x_j}{2} \right\}_{i \le j}}
This pseudomedian is a robust, distribution-free estimate of central tendency for a single sample, or a location-shift estimator for paired data.
It's resistant to outliers and compatible with rank-based inference.
This statistic is returned when \code{conf_level = 0} (for all test methods) or when confidence intervals are requested for an exact Wilcoxon test with zero-free data and tie-free ranks.
}

\subsection{Midpoint around expected signed-rank statistic (exact CI-centered estimator)}{

The exact test for data which contains zeros or whose ranks contain ties uses the Streitberg-Rohmel Shift-algorithm.
When a confidence interval is requested under this scenario, the estimated pseudomedian is a midpoint around the expected rank sum.
This midpoint is the average of the largest shift value \eqn{d} whose signed-rank statistic does not exceed the null expectation and the smallest \eqn{d} whose statistic exceeds it.

In detail, let \eqn{W^+(d)} be the Wilcoxon signed-rank sum at shift \eqn{d}, and let \eqn{E_0} denote the null expectation (e.g., \eqn{\sum r_i / 2} when ranks are \eqn{r_i}).
Then \deqn{\hat{d}_{\mathrm{mid}} = \frac{1}{2} \left( \min\{ d : W^+(d) \le \lceil E_0 \rceil \} + \max\{ d : W^+(d) > E_0 \} \right)}

This pseudomedian is a discrete-compatible point estimate that centers the exact confidence interval obtained by inverting the exact signed-rank distribution.
It may differ from the Hodges-Lehmann estimator when data are tied or rounded.
}

\subsection{Root of standardized signed-rank statistic (asymptotic CI-centered estimator)}{

A similar algorithm is used to estimate the pseudomedian when a confidence interval is requested under the asymptotic test scenario.
This pseudomedian is the value of the shift \eqn{d} for which the standardized signed-rank statistic equals zero under the asymptotic normal approximation.

In detail, let \eqn{W^+(d)} be the signed-rank sum, with null mean \eqn{E_0(d)} and null variance \eqn{\mathrm{Var}_0(d)} (with possible tie and continuity corrections).
Define \deqn{Z(d) = \frac{W^+(d) - E_0(d)}{\sqrt{\mathrm{Var}_0(d)}}.}
The pseudomedian is the root \deqn{\hat{d}_{\mathrm{root}} = \{ d : Z(d) = 0 \}.}

This pseudomedian is a continuous-compatible point estimate that centers the asymptotic confidence interval.
It's the solution to the test-inversion equation under a normal approximation.
It's sensitive to tie/zero patterns through \eqn{\mathrm{Var}_0(d)}, may include a continuity correction, and is not guaranteed to equal the Hodges-Lehmann estimator or the exact midpoint.
}

}

\subsection{Confidence intervals}{
\subsection{Exact Wilcoxon confidence interval}{

The exact Wilcoxon confidence interval is obtained by inverting the exact distribution of the signed-rank statistic.
It uses the permutation distribution of the Wilcoxon statistic and finds bounds where cumulative probabilities cross \eqn{\alpha/2} and \eqn{1-\alpha/2}.
Endpoints correspond to quantiles from \code{\link[stats:SignRank]{stats::qsignrank()}}.
This interval guarantees nominal coverage under the null hypothesis without relying on asymptotic approximations.
It respects discreteness of the data and may produce conservative intervals when the requested confidence level is not achievable (with warnings).
}

\subsection{Exact Wilcoxon confidence interval using the Shift-algorithm}{

The exact Wilcoxon confidence interval using the Shift-algorithm is obtained by enumerating all candidate shifts and inverting the exact signed-rank distribution.
In detail, it generates all pairwise averages \eqn{\frac{x_i + x_j}{2}}, evaluates the signed-rank statistic for each candidate shift, and determines bounds using \code{\link[exactRankTests:dperm]{exactRankTests::pperm()}} and \code{\link[exactRankTests:dperm]{exactRankTests::qperm()}}.
}

\subsection{Asymptotic Wilcoxon confidence interval}{

The asymptotic Wilcoxon confidence interval is obtained by inverting the asymptotic normal approximation of the signed-rank statistic.
In detail, Define a standardized statistic:
\deqn{Z(d) = \frac{W^+(d) - E_0(d) - cc}{\sqrt{\mathrm{Var}_0(d)}}}
where \eqn{W^+(d)} is the signed-rank sum at shift \eqn{d}.
Then solve for \eqn{d} such that \eqn{Z(d)} equals the normal quantiles using \code{\link[stats:uniroot]{stats::uniroot()}}.
This interval may not be reliable for small samples or heavily tied data.
}

\subsection{Bootstrap confidence intervals}{

The percentile and BCa bootstrap confidence interval methods are described in chapter 5.3 of \insertCite{davison1997;textual}{rankdifferencetest}.
}

}

\subsection{History}{

\code{\link[stats:wilcox.test]{stats::wilcox.test()}} is the canonical function for the Wilcoxon signed-rank test.
\code{\link[exactRankTests:wilcox.exact]{exactRankTests::wilcox.exact()}} implemented the Streitberg-Rohmel Shift-algorithm for exact inference when zeros and/or ties are present.
\link[coin:coin-package]{coin::coin} superseded \pkg{exactRankTests} and includes additional methods.
\code{\link[=srt]{srt()}} reimplements these functions so the best features of each is available in a fast and easy to use format.
}
}
\examples{
#----------------------------------------------------------------------------
# srt() example
#----------------------------------------------------------------------------
library(rankdifferencetest)

# Use example data from Kornbrot (1990)
data <- kornbrot_table1

# Create tall-format data for demonstration purposes
data_tall <- reshape(
  data = kornbrot_table1,
  direction = "long",
  varying = c("placebo", "drug"),
  v.names = c("time"),
  idvar = "subject",
  times = c("placebo", "drug"),
  timevar = "treatment",
  new.row.names = seq_len(prod(length(c("placebo", "drug")), nrow(kornbrot_table1)))
)

# Subject and treatment should be factors. The ordering of the treatment factor
# will determine the difference (placebo - drug).
data_tall$subject <- factor(data_tall$subject)
data_tall$treatment <- factor(data_tall$treatment, levels = c("drug", "placebo"))

# Rate transformation inverts the rank ordering.
data$placebo_rate <- 60 / data$placebo
data$drug_rate <- 60 / data$drug
data_tall$rate <- 60 / data_tall$time

# In contrast to the rank difference test, the Wilcoxon signed-rank test
# produces differing results. See table 1 and table 2 (page 245) in
# Kornbrot (1990).
## Divide p-value by 2 for one-tailed probability.
srt(
  data = data,
  formula = placebo ~ drug,
  alternative = "two.sided",
  distribution = "asymptotic",
  zero_method = "wilcoxon",
  correct = TRUE,
  conf_level = 0.95
)

srt(
  data = data_tall,
  formula = rate ~ treatment | subject,
  alternative = "two.sided",
  distribution = "asymptotic",
  zero_method = "wilcoxon",
  correct = TRUE,
  conf_level = 0.95
)

}
\references{
\insertRef{wilcoxon1945}{rankdifferencetest}

\insertRef{wilcoxon1950}{rankdifferencetest}

\insertRef{pratt1981}{rankdifferencetest}

\insertRef{pratt1959}{rankdifferencetest}

\insertRef{cureton1967}{rankdifferencetest}

\insertRef{conover1973}{rankdifferencetest}

\insertRef{hollander2014}{rankdifferencetest}

\insertRef{bauer1972}{rankdifferencetest}

\insertRef{streitberg1984}{rankdifferencetest}

\insertRef{hothorn2001}{rankdifferencetest}

\insertRef{hothorn2002}{rankdifferencetest}

\insertRef{hothorn2008}{rankdifferencetest}

\insertRef{davison1997}{rankdifferencetest}

\insertRef{exactRankTests}{rankdifferencetest}

\insertRef{R}{rankdifferencetest}
}
\seealso{
\code{\link[=rdt]{rdt()}},
\code{\link[stats:wilcox.test]{stats::wilcox.test()}},
\code{\link[coin:SymmetryTests]{coin::wilcoxsign_test()}}
}
