% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crossval.R
\name{crossval}
\alias{crossval}
\title{Calculate bias-corrected scores via cross-validation}
\usage{
crossval(data, outcome, model_fun, pred_fun, score_fun, k = 10, ...)
}
\arguments{
\item{data}{the data used in developing the model. Should contain all variables considered (i.e., even those excluded by variable selection in the development sample)}

\item{outcome}{character denoting the column name of the outcome in \code{data}.}

\item{model_fun}{a function that takes at least one argument, \code{data}. This function should implement the entire model development procedure (i.e., hyperparameter tuning, variable selection, imputation). Additional arguments can be provided via \code{...}. This function should return an object that works with \code{pred_fun}.}

\item{pred_fun}{function that takes at least two arguments, \code{model} and \code{data}. This function should return a numeric vector of predicted probabilities of the outcome with the same length as the number of rows in \code{data} so it is important to take into account how missing data is treated (e.g., \code{predict.glm} omits predictions for rows with missing values).}

\item{score_fun}{a function to calculate the metrics of interest. If this is not specified \code{\link{score_binary}} is used.}

\item{k}{number of folds. Typically scores need greater than 2 observations to be calculated so folds should be chosen with this in mind.}

\item{...}{additional arguments for \code{model_fun}, \code{pred_fun}, and/or \code{score_fun}.}
}
\value{
a list of class \code{internal_cv} containing:
\itemize{
\item{\code{apparent} - scores calculated on the original data using the original model.}
\item{\code{optimism} - estimates of optimism for each score (average difference in score for training data vs test data on each fold) which can be subtracted from 'apparent' performance calculated using the original model on the original data.}
\item{\code{cv_optimism_corrected} - 'bias corrected' scores (apparent - optimism). This is what is produced by \code{rms::validate}, \code{rms::predab.resample}.}
\item{\code{cv_average} - average of scores calculated on the test (held out) data. This is the metric described in Steyerberg et al. (2001).}
\item{\code{indices} - indices used to define test set on each fold.}
}
}
\description{
Estimate bias-corrected scores via cross-validation. CV is used to calculate optimism
which is then subtracted from apparent scores and to calculate average performance in the
out of sample (held out) data.
This function is called by \code{\link{validate}}.
}
\examples{
library(pminternal)
set.seed(456)
# simulate data with two predictors that interact
dat <- pmcalibration::sim_dat(N = 1000, a1 = -2, a3 = -.3)
mean(dat$y)
dat$LP <- NULL # remove linear predictor

# fit a (misspecified) logistic regression model
#m1 <- glm(y ~ x1 + x2, data=dat, family="binomial")

model_fun <- function(data, ...){
  glm(y ~ x1 + x2, data=data, family="binomial")
}

pred_fun <- function(model, data, ...){
  predict(model, newdata=data, type="response")
}

# CV Corrected = Apparent - CV Optimism
# CV Average = average score in held out fold
crossval(data=dat, outcome="y", model_fun=model_fun, pred_fun=pred_fun, k=10)

}
\references{
Steyerberg, E. W., Harrell Jr, F. E., Borsboom, G. J., Eijkemans, M. J. C., Vergouwe, Y., & Habbema, J. D. F. (2001). Internal validation of predictive models: efficiency of some procedures for logistic regression analysis. Journal of clinical epidemiology, 54(8), 774-781.
}
