% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/traits.summary.R
\name{traits.summary}
\alias{traits.summary}
\title{Summarizing trait data}
\usage{
traits.summary(sim, traits, fossils = NULL, selection = "all")
}
\arguments{
\item{sim}{A \code{sim} object, containing extinction times, speciation 
times, parent, and status information for each species in the simulation. 
See \code{?sim}.}

\item{traits}{List of trait data frames, usually one of the returns of 
\code{bd.sim}. \code{traits[[i]][[j]]} should correspond to the \code{j}th
trait data frame for species \code{i}. The data frames contain the following
columns

\describe{
\item{\code{value}}{A vector of trait values the species took at specific
intervals of time.}

\item{\code{max}}{A vector of time values corresponding to the upper bound
of each interval.}

\item{\code{min}}{A vector of time values corresponding to the lower bound
of each interval}}}

\item{fossils}{A data frame with a \code{"Species"} column and a
\code{SampT} column, usually an output of the \code{sample.clade}
function. Species names must contain only one number each, corresponding
to the order of the \code{sim} vectors. Note that we require it to have a
\code{SampT} column, i.e. fossils must have an exact age. This assumption
might be relaxed in the future.}

\item{selection}{Which subset of species to collect trait data for. If set
to \code{"all"}, it will return every trait value it has access to, i.e.
either all species, living or dead, or all species plus fossils if
\code{fossils} is supplied. If set to \code{"extant"}, it will return only
trait values for living species. If set to \code{"extinct"}, it will return
only trait values for extinct species, and fossils if \code{fossils} is 
supplied. If set to \code{"fossil"}, it will return values for only the
fossil species (and therefore requires a \code{fossils} parameter). If set 
to \code{"sampled"}, it will function the same as in the case for 
\code{"extant"}, except it will also return values for the fossils if 
\code{fossils} is supplied.}
}
\value{
A named list of named vectors of trait values. List element names
refer to each trait, so i.e. \code{res$traitN} will correspond to the vector
of trait values for trait \code{N}. Vector element names refer to the
species, using the default naming convention of the package (\code{tN} is
the \code{N}th species in the simulation, and \code{tN.M} is the \code{M}th
sampled fossil of that species).
}
\description{
Summarizes trait data from a \code{sim} object, usually the ouput of
\code{bd.sim} in the case where diversification rates are trait-dependent.
Returns a list of trait values at the present or the time of extinction
(depending on whether the species is alive at present), and optionally
returns values at the time of fossil sampling if provided with a fossil
record object \code{fossils}, usually the output of \code{sample.clade}. 
Does not make assumptions on the number of traits described in the
\code{traits} parameter, so that if that list has more than one trait per
species, multiple vectors will be returned by the function.
}
\examples{

###
# need a simple simulation to use as an example

# initial number of species
n0 <- 1

# maximum simulation time
tMax <- 40

# speciation, higher for state 1
lambda <- c(0.1, 0.2)

# extinction, trait-independent
mu <- 0.03

# number of traits and states (1 binary trait)
nTraits <- 1
nStates <- 2

# initial value of the trait
X0 <- 0

# transition matrix, with symmetrical transition rates
Q <- list(matrix(c(0, 0.1,
                   0.1, 0), ncol = 2, nrow = 2))

# set seed
set.seed(1)

# run the simulation
sim <- bd.sim.traits(n0, lambda, mu, tMax, nTraits = nTraits, 
                    nStates = nStates, X0 = X0, Q = Q, nFinal = c(2, Inf))
                    
# get all trait values
traitSummary <- traits.summary(sim$SIM, sim$TRAITS)
traitSummary

# could get only the extant values, instead
traitSummary <- traits.summary(sim$SIM, sim$TRAITS, selection = "extant")
traitSummary

# or all the extinct values
traitSummary <- traits.summary(sim$SIM, sim$TRAITS, selection = "extinct")
traitSummary

# set seed
set.seed(1)

# maybe we want to take a look at the traits of fossil records too
fossils <- sample.clade(sim$SIM, rho = 0.5, tMax = max(sim$SIM$TS))

# get the trait values for all extinct species, including fossil samples
traitSummary <- traits.summary(sim$SIM, sim$TRAITS, 
                               fossils = fossils, selection = "extinct")
traitSummary

# can also get the values for all sampled species, i.e. extant or fossils
traitSummary <- traits.summary(sim$SIM, sim$TRAITS, 
                               fossils = fossils, selection = "sampled")
traitSummary

# or just the fossil species
traitSummary <- traits.summary(sim$SIM, sim$TRAITS, 
                               fossils = fossils, selection = "fossil")
traitSummary

}
\author{
Bruno do Rosario Petrucci
}
