% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/degree_adoption_diagnostic.R
\name{degree_adoption_diagnostic}
\alias{degree_adoption_diagnostic}
\title{Degree and Time of Adoption Diagnostic}
\usage{
degree_adoption_diagnostic(
  graph,
  degree_strategy = c("mean", "first", "last"),
  bootstrap = TRUE,
  R = 1000,
  conf.level = 0.95,
  toa = NULL,
  t0 = NULL,
  t1 = NULL,
  name = NULL,
  behavior = NULL,
  combine = c("none", "pooled", "average", "earliest"),
  min_adopters = 3,
  valued = getOption("diffnet.valued", FALSE),
  ...
)
}
\arguments{
\item{graph}{A `[diffnet()]` object or a graph data structure (classes include
`array` (\eqn{n\times n \times T}{n*n*T}), `dgCMatrix` (sparse),
`igraph`, etc.; see [netdiffuseR-graphs]).}

\item{degree_strategy}{Character scalar. How to aggregate degree measures across
time periods:
- `"mean"` (default): Average degree across all time periods
- `"first"`: Degree in the first time period
- `"last"`: Degree in the last time period}

\item{bootstrap}{Logical scalar. Whether to compute bootstrap confidence intervals.}

\item{R}{Integer scalar. Number of bootstrap replicates (default 1000).}

\item{conf.level}{Numeric scalar. Confidence level for bootstrap intervals (default 0.95).}

\item{toa}{Integer vector of length \eqn{n} (single behavior) or an \eqn{n\times Q}{n*Q}
matrix (multi-behavior) with times of adoption. Required when `graph` is not a `diffnet`.}

\item{t0, t1}{Optional integer scalars defining the first and last observed
periods. If missing and `toa` is provided, `t0` defaults to 1
and `t1` to `max(toa, na.rm=TRUE)`.}

\item{name}{Optional character scalars used only when coercing
inputs into a `diffnet` object (passed to `new_diffnet`).}

\item{behavior}{Which behaviors to include when `toa` is a matrix (multi-diffusion).
Can be `NULL` (all), a numeric index vector, or a character vector matching `colnames(toa)`.}

\item{combine}{Character scalar. How to combine multiple behaviors when `toa` is a matrix:
- `"none"` (analyze each behavior separately)
- `"pooled"` (stack rows across behaviors)
- `"average"` (per-actor mean of TOA across selected behaviors)
- `"earliest"` (per-actor minimum TOA)
Ignored for single-behavior.}

\item{min_adopters}{Integer scalar. Minimum number of adopters required to compute correlations
for any analysis cell (default 3).}

\item{valued}{Logical scalar. Whether to use edge weights in degree calculations.}

\item{...}{Additional arguments passed on when coercing to `diffnet`.}
}
\value{
When analyzing a single behavior (or when `combine!="none"`), a list with:
\item{correlations}{Named numeric vector with correlations between in-degree/out-degree and time of adoption}
\item{bootstrap}{List with bootstrap results when `bootstrap = TRUE`, otherwise `NULL`}
\item{call}{The matched call}
\item{degree_strategy}{The degree aggregation strategy used}
\item{sample_size}{Number of rows included in the analysis (adopter rows)}
\item{combine}{`NULL` for single-behavior; otherwise the combination rule used.}

When `combine="none"` with multiple behaviors, returns the same structure, except:
- `correlations` is a \eqn{2\times Q^*}{2 x Q*} matrix with rows `c("indegree_toa","outdegree_toa")`
  and one column per analyzed behavior.
- `bootstrap` is a named list with one entry per behavior (each like the single-behavior case), or `NULL` if `bootstrap=FALSE`.
- `sample_size` is an integer vector named by behavior.
- `combine` is `"none"`.
}
\description{
Analyzes the correlation between in-degree, out-degree, and time of adoption
to identify whether opinion leaders were early adopters (supporters) or late
adopters (opposers).
}
\details{
This diagnostic function computes correlations between degree centrality measures
(in-degree and out-degree) and time of adoption. Positive correlations suggest
that central actors (opinion leaders) adopted early, while negative correlations
suggest they adopted late.

When `bootstrap = TRUE`, the function uses the `boot` package to
compute bootstrap confidence intervals for the correlations.

When `toa` is a matrix (multi-diffusion), degree vectors are computed once and
reused; the time of adoption is combined according to `combine`:
- `"none"`: computes separate results per behavior (see Value).
- `"pooled"`: stacks (actor, behavior) rows for adopters and runs a single analysis.
- `"average"`: one row per actor using the mean TOA of adopted behaviors.
- `"earliest"`: one row per actor using the minimum TOA of adopted behaviors.
}
\examples{
# Basic usage with Korean Family Planning data
data(kfamilyDiffNet)
result_basics <- degree_adoption_diagnostic(kfamilyDiffNet, bootstrap = FALSE)
print(result_basics)

# With bootstrap confidence intervals
result_boot <- degree_adoption_diagnostic(kfamilyDiffNet)
print(result_boot)

# Different degree aggregation strategies
result_first <- degree_adoption_diagnostic(kfamilyDiffNet, degree_strategy = "first")
result_last  <- degree_adoption_diagnostic(kfamilyDiffNet, degree_strategy = "last")

# Multi-diffusion (toy) ----------------------------------------------------
set.seed(999)
n <- 40; t <- 5; q <- 2
garr <- rgraph_ws(n, t, p=.3)
diffnet_multi <- rdiffnet(seed.graph = garr, t = t, seed.p.adopt = rep(list(0.1), q))

# pooled (one combined analysis)
degree_adoption_diagnostic(diffnet_multi, combine = "pooled", bootstrap = FALSE)

# per-behavior (matrix of correlations; one column per behavior)
degree_adoption_diagnostic(diffnet_multi, combine = "none", bootstrap = FALSE)

}
\seealso{
`[dgr()]`, `[diffreg()]`, `[exposure()]`

Other statistics: 
\code{\link{bass}},
\code{\link{classify_adopters}()},
\code{\link{cumulative_adopt_count}()},
\code{\link{dgr}()},
\code{\link{ego_variance}()},
\code{\link{exposure}()},
\code{\link{hazard_rate}()},
\code{\link{infection}()},
\code{\link{moran}()},
\code{\link{struct_equiv}()},
\code{\link{threshold}()},
\code{\link{vertex_covariate_dist}()}
}
\concept{statistics}
