#' Local mark correlation functions for inhomogeneous point patterns with function-valued marks.
#'
#' Local mark correlation functions for inhomogeneous point patterns with function-valued marks.
#'
#' @usage lfmcorrinhom(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart",
#'  "isham", "stoyancov", "schlather"),
#' r = NULL,
#' lambda = NULL,
#' method_lambda = c("kernel", "Voronoi"),
#' bw = NULL,
#' f = NULL,
#' method = c("density", "loess"),
#' correction = c("Ripley", "translate", "none"),
#' normalise = TRUE,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class ppp or lpp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \code{r} at which the mark correlation function should be evaluated.
#' @param lambda Estimated intensity at data points. If not given, it will be estimated internally. See details.
#' @param method_lambda The method to be used for estimating intensity at data points, if \code{lambda = NULL}.
#' @param bw Bandwidth method to be used for estimating intensity at data points if \code{lambda = NULL} and \code{method_lambda = "kernel"}.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param method Type of smoothing, either \code{density} or \code{loess}.
#' @param correction Type of edge correction to be applied, either of \code{"Ripley", "translate", "none"}. This is used only if \code{X} is of class ppp.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param tol Tolerance used in the calculation of the conditional mean of marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' 
#' 
#' @details
#' 
#' This function computes local mark correlation functions for an inhomogeneous point pattern with a function-valued mark. See the details of test functions used in \code{\link[markstat]{fmcorrinhom}}. Technical details are given in Eckardt and Moradi (2025).
#' 
#' @examples
#'  library(spatstat.random)
#'  library(spatstat.geom)
#'  library(spatstat.explore)
#'  X <- rpoispp(100)
#'  marks(X) <- data.frame(
#'  t1 = runif(npoints(X),1,10),
#'  t2 = runif(npoints(X),1,10),
#'  t3 = runif(npoints(X),1,10),
#'  t4 = runif(npoints(X),1,10),
#'  t5 = runif(npoints(X),1,10))
#'  lfmcorrinhom(X, ftype = "stoyan", method = "density", method_lambda = "kernel", bw = bw.scott)
#'  
#'  
#' @return a data.frame which gives the estimated overall local mark correlation function and the distance vector \eqn{r} at which the local mark correlation function is estimated. The outputs of the local mark correlation functions for each time point are stored as an attribute, which can be extracted as \code{attr(., "ests.time")}.
#' The outputs of the local mark correlation functions for each data point are stored as an attribute, which can be extracted as \code{attr(., "ests.points")}.
#' @references 
#' Eckardt, M., Mateu, J., & Moradi, M. (2024). Function‐Valued Marked Spatial Point Processes on Linear Networks: Application to Urban Cycling Profiles. Stat, 13(4), e70013.
#' 
#' Eckardt, M., & Moradi, M. (2025). Local indicators of mark association for marked spatial point processes.
#' 
#' Moradi, M., & Eckardt, M. (2025). Inhomogeneous mark correlation functions for general marked point processes. arXiv e-prints, arXiv-2505.
#' 
#' @seealso \code{\link[markstat]{fmcorr}}, \code{\link[markstat]{mcorrinhom.ppp}}, \code{\link[markstat]{mcorrinhom.lpp}}.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt


#' @import spatstat.univar
#' @import spatstat.random
#' @import spatstat.linnet
#' @import spatstat.geom
#' @import spatstat.explore
#' @import spatstat.utils
#' @import stats
#' @export

lfmcorrinhom <- function(X,
                         ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                         r = NULL,
                         lambda = NULL,
                         method_lambda = c("kernel", "Voronoi"),
                         bw = NULL,
                         f = NULL,
                         method = c("density", "loess"),
                         correction = c("Ripley", "translate", "none"),
                         normalise = TRUE,
                         tol = 0.01,
                         ...){

  if (all(class(X) != "lpp" & class(X) != "ppp")) stop("object X should be of class lpp or ppp.")
  
  if(all(class(marks(X)) != "data.frame" & class(marks(X)) != "hyperframe")) stop("object X should have a function-valued mark as a data.frame where each col represents a time point.")
  
  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")
  
  if (missing(method)) stop("smoothing method should be chosen.")
  
  lambda_given <- lambda
  
  correction <- match.arg(correction,correction)
  
  n <- npoints(X)
  d <- pairdist(X)

  if(is.null(r)){

    if(any(class(X)=="ppp")){

      W <- X$window
      rmaxdefault <- rmax.rule("K", W, n/area(W))
      if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
      breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
      r <- breaks$r

    }else if(any(class(X)=="lpp")){

      L <- X$domain
      rmaxdefault <- 0.98 * boundingradius(L)
      if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
      W <- Window(L)
      breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
      r <- breaks$r

    }else {
      stop("object X should be of class lpp or ppp.")
    }
  }

  rmax <- max(r)

  if(is.null(lambda)){

    if(any(class(X)=="ppp")){

      if(method_lambda=="kernel"){

        lambda <- as.numeric(density(unmark(X), sigma = bw(X), at="points", diggle=T))

      }else if(method_lambda=="Voronoi"){

        lambda <- as.numeric(densityVoronoi(X, f=0.2, nrep = 100)[X])

      }else{

        stop("You need to pick a method for intensity estimation!")

      }
    }
    else if(any(class(X)=="lpp")){

      if(method_lambda=="kernel"){

        lambda  <- as.numeric(densityQuick.lpp(unmark(X), sigma = bw(X), at="points", diggle=T))

      }else if(method_lambda=="Voronoi"){

        lambda <- as.numeric(densityVoronoi.lpp(X, f=0.2, nrep = 100)[X])

      }else{

        stop("You need to pick a method for intensity estimation!")

      }
    }
  }else{
    lambda <- lambda
  }

  m <- as.data.frame(marks(X))


  nf <- dim(m)[1]
  f.len <- dim(m)[2]


  if(any(class(X)=="ppp")){

    out <- list()
    for (i in 1:ncol(m)) {

      marks(X) <- as.numeric(m[,i])
      out[[i]] <- lmcorrinhom.ppp(X, lambda = lambda, r = r,
                                  correction = correction,
                                  normalise = normalise, f = f, ftype = ftype, method = method, tol = tol, ...)
    }

    r <- out[[1]][,"r"]
    
    col_names <- colnames(out[[1]])
    
    out.points <- lapply(X=1:length(col_names), function(i) {
      rr <- do.call(cbind, lapply(out, function(df) df[,col_names[i]]))
      colnames(rr) <- paste("p", rep(col_names[i], length(out))," ", colnames(m), sep = "")
      return(rr)
    })
    
    out.points.overall <- lapply(X=1:length(col_names), function(i){
      apply(out.points[[i]], 1, mean)
    })
    
    out.points.overall <- do.call(cbind, out.points.overall)
    
    colnames(out.points.overall) <- colnames(out[[1]])

  }else if(any(class(X)=="lpp")){

    out <- list()
    for (i in 1:ncol(m)) {

      marks(X) <- as.numeric(m[,i])
      out[[i]] <- lmcorrinhom.lpp(X, lambda = lambda, r = r,
                                  normalise = normalise, f = f, ftype = ftype, method = method, tol = tol, ...)
    }

    r <- out[[1]][,"r"]
    
    col_names <- colnames(out[[1]])
    
    out.points <- lapply(X=1:length(col_names), function(i) {
      rr <- do.call(cbind, lapply(out, function(df) df[,col_names[i]]))
      colnames(rr) <- paste("p", rep(col_names[i], length(out))," ", colnames(m), sep = "")
      return(rr)
    })
    
    out.points.overall <- lapply(X=1:length(col_names), function(i){
      apply(out.points[[i]], 1, mean)
    })
    
    out.points.overall <- do.call(cbind, out.points.overall)
    
    colnames(out.points.overall) <- colnames(out[[1]])

  }else {
    stop("object X should be of class lpp or ppp.")
  }

  names(out) <- colnames(m)
  names(out.points) <- colnames(out[[1]])
 
  
  if(ncol(out.points.overall) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out.points.overall) <- "mc"
  
  attr(out.points.overall, "ests.points") <- out.points
  attr(out.points.overall, "ests.time") <- out
  
  attr(out.points.overall, "mtype") <- "function-valued"
  attr(out.points.overall, "type") <- type
  attr(out.points.overall, "ests") <- out
  attr(out.points.overall, "ftype") <- ftype
  attr(out.points.overall, "method") <- method
  attr(out.points.overall, "lambda") <- lambda_given
  attr(out.points.overall, "normalise") <- normalise
  attr(out.points.overall, "method_lambda") <- method_lambda
  if(any(class(X)=="ppp")){
    attr(out.points.overall, "correction") <- correction  
  }
  attr(out.points.overall, "bw") <- bw
  

  return(out.points.overall)
}
