% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ves.R
\name{ves}
\alias{ves}
\title{Vector Exponential Smoothing in SSOE state space model}
\usage{
ves(data, model = "PPP", lags = c(frequency(data)),
  persistence = c("common", "individual", "dependent"),
  transition = c("common", "individual", "dependent"), phi = c("common",
  "individual"), initial = c("individual", "common"),
  initialSeason = c("common", "individual"), loss = c("likelihood",
  "diagonal", "trace"), ic = c("AICc", "AIC", "BIC", "BICc"), h = 10,
  holdout = FALSE, occurrence = c("none", "fixed", "logistic"),
  bounds = c("admissible", "usual", "none"), silent = TRUE, ...)
}
\arguments{
\item{data}{The matrix with the data, where series are in columns and
observations are in rows.}

\item{model}{The type of ETS model. Can consist of 3 or 4 chars: \code{ANN},
\code{AAN}, \code{AAdN}, \code{AAA}, \code{AAdA}, \code{MMdM} etc.
\code{PPP} means that the best pure model will be selected based on the chosen
information criteria type.
ATTENTION! ONLY PURE ADDITIVE AND PURE MULTIPLICATIVE MODELS ARE
AVAILABLE!
Pure multiplicative models are done as additive model applied to log(y).

Also \code{model} can accept a previously estimated VES model and use all its
parameters.}

\item{lags}{The lags of the model. Needed for seasonal models.}

\item{persistence}{Persistence matrix \eqn{G}, containing smoothing
parameters. Can be:
\itemize{
\item \code{"independent"} - each series has its own smoothing parameters
and no interactions are modelled (all the other values in the matrix are set
to zero);
\item \code{"dependent"} - each series has its own smoothing parameters, but
interactions between the series are modelled (the whole matrix is estimated);
\item \code{"group"} each series has the same smoothing parameters for respective
components (the values of smoothing parameters are repeated, all the other values
in the matrix are set to zero).
\item \code{"seasonal"} - each component has its own smoothing parameter, except
for the seasonal one, which is common across the time series.
\item provided by user as a vector or as a matrix. The value is used by the model.
}
You can also use the first letter instead of writing the full word.}

\item{transition}{Transition matrix \eqn{F}. Can be:
\itemize{
\item \code{"independent"} - each series has its own preset transition matrix
and no interactions are modelled (all the other values in the matrix are set
to zero);
\item \code{"dependent"} - each series has its own transition matrix, but
interactions between the series are modelled (the whole matrix is estimated). The
estimated model behaves similar to VAR in this case;
\item \code{"group"} each series has the same transition matrix for respective
components (the values are repeated, all the other values in the matrix are set to
zero).
\item provided by user as a vector or as a matrix. The value is used by the model.
}
You can also use the first letter instead of writing the full word.}

\item{phi}{In cases of damped trend this parameter defines whether the \eqn{phi}
should be estimated separately for each series (\code{"individual"}) or for the whole
set (\code{"common"}). If vector or a value is provided here, then it is used by the
model.}

\item{initial}{Can be either character or a vector / matrix of initial states.
If it is character, then it can be \code{"individual"}, individual values of
the initial non-seasonal components are used, or \code{"common"}, meaning that
the initials for all the time series are set to be equal to the same value.
If vector of states is provided, then it is automatically transformed into
a matrix, assuming that these values are provided for the whole group.}

\item{initialSeason}{Can be either character or a vector / matrix of initial
states. Treated the same way as \code{initial}. This means that different time
series may share the same initial seasonal component.}

\item{loss}{Type of Loss Function used in optimization. \code{loss} can
be:
\itemize{
\item \code{"likelihood"} - which implies the maximisation of likelihood of
multivariate normal distribution (or log Normal if the multiplicative model
is constructed);
\item \code{"diagonal"} - similar to \code{"likelihood"}, but assumes that
covariances between the error terms are zero.
\item \code{"trace"} - the trace of the covariance matrix of errors.
The sum of variances is minimised in this case.
\item Provided by user as a custom function of \code{actual}, \code{fitted} and
\code{B}. Note that internally function transposes the data, so that \code{actual}
and \code{fitted} contain observations in columns and series in rows.
}
An example of the latter option is:
   \code{lossFunction <- function(actual,fitted,B){return(mean(abs(actual - fitted)))}}
followed by \code{loss=lossFunction}.}

\item{ic}{The information criterion used in the model selection procedure.}

\item{h}{Length of forecasting horizon.}

\item{holdout}{If \code{TRUE}, holdout sample of size \code{h} is taken from
the end of the data.}

\item{occurrence}{Defines type of occurrence model used. Can be:
\itemize{
\item \code{none}, meaning that the data should be considered as non-intermittent;
\item \code{fixed}, taking into account constant Bernoulli distribution of
demand occurrences;
\item \code{logistic}, based on logistic regression.
}
In this case, the ETS model inside the occurrence part will correspond to
\code{model} and \code{probability="dependent"}.
Alternatively, model estimated using \link[legion]{oves} function can be provided
here.}

\item{bounds}{What type of bounds to use in the model estimation. The first
letter can be used instead of the whole word. \code{"admissible"} means that the
model stability is ensured, while \code{"usual"} means that the all the parameters
are restricted by the (0, 1) region.}

\item{silent}{If \code{silent="none"}, then nothing is silent, everything is
printed out and drawn. \code{silent="all"} means that nothing is produced or
drawn (except for warnings). In case of \code{silent="graph"}, no graph is
produced. If \code{silent="legend"}, then legend of the graph is skipped.
And finally \code{silent="output"} means that nothing is printed out in the
console, but the graph is produced. \code{silent} also accepts \code{TRUE}
and \code{FALSE}. In this case \code{silent=TRUE} is equivalent to
\code{silent="all"}, while \code{silent=FALSE} is equivalent to
\code{silent="none"}. The parameter also accepts first letter of words ("n",
"a", "g", "l", "o").}

\item{...}{Other non-documented parameters. For example \code{FI=TRUE} will
make the function also produce Fisher Information matrix, which then can be
used to calculated variances of smoothing parameters and initial states of
the model. The vector of initial parameter for the optimiser can be provided
here as the variable \code{B}. The upper bound for the optimiser is provided
via \code{ub}, while the lower one is \code{lb}. Also, the options for nloptr can be
passed here:
\itemize{
\item \code{maxeval=40*k} is the default number of iterations for both optimisers
used in the function (k is the number of parameters to estimate).
\item \code{algorithm1="NLOPT_LN_BOBYQA"} is the algorithm used in the first
optimiser, while \code{algorithm2="NLOPT_LN_NELDERMEAD"} is the second one.
\item \code{xtol_rel1=1e-8} is the relative tolerance in the first optimiser,
while \code{xtol_rel2=1e-6} is for the second one. All of this can be amended and
passed in ellipsis for finer tuning.
\item \code{print_level} - the level of output for the optimiser (0 by default).
If equal to 41, then the detailed results of the optimisation are returned.}}
}
\value{
Object of class "legion" is returned. It contains the following list of
values:
\itemize{
\item \code{model} - The name of the fitted model;
\item \code{timeElapsed} - The time elapsed for the construction of the model;
\item \code{states} - The matrix of states with components in columns and time in rows;
\item \code{persistence} - The persistence matrix;
\item \code{transition} - The transition matrix;
\item \code{measurement} - The measurement matrix;
\item \code{phi} - The damping parameter value;
\item \code{lagsAll} - The vector of the internal lags used in the model;
\item \code{B} - The vector of all the estimated coefficients;
\item \code{initial} - The initial values of the non-seasonal components;
\item \code{initialSeason} - The initial values of the seasonal components;
\item \code{nParam} - The number of estimated parameters;
\item \code{occurrence} - The occurrence part of the model estimated with VES;
\item \code{data} - The matrix with the original data;
\item \code{fitted} - The matrix of the fitted values;
\item \code{holdout} - The matrix with the holdout values (if \code{holdout=TRUE} in
the estimation);
\item \code{residuals} - The matrix of the residuals of the model;
\item \code{Sigma} - The covariance matrix of the errors (estimated with the correction
for the number of degrees of freedom);
\item \code{forecast} - The matrix of point forecasts;
\item \code{ICs} - The values of the information criteria;
\item \code{logLik} - The log-likelihood function;
\item \code{lossValue} - The value of the loss function;
\item \code{loss} - The type of the used loss function;
\item \code{lossFunction} - The loss function if the custom was used in the process;
\item \code{accuracy} - the values of the error measures. Currently not available.
\item \code{FI} - Fisher information if user asked for it using \code{FI=TRUE}.
}
}
\description{
Function constructs vector ETS model and returns forecast, fitted values, errors
and matrix of states along with other useful variables.
}
\details{
Function estimates vector ETS in a form of the Single Source of Error state space
model of the following type:

\deqn{
\mathbf{y}_{t} = (\mathbf{W} \mathbf{v}_{t-l} + \mathbf{x}_t
\mathbf{a}_{t-1} + \mathbf{\epsilon}_{t})
}{
y_{t} = (W v_{t-l} + x_t a_{t-1} + \epsilon_{t})
}

\deqn{
\mathbf{v}_{t} = \mathbf{F} \mathbf{v}_{t-l} + \mathbf{G}
\mathbf{\epsilon}_{t}
}{
v_{t} = F v_{t-l} + G \epsilon_{t}
}

\deqn{\mathbf{a}_{t} = \mathbf{F_{X}} \mathbf{a}_{t-1} + \mathbf{G_{X}}
\mathbf{\epsilon}_{t} / \mathbf{x}_{t}}{a_{t} = F_{X} a_{t-1} + G_{X} \epsilon_{t}
/ x_{t}}

Where \eqn{y_{t}} is the vector of time series on observation \eqn{t},
\eqn{\mathbf{v}_{t}} is the matrix of
states and \eqn{l} is the matrix of lags, \eqn{\mathbf{x}_t} is the vector of
exogenous variables. \eqn{\mathbf{W}} is the measurement matrix, \eqn{\mathbf{F}}
is the transition matrix and \eqn{\mathbf{G}} is the persistence matrix.
Finally, \eqn{\epsilon_{t}} is the vector of error terms.

Conventionally we formulate values as:

\deqn{\mathbf{y}'_t = (y_{1,t}, y_{2,t}, \dots, y_{m,t})}{y_t = (y_{1,t}, y_{2,t},
\dots, y_{m,t}),}
where \eqn{m} is the number of series in the group.
\deqn{\mathbf{v}'_t = (v_{1,t}, v_{2,t}, \dots, v_{m,t})}{v'_t = (v_{1,t}, v_{2,t},
\dots, v_{m,t}),}
where \eqn{v_{i,t}} is vector of components for i-th time series.
\deqn{\mathbf{W}' = (w_{1}, \dots , 0;
\vdots , \ddots , \vdots;
0 , \vdots , w_{m})}{W' = (w_{1}, ... , 0;
... , ... , ...;
0 , ... , w_{m})} is matrix of measurement vectors.

For the details on the additive model see Hyndman et al. (2008),
chapter 17.

In case of multiplicative model, instead of the vector y_t we use its logarithms.
As a result the multiplicative model is much easier to work with.

For some more information about the model and its implementation, see the
vignette: \code{vignette("ves","legion")}
}
\examples{

Y <- ts(cbind(rnorm(100,100,10),rnorm(100,75,8)),frequency=12)

# The simplest model applied to the data with the default values
ves(Y,model="ANN",h=10,holdout=TRUE)

# Damped trend model with the dependent persistence
ves(Y,model="AAdN",persistence="d",h=10,holdout=TRUE)

# Multiplicative damped trend model with individual phi
\donttest{ves(Y,model="MMdM",persistence="i",h=10,holdout=TRUE,initialSeason="c")}

# Automatic selection between pure models
\donttest{ves(Y,model="PPP",persistence="i",h=10,holdout=TRUE,initialSeason="c")}

# Intermittent demand vector model
Y <- cbind(c(rpois(25,0.1),rpois(25,0.5),rpois(25,1),rpois(25,5)),
           c(rpois(25,0.1),rpois(25,0.5),rpois(25,1),rpois(25,5)))

\donttest{ves(Y,model="MNN",h=10,holdout=TRUE,occurrence="l")}

}
\references{
\itemize{
\item de Silva A., Hyndman R.J. and Snyder, R.D. (2010). The vector
innovations structural time series framework: a simple approach to
multivariate forecasting. Statistical Modelling, 10 (4), pp.353-374
\item Hyndman, R.J., Koehler, A.B., Ord, J.K., and Snyder, R.D. (2008)
Forecasting with exponential smoothing: the state space approach,
Springer-Verlag.
\item Lütkepohl, H. (2005). New Introduction to Multiple Time Series
Analysis. New introduction to Multiple Time Series Analysis. Berlin,
Heidelberg: Springer Berlin Heidelberg.
\doi{10.1007/978-3-540-27752-1}
\item Svetunkov, I., Chen, H., & Boylan, J. E. (2023).
A new taxonomy for vector exponential smoothing and its application
to seasonal time series. European Journal of Operational Research,
304(3), 964–980. \doi{10.1016/j.ejor.2022.04.040}
}
}
\seealso{
\code{\link[legion]{vets}, \link[smooth]{es}, \link[smooth]{adam}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.com}
}
\keyword{models}
\keyword{multivariate}
\keyword{nonlinear}
\keyword{regression}
\keyword{ts}
