% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pool-LRT.R
\name{lavTestLRT.mi}
\alias{lavTestLRT.mi}
\alias{anova,lavaan.mi-method}
\title{Likelihood Ratio Test for Multiple Imputations}
\usage{
lavTestLRT.mi(
  object,
  ...,
  modnames = NULL,
  asANOVA = TRUE,
  pool.method = c("D4", "D3", "D2"),
  omit.imps = c("no.conv", "no.se"),
  asymptotic = FALSE,
  pool.robust = FALSE
)

\S4method{anova}{lavaan.mi}(object, ...)
}
\arguments{
\item{object}{An object of class \linkS4class{lavaan.mi}}

\item{...}{Additional objects of class \linkS4class{lavaan.mi}, as
well as arguments passed to \code{\link[lavaan:lavTestLRT]{lavaan::lavTestLRT()}} when
\code{pool.method = "D2"} and \code{pool.robust = TRUE}.}

\item{modnames}{Optional \code{character} of model names to use as row names
in the resulting matrix of results (when more than 2 models are compared)}

\item{asANOVA}{\code{logical} indicating whether to return an object of
class \code{"anova"}.  If \code{FALSE}, a numeric vector is returned for
one (pair of) model(s), or a \code{data.frame} is returned for multiple
pairs of models.}

\item{pool.method}{\code{character} indicating which pooling method to use.
\itemize{
\item \code{"D4"}, \code{"new.LRT"}, \code{"cm"}, or \code{"chan.meng"}
requests the method described by Chan & Meng (2022).
This is currently the default.
\item \code{"D3"}, \code{"old.LRT"}, \code{"mr"}, or \code{"meng.rubin"}
requests the method described by Meng & Rubin (1992).
\item \code{"D2"}, \code{"LMRR"}, or \code{"Li.et.al"} requests the
complete-data LRT statistic should be calculated using each
imputed data set, which will then be pooled across imputations, as
described in Li, Meng, Raghunathan, & Rubin (1991).
}
Find additional details in Enders (2010, chapter 8).}

\item{omit.imps}{\code{character} vector specifying criteria for omitting
imputations from pooled results.  Can include any of
\code{c("no.conv", "no.se", "no.npd")}, the first 2 of which are the
default setting, which excludes any imputations that did not
converge or for which standard errors could not be computed.  The
last option (\code{"no.npd"}) would exclude any imputations which
yielded a nonpositive definite covariance matrix for observed or
latent variables, which would include any "improper solutions" such
as Heywood cases. Specific imputation numbers can also be included in this
argument, in case users want to  apply their own custom omission criteria
(or simulations can use different numbers of imputations without
redundantly refitting the model).}

\item{asymptotic}{\code{logical}. If \code{FALSE} (default), the pooled test
will be returned as an \emph{F}-distributed statistic with numerator
(\code{df1}) and denominator (\code{df2}) degrees of freedom.
If \code{TRUE}, the pooled \emph{F} statistic will be multiplied by its
\code{df1} on the assumption that its \code{df2} is sufficiently large
enough that the statistic will be asymptotically \eqn{\chi^2} distributed
with \code{df1}.}

\item{pool.robust}{\code{logical}. Ignored unless \code{pool.method = "D2"} and a
robust test was requested. If \code{pool.robust = TRUE}, the robust test
statistic is pooled, whereas \code{pool.robust = FALSE} will pool
the naive test statistic (or difference statistic) and apply the average
scale/shift parameter to it. The harmonic mean is applied to the scaling
factor, whereas the arithmetic mean is applied to the shift parameter.}
}
\value{
\itemize{
\item When \code{asANOVA=TRUE}, returns an object of class \link[stats:anova]{stats::anova} with a
a test of model fit for a single model (\code{object}) or test(s) of the
difference(s) in fit between nested models passed via \code{...} (either an
\code{F} or \eqn{\chi^2} statistic, depending on the \code{asymptotic} argument),
its degrees of freedom, its \emph{p} value, and 2 missing-data diagnostics:
the relative increase in variance (RIV = FMI / (1 \eqn{-} FMI)) and the
fraction of missing information (FMI = RIV / (1 + RIV)).
\item When \code{asANOVA=FALSE}, returns a vector containing the LRT statistic for
a single model or comparison of a single pair of models, or a
\code{data.frame} of multiple model comparisons. Robust statistics will also
include the average (across imputations) scaling factor and
(if relevant) shift parameter(s), unless \code{pool.robust = TRUE}.
When using \code{pool.method = "D3"} or \code{"D4"}, the vector for a single
model also includes its average log-likelihood and information criteria.
}
}
\description{
Likelihood ratio test (LRT) for lavaan models fitted to multiple imputed
data sets.
}
\details{
The \code{"D2"} method is available using any estimator and test statistic.
When using a likelihood-based estimator, 2 additional methods are available
to pool the LRT.
\itemize{
\item The Meng & Rubin (1992) method, commonly referred to as \code{"D3"}.
This method has many problems, discussed in Chan & Meng (2022).
\item The Chan & Meng (2022) method, referred to as \code{"D4"} by
Grund et al. (2023), resolves problems with \code{"D3"}.
}
When \code{"D2"} is not explicitly requested in situations it is the only
applicable method, (e.g., DWLS for categorical outcomes), users are notified
that \code{pool.method} was set to \code{"D2"}.

\code{pool.method = "Mplus"} implies \code{"D3"} and \code{asymptotic = TRUE}
(see Asparouhov & Muthen, 2010).

Note that the \code{anova()} method simply calls \code{lavTestLRT.mi()}.
}
\examples{
data(HS20imps) # import a list of 20 imputed data sets

## specify CFA model from ?lavaan::cfa help page
HS.model <- '
  visual  =~ x1 + x2 + x3
  textual =~ x4 + x5 + x6
  speed   =~ x7 + x8 + x9
'

fit1 <- cfa.mi(HS.model, data = HS20imps, estimator = "mlm")

## By default, pool.method = "D4".
## Must request an asymptotic chi-squared statistic
## in order to accommodate a robust correction.
lavTestLRT.mi(fit1, asymptotic = TRUE)
## or   anova(fit1, asymptotic = TRUE)

## Comparison with more constrained (nested) models: parallel indicators
HS.parallel <- '
  visual  =~ x1 + 1*x2 + 1*x3
  textual =~ x4 + 1*x5 + 1*x6
  speed   =~ x7 + 1*x8 + 1*x9
'

fitp <- cfa.mi(HS.parallel, data = HS20imps, estimator = "mlm")

## Even more constrained model: orthogonal factors
fit0 <- cfa.mi(HS.parallel, data = HS20imps, estimator = "mlm",
               orthogonal = TRUE)

## Compare 3 models, and pass the lavTestLRT(method=) argument
\donttest{
lavTestLRT.mi(fit1, fit0, fitp, asymptotic = TRUE,
              method = "satorra.bentler.2010")

## For a single model, you can request a vector instead of an anova-class
## table in order to see naive information criteria (only using D3 or D4),
## which are calculated using the average log-likelihood across imputations.
lavTestLRT.mi(fit1, asANOVA = FALSE)



## When using a least-squares (rather than maximum-likelihood) estimator,
## only the D2 method is available.  For example, ordered-categorical data:
data(binHS5imps) # import a list of 5 imputed data sets

## fit model using default DWLS estimation
fit1c <- cfa.mi(HS.model   , data = binHS5imps, ordered = TRUE)
fit0c <- cfa.mi(HS.parallel, data = binHS5imps, ordered = TRUE,
                orthogonal = TRUE)

## Using D2, you can either robustify the pooled naive statistic ...
lavTestLRT.mi(fit1c, fit0c, asymptotic = TRUE, pool.method = "D2")
## ... or pool the robust chi-squared statistic (NOT recommended)
lavTestLRT.mi(fit1c, fit0c, asymptotic = TRUE, pool.method = "D2",
              pool.robust = TRUE)

## When calculating fit indices, you can pass lavTestLRT.mi() arguments:
fitMeasures(fit1c, output = "text",
            # lavTestLRT.mi() arguments:
            pool.method = "D2", pool.robust = TRUE)
}

}
\references{
Asparouhov, T., & Muthen, B. (2010). \emph{Chi-square statistics
with multiple imputation}. Technical Report. Retrieved from
\url{http://www.statmodel.com/}

Chan, K. W., & Meng, X. L. (2022). Multiple improvements of multiple
imputation likelihood ratio tests. \emph{Statistica Sinica, 32},
1489--1514. \doi{10.5705/ss.202019.0314}

Enders, C. K. (2010). \emph{Applied missing data analysis}.
New York, NY: Guilford.

Grund, S., Lüdtke, O., & Robitzsch, A. (2023). Pooling methods for
likelihood ratio tests in multiply imputed data sets.
\emph{Psychological Methods, 28}(5), 1207--1221.
\doi{10.1037/met0000556}

Li, K.-H., Meng, X.-L., Raghunathan, T. E., & Rubin, D. B. (1991).
Significance levels from repeated \emph{p}-values with multiply-imputed
data. \emph{Statistica Sinica, 1}(1), 65--92. Retrieved from
\url{https://www.jstor.org/stable/24303994}

Meng, X.-L., & Rubin, D. B. (1992). Performing likelihood ratio tests with
multiply-imputed data sets. \emph{Biometrika, 79}(1), 103--111.
\doi{10.2307/2337151}

Rubin, D. B. (1987). \emph{Multiple imputation for nonresponse in surveys}.
New York, NY: Wiley. \doi{10.1002/9780470316696}
}
\seealso{
\code{\link[lavaan:lavTestLRT]{lavaan::lavTestLRT()}} for arguments that can be passed via \dots,
and use \code{\link[lavaan:fitMeasures]{lavaan::fitMeasures()}} to obtain fit indices calculated from pooled
test statistics.
}
\author{
Terrence D. Jorgensen (University of Amsterdam;
\email{TJorgensen314@gmail.com})

Based on source code for \code{\link[lavaan:lavTestLRT]{lavaan::lavTestLRT()}} by Yves Rosseel.
}
