#' Compute the optimal sample size for the weighted Hochberg procedure
#'
#' @param alpha the significance level
#' @param betavec a numeric vector of two values, including one minus the desired power for rejecting H1 and one minus the desired power for rejecting H2
#' @param deltavec a numeric vector of two values representing the effect sizes for the two hypotheses
#' @param rho the correlation coefficient between two test statistics
#' @param ninterval a vector containing the end-points of the interval to be searched for optimal sample size
#' @param alphalist a vector of discrete alpha values
#'
#' @returns the overall optimal sample size for the weighted Hochberg procedure
#' @export
#' @importFrom stats uniroot
#' @author Jiangtao Gou
#' @author Fengqing Zhang
#' @references
#' Gou, J., Chang, Y., Li, T., and Zhang, F. (2025). Improved trimmed weighted Hochberg procedures with two endpoints and sample size optimization. Technical Report.
#' @examples
#' psivec <- c(0.76, 0.72)
#' thetavec <- log(psivec)
#' deltavec <- (-thetavec)/2
#' result <- optsamplesize_wHp(alpha = 0.05, betavec = c(0.20, 0.10),
#' deltavec = deltavec , rho = -0.1)
#' result$nopt
optsamplesize_wHp <- function (alpha, betavec, deltavec, rho, ninterval = c(2,2000), alphalist = seq(from = 0, to = alpha, by = 0.005)) {
  # R20241124b.R
  NNN <- length(alphalist)
  n1vec <- rep(NA, times = NNN)
  n2vec <- rep(NA, times = NNN)
  #
  for (iii in 1:NNN) {
    #
    alpha1 <- alphalist[iii]
    #
    find_n_result1 <- uniroot(f = wHpTarget1, interval = ninterval, tol = .Machine$double.eps^0.8, alpha1 = alpha1, alpha = alpha, beta1 = betavec[1], deltavec = deltavec, rho = rho)
    n_opt_1 <- find_n_result1$root
    n1vec[iii] <- n_opt_1
    #
    find_n_result2 <- uniroot(f = wHpTarget2, interval = ninterval, tol = .Machine$double.eps^0.8, alpha1 = alpha1, alpha = alpha, beta2 = betavec[2], deltavec = deltavec, rho = rho)
    n_opt_2 <- find_n_result2$root
    n2vec[iii] <- n_opt_2
  }
  #
  values <- n1vec - n2vec
  x <- alphalist
  interpolate_zero(values, x)
  nloccont <- interpolate_zero(values, 1:NNN)
  nlocflr <- floor(nloccont)
  n1opt <- n1vec[nloccont] + (n1vec[nloccont+1] - n1vec[nloccont])*(nloccont - nlocflr)
  n2opt <- n2vec[nloccont] + (n2vec[nloccont+1] - n2vec[nloccont])*(nloccont - nlocflr)
  nopt <- (n1opt + n2opt)/2
  #
  return (list(nopt = nopt, n1opt = n1opt, n2opt = n2opt, alphalist = alphalist, n1vec = n1vec, n2vec = n2vec))
}
