% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api_metadata.R
\name{get_metadata}
\alias{get_metadata}
\title{Retrieve detailed metadata about an IPUMS data source}
\usage{
get_metadata(
  collection,
  dataset = NULL,
  data_table = NULL,
  time_series_table = NULL,
  api_key = Sys.getenv("IPUMS_API_KEY")
)
}
\arguments{
\item{collection}{Character string indicating the IPUMS collection for which
to retrieve metadata.}

\item{dataset}{Name of an individual dataset from an IPUMS aggregate data
collection for which to retrieve metadata.}

\item{data_table}{Name of an individual data table from an IPUMS aggregate
data collection for which to retrieve metadata. If provided and
\code{collection = "nhgis"}, an associated \code{dataset} must also be specified.}

\item{time_series_table}{If \code{collection = "nhgis"}, name of an individual
time series table from IPUMS NHGIS for which to retrieve metadata.}

\item{api_key}{API key associated with your user account. Defaults to the
value of the \code{IPUMS_API_KEY} environment variable. See
\code{\link[=set_ipums_api_key]{set_ipums_api_key()}}.}
}
\value{
A named list of metadata for the specified data source.
}
\description{
Retrieve metadata containing API codes and descriptions for an IPUMS data
source. See the \href{https://developer.ipums.org/docs/v2/workflows/explore_metadata/}{IPUMS developer documentation}
for details about the metadata provided for individual data collections
and API endpoints.

To retrieve a summary of all available data sources of a particular
type, use \code{\link[=get_metadata_catalog]{get_metadata_catalog()}}. This output can be used to identify the
names of data sources for which to request detailed metadata.

Currently, comprehensive metadata is only available for IPUMS NHGIS
and IPUMS IHGIS. See \code{\link[=get_sample_info]{get_sample_info()}} to list basic sample information
for IPUMS microdata collections.

Learn more about the IPUMS API in \code{vignette("ipums-api")}.
}
\examples{
\dontrun{
library(dplyr)

# Get detailed metadata for a single source with its associated argument:
cs5_meta <- get_metadata("nhgis", time_series_table = "CS5")
cs5_meta$geog_levels

# Use the available values when defining an NHGIS extract request
define_extract_agg(
  "nhgis",
  time_series_tables = tst_spec("CS5", geog_levels = "state")
)

# Detailed metadata is also provided for datasets and data tables
get_metadata("nhgis", dataset = "1990_STF1")
get_metadata("nhgis", data_table = "NP1", dataset = "1990_STF1")
get_metadata("ihgis", dataset = "KZ2009pop")

# Iterate over data sources to retrieve detailed metadata for several
# records. For instance, to get variable metadata for a set of data tables:
tables <- c("NP1", "NP2", "NP10")

var_meta <- purrr::map(
  tables,
  function(dt) {
    dt_meta <- get_metadata("nhgis", dataset = "1990_STF1", data_table = dt)

    # This ensures you avoid hitting rate limit for large numbers of tables
    Sys.sleep(1)

    dt_meta$variables
  }
)
}
}
\seealso{
\code{\link[=get_metadata_catalog]{get_metadata_catalog()}} to obtain a summary of available data sources for
a given IPUMS data collection.

\code{\link[=define_extract_agg]{define_extract_agg()}} to create an IPUMS aggregate data extract
definition.
}
