#' D'Hondt function
#'
#' @rdname ElecFuns
#' @param threshold Proportion of votes that a party should surpass in order to
#' be able to win seats (from 0 to 1). 
#' 
#' @section Details: The dhondt function divides parties’ vote totals 
#' successively by 1, 2, 3, 4, 5, and so on (until m). Seats are then awarded 
#' sequentially starting with the party that enjoys the largest quotient 
#' until no more seats are available.
#' 
#' @export
#'
#' @examples 
#' 
#' ## D'hondt without threshold: 
#' 
#' dhondt(v=example, m=3)
#' 
#' ## D'hondt with 30% threshold: 
#' 
#' dhondt(v=example, m=3, threshold=0.3) 

dhondt <- function(v,
                   m,
                   threshold = 0.0, 
                   ...){
  if(nrow(v) > 1){
    stop("D'Hondt undefined for ranked votes.")
  }
  if(threshold > max(v)/sum(v)){
    stop("Threshold is higher than maximum proportion of votes")
  }
  # apply threshold
  if(sum(prop.table(v) < threshold) != ncol(v)){
    v[1, prop.table(v) < threshold]  <- 0         
  }  
  div_table <- outer(c(v), 1:m, "/")
  hi_quot <- order(c(div_table), decreasing = TRUE)[1:m]
  seats <- table(factor(arrayInd(hi_quot, dim(div_table))[,1], levels = 1:ncol(v)))
  names(seats) <- colnames(v)
  return (seats)
}