/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

#include "H5VLnative_private.h" 

#ifndef H5_NO_DEPRECATED_SYMBOLS

typedef struct H5O_visit1_adapter_t {
    H5O_iterate1_t real_op;      
    unsigned       fields;       
    void          *real_op_data; 
} H5O_visit1_adapter_t;

static herr_t H5O__reset_info1(H5O_info1_t *oinfo);
static herr_t H5O__iterate1_adapter(hid_t obj_id, const char *name, const H5O_info2_t *oinfo2, void *op_data);
static herr_t H5O__get_info_old(H5VL_object_t *vol_obj, H5VL_loc_params_t *loc_params, H5O_info1_t *oinfo,
                                unsigned fields);

static herr_t
H5O__reset_info1(H5O_info1_t *oinfo)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    memset(oinfo, 0, sizeof(H5O_info1_t));
    oinfo->type = H5O_TYPE_UNKNOWN;
    oinfo->addr = HADDR_UNDEF;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__iterate1_adapter(hid_t obj_id, const char *name, const H5O_info2_t *oinfo2, void *op_data)
{
    H5O_visit1_adapter_t *shim_data = (H5O_visit1_adapter_t *)op_data;
    H5O_info1_t           oinfo;      
    unsigned              dm_fields;  
    unsigned              nat_fields; 
    H5VL_object_t        *vol_obj;
    herr_t                ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(oinfo2);
    assert(op_data);

    
    if (H5O__reset_info1(&oinfo) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't reset object data struct");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, H5_ITER_ERROR, "invalid location identifier");

    
    dm_fields = shim_data->fields & (H5O_INFO_BASIC | H5O_INFO_TIME | H5O_INFO_NUM_ATTRS);
    if (dm_fields) {
        
        if (shim_data->fields & H5O_INFO_BASIC) {
            H5I_type_t vol_obj_type = H5I_BADID; 
            void      *vol_obj_data;

            oinfo.fileno = oinfo2->fileno;
            oinfo.type   = oinfo2->type;
            oinfo.rc     = oinfo2->rc;

            
            if ((vol_obj_type = H5I_get_type(obj_id)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL, "invalid location identifier");

            
            if (NULL == (vol_obj_data = H5VL_object_data(vol_obj)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get underlying VOL object");

            
            if (H5VL_native_token_to_addr(vol_obj_data, vol_obj_type, oinfo2->token, &oinfo.addr) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNSERIALIZE, FAIL,
                            "can't deserialize object token into address");
        }
        if (shim_data->fields & H5O_INFO_TIME) {
            oinfo.atime = oinfo2->atime;
            oinfo.mtime = oinfo2->mtime;
            oinfo.ctime = oinfo2->ctime;
            oinfo.btime = oinfo2->btime;
        }
        if (shim_data->fields & H5O_INFO_NUM_ATTRS)
            oinfo.num_attrs = oinfo2->num_attrs;
    }

    
    nat_fields = shim_data->fields & (H5O_INFO_HDR | H5O_INFO_META_SIZE);
    if (nat_fields) {
        H5VL_optional_args_t               vol_cb_args;  
        H5VL_native_object_optional_args_t obj_opt_args; 
        H5VL_loc_params_t                  loc_params;   
        H5O_native_info_t                  nat_info;     

        
        loc_params.type                         = H5VL_OBJECT_BY_NAME;
        loc_params.loc_data.loc_by_name.name    = name;
        loc_params.loc_data.loc_by_name.lapl_id = H5P_LINK_ACCESS_DEFAULT;
        loc_params.obj_type                     = H5I_get_type(obj_id);

        
        obj_opt_args.get_native_info.fields = nat_fields;
        obj_opt_args.get_native_info.ninfo  = &nat_info;
        vol_cb_args.op_type                 = H5VL_NATIVE_OBJECT_GET_NATIVE_INFO;
        vol_cb_args.args                    = &obj_opt_args;

        
        if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                 H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get native info for object");

        
        if (shim_data->fields & H5O_INFO_HDR)
            H5MM_memcpy(&(oinfo.hdr), &(nat_info.hdr), sizeof(H5O_hdr_info_t));
        if (shim_data->fields & H5O_INFO_META_SIZE) {
            H5MM_memcpy(&(oinfo.meta_size.obj), &(nat_info.meta_size.obj), sizeof(H5_ih_info_t));
            H5MM_memcpy(&(oinfo.meta_size.attr), &(nat_info.meta_size.attr), sizeof(H5_ih_info_t));
        }
    }

    
    ret_value = (shim_data->real_op)(obj_id, name, &oinfo, shim_data->real_op_data);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__get_info_old(H5VL_object_t *vol_obj, H5VL_loc_params_t *loc_params, H5O_info1_t *oinfo, unsigned fields)
{
    unsigned dm_fields;           
    unsigned nat_fields;          
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(vol_obj);
    assert(loc_params);

    
    if (H5O__reset_info1(oinfo) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't reset object data struct");

    
    dm_fields = fields & (H5O_INFO_BASIC | H5O_INFO_TIME | H5O_INFO_NUM_ATTRS);
    if (dm_fields) {
        H5VL_object_get_args_t vol_cb_args; 
        H5O_info2_t            dm_info;     

        
        vol_cb_args.op_type              = H5VL_OBJECT_GET_INFO;
        vol_cb_args.args.get_info.oinfo  = &dm_info;
        vol_cb_args.args.get_info.fields = dm_fields;

        
        if (H5VL_object_get(vol_obj, loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get data model info for object");

        
        if (fields & H5O_INFO_BASIC) {
            void *vol_obj_data;

            oinfo->fileno = dm_info.fileno;
            oinfo->type   = dm_info.type;
            oinfo->rc     = dm_info.rc;

            
            if (NULL == (vol_obj_data = H5VL_object_data(vol_obj)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get underlying VOL object");

            
            if (H5VL_native_token_to_addr(vol_obj_data, loc_params->obj_type, dm_info.token, &oinfo->addr) <
                0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNSERIALIZE, FAIL,
                            "can't deserialize object token into address");
        } 
        if (fields & H5O_INFO_TIME) {
            oinfo->atime = dm_info.atime;
            oinfo->mtime = dm_info.mtime;
            oinfo->ctime = dm_info.ctime;
            oinfo->btime = dm_info.btime;
        } 
        if (fields & H5O_INFO_NUM_ATTRS)
            oinfo->num_attrs = dm_info.num_attrs;
    } 

    
    nat_fields = fields & (H5O_INFO_HDR | H5O_INFO_META_SIZE);
    if (nat_fields) {
        H5VL_optional_args_t               vol_cb_args;  
        H5VL_native_object_optional_args_t obj_opt_args; 
        H5O_native_info_t                  nat_info;     

        
        obj_opt_args.get_native_info.fields = nat_fields;
        obj_opt_args.get_native_info.ninfo  = &nat_info;
        vol_cb_args.op_type                 = H5VL_NATIVE_OBJECT_GET_NATIVE_INFO;
        vol_cb_args.args                    = &obj_opt_args;

        
        if (H5VL_object_optional(vol_obj, loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                 H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get native info for object");

        
        if (fields & H5O_INFO_HDR)
            H5MM_memcpy(&(oinfo->hdr), &(nat_info.hdr), sizeof(H5O_hdr_info_t));
        if (fields & H5O_INFO_META_SIZE) {
            H5MM_memcpy(&(oinfo->meta_size.obj), &(nat_info.meta_size.obj), sizeof(H5_ih_info_t));
            H5MM_memcpy(&(oinfo->meta_size.attr), &(nat_info.meta_size.attr), sizeof(H5_ih_info_t));
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Oopen_by_addr(hid_t loc_id, haddr_t addr)
{
    H5VL_object_t    *vol_obj;                  
    H5I_type_t        vol_obj_type = H5I_BADID; 
    H5I_type_t        opened_type;              
    void             *opened_obj = NULL;        
    H5VL_loc_params_t loc_params;               
    H5O_token_t       obj_token = {0};          
    bool              is_native_vol_obj;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");
    if (is_native_vol_obj) {
        void *vol_obj_data;

        
        if (NULL == (vol_obj_data = H5VL_object_data(vol_obj)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID, "can't retrieve pointer to native object");

        
        if (H5VL_native_addr_to_token(vol_obj_data, vol_obj_type, addr, &obj_token) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTSERIALIZE, H5I_INVALID_HID,
                        "can't serialize address into object token");
    } 
    else
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, H5I_INVALID_HID,
                    "H5Oopen_by_addr is only meant to be used with the native VOL connector");

    loc_params.type                        = H5VL_OBJECT_BY_TOKEN;
    loc_params.loc_data.loc_by_token.token = &obj_token;
    loc_params.obj_type                    = vol_obj_type;

    
    if (NULL == (opened_obj = H5VL_object_open(vol_obj, &loc_params, &opened_type, H5P_DATASET_XFER_DEFAULT,
                                               H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open object");

    
    if ((ret_value = H5VL_register(opened_type, opened_obj, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register object handle");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info1(hid_t loc_id, H5O_info1_t *oinfo )
{
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    bool              is_native_vol_obj = false;
    herr_t            ret_value         = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine if VOL object is native connector object");

    
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                    "Deprecated H5Oget_info1 is only meant to be used with the native VOL connector");

    
    if (H5O__get_info_old(vol_obj, &loc_params, oinfo, H5O_INFO_ALL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get deprecated info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info_by_name1(hid_t loc_id, const char *name, H5O_info1_t *oinfo , hid_t lapl_id)
{
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    bool              is_native_vol_obj = false;
    herr_t            ret_value         = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine if VOL object is native connector object");

    
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                    "Deprecated H5Oget_info_by_name1 is only meant to be used with the native VOL connector");

    
    if (H5O__get_info_old(vol_obj, &loc_params, oinfo, H5O_INFO_ALL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get deprecated info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info_by_idx1(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                    hsize_t n, H5O_info1_t *oinfo , hid_t lapl_id)
{
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    bool              is_native_vol_obj = false;
    herr_t            ret_value         = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!group_name || !*group_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no info struct");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = group_name;
    loc_params.loc_data.loc_by_idx.idx_type = idx_type;
    loc_params.loc_data.loc_by_idx.order    = order;
    loc_params.loc_data.loc_by_idx.n        = n;
    loc_params.loc_data.loc_by_idx.lapl_id  = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine if VOL object is native connector object");

    
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                    "Deprecated H5Oget_info_by_idx1 is only meant to be used with the native VOL connector");

    
    if (H5O__get_info_old(vol_obj, &loc_params, oinfo, H5O_INFO_ALL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get deprecated info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info2(hid_t loc_id, H5O_info1_t *oinfo , unsigned fields)
{
    H5VL_object_t    *vol_obj; 
    H5VL_loc_params_t loc_params;
    bool              is_native_vol_obj;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, H5I_INVALID_HID,
                    "Deprecated H5Oget_info2 is only meant to be used with the native VOL connector");

    
    if (H5O__get_info_old(vol_obj, &loc_params, oinfo, fields) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get deprecated info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info_by_name2(hid_t loc_id, const char *name, H5O_info1_t *oinfo , unsigned fields,
                     hid_t lapl_id)
{
    H5VL_object_t    *vol_obj; 
    H5VL_loc_params_t loc_params;
    bool              is_native_vol_obj;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, H5I_INVALID_HID,
                    "Deprecated H5Oget_info_by_name2 is only meant to be used with the native VOL connector");

    
    if (H5O__get_info_old(vol_obj, &loc_params, oinfo, fields) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get deprecated info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info_by_idx2(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                    hsize_t n, H5O_info1_t *oinfo , unsigned fields, hid_t lapl_id)
{
    H5VL_object_t    *vol_obj; 
    H5VL_loc_params_t loc_params;
    bool              is_native_vol_obj;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!group_name || !*group_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no info struct");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = group_name;
    loc_params.loc_data.loc_by_idx.idx_type = idx_type;
    loc_params.loc_data.loc_by_idx.order    = order;
    loc_params.loc_data.loc_by_idx.n        = n;
    loc_params.loc_data.loc_by_idx.lapl_id  = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, H5I_INVALID_HID,
                    "Deprecated H5Oget_info_by_idx2 is only meant to be used with the native VOL connector");

    
    if (H5O__get_info_old(vol_obj, &loc_params, oinfo, fields) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get deprecated info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ovisit1(hid_t obj_id, H5_index_t idx_type, H5_iter_order_t order, H5O_iterate1_t op, void *op_data)
{
    H5VL_object_t              *vol_obj = NULL; 
    H5VL_object_specific_args_t vol_cb_args;    
    H5VL_loc_params_t           loc_params;     
    H5O_visit1_adapter_t        shim_data;      
    herr_t                      ret_value;      
    bool                        is_native_vol_obj = false;

    FUNC_ENTER_API(FAIL)

    
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no callback operator specified");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine if VOL object is native connector object");

    
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                    "Deprecated H5Ovisit1 is only meant to be used with the native VOL connector");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(obj_id);

    
    shim_data.real_op      = op;
    shim_data.fields       = H5O_INFO_ALL;
    shim_data.real_op_data = op_data;

    
    vol_cb_args.op_type             = H5VL_OBJECT_VISIT;
    vol_cb_args.args.visit.idx_type = idx_type;
    vol_cb_args.args.visit.order    = order;
    vol_cb_args.args.visit.op       = H5O__iterate1_adapter;
    vol_cb_args.args.visit.op_data  = &shim_data;
    vol_cb_args.args.visit.fields   = H5O_INFO_ALL;

    
    if ((ret_value = H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                          H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object visitation failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ovisit_by_name1(hid_t loc_id, const char *obj_name, H5_index_t idx_type, H5_iter_order_t order,
                  H5O_iterate1_t op, void *op_data, hid_t lapl_id)
{
    H5VL_object_t              *vol_obj = NULL; 
    H5VL_object_specific_args_t vol_cb_args;    
    H5VL_loc_params_t           loc_params;     
    H5O_visit1_adapter_t        shim_data;      
    herr_t                      ret_value;      
    bool                        is_native_vol_obj = false;

    FUNC_ENTER_API(FAIL)

    
    if (!obj_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "obj_name parameter cannot be NULL");
    if (!*obj_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "obj_name parameter cannot be an empty string");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no callback operator specified");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine if VOL object is native connector object");

    
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                    "Deprecated H5Ovisit_by_name1 is only meant to be used with the native VOL connector");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = obj_name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    shim_data.real_op      = op;
    shim_data.fields       = H5O_INFO_ALL;
    shim_data.real_op_data = op_data;

    
    vol_cb_args.op_type             = H5VL_OBJECT_VISIT;
    vol_cb_args.args.visit.idx_type = idx_type;
    vol_cb_args.args.visit.order    = order;
    vol_cb_args.args.visit.op       = H5O__iterate1_adapter;
    vol_cb_args.args.visit.op_data  = &shim_data;
    vol_cb_args.args.visit.fields   = H5O_INFO_ALL;

    
    if ((ret_value = H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                          H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object visitation failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ovisit2(hid_t obj_id, H5_index_t idx_type, H5_iter_order_t order, H5O_iterate1_t op, void *op_data,
          unsigned fields)
{
    H5VL_object_t              *vol_obj;     
    H5VL_object_specific_args_t vol_cb_args; 
    H5VL_loc_params_t           loc_params;  
    H5O_visit1_adapter_t        shim_data;   
    bool                        is_native_vol_obj;
    herr_t                      ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no callback operator specified");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");

    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, H5I_INVALID_HID,
                    "Deprecated H5Ovisit2 is only meant to be used with the native VOL connector");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(obj_id);

    
    shim_data.real_op      = op;
    shim_data.fields       = fields;
    shim_data.real_op_data = op_data;

    
    vol_cb_args.op_type             = H5VL_OBJECT_VISIT;
    vol_cb_args.args.visit.idx_type = idx_type;
    vol_cb_args.args.visit.order    = order;
    vol_cb_args.args.visit.op       = H5O__iterate1_adapter;
    vol_cb_args.args.visit.op_data  = &shim_data;
    vol_cb_args.args.visit.fields   = fields;

    
    if ((ret_value = H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                          H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object iteration failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ovisit_by_name2(hid_t loc_id, const char *obj_name, H5_index_t idx_type, H5_iter_order_t order,
                  H5O_iterate1_t op, void *op_data, unsigned fields, hid_t lapl_id)
{
    H5VL_object_t              *vol_obj;     
    H5VL_object_specific_args_t vol_cb_args; 
    H5VL_loc_params_t           loc_params;  
    H5O_visit1_adapter_t        shim_data;   
    bool                        is_native_vol_obj;
    herr_t                      ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (!obj_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "obj_name parameter cannot be NULL");
    if (!*obj_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "obj_name parameter cannot be an empty string");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no callback operator specified");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");
    if (!is_native_vol_obj)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, H5I_INVALID_HID,
                    "Deprecated H5Ovisit_by_name2 is only meant to be used with the native VOL connector");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = obj_name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    shim_data.real_op      = op;
    shim_data.fields       = fields;
    shim_data.real_op_data = op_data;

    
    vol_cb_args.op_type             = H5VL_OBJECT_VISIT;
    vol_cb_args.args.visit.idx_type = idx_type;
    vol_cb_args.args.visit.order    = order;
    vol_cb_args.args.visit.op       = H5O__iterate1_adapter;
    vol_cb_args.args.visit.op_data  = &shim_data;
    vol_cb_args.args.visit.fields   = fields;

    
    if ((ret_value = H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                          H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object iteration failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

#endif 
