/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5VMprivate.h" 

H5FL_DEFINE_STATIC(H5HF_block_loc_t);

herr_t
H5HF__man_iter_init(H5HF_block_iter_t *biter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(biter);

    
    memset(biter, 0, sizeof(H5HF_block_iter_t));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__man_iter_start_offset(H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, hsize_t offset)
{
    H5HF_indirect_t *iblock;               
    haddr_t          iblock_addr;          
    unsigned         iblock_nrows;         
    H5HF_indirect_t *iblock_parent;        
    unsigned         iblock_par_entry;     
    hsize_t          curr_offset;          
    unsigned         row;                  
    unsigned         col;                  
    bool             root_block = true;    
    herr_t           ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(biter);
    assert(!biter->ready);

    
    assert(offset >= hdr->man_dtable.cparam.start_block_size);

    
    if (NULL == (biter->curr = H5FL_MALLOC(H5HF_block_loc_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for direct block free list section");

    
    do {
        bool did_protect; 

        
        for (row = 0; row < hdr->man_dtable.max_root_rows; row++)
            if ((offset >= hdr->man_dtable.row_block_off[row]) &&
                (offset < hdr->man_dtable.row_block_off[row] +
                              (hdr->man_dtable.cparam.width * hdr->man_dtable.row_block_size[row])))
                break;

        
        curr_offset = offset - hdr->man_dtable.row_block_off[row];

        
        H5_CHECK_OVERFLOW((curr_offset / hdr->man_dtable.row_block_size[row]), hsize_t, unsigned);
        col = (unsigned)(curr_offset / hdr->man_dtable.row_block_size[row]);

        
        biter->curr->row   = row;
        biter->curr->col   = col;
        biter->curr->entry = (row * hdr->man_dtable.cparam.width) + col;

        
        if (root_block) {
            iblock_addr      = hdr->man_dtable.table_addr;
            iblock_nrows     = hdr->man_dtable.curr_root_rows;
            iblock_parent    = NULL;
            iblock_par_entry = 0;

            
            biter->curr->up = NULL;

            
            root_block = false;
        } 
        else {
            hsize_t child_size; 

            
            iblock_parent    = biter->curr->up->context;
            iblock_par_entry = biter->curr->up->entry;

            
            iblock_addr = iblock_parent->ents[iblock_par_entry].addr;

            
            child_size   = hdr->man_dtable.row_block_size[biter->curr->up->row];
            iblock_nrows = (H5VM_log2_gen(child_size) - hdr->man_dtable.first_row_bits) + 1;
        } 

        
        if (NULL ==
            (iblock = H5HF__man_iblock_protect(hdr, iblock_addr, iblock_nrows, iblock_parent,
                                               iblock_par_entry, false, H5AC__NO_FLAGS_SET, &did_protect)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap indirect block");

        
        biter->curr->context = iblock;

        
        if (H5HF__iblock_incr(biter->curr->context) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL,
                        "can't increment reference count on shared indirect block");

        
        if (H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
        iblock = NULL;

        
        
        if (curr_offset == (col * hdr->man_dtable.row_block_size[row]) ||
            row < hdr->man_dtable.max_direct_rows) {
            assert(curr_offset - (col * hdr->man_dtable.row_block_size[row]) == 0);
            break; 
        }          
        
        else {
            H5HF_block_loc_t *new_loc; 

            
            if (NULL == (new_loc = H5FL_MALLOC(H5HF_block_loc_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                            "memory allocation failed for direct block free list section");

            
            new_loc->up = biter->curr;

            
            offset = curr_offset - (col * hdr->man_dtable.row_block_size[row]);

            
            biter->curr = new_loc;
        }        
    } while (1); 

    
    biter->ready = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iter_set_entry(const H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, unsigned entry)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(biter);

    
    biter->curr->entry = entry;
    biter->curr->row   = entry / hdr->man_dtable.cparam.width;
    biter->curr->col   = entry % hdr->man_dtable.cparam.width;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__man_iter_start_entry(H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, H5HF_indirect_t *iblock,
                           unsigned start_entry)
{
    H5HF_block_loc_t *new_loc   = NULL;    
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(biter);
    assert(!biter->ready);
    assert(iblock);

    
    if (NULL == (new_loc = H5FL_MALLOC(H5HF_block_loc_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for direct block free list section");

    
    new_loc->entry   = start_entry;
    new_loc->row     = start_entry / hdr->man_dtable.cparam.width;
    new_loc->col     = start_entry % hdr->man_dtable.cparam.width;
    new_loc->context = iblock;
    new_loc->up      = NULL;

    
    if (H5HF__iblock_incr(new_loc->context) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared indirect block");

    
    biter->curr = new_loc;

    
    biter->ready = true;

done:
    if (ret_value < 0 && new_loc)
        new_loc = H5FL_FREE(H5HF_block_loc_t, new_loc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iter_reset(H5HF_block_iter_t *biter)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(biter);

    
    if (biter->curr) {
        H5HF_block_loc_t *curr_loc; 
        H5HF_block_loc_t *next_loc; 

        
        curr_loc = biter->curr;
        while (curr_loc) {
            
            next_loc = curr_loc->up;

            
            if (curr_loc->context)
                if (H5HF__iblock_decr(curr_loc->context) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                                "can't decrement reference count on shared indirect block");

            
            curr_loc = H5FL_FREE(H5HF_block_loc_t, curr_loc);

            
            curr_loc = next_loc;
        } 

        
        biter->curr = NULL;
    } 

    
    biter->ready = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iter_next(H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, unsigned nentries)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(biter);
    assert(biter->curr);
    assert(biter->curr->context);
    assert(biter->curr->row < biter->curr->context->nrows);

    
    biter->curr->entry += nentries;
    biter->curr->row = biter->curr->entry / hdr->man_dtable.cparam.width;
    biter->curr->col = biter->curr->entry % hdr->man_dtable.cparam.width;
    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__man_iter_up(H5HF_block_iter_t *biter)
{
    H5HF_block_loc_t *up_loc;              
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(biter);
    assert(biter->ready);
    assert(biter->curr);
    assert(biter->curr->up);
    assert(biter->curr->context);

    
    if (H5HF__iblock_decr(biter->curr->context) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared indirect block");

    
    up_loc = biter->curr->up;

    
    biter->curr = H5FL_FREE(H5HF_block_loc_t, biter->curr);

    
    biter->curr = up_loc;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iter_down(H5HF_block_iter_t *biter, H5HF_indirect_t *iblock)
{
    H5HF_block_loc_t *down_loc  = NULL;    
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(biter);
    assert(biter->ready);
    assert(biter->curr);
    assert(biter->curr->context);

    
    if (NULL == (down_loc = H5FL_MALLOC(H5HF_block_loc_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for direct block free list section");

    
    down_loc->row     = 0;
    down_loc->col     = 0;
    down_loc->entry   = 0;
    down_loc->context = iblock;
    down_loc->up      = biter->curr;

    
    if (H5HF__iblock_incr(down_loc->context) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared indirect block");

    
    biter->curr = down_loc;

done:
    if (ret_value < 0 && down_loc)
        down_loc = H5FL_FREE(H5HF_block_loc_t, down_loc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_iter_curr(H5HF_block_iter_t *biter, unsigned *row, unsigned *col, unsigned *entry,
                    H5HF_indirect_t **block)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(biter);
    assert(biter->ready);

    
    if (row)
        *row = biter->curr->row;
    if (col)
        *col = biter->curr->col;
    if (entry)
        *entry = biter->curr->entry;
    if (block)
        *block = biter->curr->context;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

bool
H5HF__man_iter_ready(H5HF_block_iter_t *biter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(biter);

    FUNC_LEAVE_NOAPI(biter->ready)
} 
