% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_stream_smooth.R
\name{geom_stream_smooth}
\alias{geom_stream_smooth}
\alias{stat_stream_smooth}
\title{Create a Smoothed Vector Field Layer}
\usage{
geom_stream_smooth(
  mapping = NULL,
  data = NULL,
  stat = StatStreamField,
  position = "identity",
  ...,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  n = 11,
  xlim = NULL,
  ylim = NULL,
  normalize = TRUE,
  center = FALSE,
  type = "vector",
  formula = cbind(fx, fy) ~ x * y,
  eval_points = NULL,
  lineend = "butt",
  linejoin = "round",
  linemitre = 10,
  arrow = grid::arrow(angle = 20, length = unit(0.015, "npc"), type = "closed")
)

stat_stream_smooth(
  mapping = NULL,
  data = NULL,
  geom = GeomStream,
  position = "identity",
  ...,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  n = 11,
  xlim = NULL,
  ylim = NULL,
  normalize = TRUE,
  center = FALSE,
  type = "vector",
  formula = cbind(fx, fy) ~ x * y,
  eval_points = NULL,
  lineend = "butt",
  linejoin = "round",
  linemitre = 10,
  arrow = grid::arrow(angle = 20, length = unit(0.015, "npc"), type = "closed")
)
}
\arguments{
\item{mapping}{A set of aesthetic mappings created by \code{ggplot2::aes()}.
\strong{Required:} Must include \strong{\code{x}} and \strong{\code{y}}; vector displacements are defined
by \strong{\code{fx}} and \strong{\code{fy}}.}

\item{data}{A data frame containing the raw vector data.}

\item{stat}{The statistical transformation to use on the data. Defaults to
\code{"vector_smooth"}.}

\item{position}{Position adjustment, either as a string or the result of a
position adjustment function.}

\item{...}{Additional arguments passed to the layer. If a fixed parameter
\code{color} is not provided, then \code{color = "blue"} is used.}

\item{na.rm}{Logical. If \code{FALSE} (the default), missing values are removed
with a warning.}

\item{show.legend}{Logical. Should this layer be included in the legends?}

\item{inherit.aes}{Logical. If \code{FALSE}, overrides the default aesthetics
rather than combining with them.}

\item{n}{An integer vector specifying the grid resolution for smoothing.}

\item{xlim}{Numeric vector of length 2 specifying the domain limits in the
\eqn{x}-direction. Defaults to \eqn{c(-1, 1)}.}

\item{ylim}{Numeric vector of length 2 specifying the domain limits in the
\eqn{y}-direction. Defaults to \eqn{c(-1, 1)}.}

\item{normalize}{Logical. If \code{TRUE}, the vector endpoints are scaled to unit
length before being scaled by \code{L} (default: \code{TRUE}).}

\item{center}{Logical. If \code{TRUE}, the vector is recentered so that the
original \verb{(x, y)} becomes the midpoint (default is \code{TRUE} for
\code{geom_vector()} and \code{FALSE} for \code{geom_vector2()}).}

\item{type}{Character. Either \code{"stream"} (default) or \code{"vector"}. \code{"stream"}
computes a full streamline by integrating in both directions (if \code{center = TRUE}), while \code{"vector"} computes a single vector.}

\item{formula}{A formula specifying the multivariate linear model used for
smoothing. Defaults to \code{cbind(fx, fy) ~ x * y}.}

\item{eval_points}{A data frame of evaluation points, or \code{NULL}. When
provided, it specifies the grid where the smoothing model is evaluated; if
\code{NULL}, a grid is generated based on \code{n}.}

\item{lineend}{Line end style (round, butt, square).}

\item{linejoin}{Line join style (round, mitre, bevel).}

\item{linemitre}{Line mitre limit (number greater than 1).}

\item{arrow}{An optional \code{grid::arrow()} specification to add arrowheads to
the vectors (default: \code{grid::arrow(angle = 25, length = unit(0.025, "npc"), type = "closed")}).}

\item{geom}{The geometric object used to render the streamline (only used in
\code{stat_stream()}; defaults to \link{GeomStream}).}
}
\value{
A ggplot2 layer that can be added to a ggplot object to display a
smoothed vector field.

\describe{
\item{norm}{Computed as the Euclidean norm of the displacement,
\eqn{\sqrt{fx^2 + fy^2}}, this variable is used to normalize and scale the
vector lengths.}

\item{t}{The integration time or evaluation time at each computed point along
the smoothed field (when applicable).}

\item{d}{The incremental distance between consecutive computed points.}

\item{l}{The cumulative arc length along the smoothed vector field, calculated
as the cumulative sum of \code{d}.}
}
}
\description{
\code{geom_stream_smooth()} creates a ggplot2 layer that visualizes a smooth
vector field based on raw vector data. The function fits a multivariate
linear model (by default, using the formula \code{cbind(fx, fy) ~ x * y}) to
predict the vector displacements at any given location. It also handles
different input formats by converting polar coordinates or endpoint data to
vector displacements.
}
\section{Aesthetics}{
 \code{geom_stream_smooth()} supports the following aesthetics
(required aesthetics are in \strong{bold}):
\itemize{
\item \strong{\code{x}}: The x-coordinate of the vector's starting point.
\item \strong{\code{y}}: The y-coordinate of the vector's starting point.
\item \strong{\code{fx}}: The displacement along the x-axis.
\item \strong{\code{fy}}: The displacement along the y-axis.
\item \code{color}: The fixed color for the vector. Defaults to \code{"blue"}.
\item \code{linewidth}: The thickness of the vector line.
\item \code{linetype}: The type of the vector line (e.g., solid or dashed).
\item \code{alpha}: The transparency level of the vector.
\item \code{arrow}: Specifies arrowheads for the vectors.
}
}

\section{Details}{

\strong{Data Conversion:}
If \code{xend}/\code{yend} are missing or all \code{NA}, the function computes them. It
first checks for vector displacements (\code{fx} and \code{fy}); if present, it
computes \eqn{xend = x + fx,\quad yend = y + fy.} Otherwise, it checks for
polar coordinates (\code{angle} and \code{distance}) and computes \eqn{xend = x +
  distance \times \cos(angle \times 180/\pi),\quad yend = y + distance \times
  \sin(angle \times 180/\pi).} An error is thrown if neither set is
available.

\strong{Smoothing:}
The multivariate linear model is fitted using the provided \code{formula} and
\code{data}. This model is then used to predict vector displacements at any
specified grid point, generating a smooth approximation of the vector
field.

\strong{Prediction Intervals:}
Two types of prediction intervals can be displayed:
\itemize{
\item \strong{Ellipse:} Depicts the joint uncertainty (covariance) in the predicted \code{fx} and \code{fy}.
\item \strong{Wedge:} Indicates the range of possible vector directions and magnitudes.
}
}

\examples{

\donttest{
# Define a true vector field function
f <- function(u) {
  x <- u[1]; y <- u[2]
  c(x^2 - y^2, x^2 + y^2 - 2)
}

# Alternative example function
f <- function(u) c(-u[2], u[1])

# Visualize the vector field
ggplot() + geom_stream_field(fun = f, xlim = c(-2, 2), ylim = c(-2, 2))

# Generate design points
n <- 20
df <- data.frame(x = runif(n, -2, 2), y = runif(n, -2, 2))

# Sample function values at design points
fdf <- as.data.frame(t(apply(df, 1, f)))
colnames(fdf) <- c("fx", "fy")
df <- cbind(df, fdf)

# Visualize raw vector field data
ggplot(df) + geom_vector(aes(x, y, fx = fx, fy = fy))

# Add smoothed layer using default model
ggplot(df) +
  geom_vector(aes(x, y, fx = fx, fy = fy)) +
  geom_stream_smooth(formula = cbind(fx, fy) ~ x * y)

# Use a more complex polynomial model
ggplot(df) +
  geom_vector(aes(x, y, fx = fx, fy = fy)) +
  geom_stream_smooth(formula = cbind(fx, fy) ~ poly(x, 2) * poly(y, 2), data = df)

# Fit a linear model and use it for prediction
fhat <- function(u) {
  model <- lm(cbind(fx, fy) ~ x * y, data = df)
  predict(model, newdata = data.frame(x = u[1], y = u[2])) |> as.numeric()
}

# Visualize estimated field with the raw vector field
ggplot(df) +
  geom_stream_field(fun = fhat, normalize = FALSE, color = "#3366FF") +
  geom_vector(aes(x, y, fx = fx, fy = fy))

# Generate a hexagonal grid
hex_lattice <- grid_hex(xlim = c(-5, 5), ylim = c(-5, 5), d = 1)

# Use the hexagonal grid in geom_stream_field
ggplot(data = df) +
  geom_vector(aes(x, y, fx = fx, fy = fy), color = "black", normalize = FALSE) +
  geom_stream_smooth(eval_points = hex_lattice)

# user specified point

eval_pts <- data.frame(x = c(0, 1), y = c(2, -1))

ggplot(data = df) +
  geom_vector(aes(x, y, fx = fx, fy = fy), color = "black", normalize = FALSE) +
  geom_stream_smooth(eval_points = eval_pts)
}

}
