qrs.fast <- function(y, x, d, z, w = NULL, Q1, Q2, P = 10, link, family,
                     gridtheta, m = 1) {
  #' qrs.fast
  #'
  #' Estimation of Quantile Regression with Selection (QRS) using Algorithm 3 
  #' for the estimation of the quantile and copula coefficients.
  #'
  #' @param y = Dependent variable (N x 1)
  #' @param x = Regressors matrix (N x K)
  #' @param d = Participation variable (N x 1)
  #' @param z = Regressors and instruments matrix for the propensity score
  #' (N x Kz)
  #' @param w = Sample weights (N x 1)
  #' @param Q1 = Number of quantiles in reduced grid
  #' @param Q2 = Number of quantiles in large grid
  #' @param P =  Number of evaluated values of parameter with large quantile
  #' grid
  #' @param link = Link function to compute the propensity score
  #' @param family = Parametric copula family
  #' @param gridtheta = Grid of values for copula parameter (T x 1)
  #' @param m =  Parameter to select interval of observations in top and bottom
  #' groups
  #'
  #' @return gamma = Estimated gamma coefficients (Kz x 1)
  #' @return beta = Estimated beta coefficients (K x Q2)
  #' @return theta = Estimated copula parameter
  #' @return objf = Value of objective function at the optimum
  #' @return b1 = Estimated beta coefficients for the grid of values of the
  #' copula parameter with the reduced quantile grid (K x Q1 x T)
  #'
  #' @export
  #' @importFrom stats binomial
  #' 
  #' @examples
  #' \donttest{
  #' set.seed(1)
  #' 
  #' N <- 100
  #' x <- cbind(1, 2 + runif(N))
  #' z <- cbind(x, runif(N))
  #' cop <- copula::normalCopula(param = -0.5, dim = 2)
  #' copu <- copula::rCopula(N, cop)
  #' v <- copu[,1]
  #' u <- copu[,2]
  #' gamma <- c(-1.5, 0.05, 2)
  #' beta <- cbind(qnorm(u), u^0.5)
  #' prop <- exp(z %*% gamma) / (1 + exp(z %*% gamma))
  #' d <- as.numeric(v <= prop)
  #' y <- d * rowSums(x * beta)
  #' w <- matrix(1, nrow = N, ncol = 1)
  #' 
  #' Q1 <- 9
  #' Q2 <- 19
  #' P <- 2
  #' m <- 1
  #' gridtheta <- seq(from = -1, to = 0, by = .1)
  #' link <- "probit"
  #' family <- "Gaussian"
  #' 
  #' result <- qrs.fast(y, x[,-1], d, z[,-1], w, Q1, Q2, P, link, family, gridtheta, m)
  #' summary(result)
  #' }
  
  x <- cbind(rep(1,NROW(y)), x)
  
  # Estimate the propensity score
  if (!link %in% c("probit", "logit")) {
    stop("Unsupported link function. Use 'probit' or 'logit'.")
  }
  ghat <- switch(link,
                   "probit" = stats::glm(d ~ z, family = binomial(link = "probit"), weights = w),
                   "logit" = stats::glm(d ~ z, family = binomial(link = "logit"), weights = w)
  )
  prop <- ghat$fitted.values
  
  # Estimate the beta coefficients and the copula
  qrs <- .qrs.prop.fast(y[d==1], x[d==1,], prop[d==1], w[d==1], Q1, Q2, P, family, gridtheta, m)
  
  list(gamma = ghat$coefficients, beta = qrs$beta, theta = qrs$theta,
       objf = qrs$objf_min, b1 = qrs$b1)
}