#include <array>

#include "fastcpd.h"
#include "testthat.h"

using ::arma::approx_equal;
using ::arma::colvec;
using ::arma::mat;
using ::arma::ones;

using ::fastcpd::test::FastcpdTest;

// ``` r
// set.seed(1)
// x <- arima.sim(list(ar = c(0.1, -0.2, 0.6), ma = c(0.1, 0.5)), n = 200)
// ```
constexpr std::array<double, 200> kARMA32 = {
    0.16086394,  1.36633450,  0.71425776,  -0.23698470, 1.13222067,
    -0.81209066, 1.04080322,  2.50485157,  0.10240048,  0.04309484,
    1.76843576,  -0.37048506, 2.30809336,  1.49936227,  1.35265076,
    1.29760473,  0.36324437,  0.71686001,  -1.38011604, 1.31601817,
    0.23507054,  1.85294823,  1.69728898,  0.36409193,  1.58620681,
    -0.12382253, -1.15284844, 0.56023783,  -0.82866400, -0.78412375,
    0.27626904,  -0.89428088, -1.20560631, -0.26274899, 0.55851260,
    -2.08831014, 0.55245008,  0.43857619,  0.07374949,  0.21973100,
    1.14151925,  0.26646413,  -0.40061958, 1.87872196,  1.43780397,
    0.94785207,  3.17151372,  2.05753994,  -0.28716435, 1.04100239,
    -0.52417819, -1.31541151, -0.68211713, -0.37625744, -1.90734262,
    -0.43675851, -0.98220438, 0.62556888,  0.56308654,  1.20736086,
    1.21701663,  2.39376315,  0.44496334,  0.61182334,  2.47669709,
    -0.34600745, 0.28084458,  0.84441002,  -0.64229642, -0.57212145,
    0.88417372,  -0.45000449, -0.84170441, 1.74011253,  -0.26090482,
    -0.40863069, 0.82994931,  0.62104715,  -0.40145671, 0.64495703,
    -0.20479077, -0.80159791, 0.03467575,  -0.70410088, -0.05561702,
    -1.60362778, -0.15123335, -1.99275907, -1.43254808, -1.16204543,
    -1.88076120, -1.20144209, -2.68731572, -0.20020090, -2.70791602,
    -1.88487685, -1.76178438, -2.51812889, 0.42374247,  -0.66036783,
    -1.90264031, -1.56448276, -0.52916463, -1.67544799, -1.09523234,
    -1.06238132, -2.63189117, -2.39647166, -0.20191830, -2.17889216,
    -2.56192766, 1.47864928,  -0.72726882, -1.16710509, 2.16310135,
    0.88631727,  -1.04561085, 3.60342576,  0.75721680,  -1.61927610,
    1.43432190,  0.40246854,  1.03834924,  1.32430373,  1.78780989,
    0.55837792,  0.37964132,  0.89227190,  0.96549226,  2.28252559,
    2.19925399,  2.69098578,  0.60192677,  2.30626534,  1.42748530,
    -0.91065824, 1.49145646,  0.34749113,  0.89928610,  0.21630369,
    0.27574153,  -0.82828660, -0.49301554, 0.20304732,  -1.15816310,
    0.50596914,  -1.08686163, -1.65838396, 1.08743329,  -1.60742076,
    -0.34229029, 0.09191278,  -0.30952153, 1.28456656,  2.20568531,
    0.45012999,  -1.15041623, 3.22202770,  0.59657857,  0.58186733,
    2.24635394,  1.24564121,  -0.09565102, 1.74843029,  0.50099276,
    -1.55686262, 1.44386747,  1.68977984,  -0.71676002, -0.06760279,
    1.51300168,  -0.87236517, -1.84696719, 0.70403277,  -1.58031874,
    -1.80995143, -0.94833112, 0.83596631,  -1.54181203, -1.62742880,
    0.51827539,  -1.06763563, -2.04778834, -2.53620199, -1.90693103,
    -1.85012658, -1.64826101, -0.75785666, -0.33506819, -2.05035803,
    0.66580539,  -0.21328442, -0.12953955, 1.53135295,  0.73161908,
    -0.65013812, 1.89430814,  -1.56479241, -1.08874870, 0.03043624};

// ``` r
// arima(x, order = c(3, 0, 2), method = "ML", include.mean = FALSE)
// ```
constexpr std::array<double, 6> kARMA32PAR = {
    0.09403877, -0.21094448, 0.64057698, 0.08878212, 0.50335561, 0.976383};
constexpr double kARMA32VALUE = 282.2705;
constexpr std::array<double, 200> kARMA32RESIDUALS = {
    0.119657408,  1.032473704,  0.272376942,  -0.513467899, 0.360895835,
    -1.192880996, 1.433100166,  1.973922066,  -0.286091638, -1.068905105,
    0.420306848,  -0.092951436, 2.484883335,  -0.102345183, 0.694294107,
    -0.001945305, -0.783193088, 0.160456193,  -1.822140989, 1.445338741,
    0.149844825,  2.251697123,  0.454280147,  -0.728966478, 0.559100547,
    -0.966137311, -1.235482332, 0.222442392,  -0.443079174, 0.077839927,
    0.032445119,  -0.596906088, -0.524277031, -0.167988007, 1.180573061,
    -1.444229846, 0.568932014,  0.264786853,  1.176882560,  -0.286343959,
    0.288503044,  0.276744804,  -0.495424595, 1.146057717,  1.153557308,
    0.786286620,  1.531751912,  0.506440764,  -1.234790578, -0.674857496,
    -1.319207187, -0.405758548, -0.635777671, 0.006872234,  -0.853814223,
    0.172529006,  -0.687999003, 1.821840720,  0.761406944,  0.930913649,
    0.355625927,  1.673148403,  -0.624379803, -0.491417044, 1.337549093,
    -0.606278432, -0.175527827, -0.520746379, -0.306230794, -0.224191200,
    0.435623968,  -0.168224381, -0.450725594, 1.282651139,  -0.200834469,
    -0.105651058, -0.190864937, 0.694056412,  0.011426560,  -0.068302094,
    -0.747642931, -0.288368026, 0.055645114,  -0.605058777, 0.557104707,
    -1.514037514, 0.292866223,  -1.545088726, -0.260045877, -0.549998796,
    -0.617430793, -0.020385573, -1.914093097, 1.184044158,  -1.628000096,
    -0.402484407, -1.172312334, -0.708755400, 2.149329038,  0.063091100,
    -1.225576779, -1.719249381, 0.409161913,  0.092150142,  -0.261263212,
    -0.997031624, -1.469739057, -1.039146694, 0.980856089,  -0.543519749,
    -1.309960879, 1.779173562,  0.490421840,  -0.084786515, 0.932923390,
    0.862430019,  -0.471204686, 2.110810084,  -0.320183872, -1.294628320,
    -0.285838057, 0.117989406,  2.473737718,  0.113749269,  0.369227302,
    -0.085570702, -0.322313197, -0.099184080, 0.775027234,  2.117878532,
    1.038560466,  1.188933066,  -1.277664314, 0.923495622,  0.174926126,
    -1.424360706, 0.439281014,  -0.221316503, 1.563104058,  -0.777730196,
    -0.495243442, -0.949207280, -0.161961302, 0.390221806,  -0.703795583,
    0.839591621,  -1.229099166, -1.021043933, 1.399330431,  -0.973577378,
    0.482655660,  -0.464355484, 0.437586625,  1.747211428,  1.585334370,
    -0.308263370, -2.320951195, 2.373477690,  0.720107605,  0.683728936,
    -0.169649076, 0.445887060,  -0.065857428, 0.362630920,  -0.480579045,
    -1.313747291, 0.934489800,  1.482980398,  -0.175841430, -1.299513016,
    0.489612611,  0.040881864,  -1.652545737, -0.149356969, -0.632237208,
    -0.198392678, -1.226618346, 1.764426591,  -0.200281576, -1.568969725,
    0.050688596,  0.313228026,  -0.848891876, -2.983136285, -0.724352905,
    0.671855094,  0.053057034,  -0.114486630, 0.557114459,  -1.114710564,
    1.091942955,  -0.029619865, 0.797371683,  1.016160892,  0.205333539,
    -0.842648400, 1.100284213,  -2.022265971, -0.499833359, -0.348418497};

context("GetNllPelt Unit Test") {
  test_that("arma(3, 2) is correct for 200 data points") {
    const std::tuple<mat, mat, double> cost_result = FastcpdTest::GetNllPelt(
        colvec(kARMA32.data(), kARMA32.size()), 0, 199, false, R_NilValue);
    const colvec par = std::get<0>(cost_result);
    const colvec residuals = std::get<1>(cost_result).as_col();
    const double value = std::get<2>(cost_result);

    // Expected values obtained from the following R code:
    //   arima(x, order = c(3, 0, 2), include.mean = FALSE)
    const colvec expected_par(kARMA32PAR.data(), kARMA32PAR.size());
    const double expected_value(kARMA32VALUE);
    const colvec expected_residuals(kARMA32RESIDUALS.data(),
                                    kARMA32RESIDUALS.size());

    expect_true(norm(par - expected_par, "fro") < 1e-6);
    expect_true(abs(value - expected_value) < 1e-4);
    expect_true(norm(residuals - expected_residuals, "fro") < 1e-6);
  }
}

context("GetNllSen Unit Test") {
  test_that("arma(3, 2) is correct for 200 data points") {
    const double value = FastcpdTest::GetNllSen(
        colvec(kARMA32.data(), kARMA32.size()), 0, 199, 0.1 * ones<colvec>(6));
    const double expected_value = 1363.288;
    expect_true(abs(value - expected_value) < 0.001);
  }
}

context("GetGradient Unit Test") {
  test_that("arma(3, 2) is correct for 200 data points") {
    const colvec gradient = FastcpdTest::GetGradientArma(
        colvec(kARMA32.data(), kARMA32.size()), 0, 199, 0.1 * ones<colvec>(6));
    const colvec expected_gradient = {4.401258, 6.600128, -7.591818,
                                      4.151778, 7.503752, -2.806806};
    expect_true(norm(gradient - expected_gradient, "fro") < 1e-6);
  }
}

context("GetHessian Unit Test") {
  test_that("binomial is correct for a two-dimensional data") {
    const mat hessian =
        FastcpdTest::GetHessianBinomial({{1, 1, 0.2}}, 0, 0, {-0.5, 0.3});
    const mat expected_hessian = {{0.23828, 0.047656}, {0.047656, 0.0095312}};
    expect_true(norm(hessian - expected_hessian, "fro") < 1e-6);
  }

  test_that("poisson is correct for a two-dimensional data") {
    const mat hessian =
        FastcpdTest::GetHessianPoisson({{4, 1, 0.2}}, 0, 0, {-0.5, 0.3});
    const mat expected_hessian = {{0.6440364, 0.128807}, {0.128807, 0.0257616}};
    expect_true(norm(hessian - expected_hessian, "fro") < 1e-6);
  }

  test_that("arma(3, 2) is correct for 200 data points") {
    const mat hessian = FastcpdTest::GetHessianArma(
        colvec(kARMA32.data(), kARMA32.size()), 0, 199, 0.1 * ones<colvec>(6));
    const mat expected_hessian = {
        {12.406525, 18.60483, -21.40027, 4.743794, 28.98263, -44.01258},
        {18.604831, 27.89981, -32.09185, 25.380851, 27.48253, -66.00128},
        {-21.400265, -32.09185, 36.91375, -24.424268, -34.63643, 75.91818},
        {4.743794, 25.38085, -24.42427, -4.424596, 35.77934, -41.51778},
        {28.982631, 27.48253, -34.63643, 35.779338, 24.80587, -75.03752},
        {-44.012575, -66.00128, 75.91818, -41.51778, -75.03752, 106.13612}};
    expect_true(norm(hessian - expected_hessian, "fro") < 2e-5);
  }
}
