library(testthat)

# Create test data
test_that("MFric function basic functionality tests", {
  # Create simple test dataset
  set.seed(123)
  test_data <- matrix(runif(20), nrow = 4, ncol = 5)
  colnames(test_data) <- paste0("func", 1:5)
  rownames(test_data) <- paste0("site", 1:4)

  # Test default parameters
  result_default <- MFric(test_data)
  expect_s3_class(result_default, "data.frame")
  expect_equal(nrow(result_default), 4)
  expect_equal(ncol(result_default), 1)
  expect_equal(colnames(result_default), "MFric")
  expect_equal(rownames(result_default), rownames(test_data))

  # Test uncorrected mode
  result_uncor <- MFric(test_data, cor = FALSE)
  expect_equal(result_default, result_uncor)  # Default should be uncorrected

  # Test corrected mode
  result_cor <- MFric(test_data, cor = TRUE)
  expect_s3_class(result_cor, "data.frame")
  expect_equal(dim(result_cor), dim(result_default))
  expect_equal(colnames(result_cor), colnames(result_default))

  # Test weight parameter
  weights <- c(0.5, 1, 1, 2, 0.8)
  result_weighted <- MFric(test_data, weights = weights)
  expect_s3_class(result_weighted, "data.frame")

  # Test weight error handling
  expect_error(MFric(test_data, weights = c(1, 2, 3)),
               "The length of the weight vector must be equal to the number of columns in the data frame")
})

test_that("MFric handles special cases", {
  # Test single row data
  single_row <- matrix(runif(5), nrow = 1)
  result_single <- MFric(single_row)
  expect_equal(nrow(result_single), 1)

  # Test single column data
  single_col <- matrix(runif(4), ncol = 1)
  result_col <- MFric(single_col)
  expect_equal(nrow(result_col), 4)

  # Test data with zero values
  test_data_zeros <- matrix(c(0.5, 0, 0.7, 0.3, 0, 0.6), nrow = 2)
  result_zeros <- MFric(test_data_zeros)
  expect_s3_class(result_zeros, "data.frame")

  # Test data with negative values
  test_data_neg <- matrix(c(0.5, -0.2, 0.7, 0.3, -0.5, 0.6), nrow = 2)
  result_neg <- MFric(test_data_neg)
  expect_s3_class(result_neg, "data.frame")
})

test_that("MFric consistency test with example dataset", {
  # Load example data from package
  data(forestfunctions)

  # Test results on example data - uncorrected
  result_forest_uncor <- MFric(forestfunctions[,6:31], cor = FALSE)
  expect_s3_class(result_forest_uncor, "data.frame")
  expect_equal(nrow(result_forest_uncor), nrow(forestfunctions))

  # Test results on example data - corrected
  result_forest_cor <- MFric(forestfunctions[,6:31], cor = TRUE)
  expect_s3_class(result_forest_cor, "data.frame")
  expect_equal(nrow(result_forest_cor), nrow(forestfunctions))

  # Note: corrected and uncorrected results should be different
  expect_false(identical(result_forest_cor, result_forest_uncor))
})

# Test edge cases
test_that("MFric edge case handling", {
  # All zeros data - may produce warnings about standard deviation
  all_zeros <- matrix(0, nrow = 3, ncol = 4)
  suppressWarnings({
    result_all_zeros <- MFric(all_zeros)
  })
  expect_s3_class(result_all_zeros, "data.frame")

  # All same values - may produce warnings about standard deviation
  all_same <- matrix(0.5, nrow = 3, ncol = 4)
  suppressWarnings({
    result_all_same <- MFric(all_same)
  })
  expect_s3_class(result_all_same, "data.frame")

  # Test perfectly correlated functions - uncorrected
  perfect_corr <- matrix(c(1:5, 2*(1:5)), ncol = 2)
  result_corr_uncor <- MFric(perfect_corr, cor = FALSE)
  expect_s3_class(result_corr_uncor, "data.frame")

  # Test perfectly correlated functions - corrected
  result_corr_cor <- MFric(perfect_corr, cor = TRUE)
  expect_s3_class(result_corr_cor, "data.frame")
})
