% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ELHMC.R
\name{ELHMC}
\alias{ELHMC}
\title{Empirical Likelihood Hamiltonian Monte Carlo Sampling}
\usage{
ELHMC(
  initial,
  data,
  fun,
  dfun,
  prior,
  dprior,
  n.samples = 100,
  lf.steps = 10,
  epsilon = 0.05,
  p.variance = 1,
  tol = 10^-5,
  detailed = FALSE,
  print.interval = 1000,
  plot.interval = 0,
  which.plot = NULL,
  FUN,
  DFUN
)
}
\arguments{
\item{initial}{a vector containing the initial values of the parameters}

\item{data}{a matrix containing the data}

\item{fun}{the estimating function \eqn{g}. It takes in a parameter vector
\code{params} as the first argument and a data point vector \code{x} as the
 second parameter. This function returns a vector.}

\item{dfun}{a function that calculates the gradient of the estimating function
\eqn{g}. It takes in a parameter vector \code{params} as the first argument
 and a data point vector \code{x} as the second argument. This function
 returns a matrix.}

\item{prior}{a function with one argument \code{x} that returns the log joint prior
density of the parameters of interest}

\item{dprior}{a function with one argument \code{x} that returns
the gradients of the log densities of the parameters of interest}

\item{n.samples}{number of samples to draw}

\item{lf.steps}{number of leap frog steps in each Hamiltonian Monte Carlo
update}

\item{epsilon}{the leap frog step size(s). This has to be a single numeric
value or a vector of the same length as \code{initial}.}

\item{p.variance}{the covariance matrix of a multivariate normal distribution
used to generate the initial values of momentum \eqn{p} in
Hamiltonian Monte Carlo. This can also be a single numeric value or
a vector. See Details.}

\item{tol}{EL tolerance}

\item{detailed}{If this is set to \code{TRUE}, the function will return a list
with extra information.}

\item{print.interval}{the frequency at which the results would be printed on the terminal. Defaults to 1000.}

\item{plot.interval}{the frequency at which the drawn samples would be plotted. The last half of the samples drawn are plotted after 
each plot.interval steps.  The acceptance rate is also plotted. Defaults to 0, which means no plot.}

\item{which.plot}{the vector of parameters to be plotted after each plot.interval. Defaults to NULL, which means no plot.}

\item{FUN}{the same as \code{fun} but takes in a matrix \code{X} instead of
a vector \code{x} and returns a matrix so that \code{FUN(params, X)[i, ]} is
the same as \code{fun(params, X[i, ])}. Only one of \code{FUN} and
\code{fun} should be provided. If both are then \code{fun} is ignored.}

\item{DFUN}{the same as \code{dfun} but takes in a matrix \code{X} instead of
a vector \code{x} and returns an array so that \code{DFUN(params, X)[, , i]}
is the same as \code{dfun(params, X[i, ])}. Only one of \code{DFUN} and
\code{dfun} should be provided. If both are then \code{dfun} is ignored.}
}
\value{
The function returns a list with the following elements:
 \item{\code{samples}}{A matrix containing the parameter samples}
 \item{\code{acceptance.rate}}{The acceptance rate}
 \item{\code{call}}{The matched call}

 If \code{detailed = TRUE}, the list contains these extra elements:
 \item{\code{proposed}}{A matrix containing the proposed values at
   \code{n.samaples - 1} Hamiltonian Monte Carlo updates}
 \item{\code{acceptance}}{A vector of \code{TRUE}/\code{FALSE} values
   indicates whether each proposed value is accepted}
 \item{\code{trajectory}}{A list with 2 elements \code{trajectory.q} and
   \code{trajectory.p}. These are lists of matrices contraining position and
   momentum values along trajectory in each Hamiltonian Monte Carlo update.}
}
\description{
This function draws samples from a Empirical Likelihood Bayesian posterior
distribution of parameters using Hamiltonian Monte Carlo.
}
\details{
Suppose there are data \eqn{x = (x_1, x_2, ..., x_n)} where \eqn{x_i}
 takes values in \eqn{R^p} and follow probability distribution \eqn{F}.
 Also, \eqn{F} comes from a family of distributions that depends on
 a parameter \eqn{\theta = (\theta_1, ..., \theta_d)} and there is
 a smooth function
 \eqn{g(x_i, \theta) = (g_1(x_i, \theta), ...,g_q(x_i, \theta))^T} that
 satisfies \eqn{E_F[g(x_i, \theta)] = 0} for \eqn{i = 1, ...,n}.

 \code{ELHMC} draws samples from a Empirical Likelihood Bayesian
 posterior distribution of the parameter \eqn{\theta}, given the data \eqn{x}
 as \code{data}, the smoothing function \eqn{g} as \code{fun},
 and the gradient of \eqn{g} as \code{dfun} or \eqn{G(X) =
 (g(x_1), g(x_2), ..., g(x_n))^T} as \code{FUN} and the gradient of \eqn{G}
 as \code{DFUN}.
}
\examples{
\donttest{
## Suppose there are four data points (1, 1), (1, -1), (-1, -1), (-1, 1)
x = rbind(c(1, 1), c(1, -1), c(-1, -1), c(-1, 1))
## If the parameter of interest is the mean, the smoothing function and
## its gradient would be
f <- function(params, x) {
 x - params
}
df <- function(params, x) {
 rbind(c(-1, 0), c(0, -1))
}
## Draw 50 samples from the Empirical Likelihood Bayesian posterior distribution
## of the mean, using initial values (0.96, 0.97) and standard normal distributions
## as priors:
normal_prior <- function(x) {
   -0.5 * (x[1] ^ 2 + x[2] ^ 2) -log(2 * pi)
}
normal_prior_log_gradient <- function(x) {
   -x
}
set.seed(1234)
mean.samples <- ELHMC(initial = c(0.96, 0.97), data = x, fun = f, dfun = df,
                     n.samples = 50, prior = normal_prior,
                     dprior = normal_prior_log_gradient)
plot(mean.samples$samples, type = "l", xlab = "", ylab = "")
}
}
\references{
Chaudhuri, S., Mondal, D. and Yin, T. (2017)
 Hamiltonian Monte Carlo sampling in Bayesian empirical likelihood
 computation.
 \emph{Journal of the Royal Statistical Society: Series B}.

 Neal, R. (2011) MCMC for using Hamiltonian dynamics.
 \emph{Handbook of Markov Chain Monte Carlo}
 (eds S. Brooks, A.Gelman, G. L.Jones and X.-L. Meng), pp. 113-162.
 New York: Taylor and Francis.
}
