#' @importFrom stats median
#' @importFrom stats na.omit
#' @importFrom stats quantile

if (getRversion() >= "2.15.1") utils::globalVariables(".")

##' @export
median.ordered <- function(x, na.rm=FALSE, ...)
{
  ## from https://stackoverflow.com/a/7925162
  
  if (na.rm) {
    x <- na.omit(x)
  }
  m <- quantile(x, probs=0.5, type=1)
  attr(m, "names") <- NULL
  return(m)
}

#' Count Decimal Places in a Number String
#'
#' Computes the number of decimal places in the character representation of a number.
#'
#' @param x A character string representing a number.
#' @return Integer: The number of digits after the decimal point; \code{0} if there is no decimal point.
#' @examples
#' decimalplaces("3.141")   # returns 3
#' decimalplaces("42")      # returns 0
#' @export
decimalplaces <- function(x) {
  if (grepl( ".", x, fixed = TRUE)) {
    nchar(gsub("^.*\\.","",x))
  } else {
    return(0)
  }
}

#' Modified formatC with Improved Zero and Exponential Handling
#'
#' Formats a numeric vector similar to \code{formatC}, but tweaks output for certain special cases:
#' - Attempts to remove trailing decimal points for whole numbers.
#' - If format is 'fg' and output is "0", falls back to fixed format (\code{f}).
#' - Converts exponential notation to plain numbers where appropriate.
#'
#' @param x A numeric vector.
#' @param digits Integer; desired number of digits.
#' @param format Character string specifying the output format; see \code{\link[base]{formatC}}. Common are "fg" (significant digits), or "f" (fixed).
#' @param ... Further arguments passed to \code{\link[base]{formatC}}.
#' @return A character vector with formatted numbers.
#' @examples
#' formatC_mod(c(3.14159, 42, 0), digits = 2, format = "fg")
#' formatC_mod(c(1e-6, 42), digits = 2, format = "fg")
#' @export
formatC_mod <- function(x, digits = NULL, format = NULL, ...){
  res <- formatC(x = x, digits = digits, format = format, ...)
  for(i in 1:length(res))
  {
    if(format=='fg' & res[i]=="0")
      res[i] <- formatC(x = x[i], digits = max(digits-1,0), format = 'f', ...)
    if(decimalplaces(res[i])==0) {
      res[i] <- sub("\\.", "", res[i])
    } else{
      if(grepl("\\.e\\+", res[i]))
        res[i] <- as.character(as.numeric(res[i]))
    }
  }
  return(res)
}

##' generate [lower quartile; upper quartile]
##'
##' wrapper around \code{quantile()} and \code{format()} and \code{paste()}.
##' @param x vector to be summarized
##' @param digits Integer specifying the number of digits to display. Default is 2.
##' @param significant_digits Logical. If \code{TRUE}, format numbers using significant digits; if \code{FALSE}, use fixed number of decimal digits. Default is \code{TRUE}.
##' @param ... passed onto quantile() as well as to format().  meant for arguments \code{na.rm} or \code{digits}
##' @return character.
##' @author Dr. Andreas Leha
##' @export
formatIQR <- function(x, digits = 2, significant_digits = TRUE, ...)
{
  if(is.ordered(x) || inherits(x, "Date"))
    paste0("[",
           paste(
             quantile(x, probs = c(0.25, 0.75), type = 1, ...),
             collapse = "; "),
           "]")
  else
    paste0("[",
           paste(
             formatC_mod(quantile(x, probs = c(0.25, 0.75), ...), format = ifelse(significant_digits, 'fg', 'f') ,flag = "#", digits = digits),
             collapse = "; "),
           "]")
}

##' Calculate the Time Difference in Years
##'
##' Code from Dirk Eddelbuettel via Stackoverflow
##' (\url{https://stackoverflow.com/a/15569373/1844418})
##' @param t2 end time of the interval.  Will be coerced to Date
##' @param t1 starting time of the interval.  Will be coerced to Date
##' @return numeric.  t2 - t1 [years]
##' @author Andreas Leha
##' @export
##' @examples
##' difftime_years("2003-04-05", "2001-01-01")
difftime_years <- function(t2, t1)
{
  t1 <- as.Date(t1)
  t2 <- as.Date(t2)
  td <- difftime(t2, t1, units="weeks")
  td <- as.numeric(td)
  ty <- td/52.25
  return(ty)
}


##' wrap all elements in a vector in quotes (or other strings)
##'
##' @param v vector of elements to wrap
##' @param quoteChr character.  to be put around of the elements of
##'   \code{v}.  Defaults to "'".
##' @param endquoteChr character or NULL (default).  If not NULL
##'   \code{quoteChr} is put before the elements of \code{v} and
##'   \code{endquoteChr} is put after them
##' @return character vector of the elements of \code{v} wrapped
##'   between quotes
##' @author Andreas Leha
##' @export
##' @examples
##' ## default behaviour: wrap in single quotes
##' wrapQuote(1:10)
##'
##' ## change to wrap in asterisks
##' wrapQuote(1:10, "*")
##'
##' ## different front from back quotes
##' wrapQuote(1:10, "*", "/")
##'
##' ## you can also wrap with longer strings
##' wrapQuote(1:10, "quote")
wrapQuote <- function(v, quoteChr = "'", endquoteChr = NULL) {
  if (length(v) == 0) return(character(0))
  
  if (is.null(endquoteChr)) endquoteChr <- quoteChr
  
  paste0(quoteChr, v, endquoteChr)
}

##' Print a text for English prosa
##'
##' Pastes a vector and adds comma and "and" to the correct places
##' @title makeEnglishList
##' @param v vector
##' @param sep character. spearates all but last entries of \code{v}
##'   [", "]
##' @param lastsep character. spearates the last entries of \code{v}
##'   [", and "]
##' @param onlysep character. spearates the two entries of \code{v} if
##'   \code{length(v) == 2} [" and "]
##' @return character with plain text English prosa version
##' @author Andreas Leha
##' @export
##' @examples
##' ## defaut separators
##' makeEnglishList(c("foo", "bar", "baz"))
##' makeEnglishList(c("foo", "bar"))
##' makeEnglishList(c("foo"))
##'
##' ## without the 'Oxford comma'
##' makeEnglishList(c("foo", "bar", "baz"), lastsep = " and ")
##'
##' ## an 'or' list
##' makeEnglishList(c("foo", "bar", "baz"), lastsep = ", or ")
makeEnglishList <- function(v, sep = ", ", lastsep = ", and ", onlysep = " and ") {
  l <- length(v)
  
  if (l == 0)
    return("")
  
  if (l == 1)
    return(v)
  
  if (l == 2)
    return(paste(v, collapse = onlysep))
  
  ret <- paste(v[1:(l-1)], collapse=sep)
  ret <- paste(ret, v[l], sep=paste0(lastsep))
  ret
}

##' English text version of number
##'
##' Converts a number to a text version of that number
##' @title digits2text
##' @param x number to convert
##' @param mult to be appended (like a unit)
##' @return character
##' @author Graham Williams \email{Graham.Williams@@togawre.com}
##' \url{https://rattle.togaware.com/utility.R}
##' @export
digits2text <- function(x,mult="") {
  if (x == 0) return("zero")
  
  if (!requireNamespace("stringr"))
    stop("digits2text requires stringr")
  
  units <- c("one","two","three","four","five",
             "six","seven","eight","nine")
  teens <- c("ten","eleven","twelve","thirteen","fourteen",
             "fifteen","sixteen","seventeen","eighteen","nineteen")
  tens <- c("ten","twenty","thirty","forty","fifty",
            "sixty","seventy","eighty","ninety")
  
  digits <- rev(as.numeric(strsplit(as.character(x),"")[[1]]))
  digilen <- length(digits)
  
  if(digilen == 2 && digits[2] == 1) return(teens[digits[1]+1])
  
  digitext <- units[digits[1]]
  if(digilen > 1) digitext <- c(digitext, tens[digits[2]])
  if(digilen > 2) digitext <- c(digitext, "hundred", units[digits[3]])
  if(digilen > 3) digitext <- c(digitext,
                                digits2text(floor(x/1000),"thousand"))
  if(digilen > 6) digitext <- c(digitext,
                                digits2text(floor(x/1000000),"million"))
  
  return(stringr::str_trim(paste(c(rev(digitext),mult),sep="",collapse=" ")))
}
