#' Wrapper for ggplot2::ggplot()
#'
#' @param data A data frame with the data to be plotted.
#' Ignored in this case.
#' @param parameter Character. Name of the column in \code{data} to the values to be plotted.
#' Ignored in this case.
#' @param groupby Character (default: NULL). Name of the grouping (categorical) variable in \code{data}.
#' Ignored in this case.
#' @param pval.display Logical. If \code{TRUE} and there are exactly two groups in \code{groupby}, a p-value annotation is added.
#' Ignored in this case.
#' @param ldat A data frame containing the p-value(s) to annotate on the plot. Expected columns: `"p value"` and/or `"adjusted p value"`.
#' Ignored in this case.
#' @param ... Additional arguments
#' Ignored in this case.
#' @return A \code{ggplot} object.
#' @author Dr. Andreas Leha
#' @export
#' @examples
#' w.emptyplot()
w.emptyplot <- function(data, parameter, groupby = NULL, pval.display = FALSE, ldat, ...)
{
  p <- ggplot2::ggplot()
  return(p)
}

#' Wrapper for ggstatsplot::ggbetweenstats with optional p-value annotation and date formatting
#'
#' This function creates a `ggbetweenstats` plot from `ggstatsplot`, and optionally adds a p-value annotation using `geom_signif` for two-group comparisons based on values in `ldat`.
#' If requested, it can also format the y-axis as dates.
#'
#' @param data A data frame with the data to be plotted.
#' @param parameter Character. Name of the column in \code{data} to the values to be plotted.
#' @param groupby Character (default: NULL). Name of the grouping (categorical) variable in \code{data}.
#' @param pval.display Logical. If \code{TRUE} and there are exactly two groups in \code{groupby}, a p-value annotation is added.
#' @param ldat A data frame containing the p-value(s) to annotate on the plot. Expected columns: `"p value"` and/or `"adjusted p value"`.
#' @param ... Additional arguments passed to \code{\link[ggstatsplot]{ggbetweenstats}}.
#'
#' @details
#' If \code{groupby} is not \code{NULL} and there are exactly two levels in the grouping variable, and \code{pval.display = TRUE}, a p-value from \code{ldat} is added between the two groups using \code{geom_signif}. If \code{"adjusted p value"} is present in \code{ldat}, it is used; otherwise, \code{"p value"} is used. If \code{wasDate} is \code{TRUE}, the y-axis labels are formatted as dates.
#'
#' @return A \code{ggplot} object.
#' @author Fabian Kück
#' @export
#' @examples
#' ## ungrouped plot
#' p <- w.ggbetweenstats(iris, "Sepal.Length")
#' if (FALSE) print(p)
#'
#' ## grouped plot
#' p <- w.ggbetweenstats(iris, "Sepal.Length", groupby = "Species",
#'                       bf.message = FALSE, pairwise.display = "none")
#' if (FALSE) print(p)
#'
#' ## grouped plot with p value annotation for >2 groups
#' pval <- anova(lm(Sepal.Length ~ Species, data = iris))['Species', 'Pr(>F)']
#' pval <- descutils::prettyPvalues(pval, digits = 3, orgbold = FALSE, lhs = "p")
#' ldat <- data.frame(`p value` = pval, check.names = FALSE)
#' p <- w.ggbetweenstats(iris, "Sepal.Length", groupby = "Species", pval.display = TRUE, ldat = ldat,
#'                       bf.message = FALSE, pairwise.display = "none")
#' if (FALSE) print(p)
#'
#' ## setup simple data with only two groups
#' iris2 <- iris[iris$Species %in% c("setosa", "virginica"),]
#'
#' ## grouped plot with p value annotation for two groups
#' pval <- t.test(Sepal.Length ~ Species, data = iris2)$p.value
#' pval <- descutils::prettyPvalues(pval, digits = 3, orgbold = FALSE, lhs = "p")
#' ldat <- data.frame(`p value` = pval, check.names = FALSE)
#' p <- w.ggbetweenstats(iris2, "Sepal.Length", groupby = "Species", pval.display = TRUE, ldat = ldat,
#'                       bf.message = FALSE, pairwise.display = "none")
#' if (FALSE) print(p)
w.ggbetweenstats <- function(data, parameter, groupby = NULL, pval.display = FALSE, ldat, ...)
{
  as.numericfactor <- function(x) as.numeric(as.factor(x))

  ## setup plot data
  if (inherits(data[[parameter]], "Date")) {
    data <- data %>% dplyr::mutate_at(parameter, as.numeric)
    wasDate <- TRUE
  }
  if (is.factor(data[[parameter]]))
    data <- data %>% dplyr::mutate_at(parameter, as.numeric)
  if (is.character(data[[parameter]]))
    data <- data %>% dplyr::mutate_at(parameter, as.numericfactor)
  if (is.null(groupby))
    data <- data %>% dplyr::mutate(...dummy = "")

  ## generate plot
  p <- eval(substitute(ggstatsplot::ggbetweenstats(
                                      data = data,
                                      x = XXX,
                                      y = YYY,
                                      ...),
                       list(XXX = ifelse(is.null(groupby), "...dummy", groupby),
                            YYY = parameter)))

  ## add p values
  if (!is.null(groupby) &&
      pval.display &&
      length(unique(na.omit(data[[groupby]][!is.na(data[[parameter]])]))) > 1 &&
      ("adjusted p value" %in% colnames(ldat) || "p value" %in% colnames(ldat))) {
    p_annotation <- ifelse("adjusted p value" %in% colnames(ldat), ldat$`adjusted p value`, ldat$`p value`)
    n_groups <- length(unique(na.omit(data[[groupby]])))
    if (n_groups == 2) {
        p <- p + ggsignif::geom_signif(comparisons = list(c(1, 2)), annotation = p_annotation)
    } else if (n_groups > 2) {
        p <- p + ggplot2::labs(subtitle = c(p_annotation)) +
          ggplot2::theme(plot.subtitle = ggplot2::element_text(hjust = 0.5))      # center subtitle
    }
  }

  ## final adjustments
  ## Date scale annotation
  if (exists("wasDate") && wasDate)
    p <- p + ggplot2::scale_y_continuous(labels = function(x) zoo::as.Date(as.numeric(x)))
  ## remove x axis annotation if ungrouped
  if (is.null(groupby))
    p <- p +
      ggplot2::theme(
                 axis.title.x       = ggplot2::element_blank(),
                 axis.ticks.x       = ggplot2::element_blank(),
                 panel.grid.major.x = ggplot2::element_blank())

  ## done!
  return(p)
}





#' Wrapper for ggstatsplot::ggbarstats with optional p-value annotation
#'
#' This function creates a `ggbarstats` plot from `ggstatsplot`, and optionally adds a p-value annotation using `geom_signif` for two-group comparisons based on values in `ldat`.
#'
#' @param data A data frame with the data to be plotted.
#' @param parameter Character. Name of the column in \code{data} to the values to be plotted.
#' @param groupby Character (default: NULL). Name of the grouping (categorical) variable in \code{data}.
#' @param pval.display Logical. If \code{TRUE} and there are exactly two groups in \code{x}, a p-value annotation is added.
#' @param ldat A data frame containing the p-value(s) to annotate on the plot. Expected columns: `"p value"` and/or `"adjusted p value"`.
#' @param tip_length Numeric. Length of the annotation tips for \code{geom_signif} (default: 0.0004).
#' @param pval_y_position Numeric. Vertical position for the p-value annotation (default: 1.05).
#' @param ... Additional arguments passed to \code{\link[ggstatsplot]{ggbarstats}}.
#'
#' @details
#' If \code{groupby} is not \code{NULL} and there are exactly two levels in the grouping variable, and \code{pval.display = TRUE}, a p-value from \code{ldat} is added between the two groups using \code{geom_signif}. If \code{"adjusted p value"} is present in \code{ldat}, it is used; otherwise, \code{"p value"} is used.
#'
#' @return A \code{ggplot} object.
#' @author Fabian Kück
#' @export
#' @examples
#' ## ungrouped plot
#' p <- w.ggbarstats(mtcars, "gear",
#'                   bf.message = FALSE,
#'                   proportion.test = FALSE)
#' if (FALSE) print(p)
#'
#' ## grouped plot
#' p <- w.ggbarstats(mtcars, "gear", groupby = "cyl",
#'                   bf.message = FALSE,
#'                   proportion.test = FALSE)
#' if (FALSE) print(p)
#'
#' ## grouped plot with p value annotation for >2 groups
#' pval <- fisher.test(mtcars$gear, mtcars$cyl)$p.value
#' pval <- descutils::prettyPvalues(pval, digits = 3, orgbold = FALSE, lhs = "p")
#' ldat <- data.frame(`p value` = pval, check.names = FALSE)
#' p <- w.ggbarstats(mtcars, "gear", groupby = "cyl", pval.display = TRUE, ldat = ldat,
#'                   bf.message = FALSE,
#'                   proportion.test = FALSE)
#' if (FALSE) print(p)
#'
#' ## grouped plot with p value annotation for two groups
#' pval <- fisher.test(mtcars$gear, mtcars$vs)$p.value
#' pval <- descutils::prettyPvalues(pval, digits = 3, orgbold = FALSE, lhs = "p")
#' ldat <- data.frame(`p value` = pval, check.names = FALSE)
#' p <- w.ggbarstats(mtcars, "gear", groupby = "vs", pval.display = TRUE, ldat = ldat,
#'                   bf.message = FALSE,
#'                   proportion.test = FALSE)
#' if (FALSE) print(p)
w.ggbarstats <- function(data, parameter, groupby = NULL, pval.display = FALSE, ldat, tip_length = 0.0004, pval_y_position = 1.05, ...)
{
  ## setup plot data
  if (is.null(groupby))
    data <- data %>% dplyr::mutate(...dummy = "")

  ## generate plot
  p <- eval(substitute(ggstatsplot::ggbarstats(
                                      data = data,
                                      x = XXX,
                                      y = YYY,
                                      ...),
                       list(XXX = parameter,
                            YYY = ifelse(is.null(groupby), "...dummy", groupby))))

  ## add p values
  if(!is.null(groupby) &&
     pval.display &&
     length(unique(na.omit(data[[groupby]][!is.na(data[[parameter]])]))) > 1 &&
     ("adjusted p value" %in% colnames(ldat) || "p value" %in% colnames(ldat))) {
    p_annotation <- ifelse("adjusted p value" %in% colnames(ldat), ldat$`adjusted p value`, ldat$`p value`)
    n_groups <- length(unique(stats::na.omit(data[[groupby]])))
    if (n_groups == 2) {
      p <- p + ggsignif::geom_signif(y_position = pval_y_position, # 5% above tallest bar
                                     xmin = 1,
                                     xmax = 2,
                                     annotation = p_annotation,
                                     tip_length = tip_length)
    } else if (n_groups > 2) {
      p <- p +
        ggplot2::labs(subtitle = p_annotation) +
        ggplot2::theme(plot.subtitle = ggplot2::element_text(hjust = 0.5, vjust = -5))      # center subtitle
    }
  }

  ## final adjustments
  ## remove x axis annotation if ungrouped
  if (is.null(groupby))
    p <- p +
      ggplot2::theme(
                 axis.title.x       = ggplot2::element_blank(),
                 axis.ticks.x       = ggplot2::element_blank(),
                 panel.grid.major.x = ggplot2::element_blank())

  ## done!
  return(p)
}



#' Wrapper for ggstatsplot::ggscatterstats with optional p-value annotation
#'
#' This function creates a `ggscatterstats` plot from `ggstatsplot`, and optionally adds a p-value annotation based on values in `ldat`.
#' It can also format the y-axis as dates.
#'
#' @param data A data frame with the data to be plotted.
#' @param parameter Character. Name of the column in \code{data} to the values to be plotted.
#' @param groupby Character. Name of the grouping (categorical) variable in \code{data}.
#' @param pval.display Logical. If \code{TRUE} and there are exactly two groups in \code{groupby}, a p-value annotation is added.
#' @param ldat A data frame containing the p-value(s) to annotate on the plot. Expected columns: `"p value"` and/or `"adjusted p value"`.
#' @param ... Additional arguments passed to \code{\link[ggstatsplot]{ggscatterstats}}.
#'
#' @details
#' If \code{groupby} is not \code{NULL} and there are exactly two levels in the grouping variable, and \code{pval.display = TRUE}, a p-value from \code{ldat} is added between the two groups using \code{geom_signif}. If \code{"adjusted p value"} is present in \code{ldat}, it is used; otherwise, \code{"p value"} is used. If \code{wasDate} is \code{TRUE}, the y-axis labels are formatted as dates.
#'
#' @return A \code{ggplot} object.
#' @author Fabian Kück
#' @export
#' @examples
#' ## plot
#' p <- w.ggscatterstats(mtcars, parameter = "cyl", groupby = "carb",
#'                       bf.message = FALSE,)
#' if (FALSE) print(p)
#'
#' ## plot with p value annotation
#' pval <- cor.test(~ cyl + carb, data = mtcars, method = "kendall")$p.value
#' pval <- descutils::prettyPvalues(pval, digits = 3, orgbold = FALSE, lhs = "p")
#' ldat <- data.frame(`p value` = pval, check.names = FALSE)
#' p <- w.ggscatterstats(mtcars, parameter = "cyl", groupby = "carb", pval.display = TRUE, ldat = ldat,
#'                       bf.message = FALSE,)
#' if (FALSE) print(p)
w.ggscatterstats <- function(data, parameter, groupby, pval.display = FALSE, ldat, ...)
{
  as.numericfactor <- function(x) as.numeric(as.factor(x))

  ## setup plot data
  if (inherits(data[[parameter]], "Date")) {
    data <- data %>% dplyr::mutate_at(parameter, as.numeric)
    wasDate <- TRUE
  }
  if (is.character(data[[parameter]]))
    data <- data %>% dplyr::mutate_at(parameter, as.numericfactor)
  if (!is.numeric(data[[parameter]]))
    data <- data %>% dplyr::mutate_at(parameter, as.numeric)
  if (is.character(data[[groupby]]))
    data <- data %>% dplyr::mutate_at(groupby, as.numericfactor)
  if (!is.numeric(data[[groupby]]))
    data <- data %>% dplyr::mutate_at(groupby, as.numeric)

  ## generate plot
  p <- eval(substitute(ggstatsplot::ggscatterstats(data = data,
                                                   x = XXX,
                                                   y = YYY,
                                                   ...),
                       list(XXX = groupby,
                            YYY = parameter)))

  ## add p values
  if (!is.null(groupby) &&
      pval.display &&
      length(unique(na.omit(data[[groupby]][!is.na(data[[parameter]])]))) > 1 &&
      ("adjusted p value" %in% colnames(ldat) || "p value" %in% colnames(ldat))) {
    p_annotation <- ifelse("adjusted p value" %in% colnames(ldat), ldat$`adjusted p value`, ldat$`p value`)
    p <- p + ggplot2::labs(subtitle = c(p_annotation)) +
      ggplot2::theme(plot.subtitle = ggplot2::element_text(hjust = 0.5))      # center subtitle
  }

  ## final adjustments
  ## Date scale annotation
  if (exists("wasDate") && wasDate)
    p <- p + ggplot2::scale_y_continuous(labels = function(x) zoo::as.Date(as.numeric(x)))

  ## done!
  return(p)
}
