% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R, R/model-class.R
\name{validate_ddm_parameters}
\alias{validate_ddm_parameters}
\alias{simulate_ddm_trials}
\title{Simulate Design-based Decision Diffusion Model (DDM) Trials}
\usage{
validate_ddm_parameters(rt_model_r, parameters_r, debug = FALSE)

simulate_ddm_trials(
  rt_model_r,
  parameters_r,
  time_parameters_r = as.numeric(c()),
  n_trial = 1L,
  debug = FALSE
)
}
\arguments{
\item{rt_model_r}{An S4 object of class \code{ddm} specifying the DDM
configuration. Must contain slots for boundary separation, drift rate,
and other DDM-relevant parameters.}

\item{parameters_r}{A numeric vector of DDM parameters. Expected elements:
   \describe{
     \item{a}{Boundary separation (threshold)}
     \item{v}{Drift rate (evidence accumulation speed)}
     \item{t0}{Non-decision time (encoding + motor response baseline)}
     \item{z}{Starting point (bias; must satisfy \code{0 < z < a})}
     \item{d}{(Optional) Difference in motor execution time between
boundaries}
     \item{sv}{(Optional) Inter-trial variability in drift rate}
     \item{st0}{(Optional) Inter-trial variability in non-decision time}
     \item{sz}{(Optional) Inter-trial variability in starting point}
   }}

\item{debug}{Logical. If \code{TRUE}, prints internal simulation states
and timing diagnostics.}

\item{time_parameters_r}{A numeric vector controlling simulation's
temporal dynamics:
       \describe{
         \item{[1]}{Minimum time (default: \code{-0.5})}
         \item{[2]}{Maximum time (default: \code{0.5})}
         \item{[3]}{Time step for numerical integration (default:
\code{0.01})}
       }}

\item{n_trial}{Integer. Number of trials to simulate (default: \code{1}).}
}
\value{
A \code{data.frame} with the following columns:
\describe{
  \item{rt}{Simulated response times in seconds}
  \item{response}{Binary decision outcome
(0 = lower boundary, 1 = upper boundary)}
}
}
\description{
Generates synthetic response time (RT) and choice data using a diffusion
decision model (DDM) with specified parameters. The simulation allows for
flexible parameter settings and time-domain controls.
}
\details{
The core simulation is implemented in C++ via \code{simulate_ddm_trials}
for high performance in large-scale experiments. A complementary R
wrapper (\code{.simulate_ddm_trials}) provides additional functionality,
such as random seed control and input validation.

The function \code{validate_ddm_parameters} checks whether the input
parameter vector satisfies the constraints of the diffusion model.
}
\examples{
# Basic simulation with default parameters
if (requireNamespace("ggdmcModel", quietly = TRUE)) {
  BuildModel <- getFromNamespace("BuildModel", "ggdmcModel")
  model <- ggdmcModel::BuildModel(
    p_map = list(
      a = "1", v = "1", z = "1", d = "1", sz = "1", sv = "1",
      t0 = "1", st0 = "1", s = "1", precision = "1"
    ),
    match_map = list(M = list(s1 = "r1", s2 = "r2")),
    factors = list(S = c("s1", "s2")),
    constants = c(d = 0, s = 1, st0 = 0, sv = 0, precision = 3),
    accumulators = c("r1", "r2"),
    type = "fastdm"
  )
}
sub_model <- setDDM(model)
p_vector <- c(a = 1, sz = 0.25, t0 = 0.15, v = 2.5, z = .38)
time_parameters <- c(t_min = -0.5, tmax = 0.5, dt = 0.01)

# Simulation with custom time parameters
sim_data <- simulate_ddm_trials(
  rt_model_r = sub_model,
  parameters_r = p_vector,
  time_parameters_r = time_parameters,
  n_trial = 32
)

}
\references{
Ratcliff, R., & McKoon, G. (2008). The diffusion decision model: Theory
and data for two-choice decision tasks. Neural Computation, 20(4), 873-922.
}
\seealso{
\code{\link{ddm-class}}, \code{\link{setDDM}}, \code{\link{validate_ddm_parameters}}
}
