#' ctablerseh
#'
#' Processes survey data and displays the estimation results along with the relative standard error in the form of a table that includes the number of samples with a t-distribution approach to produce confidence intervals as in SPSS.
#'
#' @param numerator is a variable that contains the numerator value or main value of the observation, the estimated value of which will then be calculated, either in the form of a proportion, ratio or average.
#' @param denominator is a variable that contains the denominator or divisor value which is usually all observations for proportion cases, or observations with different conditions for ratio cases, or is given the NA value if you only want to calculate the average.
#' @param disaggregation is a variable that contains the grouping code for each observation, which can be the smallest group such as district, province or if you want to produce one aggregate estimate value, you can use the same grouping code for all observations.
#' @param survey.design is a parameter that contains the survey design created with the "svydesign" function from the "survey" package.
#' @param survey.data is the data used in calculating the estimated survey results.
#' @param survey.type is a parameter that contains the survey type options, namely proportion (output in percentage form) by selecting "prop", or ratio (output in percentage form) by selecting "ratio" or average by selecting "mean".
#' @export
#'
#' @import survey stats
#' @importFrom stats as.formula qt
#' @importFrom survey degf svyby svytotal
#'
#'
#' @return This function returns a dataframe containing the estimated value, standard error, relative standard error, confidence interval, number of samples, and the population represented.
#'
#' @examplesIf requireNamespace("survey", quietly = TRUE)
#' susenas.design = survey::svydesign(id=~psu, strata=~strata, data = datex, weights=~FWT)
#' ctablerseh(numerator = INDI, denominator = denom, disaggregation = R101,
#' survey.design = susenas.design, survey.data = datex, survey.type = "prop")

ctablerseh = function(numerator, denominator, disaggregation, survey.design, survey.data, survey.type){

  if(survey.type=="ratio"){
    # Mendapatkan nama kolom berdasarkan input
    num_name <- deparse(substitute(numerator))
    denom_name <- deparse(substitute(denominator))
    disag_name <- deparse(substitute(disaggregation))
    fname.est <- paste(num_name, "/", denom_name, sep = "")
    fname.se <- paste("se.",num_name, "/", denom_name, sep = "")

    # Mengonversi nama kolom menjadi formula yang sesuai untuk svyby
    formula_num <- stats::as.formula(paste("~", num_name))
    formula_denom <- stats::as.formula(paste("~", denom_name))
    formula_disag <- stats::as.formula(paste("~", disag_name))

    # Menjalankan survey::svyby dengan parameter yang fleksibel
    hasil_INDI <- survey::svyby(
      formula = formula_num,
      denom = formula_denom,
      by = formula_disag,
      design = survey.design,
      data = survey.data,
      FUN = survey::svyratio,
      vartype = c("se", "cvpct")
    )

    #  return(hasil_INDI)

    hasil_INDI$theta = round(hasil_INDI[[fname.est]],4)
    hasil_INDI$SE = round(hasil_INDI[[fname.se]],4)
    hasil_INDI$CI_LOWER = round((hasil_INDI[[fname.est]] - stats::qt(0.975, df=survey::degf(survey.design))*hasil_INDI[[fname.se]]),4)
    hasil_INDI$CI_UPPER = round((hasil_INDI[[fname.est]] + stats::qt(0.975, df=survey::degf(survey.design))*hasil_INDI[[fname.se]]),4)
    hasil_INDI$RSE = round(hasil_INDI$`cv%`,4)
    output_INDI = as.data.frame(cbind(hasil_INDI[[disag_name]], hasil_INDI$theta, hasil_INDI$SE, hasil_INDI$RSE, hasil_INDI$CI_LOWER, hasil_INDI$CI_UPPER))
    names(output_INDI) = c("Characteristics","Estimation","SE","RSE","CI LOWER","CI UPPER")

    result_unweighted <- as.data.frame(table(survey.design$variables[[disag_name]])) #yg fleksible denom menyesuaikan disagregasi
    colnames(result_unweighted) <- c("Characteristics", "Unweighted Count")
    survey.design$variables$one <- 1
    result_weighted <- as.data.frame(survey::svyby(~one, formula_disag, design = survey.design, survey::svytotal)[,1:2]) #yg fleksible denom menyesuaikan disagregasi
    colnames(result_weighted) <- c("Characteristics", "Population Size")
    final_result <- merge(result_unweighted, result_weighted, by = "Characteristics") #yg fleksibel Kategori
    output_INDI = merge(output_INDI, final_result, by = "Characteristics")
    output_INDI[order(output_INDI$Characteristics, decreasing = FALSE), ]
    return(output_INDI)
  }else{
    if(survey.type=="prop"){
      # Mendapatkan nama kolom berdasarkan input
      num_name <- deparse(substitute(numerator))
      denom_name <- deparse(substitute(denominator))
      disag_name <- deparse(substitute(disaggregation))
      fname.est <- paste(num_name, "/", denom_name, sep = "")
      fname.se <- paste("se.",num_name, "/", denom_name, sep = "")

      # Mengonversi nama kolom menjadi formula yang sesuai untuk svyby
      formula_num <- stats::as.formula(paste("~", num_name))
      formula_denom <- stats::as.formula(paste("~", denom_name))
      formula_disag <- stats::as.formula(paste("~", disag_name))

      # Menjalankan survey::svyby dengan parameter yang fleksibel
      hasil_INDI <- survey::svyby(
        formula = formula_num,
        denom = formula_denom,
        by = formula_disag,
        design = survey.design,
        data = survey.data,
        FUN = survey::svyratio,
        vartype = c("se", "cvpct")
      )

      #  return(hasil_INDI)

      hasil_INDI$theta = round(hasil_INDI[[fname.est]]*100,4)
      hasil_INDI$SE = round(hasil_INDI[[fname.se]]*100,4)
      hasil_INDI$CI_LOWER = round((hasil_INDI[[fname.est]] - stats::qt(0.975, df=survey::degf(survey.design))*hasil_INDI[[fname.se]])*100,4)
      hasil_INDI$CI_UPPER = round((hasil_INDI[[fname.est]] + stats::qt(0.975, df=survey::degf(survey.design))*hasil_INDI[[fname.se]])*100,4)
      hasil_INDI$RSE = round(hasil_INDI$`cv%`,4)
      output_INDI = as.data.frame(cbind(hasil_INDI[[disag_name]], hasil_INDI$theta, hasil_INDI$SE, hasil_INDI$RSE, hasil_INDI$CI_LOWER, hasil_INDI$CI_UPPER))
      names(output_INDI) = c("Characteristics","Estimation","SE","RSE","CI LOWER","CI UPPER")

      result_unweighted <- as.data.frame(table(survey.design$variables[[disag_name]])) #yg fleksible denom menyesuaikan disagregasi
      colnames(result_unweighted) <- c("Characteristics", "Unweighted Count")
      survey.design$variables$one <- 1
      result_weighted <- as.data.frame(survey::svyby(~one, formula_disag, design = survey.design, survey::svytotal)[,1:2]) #yg fleksible denom menyesuaikan disagregasi
      colnames(result_weighted) <- c("Characteristics", "Population Size")
      final_result <- merge(result_unweighted, result_weighted, by = "Characteristics") #yg fleksibel Kategori
      output_INDI = merge(output_INDI, final_result, by = "Characteristics")
      output_INDI[order(output_INDI$Characteristics, decreasing = FALSE), ]
      return(output_INDI)
    } else{if(survey.type=="mean"){
      # Mendapatkan nama kolom berdasarkan input
      num_name <- deparse(substitute(numerator))
      denom_name <- deparse(substitute(denominator))
      disag_name <- deparse(substitute(disaggregation))
      fname.est <- paste(num_name, "/", denom_name, sep = "")
      fname.se <- paste("se.",num_name, "/", denom_name, sep = "")

      # Mengonversi nama kolom menjadi formula yang sesuai untuk svyby
      formula_num <- stats::as.formula(paste("~", num_name))
      formula_denom <- stats::as.formula(paste("~", denom_name))
      formula_disag <- stats::as.formula(paste("~", disag_name))

      # Menjalankan survey::svyby dengan parameter yang fleksibel
      hasil_INDI <- survey::svyby(
        formula = formula_num,
        denom = formula_denom,
        by = formula_disag,
        design = survey.design,
        data = survey.data,
        FUN = survey::svymean,
        vartype = c("se", "cvpct")
      )

      #  return(hasil_INDI)

      hasil_INDI$theta = round(hasil_INDI[[num_name]],4)
      hasil_INDI$SE = round(hasil_INDI[["se"]],4)
      hasil_INDI$CI_LOWER = round((hasil_INDI[[num_name]] - stats::qt(0.975, df=survey::degf(survey.design))*hasil_INDI[["se"]]),4)
      hasil_INDI$CI_UPPER = round((hasil_INDI[[num_name]] + stats::qt(0.975, df=survey::degf(survey.design))*hasil_INDI[["se"]]),4)
      hasil_INDI$RSE = round(hasil_INDI$`cv%`,4)
      output_INDI = as.data.frame(cbind(hasil_INDI[[disag_name]], hasil_INDI$theta, hasil_INDI$SE, hasil_INDI$RSE, hasil_INDI$CI_LOWER, hasil_INDI$CI_UPPER))
      names(output_INDI) = c("Characteristics","Estimation","SE","RSE","CI LOWER","CI UPPER")

      result_unweighted <- as.data.frame(table(survey.design$variables[[disag_name]])) #yg fleksible denom menyesuaikan disagregasi
      colnames(result_unweighted) <- c("Characteristics", "Unweighted Count")
      survey.design$variables$one <- 1
      result_weighted <- as.data.frame(survey::svyby(~one, formula_disag, design = survey.design, survey::svytotal)[,1:2]) #yg fleksible denom menyesuaikan disagregasi
      colnames(result_weighted) <- c("Characteristics", "Population Size")
      final_result <- merge(result_unweighted, result_weighted, by = "Characteristics") #yg fleksibel Kategori
      output_INDI = merge(output_INDI, final_result, by = "Characteristics")
      output_INDI[order(output_INDI$Characteristics, decreasing = FALSE), ]
      return(output_INDI)
    }
    }
  }
}
