% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cat_cox_bayes.R
\name{cat_cox_bayes}
\alias{cat_cox_bayes}
\title{Bayesian Estimation for Catalytic Cox Proportional-Hazards Model (COX) with Fixed tau}
\usage{
cat_cox_bayes(
  formula,
  cat_init,
  tau = NULL,
  hazard_beta = 2,
  chains = 4,
  iter = 2000,
  warmup = 1000,
  ...
)
}
\arguments{
\item{formula}{A formula specifying the Cox model. Should at least include response variables (e.g. \code{~.}).}

\item{cat_init}{A list generated from \code{cat_cox_initialization}.}

\item{tau}{Optional numeric scalar controlling the weight of the synthetic data in the coefficient estimation, defaults to \code{ncol(cat_init$obs_x)}.}

\item{hazard_beta}{Numeric, default \code{2}. Shape parameter for the Gamma distribution in the hazard model.}

\item{chains}{Integer, default \code{4}. Number of Markov chains to be run in the RStan sampling.}

\item{iter}{Integer, default \code{2000}. Number of iterations per chain in the RStan sampling.}

\item{warmup}{Integer, default \code{1000}. Number of warm-up (or burn-in) iterations for each chain.}

\item{...}{Additional arguments passed to RStan’s \code{rstan::sampling} function.}
}
\value{
A list containing the values of all the arguments and the following components:
\item{stan_data}{A data list used for fitting RStan sampling model.}
\item{stan_model}{Compiled RStan model object for Cox regression.}
\item{stan_sample_model}{Fitted RStan sampling model containing posterior samples.}
\item{coefficients}{Mean posterior estimates of model coefficients from \code{stan_sample_model}.}
\item{increment_cumulative_baseline_hazard}{Mean posterior estimates of
Estimated cumulative hazard increments over time intervals from \code{stan_sample_model}.}
}
\description{
This function performs Bayesian estimation for a catalytic Cox proportional-hazards model (COX) using RStan
by given a single tau value. It allows users to estimate the coefficients and cumulative baseline hazard increments
over specified time intervals with Bayesian sampling.
}
\examples{
\donttest{
library(survival)
data("cancer")
cancer$status[cancer$status == 1] <- 0
cancer$status[cancer$status == 2] <- 1

cat_init <- cat_cox_initialization(
  formula = Surv(time, status) ~ 1, # formula for simple model
  data = cancer,
  syn_size = 100, # Synthetic data size
  hazard_constant = 0.1, # Hazard rate value
  entry_points = rep(0, nrow(cancer)), # Entry points of each observation
  x_degree = rep(1, ncol(cancer) - 2), # Degrees for polynomial expansion of predictors
  resample_only = FALSE, # Whether to perform resampling only
  na_replace = stats::na.omit # How to handle NA values in data
)

cat_model <- cat_cox_bayes(
  formula = ~., # Should at least include response variables
  cat_init = cat_init, # Only accept object generated from `cat_cox_initialization`
  tau = 1, # Weight for synthetic data
  hazard_beta = 2, # Shape parameter for the Gamma distribution in the hazard model
  chains = 1, # Number of Markov chains to be run in the RStan sampling
  iter = 10, # Number of iterations per chain in the RStan sampling
  warmup = 5 # Number of warm-up (or burn-in) iterations for each chain
)
cat_model
}
}
