## ------------------------------------------------------------------------
# 
# "per_boo(x, B, taper0=0) " 
# 
# Function compute the bootstrap periodograms at Fourier frequencies 
#
# ------------------------------------------------------------------------
##


#' @aliases per_boo
#' @title Bootstrap of Periodogram 
#' @description Computes periodogram values at Fourier frequencies for a time series, smooths 
#'              the periodogram to estimate the spectral density, and generates the bootstrap 
#'              version of the periodogram.
#' @param x A vector or a time series.
#' @param B A positive integer; the number of bootstrap replications.
#' @param taper0 A numeric value; specifies the proportion of data to taper. 
#' The default value is 0, that is there is no tapering.
#' @details The function first centers the input time series and calculates the values of
#' the periodogram 
#' at Fourier frequencies using \code{spec.pgram}. Spectral density is then estimated by applying 
#' a kernel smoother to the periodogram values, with the smoothing bandwidth computed as 
#' \eqn{sd(x) * n^(-1/3)}. Bootstrap is then performed by resampling periodogram ordinates.
#' 
#' The function outputs a graph of the histogram of the periodogram ordinates which should be close to
#' an exponential density.
#' @return A list containing:
#' \itemize{
#'   \item `obj`: A list of class "boodd" containing the bootstrap periodograms.
#'   \item `freqs`: A vector of Fourier frequencies used in the periodogram estimation.
#' }
#' @references  Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#' 
#' Bertail, P. and  Dudek, A.E. (2021). Consistency of the Frequency Domain
#' Bootstrap for differentiable functionals, \emph{Electron. J. Statist.}, \bold{15}, 1-36.
#' 
#' Hurvich, C. M. and Zeger, S. L. (1987). Frequency domain bootstrap 
#' methods for time series, Technical Report 87-115, Graduate School of Business
#' Administration, New York Univ.
#' 
#' Lahiri, S.N. (2003). \emph{Resampling Methods for Dependent Data}. Springer,
#' New York.
#' @seealso \code{\link{tft_boot}}, \code{\link{func_fdb}}, \code{\link{freqboot}}.
#' @keywords "Frequency bootstrap" Periodogram "Periodogram ordinates"
#' @export
#' @examples 
#' set.seed(12345)
#' x=arima.sim(model=list(ar=0.8),n=200)
#' boo1=per_boo(x,99)
#' 
#' fn=length(boo1[[2]])
#' spec.pgram(x, plot=TRUE)
#' # Superimposed plots of 99 bootstrap periodograms
#' for ( i in (1:99)) {
#' lines(boo1[[2]],t(boo1[[1]]$s)[,i], type="l", col=i)
#' }


per_boo<-function (x, B, taper0=0) {
  x=x-mean(x)
  n <- length(x)
  bandwidth <- sd(x) * n^(-1/3)
  kernel="epanechnikov"
  x <- ts(x, frequency = 1)
  # computation of periodogram at Fourier frequencies
  P <- spec.pgram(x, plot = FALSE, taper = taper0, fast = FALSE, 
                  detrend = FALSE)
  # values of Periodogram
  specs <- P$spec/(2 * pi)
  # values of frequencies
  freqs <- P$freq * 2 * pi
  n0 <- length(specs)
  # estimation of density by smoothing
  smc <- smoothingCoefficients(n, bandwidth, kernel)
  # P0 <- sum(x^2)/(2 * pi * n)
  In <- c(rev(specs), specs)
  ljn <- c(-rev(freqs), freqs)
  fnhat <- numeric(n0)
  # compute smoothing weights for spectral density estimator
  for (i in 1:n0) {
    K <- smc[(n0 - i + 1):(3 * n0 - i)]
    fnhat[i] <- 2 * pi * mean(K * In)/bandwidth
  }
  res <- matrix(nrow = B, ncol = n0)
  eps_hat <- specs/fnhat
  eps_tilde <- eps_hat/mean(eps_hat)
  
  hist(eps_tilde,nclass=35,freq =FALSE)
  
  for (j in 1:B) {
    fnhat_star <- numeric(n0)
    eps_star <- sample(eps_tilde, n0, replace = TRUE)
    I_star <- fnhat * eps_star
    res[j,] = I_star[1:n0]
  }
  # returns results and frequencies
  obj <- list(s = res, Tn =specs )
  class(obj) <- "boodd"
  attr(obj, "kind") <- "Periodogram"
  attr(obj, "func") <- "spec.gram"
  return(list(obj, freqs))
}
