
##
# ------------------------------------------------------------------------
#
# "field.sub(arr,func,length.block,...)" --
#
# subsampling of discrete multidimensional random fields.
#
# ------------------------------------------------------------------------
##
#' @aliases field.sub
#' @title Subsampling of Random Fields
#' @description Performs subsampling of a multidimensional array representing a
#' random field on a lattice. 
#' This function applies a specified function to each subsample.
#' @param arr A multidimensional real-valued array; it represents a random field on a grid of dimension
#' equals to dimension of the \code{arr}. 
#' @param func A function applied to each subsample. 
#' The function should accept the subsample as input and return a vector.
#' @param length.block An integer or vector of integers; it specified the block lengths for subsampling.
#'  If a scalar is provided, the same block length is used for all dimensions
#' @param ... An optional additional arguments for the \code{func} function.
#' @details 
#' The \code{field.sub} function is designed for subsampling a multidimensional array. 
#' The \code{length.block} argument defines the size of each subsample. If \code{length.block} 
#' is a scalar, 
#' it applies uniformly across all dimensions of the array. Otherwise, it must 
#' be a vector with a length equal to the number of dimensions in \code{arr}. 
#' The function \code{func} is applied to each subsample, 
#' and the results are returned in a matrix or vector, depending on the output of \code{func}.
#' @return Returns a matrix or vector, depending on the output of the \code{func} 
#' function. Each row in the matrix 
#' corresponds to the result of applying \code{func} to a subsample.
#' Since it is not a \code{boodd} object, one cannot apply, for example, 
#' the \code{confint} function to construct a confidence intervals (which depends on the rate of 
#' convergence of the statistic of interest), see example below.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Politis, D.N. Romano, J.P. Wolf, M.  (1999). \emph{Subsampling}, Springer, New York.
#' 
#' @seealso \code{\link{fieldboot}}, \code{\link{fieldbootP}}, \code{\link{blockboot}}, \code{\link{jackVarField}}. 
#' @keywords Bootstrap "Random fields" Subsampling
#' @export
#' @examples 
#' \donttest{
#' set.seed(1)
#' # 2-dims array
#' dlens <- c(100,50)
#' blens <- c(6,3)
#' arr <- array(round(rnorm(prod(dlens)),2),dim=dlens)
#' resu<-field.sub(arr,mean,blens)
#' hist(resu,nclass=25)
#' N=length(resu)
#' lB=length(blens)
#' conf=mean(arr)-(quantile(resu,c(0.025,0.975))-mean(arr))*sqrt(lB)/sqrt(N)
#' conf[2:1]
#' }
##
field.sub <- function(arr,func,length.block,...) {
  # Test the value returned by func
  y <- func(arr,...)
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  
  # Data lengths n_1, n_2, ..., n_d
  dlens <- dim(arr)
  ndims <- length(dlens)
  if (ndims < 2) {
    stop("expected at least 2-dimensional array")
  }
  if (length(length.block) == 1) {
    length.block = rep(length.block,ndims)
  }
  if (length(length.block) != ndims) {
    stop("wrong number of block lengths")
  }
  if (any(length.block > dlens)) {
    stop("block lengths must be less than data lengths in all dimensions")
  }
  if (any(length.block <= 0)) {
    stop("block lengths must be positive")
  }
  
  # Block lengths b_1, b_2, ..., b_d
  blens <- length.block
  last <- dlens-blens+1
  fulldims <- ((dlens-1)%/%blens +1)*blens
  
  # Array indices as string "1:n_1,1:n_2,...,1:n_d"
  indstr <- blockString(rep(1,ndims),dlens)
  
  # Build the tile indices (as strings) defining the covering
  # of the reconstructed array
  tiles <- blockTiles(blens,last)
  tnum <- length(tiles)
  
  # Initialize results 
  res <- matrix(nrow=tnum,ncol=len)
  
  # Initialize the new array (a slightly bigger array containing an
  # entire number of tiles that will be trimmed later)
  narr <- array(0,dim=fulldims)
  origs <- integer(ndims)
  
  
  for (j in 1:tnum) {
    tile <- tiles[j]
    
    # Apply the statistics to the rebuilt array (trimmed to the
    # original dimensions)
    resCmd <- paste0("res[",j,",] <- func(arr[",tile,"],...)")
    eval(parse(text=resCmd))
  }
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  return(res)
}