% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bfastpp.R
\name{bfastpp}
\alias{bfastpp}
\title{Time Series Preprocessing for BFAST-Type Models}
\usage{
bfastpp(
  data,
  order = 3,
  lag = NULL,
  slag = NULL,
  na.action = na.omit,
  stl = c("none", "trend", "seasonal", "both"),
  decomp = c("stl", "stlplus"),
  sbins = 1
)
}
\arguments{
\item{data}{A time series of class \code{\link[stats]{ts}}, or another
object that can be coerced to such. For seasonal components, a frequency
greater than 1 is required.}

\item{order}{numeric. Order of the harmonic term, defaulting to \code{3}.}

\item{lag}{numeric. Orders of the autoregressive term, by default omitted.}

\item{slag}{numeric. Orders of the seasonal autoregressive term, by default
omitted.}

\item{na.action}{function for handling \code{NA}s in the data (after all
other preprocessing).}

\item{stl}{character. Prior to all other preprocessing, STL (season-trend
decomposition via LOESS smoothing) can be employed for trend-adjustment
and/or season-adjustment.  The \code{"trend"} or \code{"seasonal"} component
or both from \code{\link[stats]{stl}} are removed from each column in
\code{data}. By default (\code{"none"}), no STL adjustment is used.}

\item{decomp}{"stlplus" or "stl": use the NA-tolerant decomposition package
or the reference package (which can make use of time series with 2-3
observations per year)}

\item{sbins}{numeric. Controls the number of seasonal dummies. If integer > 1,
sets the number of seasonal dummies to use per year.
If <= 1, treated as a multiplier to the number of observations per year, i.e.
\code{ndummies = nobs/year * sbins}.}
}
\value{
If no formula is provided, \code{bfastpp} returns a
\code{"data.frame"} with the following variables (some of which may be
matrices).  \item{time}{numeric vector of time stamps,}
\item{response}{response vector (first column of \code{data}),}
\item{trend}{linear time trend (running from 1 to number of observations),}
\item{season}{factor indicating season period,} \item{harmon}{harmonic
seasonal terms (of specified \code{order}),} \item{lag}{autoregressive terms
(or orders \code{lag}, if any),} \item{slag}{seasonal autoregressive terms
(or orders \code{slag}, if any),} \item{xreg}{covariate regressor (all
columns of \code{data} except the first, if any).}

If a formula is given, \code{bfastpp} returns a \code{list} with components
\code{X}, \code{y}, and \code{t}, where \code{X} is the design matrix of the
model, \code{y} is the response vector, and \code{t} represents the time of
observations. \code{X} will only contain variables that occur in the
formula. Columns of \code{X} have names as decribed above.
}
\description{
Time series preprocessing for subsequent regression modeling.  Based on a
(seasonal) time series, a data frame with the response, seasonal terms, a
trend term, (seasonal) autoregressive terms, and covariates is computed.
This can subsequently be employed in regression models.
}
\details{
To facilitate (linear) regression models of time series data, \code{bfastpp}
facilitates preprocessing and setting up regressor terms. It returns a
\code{data.frame} containing the first column of the \code{data} as the
\code{response} while further columns (if any) are used as covariates
\code{xreg}. Additionally, a linear trend, seasonal dummies, harmonic
seasonal terms, and (seasonal) autoregressive terms are provided.

Optionally, each column of \code{data} can be seasonally adjusted and/or
trend-adjusted via STL (season-trend decomposition via LOESS smoothing)
prior to preprocessing. The idea would be to capture season and/or trend
nonparametrically prior to regression modelling.
}
\examples{
## set up time series
ndvi <- as.ts(zoo::zoo(cbind(a = som$NDVI.a, b = som$NDVI.b), som$Time))
ndvi <- window(ndvi, start = c(2006, 1), end = c(2009, 23))

## parametric season-trend model
d1 <- bfastpp(ndvi, order = 2)
d1lm <- lm(response ~ trend + harmon, data = d1)
summary(d1lm)
# plot visually (except season, as it's a factor)
plot(zoo::read.zoo(d1)[,-3],
  # Avoid clipping plots for pretty output
  ylim = list(c(min(d1[,2]), max(d1[,2])),
            c(min(d1[,3]), max(d1[,3])),
            c(-1, 1), c(-1, 1), c(-1, 1), c(-1, 1),
            c(min(d1[,6]), max(d1[,6]))
       ))

## autoregressive model (after nonparametric season-trend adjustment)
d2 <- bfastpp(ndvi, stl = "both", lag = 1:2)
d2lm <- lm(response ~ lag, data = d2)
summary(d2lm)

## use the lower level lm.fit function
d3 <- bfastpp(ndvi, stl = "both", lag = 1:2)
d3mm <- model.matrix(response ~ lag, d3)
d3lm <- lm.fit(d3mm, d3$response)
d3lm$coefficients
}
\references{
\insertRef{janbfastmonitor}{bfast}
}
\seealso{
\code{\link[bfast]{bfastmonitor}}
}
\author{
Achim Zeileis
}
\keyword{ts}
