% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/log_rss.R
\name{log_rss}
\alias{log_rss}
\alias{log_rss.glm}
\alias{log_rss.fit_clogit}
\title{Calculate log-RSS for a fitted model}
\usage{
log_rss(object, ...)

\method{log_rss}{glm}(object, x1, x2, ci = NA, ci_level = 0.95, n_boot = 1000, ...)

\method{log_rss}{fit_clogit}(object, x1, x2, ci = NA, ci_level = 0.95, n_boot = 1000, ...)
}
\arguments{
\item{object}{\verb{[fit_logit, fit_clogit]} \cr A fitted RSF or (i)SSF model.}

\item{...}{Further arguments, none implemented.}

\item{x1}{\verb{[data.frame]} \cr A \code{data.frame} representing the habitat values
at location x_1. Must contain all fitted covariates as expected by
\code{predict()}.}

\item{x2}{\verb{[data.frame]} \cr A 1-row \code{data.frame} representing the single
hypothetical location of x_2. Must contain all fitted covariates as expected
by \code{predict()}.}

\item{ci}{\verb{[character]} \cr Method for estimating confidence intervals around
log-RSS. \code{NA} skips calculating CIs. Character string \code{"se"} uses standard error
method and \code{"boot"} uses empirical bootstrap method.}

\item{ci_level}{\verb{[numeric]} \cr Level for confidence interval. Defaults to 0.95
for a 95\% confidence interval.}

\item{n_boot}{\verb{[integer]} \cr Number of bootstrap samples to estimate confidence
intervals. Ignored if \code{ci != "boot"}.}
}
\value{
Returns a \code{list} of class \code{log_rss} with four entries:
\itemize{
\item \code{df}: A \code{data.frame} with the covariates and the \code{log_rss}
\item \code{x1}: A \code{data.frame} with covariate values for \code{x1}.
\item \code{x2}: A \code{data.frame} with covariate values for \code{x2}.
\item \code{formula}: The formula used to fit the model.
}
}
\description{
Calculate log-RSS(x1, x2) for a fitted RSF or (i)SSF
}
\details{
This function assumes that the user would like to compare relative
selection strengths from at least one proposed location (\code{x1}) to exactly
one reference location (\code{x2}).

The objects \code{object$model}, \code{x1}, and \code{x2} will be passed to
\code{predict()}. Therefore, the columns of \code{x1} and \code{x2} must match
the terms in the model formula exactly.
}
\examples{

\donttest{
# RSF -------------------------------------------------------
# Fit an RSF, then calculate log-RSS to visualize results.

# Load packages
library(ggplot2)

# Load data
data("amt_fisher")
amt_fisher_covar <- get_amt_fisher_covars()

# Prepare data for RSF
rsf_data <- amt_fisher |>
  filter(name == "Lupe") |>
  make_track(x_, y_, t_) |>
  random_points() |>
  extract_covariates(amt_fisher_covar$elevation) |>
  extract_covariates(amt_fisher_covar$popden) |>
  extract_covariates(amt_fisher_covar$landuse) |>
  mutate(lu = factor(landuse))

# Fit RSF
m1 <- rsf_data |>
  fit_rsf(case_ ~ lu + elevation + popden)

# Calculate log-RSS
# data.frame of x1s
x1 <- data.frame(lu = factor(50, levels = levels(rsf_data$lu)),
                 elevation = seq(90, 120, length.out = 100),
                 popden = mean(rsf_data$popden))
# data.frame of x2 (note factor levels should be same as model data)
x2 <- data.frame(lu = factor(50, levels = levels(rsf_data$lu)),
                 elevation = mean(rsf_data$elevation),
                 popden = mean(rsf_data$popden))
# Calculate (use se method for confidence interval)
logRSS <- log_rss(object = m1, x1 = x1, x2 = x2, ci = "se")

# Plot
ggplot(logRSS$df, aes(x = elevation_x1, y = log_rss)) +
  geom_hline(yintercept = 0, linetype = "dashed", color = "gray") +
  geom_ribbon(aes(ymin = lwr, ymax = upr), fill = "gray80") +
  geom_line() +
  xlab(expression("Elevation " * (x[1]))) +
  ylab("log-RSS") +
  ggtitle(expression("log-RSS" * (x[1] * ", " * x[2]))) +
  theme_bw()

# SSF -------------------------------------------------------
# Fit an SSF, then calculate log-RSS to visualize results.

# Load data
data(deer)
sh_forest <- get_sh_forest()

# Prepare data for SSF
ssf_data <- deer |>
  steps_by_burst() |>
  random_steps(n = 15) |>
  extract_covariates(sh_forest) |>
  mutate(forest = factor(forest, levels = 1:0,
                    labels = c("forest", "non-forest")),
  cos_ta = cos(ta_),
  log_sl = log(sl_))

# Fit an SSF (note model = TRUE necessary for predict() to work)
m2 <- ssf_data |>
  fit_clogit(case_ ~ forest + strata(step_id_), model = TRUE)

# Calculate log-RSS
# data.frame of x1s
x1 <- data.frame(forest = factor(c("forest", "non-forest")))
# data.frame of x2
x2 <- data.frame(forest = factor("forest", levels = levels(ssf_data$forest)))
# Calculate
logRSS <- log_rss(object = m2, x1 = x1, x2 = x2, ci = "se")

# Plot
ggplot(logRSS$df, aes(x = forest_x1, y = log_rss)) +
  geom_hline(yintercept = 0, linetype = "dashed", color = "gray") +
  geom_errorbar(aes(ymin = lwr, ymax = upr), width = 0.25) +
  geom_point(size = 3) +
  xlab(expression("Forest Cover " * (x[1]))) +
  ylab("log-RSS") +
  ggtitle(expression("log-RSS" * (x[1] * ", " * x[2]))) +
  theme_bw()
}
}
\references{
\itemize{
\item Avgar, T., Lele, S.R., Keim, J.L., and Boyce, M.S.. (2017). Relative Selection
Strength: Quantifying effect size in habitat- and step-selection inference.
\emph{Ecology and Evolution}, 7, 5322–5330.
\item Fieberg, J., Signer, J., Smith, B., & Avgar, T. (2021). A "How to" guide for interpreting parameters in habitat-selection analyses. \emph{Journal of Animal Ecology}, 90(5), 1027-1043.
}
}
\seealso{
See Avgar \emph{et al.} 2017 for details about relative
selection strength.

Default plotting method available: \code{\link{plot.log_rss}()}
}
\author{
Brian J. Smith
}
