% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeTemplateFunction}
\alias{computeTemplateFunction}
\title{Compute a Vectorization of a Persistence Diagram based on Tent Template Functions}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeTemplateFunction()} computes a vectorization using a collection of tent template functions defined by

\deqn{G_{(b,p),\delta}(D) = \sum_{i=1}^N\max \left\{ 0, 1 - \frac{1}{\delta} \max \left( |b_i - b|, |p_i - p| \right) \right\},}
where \eqn{p_i=d_i-b_i} (persistence), \eqn{b\geq 0}, \eqn{p>0} and  \eqn{0<\delta<p}. The point \eqn{(b,p)} is referred to as the center. Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computeTemplateFunction(D, homDim, delta, d, epsilon)
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}

\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}

\item{delta}{a positive scalar representing the increment size used in the computation of the template function.}

\item{d}{a positive integer specifying the number of bins along each axis in the grid.}

\item{epsilon}{a positive scalar indicating the vertical shift applied to the grid.}
}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes the tent template function on a discretized grid determined by \code{delta}, \code{d}, and \code{epsilon}. The number of tent functions is controlled by \code{d}. The value of \eqn{\delta} is chosen such that the box \eqn{[0, \delta d] \times [\epsilon,\delta d + \epsilon]} contains all the points of the diagrams considered in the birth-persistence plane. \eqn{\epsilon} should be smaller than the minimum persistence value across all the diagrams under consideration. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.

If \code{homDim=0} and all the birth values are equal (e.g., zero), one-dimensional tent template functions are used instead for vectorization:

\deqn{G_{p,\delta}(D) = \sum_{i=1}^N\max (0, 1 - \frac{|p_i - p|}{\delta}).}
}

\value{
A numeric vector of dimension \eqn{(d+1)d}, containing the values of the tent template functions centered at the grid points \eqn{\{(\delta i, \delta j + \epsilon)\}_{i=0,j=1}^{d,d}}:
\deqn{
\{ G_{(\delta i, \delta j + \epsilon), \delta}(D) \mid 0 \leq i \leq d, \, 1 \leq j \leq d \}.
}
When one-dimensional tent template functions are used, the returned vector has a dimension of \eqn{d}:
\deqn{
\{ G_{\delta j + \epsilon, \delta}(D) \mid 1 \leq j \leq d \}.
}
}

\author{Umar Islambekov}

\references{
1. Perea, J.A., Munch, E. and Khasawneh, F.A., (2023). Approximating continuous functions on persistence diagrams using template functions. Foundations of Computational Mathematics, 23(4), pp.1215-1272.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

# Compute a vectorizaton based on tent template functions for homological dimension H_0
computeTemplateFunction(D, homDim = 0, delta = 0.1, d = 20, epsilon = 0.01)

# Compute a vectorizaton based on tent template functions for homological dimension H_1
computeTemplateFunction(D, homDim = 1, delta = 0.1, d = 9, epsilon = 0.01)
}
