% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation-spades.R
\name{defineEvent}
\alias{defineEvent}
\title{Alternative way to define events in SpaDES.core}
\usage{
defineEvent(
  sim,
  eventName = "init",
  code,
  moduleName = NULL,
  envir = parent.frame()
)
}
\arguments{
\item{sim}{A \code{simList}}

\item{eventName}{Character string of the desired event name to define. Default is "init"}

\item{code}{An expression that defines the code to execute during the event. This will
be captured, and pasted into a new function (\code{doEvent.moduleName.eventName}),
remaining unevaluated until that new function is called.}

\item{moduleName}{Character string of the name of the module. If this function is
used within a module, then it will try to find the module name.}

\item{envir}{An optional environment to specify where to put the resulting function.
The default will place a function called \code{doEvent.moduleName.eventName} in the
module function location, i.e., \code{sim$.mods[[moduleName]]}. However, if this
location does not exist, then it will place it in the \code{parent.frame()}, with a message.
Normally, especially, if used within SpaDES module code, this should be left missing.}
}
\description{
There are two ways to define what occurs during an event: defining a function
called \code{doEvent.moduleName}, where \code{moduleName} is the actual module name.
This approach is the original approach used in SpaDES.core, and it must have an
explicit \code{switch} statement branching on \code{eventType}.
The newer approach (still experimental) uses \code{defineEvent()}.
Instead of creating, \code{doEvent.moduleName()}, it creates one function
for each event, each with the name \code{doEvent.moduleName.eventName}.
This may be a little bit cleaner, but both with still work.
}
\examples{
sim <- simInit()

# these put the functions in the parent.frame() which is .GlobalEnv for an interactive user
defineEvent(sim, "init", moduleName = "thisTestModule", code = {
  sim <- Init(sim) # initialize
  # Now schedule some different event for "current time", i.e., will
  #   be put in the event queue to run *after* this current event is finished
  sim <- scheduleEvent(sim, time(sim), "thisTestModule", "grow")
}, envir = envir(sim))

defineEvent(sim, "grow", moduleName = "thisTestModule", code = {
  sim <- grow(sim) # grow
  # Now schedule this same event for "current time plus 1", i.e., a "loop"
  sim <- scheduleEvent(sim, time(sim) + 1, "thisTestModule", "grow") # for "time plus 1"
})

Init <- function(sim) {
  sim$messageToWorld <- "Now the sim has an object in it that can be accessed"
  sim$size <- 1 # initializes the size object --> this can be anything, Raster, list, whatever
  message(sim$messageToWorld)
  return(sim)   # returns all the things you added to sim as they are in the simList
}

grow <- function(sim) {
  sim$size <- sim$size + 1 # increments the size
  message(sim$size)
  return(sim)
}

# schedule that first "init" event
sim <- scheduleEvent(sim, 0, "thisTestModule", "init")
# Look at event queue
events(sim) # shows the "init" we just added
\donttest{
  # this is skipped when running in automated tests; it is fine in interactive use
  out <- spades(sim)
}

}
\seealso{
\code{\link[=defineModule]{defineModule()}}, \code{\link[=simInit]{simInit()}}, \code{\link[=scheduleEvent]{scheduleEvent()}}
}
