#' catch_adjust
#'
#' @description
#' Calculates the targeted proportions under historical (old) and proposed (new)
#' harvest slot limits using the same survivorship-by-length framework as
#' \code{\link{prop_target}}. The ratio ("catch adjustment") is returned, and (optionally)
#' a historical catch value is scaled by the adjustment. The optional plot overlays old/new
#' in-slot proportions on the normalized survivorship curve with arrows
#' indicating the direction of change (old → new).
#'
#' @param old_minLS,old_maxLS,old_Lc Numeric. Historical slot limits and length at first capture.
#' @param new_minLS,new_maxLS,new_Lc Numeric. New slot limits and length at first capture. If
#'   `new_Lc` is `NULL`, `old_Lc` is used.
#' @param catch Optional numeric. Historical catch to be adjusted. If provided,
#'   `adjusted_catch = catch * (prop_new/prop_old)` is also computed.
#' @param M Numeric or `NULL`. Natural mortality. If `NULL`,
#'   defaults to `M = 4.899 * tmax^-0.916`.
#' @param growth_model One of `"vb"`, `"gompertz"`, `"schnute"`.
#' @param Linf,K,l0 von Bertalanffy parameters; `l0` is start length (default 0).
#' @param Gom_Linf,Gom_K,Gom_l0 Gompertz parameters; requires `0 < Gom_l0 < Gom_Linf`.
#' @param g1,g2,l2 Schnute parameters; `l2` is length at `tmax`; requires `g1>0`, `l2>0`,
#'   and this parameterization assumes `g2 != 0`.
#' @param tmax Numeric. The maximum observed age used to bound the integrals via `l(tmax)` and
#'   in the default mortality estimator `M = 4.899 * tmax^-0.916`.
#' @param Lmin Optional numeric. Lower bound for the curve grid. If `NULL` it uses the
#'   model’s start length (`l0`, `Gom_l0`, or `0`).
#' @param plot Logical. If `TRUE`, return a `ggplot2` plot.
#'   Default `FALSE` (returns numeric catch adjustment only).
#' @param length_units Optional character scalar. Units to show in the x-axis
#'   label when `plot = TRUE` (e.g., "mm" or "cm"). If `NULL` (default),
#'   the label is simply "Length".
#'
#' @return
#' If `plot = FALSE` (default): a numeric scalar `adjust_factor = prop_new/prop_old`.
#' If `plot = TRUE`: a list with
#' \itemize{
#'   \item `prop_old`, `prop_new` — targeted proportions under old/new slots,
#'   \item `adjust_factor` — `prop_new/prop_old`,
#'   \item `adjusted_catch` — only if `catch` provided,
#'   \item `plot` — the `ggplot2` object (or `NULL` if \pkg{ggplot2} is unavailable).
#' }
#'
#' @examples
#' # numeric only
#' catch_adjust(old_minLS = 130, old_maxLS = 280, old_Lc = 80,
#'              new_minLS = 100, new_maxLS = 240,
#'              growth_model = "vb", Linf = 405, K = 0.118, l0 = 0, tmax = 34)
#'
#' \donttest{
#' # with plot (requires ggplot2)
#' catch_adjust(old_minLS = 130, old_maxLS = 280, old_Lc = 80,
#'              new_minLS = 100, new_maxLS = 240,
#'              growth_model = "vb", Linf = 405, K = 0.118, l0 = 0,
#'              tmax = 34, plot = TRUE, length_units = "mm")
#' # note that overlapping portions stray from color in legend due to alpha value
#' catch_adjust(old_minLS = 100, old_maxLS = 150, old_Lc = 80,
#'              new_minLS = 160, new_maxLS = 300,
#'              growth_model = "vb", Linf = 405, K = 0.118, l0 = 0,
#'              tmax = 34, plot = TRUE, length_units = "mm")
#' }
#'
#' @export
catch_adjust <- function(
    old_minLS = NULL, old_maxLS = NULL, old_Lc = NULL,
    new_minLS = NULL, new_maxLS = NULL, new_Lc = NULL,
    catch = NULL,
    M = NULL,
    growth_model = c("vb", "gompertz", "schnute"),
    Linf = NULL, K = NULL, l0 = 0,
    tmax = NULL,
    Gom_Linf = NULL, Gom_K = NULL, Gom_l0 = NULL,
    g1 = NULL, g2 = NULL, l2 = NULL,
    Lmin = NULL,
    plot = FALSE,
    length_units = NULL
) {
  growth_model <- match.arg(growth_model)

  # basic validation
  if (!is.numeric(tmax) || length(tmax) != 1L || !is.finite(tmax) || tmax <= 0)
    stop("`tmax` must be a positive, finite number.", call. = FALSE)

  if (is.null(M)) M <- 4.899 * tmax^-0.916
  if (!is.numeric(M) || length(M) != 1L || !is.finite(M) || M <= 0)
    stop("`M` must be positive and finite.", call. = FALSE)

  if (!is.logical(plot) || length(plot) != 1L || is.na(plot))
    stop("`plot` must be a single logical (TRUE/FALSE).", call. = FALSE)

  if (!is.null(length_units)) {
    if (!is.character(length_units) || length(length_units) != 1L ||
        is.na(length_units) || !nzchar(trimws(length_units))) {
      stop("`length_units` must be a non-empty character scalar or NULL.", call. = FALSE)
    }
    length_units <- trimws(length_units)
  }
  if (!is.null(Lmin)) {
    if (!is.numeric(Lmin) || length(Lmin) != 1L || !is.finite(Lmin)) {
      stop("`Lmin` must be a single finite numeric or NULL.", call. = FALSE)
    }
  }

  if (is.null(new_Lc)) new_Lc <- old_Lc

  # scalar, finite numerics
  scalar_ok <- function(x) is.numeric(x) && length(x) == 1L && is.finite(x)
  for (nm in c("old_minLS","old_maxLS","old_Lc","new_minLS","new_maxLS","new_Lc")) {
    if (!scalar_ok(get(nm, inherits = FALSE)))
      stop(sprintf("`%s` must be a single finite numeric.", nm), call. = FALSE)
  }
  if (old_maxLS <= old_minLS) stop("Require old_maxLS > old_minLS.", call. = FALSE)
  if (new_maxLS <= new_minLS) stop("Require new_maxLS > new_minLS.", call. = FALSE)

  eps <- 1e-7

  # model validation
  if (growth_model == "vb") {
    if (is.null(Linf) || is.null(K)) stop("VB: Linf and K must be provided.", call. = FALSE)
    if (Linf <= 0 || K <= 0) stop("VB: Linf and K must be positive.", call. = FALSE)
    if (l0 < 0 || l0 >= Linf) stop("VB: require 0 <= l0 < Linf.", call. = FALSE)
  } else if (growth_model == "gompertz") {
    if (is.null(Gom_Linf) || is.null(Gom_K)) stop("Gompertz: Gom_Linf and Gom_K must be provided.", call. = FALSE)
    if (Gom_Linf <= 0 || Gom_K <= 0) stop("Gompertz: Gom_Linf and Gom_K must be positive.", call. = FALSE)
    if (is.null(Gom_l0)) stop("Gompertz: require Gom_l0 and 0 < Gom_l0 < Gom_Linf.", call. = FALSE)
    if (Gom_l0 <= 0 || Gom_l0 >= Gom_Linf) stop("Gompertz: require 0 < Gom_l0 < Gom_Linf.", call. = FALSE)
  } else { # schnute
    if (is.null(g1) || is.null(g2) || is.null(l2)) stop("Schnute: g1, g2, and l2 must be provided.", call. = FALSE)
    if (g1 <= 0) stop("Schnute: require g1 > 0.", call. = FALSE)
    if (l2 <= 0) stop("Schnute: require l2 > 0.", call. = FALSE)
    if (abs(g2) < .Machine$double.eps) stop("Schnute: require g2 != 0 for this parameterization.", call. = FALSE)
  }

  # start-length anchor
  anchor_L <- switch(growth_model,
                     "vb"       = l0,
                     "gompertz" = Gom_l0,
                     "schnute"  = 0
  )
  if (is.null(Lmin)) Lmin <- anchor_L

  # inverse age-from-length
  age_from_length <- switch(
    growth_model,
    "vb" = {
      function(L) {
        L_eff <- pmin(pmax(L, l0), Linf - eps)
        val   <- (Linf - L_eff) / (Linf - l0)
        age   <- -log(pmax(val, eps)) / K
        age[L <= l0] <- 0
        pmin(age, tmax)
      }
    },
    "gompertz" = {
      denom_ln <- log(Gom_l0 / Gom_Linf)  # < 0
      function(L) {
        L_eff     <- pmin(pmax(L, Gom_l0), Gom_Linf - eps)
        log_ratio <- log(L_eff / Gom_Linf) / denom_ln
        age       <- -log(pmax(log_ratio, eps)) / Gom_K
        age[L <= Gom_l0] <- 0
        pmin(age, tmax)
      }
    },
    "schnute" = {
      denom <- (1 - exp(-g1 * tmax))
      function(L) {
        L_eff <- pmin(pmax(L, 0), l2 - eps)
        inside <- 1 - ((L_eff^g2) / (l2^g2)) * denom
        inside <- pmin(pmax(inside, eps), 1)
        age <- -log(inside) / g1
        age[L <= 0] <- 0
        pmin(age, tmax)
      }
    }
  )

  # survivorship normalized at model start
  S0 <- exp(-M * age_from_length(anchor_L))
  S_norm <- function(L) {
    age <- age_from_length(L)
    val <- exp(-M * age) / S0
    ifelse(is.finite(val), val, 0)
  }

  # length at tmax
  length_at_t_max <- switch(
    growth_model,
    "vb"       = Linf * (1 - (1 - l0 / Linf) * exp(-K * tmax)),
    "gompertz" = Gom_Linf * ((Gom_l0 / Gom_Linf) ^ (exp(-Gom_K * tmax))),
    "schnute"  = l2
  )

  if (old_Lc > length_at_t_max || new_Lc > length_at_t_max) {
    warning("Lc exceeds l(tmax); check that growth-parameter units match slot units.", call. = FALSE)
  }

  # proportions & adjustment
  prop_for_slot <- function(MinLS, MaxLS, Lc) {
    lower_den <- max(Lc, Lmin)
    upper_den <- max(length_at_t_max, lower_den + eps)
    lower_num <- max(MinLS, Lc)
    upper_num <- min(MaxLS, length_at_t_max)
    if (upper_num <= lower_num) return(0)

    total_exploitable <- try(stats::integrate(S_norm, lower = lower_den, upper = upper_den, rel.tol = 1e-6)$value, silent = TRUE)
    in_slot           <- try(stats::integrate(S_norm, lower = lower_num, upper = upper_num, rel.tol = 1e-6)$value, silent = TRUE)
    if (inherits(total_exploitable, "try-error") || inherits(in_slot, "try-error")) {
      stop("Numerical integration failed; check parameter values and ranges.", call. = FALSE)
    }
    in_slot / total_exploitable
  }

  prop_old <- prop_for_slot(old_minLS, old_maxLS, old_Lc)
  prop_new <- prop_for_slot(new_minLS, new_maxLS, new_Lc)

  adjust_factor  <- if (prop_old > .Machine$double.eps) prop_new / prop_old else NA_real_
  adjusted_catch <- if (!is.null(catch) && is.finite(adjust_factor)) catch * adjust_factor else NULL

  if (!isTRUE(plot)) return(adjust_factor)

  # plotting (returned, not printed)
  gg <- NULL
  if (requireNamespace("ggplot2", quietly = TRUE)) {
    .use_linewidth <- function() {
      requireNamespace("ggplot2", quietly = TRUE) &&
        utils::packageVersion("ggplot2") >= "3.4.0"
    }

    by_step <- max(0.1, (length_at_t_max - anchor_L) / 1000)
    length_values <- seq(anchor_L, length_at_t_max, by = by_step)
    survivorship_values <- exp(-M * age_from_length(length_values))
    df <- data.frame(Length = length_values, Survivorship = survivorship_values / S0)

    min_Lc_both <- min(old_Lc, new_Lc)
    df_below    <- subset(df, Length >= anchor_L & Length < min_Lc_both)
    df_old_slot <- subset(df, Length >= max(old_minLS, old_Lc) & Length <= min(old_maxLS, length_at_t_max))
    df_new_slot <- subset(df, Length >= max(new_minLS, new_Lc) & Length <= min(new_maxLS, length_at_t_max))

    x_lines <- c(old_minLS, old_maxLS, new_minLS, new_maxLS)
    y_lines <- ifelse(
      x_lines < anchor_L | x_lines > length_at_t_max,
      0,
      stats::approx(df$Length, df$Survivorship, xout = x_lines, ties = "ordered")$y
    )
    segs <- data.frame(x = x_lines, y0 = 0, y1 = y_lines)

    arrow_df <- data.frame(
      x    = c(old_minLS, old_maxLS),
      xend = c(new_minLS, new_maxLS),
      y    = 0.9, yend = 0.9
    )
    arrow_df <- subset(arrow_df, abs(x - xend) > .Machine$double.eps)

    title_label <- switch(growth_model, "vb" = "vB", "gompertz" = "Gompertz", "schnute" = "Schnute")
    x_lab <- if (is.null(length_units)) "Length" else paste0("Length (", length_units, ")")

    gg <- ggplot2::ggplot(df, ggplot2::aes(x = Length, y = Survivorship)) +
      ggplot2::geom_ribbon(
        data = df_below,
        ggplot2::aes(ymin = 0, ymax = Survivorship, fill = "<Lc"), alpha = 0.25
      ) +
      ggplot2::geom_ribbon(
        data = df_old_slot,
        ggplot2::aes(ymin = 0, ymax = Survivorship, fill = "Old in-slot"), alpha = 0.8
      ) +
      ggplot2::geom_ribbon(
        data = df_new_slot,
        ggplot2::aes(ymin = 0, ymax = Survivorship, fill = "New in-slot"), alpha = 0.8
      ) +
      ggplot2::geom_segment(
        data = segs,
        ggplot2::aes(x = x, xend = x, y = y0, yend = y1),
        inherit.aes = FALSE,
        size = if (.use_linewidth()) NULL else 0.2,
        linewidth = if (.use_linewidth()) 0.2 else NULL,
        linetype = "solid", colour = "black", show.legend = FALSE
      ) +
      ggplot2::geom_segment(
        data = arrow_df,
        ggplot2::aes(x = x, xend = xend, y = y, yend = yend),
        inherit.aes = FALSE,
        arrow = grid::arrow(length = grid::unit(3, "mm"), angle = 20, type = "closed"),
        size = if (.use_linewidth()) NULL else 0.6,
        linewidth = if (.use_linewidth()) 0.6 else NULL,
        colour = "black", show.legend = FALSE
      ) +
      ggplot2::geom_line(
        size = if (.use_linewidth()) NULL else 1,
        linewidth = if (.use_linewidth()) 1 else NULL,
        col = "darkgreen"
      ) +
      ggplot2::labs(
        title = paste0("Growth model: ", title_label),
        subtitle = sprintf(
          "Old P = %.2f   |   New P = %.2f   |   Adjustment = %s*Cy",
          prop_old, prop_new,
          ifelse(is.finite(adjust_factor), sprintf("%.3f", adjust_factor), "NA")
        ),
        x = x_lab, y = "Normalized survivorship", fill = ""
      ) +
      ggplot2::scale_fill_manual(
        values = c("<Lc" = "grey70", "Old in-slot" = "darkred", "New in-slot" = "lightyellow"),
        breaks = c("<Lc", "Old in-slot", "New in-slot"),
        labels = c("< Lc", "Old in-slot", "New in-slot")
      ) +
      ggplot2::coord_cartesian(
        ylim = c(0, 1),
        xlim = c(0, max(length_at_t_max, old_maxLS, new_maxLS, old_Lc, new_Lc)),
        expand = FALSE
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(hjust = 0.5),
        plot.margin = grid::unit(c(0.25, 0.25, 0.25, 0.25), "cm"),
        text = ggplot2::element_text(size = 15),
        plot.subtitle = ggplot2::element_text(size = 10, hjust = 0.5),
        legend.position = "top",
        legend.direction = "horizontal",
        legend.background = ggplot2::element_blank(),
        legend.key = ggplot2::element_blank()
      )
  } else {
    warning("Package 'ggplot2' not installed; returning values without a plot.", call. = FALSE)
  }

  list(
    prop_old = prop_old,
    prop_new = prop_new,
    adjust_factor = adjust_factor,
    adjusted_catch = adjusted_catch,
    plot = gg
  )
}
