% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/source_varsel.R
\name{sblm_modelsel}
\alias{sblm_modelsel}
\title{Model selection for semiparametric Bayesian linear regression}
\usage{
sblm_modelsel(
  y,
  X,
  prob_inclusion = 0.5,
  psi = length(y),
  fixedX = (length(y) >= 500),
  init_screen = NULL,
  nsave = 1000,
  override = FALSE,
  ngrid = 100,
  verbose = TRUE
)
}
\arguments{
\item{y}{\code{n x 1} response vector}

\item{X}{\code{n x p} matrix of predictors (no intercept)}

\item{prob_inclusion}{prior inclusion probability for each variable}

\item{psi}{prior variance (g-prior)}

\item{fixedX}{logical; if TRUE, treat the design as fixed (non-random) when sampling
the transformation; otherwise treat covariates as random with an unknown distribution}

\item{init_screen}{for the initial approximation, number of covariates
to pre-screen (necessary when \code{p > n}); if NULL, use \code{n/log(n)}}

\item{nsave}{number of Monte Carlo simulations}

\item{override}{logical; if TRUE, the user may override the default
cancellation of the function call when \code{p > 15}}

\item{ngrid}{number of grid points for inverse approximations}

\item{verbose}{logical; if TRUE, print time remaining}
}
\value{
a list with the following elements:
\itemize{
\item \code{post_probs} the posterior probabilities for each model
\item \code{all_models}: \code{2^p x p} matrix where each row corresponds to a model
from \code{post_probs} and each column indicates inclusion (TRUE) or exclusion (FALSE) for that variable
\item \code{model}: the model fit (here, \code{sblm_modelsel})
}
as well as the arguments passed in.
}
\description{
Compute model probabilities for semiparametric Bayesian linear regression with
1) an unknown (nonparametric) transformation and 2) a sparsity prior on
the regression coefficients. The model probabilities are computed
using direct Monte Carlo (not MCMC) sampling.
}
\details{
This function provides fully Bayesian model selection for a
transformed linear model with sparse g-priors on the regression coefficients.
The transformation is modeled as unknown and learned jointly
with the model probabilities. This model applies for real-valued data, positive data, and
compactly-supported data (the support is automatically deduced from the observed \code{y} values).
By default, \code{fixedX} is set to \code{FALSE} for smaller datasets (\code{n < 500})
and \code{TRUE} for larger datasets.

Enumeration of all possible subsets is computationally demanding and
should be reserved only for small \code{p}. The function will exit for
\code{p > 15} unless \code{override = TRUE}.

This function exclusively computes model probabilities and does not
provide other coefficient inference or prediction. These additions
would be straightforward, but are omitted to save on computing time.
For prediction, inference, and computation
with moderate to large \code{p}, use \code{\link{sblm_ssvs}}.
}
\note{
The location (intercept) and scale (\code{sigma_epsilon}) are
not identified, so any intercept in \code{X} will be removed.
The model-fitting *does* include an internal location-scale
adjustment, but the model probabilities only refer to the
non-intercept variables in \code{X}.
}
\examples{
\donttest{
# Simulate data from a transformed (sparse) linear model:
dat = simulate_tlm(n = 100, p = 5, g_type = 'beta')
y = dat$y; X = dat$X

hist(y, breaks = 25) # marginal distribution

# Package for conveniently computing all subsets:
library(plyr)

# Fit the semiparametric Bayesian linear model with model selection:
fit = sblm_modelsel(y = y, X = X)
names(fit) # what is returned

# Summarize the probabilities of each model (by size):
plot(rowSums(fit$all_models), fit$post_probs,
     xlab = 'Model sizes', ylab = 'p(model | data)',
     main = 'Posterior model probabilities', pch = 2, ylim = c(0,1))

# Highest probability model:
hpm = which.max(fit$post_probs)
fit$post_probs[hpm] # probability
which(fit$all_models[hpm,]) # which variables
which(dat$beta_true != 0) # ground truth
}

}
