#' Dunn's index
#'
#'
#' This function calculates the Dunn's index as is defined in Bezdek and Pal (1995) without imposing that the use of the euclidean distance. This function allows calculating the Dunn's index using different distances.
#'
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param FHW_output List. List with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'
#' @return Returns a number, the value of the Dunn's index.
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'          matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'          matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'FHW_output=Hartigan_and_Wong(data,
#'                             Euclideandistance,
#'                             k,
#'                             centers_function_mean,
#'                             init_centers_random,
#'                             seed=seed,
#'                             10)
#'
#'DunnIndex(data, FHW_output, Euclideandistance)
#'
#'@references  Bezdek, J. C., & Pal, N. R. (1995, November). Cluster validation with generalized Dunn's indices. In Proceedings 1995 second New Zealand international two-stream conference on artificial neural networks and expert systems (pp. 190-193). IEEE.
#'
#' @export



DunnIndex<-function(data, FHW_output, distance){

  Q_clusters=c(1:length(FHW_output$grouping))
  mat=utils::combn(Q_clusters, 2)

  dist_mat=rep(0, dim(mat)[2])
  for (i in 1:(dim(mat)[2])){
    dist_mat[i]=DistanceBetweenGroups(Q_clusters[mat[1,i]] , Q_clusters[mat[2,i]], FHW_output, distance, data)
  }

  num=min(dist_mat)

  max_dist_group=rep(0,length(FHW_output$grouping))
  for (i in 1:length(FHW_output$grouping)){
    max_dist_group[i]=DistanceSameGroup(i,  FHW_output, data,  distance)
  }

  dem=max(max_dist_group)

  Dunn=num/dem


  return(Dunn)
}
