% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RDML.R
\docType{class}
\name{RDML}
\alias{RDML}
\title{R6 class \code{RDML} -- contains methods to read and overview fluorescence 
data from RDML v1.1 and v1.2 format files}
\format{
An \code{\link[R6]{R6Class}} generator object.
}
\description{
This class is a container for RDML format data (Lefever et al. 
2009). The data may be further transformed to the appropriate format of the 
\code{qpcR} (Ritz et al. 2008, Spiess et al. 2008) and \code{chipPCR} 
(Roediger et al. 2015) packages (see \link{RDML.new} for import details). 
Real-time PCR Data Markup Language (RDML) is the recommended file format 
element in the Minimum Information for Publication of Quantitative Real-Time 
PCR Experiments (MIQE) guidelines (Bustin et al. 2009). The inner structure of 
imported data faithfully reflects the structure of RDML file v1.2. All data with 
the exception for fluorescence values can be represented as \code{data.frame} by 
method \code{AsTable}. Such possibility of data representation streamlines 
sample filtering (by targets, types, etc.) and serves as request for \code{GetFData}
method, which extracts fluorescence data for specified samples.
}
\section{Fields}{
 Type, structure of data and description of fields can be 
  viewed at RDML v1.2 file description. Names of fields are first level of 
  XML tree.
}

\section{Methods}{
 \describe{
\item{new}{creates a new instance of \code{RDML} class object (see \link{RDML.new})} 
\item{AsTable}{represent RDML data as \code{data.frame} (see \link{RDML.AsTable})}
\item{GetFData}{gets fluorescence data (see \link{RDML.GetFData})}
\item{SetFData}{sets fluorescence data (see \link{RDML.SetFData})}
\item{Merge}{merges two \code{RDML} to one (see \link{MergeRDMLs})}
\item{AsDendrogram}{represents structure of \code{RDML} object as dendrogram(see 
  \link{RDML.AsDendrogram})}
 }
}

\examples{
## EXAMPLE 1:
## internal dataset lc96_bACTXY.rdml (in 'data' directory)
## generated by Roche LightCycler 96. Contains qPCR data
## with four targets and two types.
## Import with default settings.
PATH <- path.package("RDML")
filename <- paste(PATH, "/extdata/", "lc96_bACTXY.rdml", sep ="")
lc96 <- RDML$new(filename)

tab <- lc96$AsTable(name.pattern = paste(sample[[react$sample$id]]$description,
                                         react$id$id), 
                    quantity = sample[[react$sample$id]]$quantity$value)
## Show dyes names
unique(tab$target.dyeId)
## Show types of the samples for dye 'FAM'
library(dplyr)
unique(filter(tab, target.dyeId == "FAM")$sample.type)

## Show template quantities for dye 'FAM' type 'std' 
COPIES <- filter(tab, target.dyeId == "FAM", sample.type == "std")$quantity
## Define calibration curves (type of the samples - 'std').
## No replicates.
library(qpcR)
CAL <- modlist(lc96$GetFData(filter(tab,
                                    target.dyeId == "FAM", 
                                    sample.type == "std")),
               baseline="lin", basecyc=8:15)
## Define samples to predict (first two samples with the type - 'unkn').
PRED <- modlist(lc96$GetFData(filter(tab, 
                                    target.dyeId == "FAM", 
                                    sample.type == "unkn")),
               baseline="lin", basecyc=8:15)
## Conduct quantification.
calib(refcurve = CAL, predcurve = PRED, thresh = "cpD2",
      dil = COPIES)

## EXAMPLE 2:
\donttest{
## internal dataset lc96_bACTXY.rdml (in 'data' directory)
## generated by Roche LightCycler 96. Contains qPCR data
## with four targets and two types.
## Import with default settings.
library(chipPCR)                        
PATH <- path.package("RDML")
filename <- paste(PATH, "/extdata/", "lc96_bACTXY.rdml", sep ="")
lc96 <- RDML$new(filename)

tab <- lc96$AsTable(name.pattern = paste(sample[[react$sample$id]]$description,
                                         react$id$id), 
                    quantity = sample[[react$sample$id]]$quantity$value)
## Show targets names
unique(tab$target)
## Fetch cycle dependent fluorescence for HEX chanel
tmp <- lc96$GetFData(filter(tab, target == "bACT", sample.type == "std"))
## Fetch vector of dillutions 
dilution <- filter(tab, target.dyeId == "FAM", sample.type == "std")$quantity

## Use plotCurves function from the chipPCR package to 
## get an overview of the amplification curves
tmp <- as.data.frame(tmp)
plotCurves(tmp[,1], tmp[,-1])
oldpar <- par()
par(mfrow = c(1,1))
## Use inder function from the chipPCR package to 
## calculate the Cq (second derivative maximum, SDM)
SDMout <- sapply(2L:ncol(tmp), function(i) {
  SDM <- summary(inder(tmp[, 1], tmp[, i]), print = FALSE)[2]
})

## Use the effcalc function from the chipPCR package and 
## plot the results for the calculation of the amplification
## efficiency analysis.
plot(effcalc(dilution, SDMout), CI = TRUE)
par(oldpar)
}
## EXAMPLE 3:
## internal dataset BioRad_qPCR_melt.rdml (in 'data' directory)
## generated by Bio-Rad CFX96. Contains qPCR and melting data.
## Import with custom name pattern.
PATH <- path.package("RDML")
filename <- paste(PATH, "/extdata/", "BioRad_qPCR_melt.rdml", sep ="")
cfx96 <- RDML$new(filename)
## Use plotCurves function from the chipPCR package to 
## get an overview of the amplification curves
library(chipPCR)
## Extract all qPCR data 
tab <- cfx96$AsTable()
cfx96.qPCR <- as.data.frame(cfx96$GetFData(tab))
plotCurves(cfx96.qPCR[,1], cfx96.qPCR[,-1], type = "l")

## Extract all melting data 
cfx96.melt <- cfx96$GetFData(tab, dp.type = "mdp")
## Show some generated names for samples.
colnames(cfx96.melt)[2L:5]
## Select columns that contain
## samples with dye 'EvaGreen' and have type 'pos'
## using filtering by names.
cols <- cfx96$GetFData(filter(tab, grepl("pos_EvaGreen$", fdata.name)),
                       dp.type = "mdp")
## Conduct melting curve analysis.
library(qpcR)
invisible(meltcurve(cols, fluos = 2,
          temps = 1))
}
\references{
RDML format http://www.rdml.org/ \code{R6} package 
  http://cran.r-project.org/web/packages/R6/index.html
  
  \code{qpcR} package http://cran.r-project.org/web/packages/qpcR/index.html
  
  \code{chipPCR} package: 
  http://cran.r-project.org/web/packages/chipPCR/index.html
  
  Roediger S, Burdukiewicz M and Schierack P (2015). chipPCR: an R Package 
  to Pre-Process Raw Data of Amplification Curves. \emph{Bioinformatics} first 
  published online April 24, 2015 doi:10.1093/bioinformatics/btv205
  
  Ritz, C., Spiess, A.-N., 2008. qpcR: an R package for sigmoidal model 
  selection in quantitative real-time polymerase chain reaction analysis. 
  \emph{Bioinformatics} 24, 1549--1551. 
  doi:10.1093/bioinformatics/btn227
  
  Spiess, A.-N., Feig, C., Ritz, C., 2008. Highly accurate sigmoidal fitting 
  of real-time PCR data by introducing a parameter for asymmetry. \emph{BMC 
  Bioinformatics} 9, 221. doi:10.1186/1471-2105-9-221
  
  Bustin, S.A., Benes, V., Garson, J.A., Hellemans, J., Huggett, J., Kubista,
  M., Mueller, R., Nolan, T., Pfaffl, M.W., Shipley, G.L., Vandesompele, J., 
  Wittwer, C.T., 2009. The MIQE guidelines: minimum information for 
  publication of quantitative real-time PCR experiments. \emph{Clin. Chem.} 
  55, 611--622.  doi:10.1373/clinchem.2008.112797
  
  Lefever, S., Hellemans, J., Pattyn, F., Przybylski, D.R., Taylor, C., 
  Geurts, R., Untergasser, A., Vandesompele, J., RDML consortium, 2009. RDML:
  structured language and reporting guidelines for real-time quantitative PCR
  data.  \emph{Nucleic Acids Res.} 37, 2065--2069. doi:10.1093/nar/gkp056
}
\author{
Konstantin A. Blagodatskikh <k.blag@yandex.ru>, Stefan Roediger 
  <stefan.roediger@b-tu.de>, Michal Burdukiewicz 
  <michalburdukiewicz@gmail.com>
}
\keyword{Bio--Rad}
\keyword{CFX96}
\keyword{IO}
\keyword{LightCycler}
\keyword{RDML}
\keyword{StepOne}
\keyword{file}
\keyword{qPCR}
\section{Super class}{
\code{\link[RDML:rdmlBaseType]{RDML::rdmlBaseType}} -> \code{RDML}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-RDML-AsDendrogram}{\code{RDML$AsDendrogram()}}
\item \href{#method-RDML-AsTable}{\code{RDML$AsTable()}}
\item \href{#method-RDML-GetFData}{\code{RDML$GetFData()}}
\item \href{#method-RDML-new}{\code{RDML$new()}}
\item \href{#method-RDML-AsXML}{\code{RDML$AsXML()}}
\item \href{#method-RDML-SetFData}{\code{RDML$SetFData()}}
\item \href{#method-RDML-clone}{\code{RDML$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="RDML" data-topic="rdmlBaseType" data-id=".asXMLnodes"><a href='../../RDML/html/rdmlBaseType.html#method-rdmlBaseType-.asXMLnodes'><code>RDML::rdmlBaseType$.asXMLnodes()</code></a></span></li>
<li><span class="pkg-link" data-pkg="RDML" data-topic="rdmlBaseType" data-id="copy"><a href='../../RDML/html/rdmlBaseType.html#method-rdmlBaseType-copy'><code>RDML::rdmlBaseType$copy()</code></a></span></li>
<li><span class="pkg-link" data-pkg="RDML" data-topic="rdmlBaseType" data-id="print"><a href='../../RDML/html/rdmlBaseType.html#method-rdmlBaseType-print'><code>RDML::rdmlBaseType$print()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-AsDendrogram"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-AsDendrogram}{}}}
\subsection{Method \code{AsDendrogram()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$AsDendrogram(plot.dendrogram = TRUE)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-AsTable"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-AsTable}{}}}
\subsection{Method \code{AsTable()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$AsTable(
  .default = list(exp.id = experiment$id$id, run.id = run$id$id, react.id = react$id$id,
    position = react$position, sample = react$sample$id, target = data$tar$id,
    target.dyeId = target[[data$tar$id]]$dyeId$id, sample.type =
    sample[[react$sample$id]]$type$value, adp = !is.null(data$adp), mdp =
    !is.null(data$mdp)),
  name.pattern = paste(react$position, react$sample$id,
    private$.sample[[react$sample$id]]$type$value, data$tar$id, sep = "_"),
  add.columns = list(),
  treat.null.as.na = FALSE,
  ...
)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-GetFData"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-GetFData}{}}}
\subsection{Method \code{GetFData()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$GetFData(request, dp.type = "adp", long.table = FALSE)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-new"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-new}{}}}
\subsection{Method \code{new()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$new(
  filename,
  show.progress = TRUE,
  conditions.sep = NULL,
  format = "auto"
)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-AsXML"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-AsXML}{}}}
\subsection{Method \code{AsXML()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$AsXML(file.name)}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-SetFData"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-SetFData}{}}}
\subsection{Method \code{SetFData()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$SetFData(fdata, description, fdata.type = "adp")}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RDML-clone"></a>}}
\if{latex}{\out{\hypertarget{method-RDML-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RDML$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
