% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clustering_functions.R
\name{pkbc}
\alias{pkbc}
\alias{pkbc,ANY-method}
\alias{show,pkbc-method}
\title{Poisson kernel-based clustering on the sphere}
\usage{
pkbc(
  dat,
  nClust,
  maxIter = 300,
  stoppingRule = "loglik",
  initMethod = "sampleData",
  numInit = 10
)

\S4method{pkbc}{ANY}(
  dat,
  nClust,
  maxIter = 300,
  stoppingRule = "loglik",
  initMethod = "sampleData",
  numInit = 10
)

\S4method{show}{pkbc}(object)
}
\arguments{
\item{dat}{\eqn{(n \times d)}-data matrix or data.frame of data points on the
sphere to be clustered. The observations in \code{dat} are
normalized by dividing with the length of the vector to ensure
that they lie on the \eqn{d}-dimensional sphere. Note that
\eqn{d > 1}.}

\item{nClust}{Number of clusters. It can be a single value or a numeric
vector.}

\item{maxIter}{The maximum number of iterations before a run is terminated.}

\item{stoppingRule}{String describing the stopping rule to be used within
each run. Currently must be either \code{'max'},
\code{'membership'}, or \code{'loglik'}.}

\item{initMethod}{String describing the initialization method to be used.
Currently must be \code{'sampleData'}.}

\item{numInit}{Number of initialization.}

\item{object}{Object of class \code{pkbc}}
}
\value{
An S4 object of class \code{pkbc} containing the results of the
clustering procedure based on Poisson kernel-based distributions. The object
contains the following slots:

\code{res_k}: List of results of the Poisson kernel-based clustering
algorithm for each value of number of clusters specified in \code{nClust}.
Each object in the list contains:
\itemize{
\item \code{postProbs} Posterior probabilities of each observation for
the indicated clusters.
\item \code{LogLik} Maximum value of log-likelihood function
\item \code{wcss} Values of within-cluster sum of squares computed with
Euclidean distance and cosine similarity, respectively.
\item \code{params} List of estimated parameters of the mixture model
\itemize{
\item \code{mu} estimated centroids
\item \code{rho} estimated concentration parameters rho
\item \code{alpha} estimated mixing proportions
}
\item \code{finalMemb} Vector of final memberships
\item \code{runInfo} List of information of the EM algorithm iterations
\itemize{
\item \code{lokLikVec} vector of log-likelihood values
\item \code{numIterPerRun} number of E-M iterations per run
}
}
\code{input}: List of input information.
}
\description{
The function \code{pkbc()} performs the Poisson kernel-based clustering
algorithm on the sphere proposed by Golzy and Markatou (2020).
The proposed algorithm is based on a mixture, with \eqn{M} components, of
Poisson kernel-based densities on the hypersphere \eqn{\mathcal{S}^{d-1}}
given by
\deqn{f(x|\Theta) = \sum_{j=1}^M \alpha_j f_j(x|\rho_j, \mu_j)}
where \eqn{\alpha_j}'s are the mixing proportions and \eqn{f_j(x|\rho_j, 
\mu_j)}'s denote the probability density function of a \eqn{d}-variate
Poisson kernel-based density given as
\deqn{f(\mathbf{x}|\rho, \mathbf{\mu}) = \frac{1-\rho^2}{\omega_d 
||\mathbf{x} - \rho \mathbf{\mu}||^d}.}
The parameters \eqn{\alpha_j, \mu_j, \rho_j} are estimated through a
iterative reweighted EM algorithm. \cr
The proposed clustering algorithm exhibits excellent results when
(1) the clusters are not well separated; (2) the data points
are fairly well concentrated around the vectors \eqn{\mu_j} of each cluster;
(3) the percentage of noise in the data increases.
}
\details{
We set all concentration parameters equal to 0.5 and all mixing proportions
to be equal. \cr
The initialization method \code{'sampleData'} indicates that observation
points are randomly chosen as initializers of the centroids \eqn{\mu_j}.
This random starts strategy has a chance of not obtaining initial
representatives from the underlying clusters, then the clustering is
performed \code{numInit} times and the random start with the highest
likelihood is chosen as the final estimate of the parameters.

The possible \code{stoppingRule} for each iteration are: \cr
\itemize{
\item \code{'loglik'} run the algorithm until the change in log-likelihood from
one iteration to the next is less than a given threshold (1e-7) \cr
\item \code{'membership'} run the algorithm until the membership is unchanged for
all points from one iteration to the next \cr
\item \code{'max'} reach a maximum number of iterations \code{maxIter}
}

The obtained estimates are used for assigning final memberships, identifying
the \code{nClust} clusters, according to the following rule
\deqn{P(x_i, \Theta) = \arg\max_{j \in \{1, \ldots, k\}} \{ \frac{\alpha_j 
f_j(x_i|\mu_j, \rho_j)}{f(x_i, \Theta)}\}.}
The number of clusters \code{nClust} must be provided as input to the
clustering algorithm.
}
\note{
The clustering algorithm is tailored for data points on the sphere
\eqn{\mathcal{S}^{d-1}}, but it can also be performed on spherically
transformed observations, i.e. data points on the Euclidean space
\eqn{\mathbb{R}^d} that are normalized such that they lie on the
corresponding \eqn{(d-1)}-dimensional sphere \eqn{\mathcal{S}^{d-1}}.
}
\examples{
# We generate three samples of 100 observations from 3-dimensional
# Poisson kernel-based densities with rho=0.8 and different mean directions
size <- 100
groups <- c(rep(1, size), rep(2, size), rep(3, size))
rho <- 0.8
set.seed(081423)
data1 <- rpkb(size, c(1, 0, 0), rho)
data2 <- rpkb(size, c(0, 1, 0), rho)
data3 <- rpkb(size, c(0, 0, 1), rho)
dat <- rbind(data1, data2, data3)

# Perform the clustering algorithm with number of clusters k=3.
pkbd <- pkbc(dat = dat, nClust = 3)
show(pkbd)

}
\references{
Golzy, M. and Markatou, M. (2020) Poisson Kernel-Based Clustering on the
Sphere: Convergence Properties, Identifiability, and a Method of Sampling,
Journal of Computational and Graphical Statistics, 29:4, 758-770,
DOI: 10.1080/10618600.2020.1740713.
}
\seealso{
\code{\link[=dpkb]{dpkb()}} and \code{\link[=rpkb]{rpkb()}} for more information on the Poisson
kernel-based distribution. \cr
\linkS4class{pkbc} for the class definition.
}
