% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/performance_analytics.R
\name{calculate_daily_values}
\alias{calculate_daily_values}
\title{Daily equity curve from positions and daily prices}
\usage{
calculate_daily_values(
  positions,
  daily_prices,
  strategy_dates,
  initial_capital,
  cash_values
)
}
\arguments{
\item{positions}{A \code{data.frame}/\code{data.table} of portfolio positions with
columns \code{Date} + symbols. Values should be the backtest's \strong{position
inventory} per symbol at each rebalance date (typically shares or notional
units consistent with your backtest's accounting).}

\item{daily_prices}{A \code{data.frame}/\code{data.table} of \strong{daily} prices with
columns \code{Date} + the same symbol set present in \code{positions} (at least the
intersection).}

\item{strategy_dates}{A \code{Date} vector of the backtest's decision/rebalance
calendar (e.g., \code{backtest_result$dates}).}

\item{initial_capital}{Numeric scalar. Starting cash used for days \strong{before}
the first position exists (typically \code{backtest_result$initial_capital}).}

\item{cash_values}{Optional numeric vector of cash balances at the strategy
dates (e.g., \code{backtest_result$cash}). If \code{NULL}, leading days are treated
as all-cash (= \code{initial_capital}) and post-rebalance cash defaults to 0.}
}
\value{
A list with components:
\itemize{
\item \code{dates} Daily dates within the strategy span.
\item \code{portfolio_values} Daily total portfolio value (positions + cash).
\item \code{positions_value} Daily mark-to-market of positions only.
\item \code{cash} Daily carried cash series.
}
}
\description{
Carries portfolio positions (from a weekly or lower-frequency backtest)
forward to daily dates, multiplies by daily prices, and combines with cash
to produce a daily portfolio value series for monitoring and analytics.
}
\examples{
\donttest{
  # Minimal end-to-end example using bundled data and a simple weekly backtest
  library(PortfolioTesteR)
  data(sample_prices_weekly); data(sample_prices_daily)

  # Build a tiny strategy: momentum -> top-3 -> equal weights
  mom <- calc_momentum(sample_prices_weekly, lookback = 12)
  sel <- filter_top_n(mom, n = 3)
  W   <- weight_equally(sel)
  bt  <- run_backtest(sample_prices_weekly, W, name = "Demo")

  # Compute daily monitoring values from positions + cash
  vals <- calculate_daily_values(
    positions       = bt$positions,
    daily_prices    = sample_prices_daily,
    strategy_dates  = bt$dates,
    initial_capital = bt$initial_capital,
    cash_values     = bt$cash
  )

  # Quick sanity checks
  head(vals$dates)
  head(vals$portfolio_values)
}

}
