#ifndef MinTriadicClosure_H
#define MinTriadicClosure_H

#include <Rcpp.h>
#include <lolog.h>
#include <cmath>

namespace lologext {

using namespace Rcpp;
using namespace std;

template<class Engine>
class MinTriadicClosure : public lolog::BaseStat<Engine> {
public:
    int triadDegree;
    double smoothing_k;
    vector<int> triadDegs;

    // Default constructor
    MinTriadicClosure()
    : triadDegree(0), smoothing_k(1.0) {}

    // R-facing constructor
    MinTriadicClosure(List params) {
        try {
            triadDegree = as<int>(params(0));
            smoothing_k = as<double>(params(1));
        } catch (...) {
            Rf_error("minTriadicClosure error: please specify "
                     "triad degree threshold and smoothing (k).");
        }
    }

    virtual string name() { return "minTriadicClosure"; }
    vector<string> statNames() { return { "minTriadicClosure" }; }

    // Clamped sigmoid to prevent flat gradients and numerical issues
    double sigmoid(double x) const {
        double s = 1.0 / (1.0 + std::exp(-smoothing_k * (x - triadDegree)));
        const double eps = 1e-6;
        return std::max(eps, std::min(1.0 - eps, s));
    }

    // Calculate: compute both smoothed and integer count for reporting
    virtual void calculate(const lolog::BinaryNet<Engine>& net) {
        int n = net.size();
        triadDegs.assign(n, 0);
        double smoothCount = 0.0;
        int hardCount = 0;  // For internal use, no printing now

        for (int i = 0; i < n; ++i) {
            if (net.degree(i) < 2) continue;

            vector<int> neighbors;
            for (int j = 0; j < n; ++j)
                if (net.hasEdge(i, j))
                    neighbors.push_back(j);

            for (size_t a = 0; a + 1 < neighbors.size(); ++a) {
                for (size_t b = a + 1; b < neighbors.size(); ++b) {
                    if (net.hasEdge(neighbors[a], neighbors[b])) {
                        triadDegs[i]++;
                    }
                }
            }

            smoothCount += sigmoid(double(triadDegs[i]));
            if (triadDegs[i] >= triadDegree)
                hardCount++;  // Used internally but not printed
        }

        this->stats     = { double(hardCount) };   // What gets printed in summaries
        this->lastStats = { smoothCount };         // What gets used internally

        if (this->thetas.size() != 1)
            this->thetas = { 0.0 };

    }

    // When an edge toggles, update smoothed statistic for inference
    virtual void dyadUpdate(const lolog::BinaryNet<Engine>& net,
                            const int& from,
                            const int& to,
                            const vector<int>& /*order*/,
                            int /*actorIndex*/) {
        updateNode(net, from, to);
        updateNode(net, to, from);
    }

    // Adjust by the difference in smoothed contribution
    void updateNode(const lolog::BinaryNet<Engine>& net, int u, int v) {
        double oldVal = sigmoid(double(triadDegs[u]));

        vector<int> neighbors;
        for (int i = 0; i < net.size(); ++i)
            if (i != v && net.hasEdge(u, i))
                neighbors.push_back(i);

        int delta = 0;
        for (int w : neighbors)
            if (net.hasEdge(w, v))
                delta++;

        bool adding = !net.hasEdge(u, v);
        triadDegs[u] += adding ? delta : -delta;

        double newVal = sigmoid(double(triadDegs[u]));
        this->stats[0] += (newVal - oldVal);
    }

    bool isOrderIndependent() { return true; }
    bool isDyadIndependent() { return false; }

    virtual lolog::BaseStat<Engine>* clone() const {
        return new MinTriadicClosure<Engine>(*this);
    }
};

typedef lolog::Stat<lolog::Undirected, MinTriadicClosure<lolog::Undirected>>
    UndirectedMinTriadicClosure;

} // namespace lologext

void registerMinTriadicClosure();

#endif
